/**********************************************************************
 * Copyright (c) 2005 Scapa Technologies Limited and others
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * Scapa Technologies Limited - Initial API and implementation
 **********************************************************************/

package org.eclipse.stp.b2j.core.jengine.internal.transport.session;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.EOFException;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.security.MessageDigest;
import java.security.SecureRandom;
import java.util.Arrays;

import org.eclipse.stp.b2j.core.jengine.internal.multiplex.ByteArrayInBuffer;
import org.eclipse.stp.b2j.core.jengine.internal.multiplex.ByteArrayOutBuffer;
import org.eclipse.stp.b2j.core.jengine.internal.multiplex.MultiplexerInputStream;
import org.eclipse.stp.b2j.core.jengine.internal.multiplex.MultiplexerOutputStream;
import org.eclipse.stp.b2j.core.jengine.internal.utils.Logger;
import org.eclipse.stp.b2j.core.jengine.internal.utils.StreamUtils;
import org.eclipse.stp.b2j.core.publicapi.extension.sessiontransport.SessionTransport;
import org.eclipse.stp.b2j.core.publicapi.transport.session.SessionAddress;

/**
 * 
 * @author amiguel
 *
 * A class to represent a single communication session
 */
public class Session {
	
	private static final boolean DEBUG = false;
	
	public static final int RESERVED_STREAM_MIN = -84808480;
	public static final int RESERVED_STREAM_MAX = -84808488;
	
	public static final short DEFAULT_STREAM_INDEX = 0;
	private static final short CONTROL_STREAM_INDEX = 1;
	
	private static final int DEFAULT_READ_BUFFER_BYTES = 2048;
	private static final int DEFAULT_WRITE_BUFFER_BYTES = 4096;

//	private static final int ACK_COUNT = 2;
//	private static final int ACK_OFFSET = 1;
//	private static final int LOGICAL_CLOCK_MAX = 10;
	private static final int ACK_COUNT = 100;	//ACK every 100 messages
	private static final int ACK_OFFSET = ACK_COUNT-1;	//Wait for an ack just before we send enough infomation to expect another one
	private static final int LOGICAL_CLOCK_MAX = 10000;
	
	Object finish_LOCK = new Object();
	Object reconnection_LOCK = new Object();
	
	boolean started = false;
	boolean finished = false;
	
	boolean multiplexed = true;
	
	boolean use_acks = false;
	
	ByteArrayInBuffer bin;
	ByteArrayOutBuffer bout;
	
	MultiplexerOutputStream outstreams;
	MultiplexerInputStream instreams;
	
	public SessionTransport transport;
	
	SessionAddress address;
	
//	ReconnectThread reconnect_thread;
	ReaderThread reader_thread;
	WriterThread writer_thread;
	
	long session_id = System.currentTimeMillis();
	long remote_session_id = -1;
	String session_ids = session_id+"-(never connected)";
	
	SecureRandom sr = new SecureRandom();
	
	Thread begin_thread = null;
	
	public Session(SessionAddress address, SessionTransport transport) throws Exception {
		this.address = address;
		this.transport = transport;

		this.use_acks = address.getRequiresLinkReconnection();
		
		multiplexed = address.getRequiresMultipleStreams();
		
		bin = new ByteArrayInBuffer();
		bout = new ByteArrayOutBuffer();
		
		if (multiplexed) {
			instreams = new MultiplexerInputStream(bin);
			outstreams = new MultiplexerOutputStream(bout);
		}
		
	}
	
	public String getTransportImplementationName() {
		return transport.getClass().getName();
	}
	
	public void beginNonBlocking() throws Exception {
		begin_thread = new SessionBeginThread(this);
		begin_thread.start();
	}
	
	public void waitUntilSessionTransportReady() {
		try {
			begin_thread.join();
		} catch (Exception e) {
		}
	}
	
	public void waitUntilSessionTransportBound() throws Exception {
		while (transport.getActualAddress() == null) {
			try {
				//TODO would be nice not to poll here
				Thread.sleep(100);
			} catch (Exception e) {
			}
			if (begin_thread != null) {
				if (begin_thread instanceof SessionBeginThread) {
					SessionBeginThread sbthread = (SessionBeginThread)begin_thread;
					if (sbthread.error != null) {
						throw sbthread.error;
					}
				}
			}
		}
	}
	
	public void begin() throws Exception {
		
		begin_thread = Thread.currentThread();
		
		if (!started) {
			//start the pipe threads (once only)
			reader_thread = new ReaderThread();
			writer_thread = new WriterThread();
			reader_thread.start();
			writer_thread.start();
			
			//connect the transport link
			restartTransport(true);

//			if (use_acks) {
//				reconnect_thread = new ReconnectThread();
//				reconnect_thread.start();
//			}
			
			started = true;
		} else {
			//connect the transport link
			restartTransport(true);
		}
		
		//Once we have connected on a specific port etc we don't change later on when we're reconnecting
		transport.setSessionAddress(transport.getActualAddress());
	}
	
	public void restartTransport(boolean startup) throws Exception {
		
		if (!finished) {
			//dont go restarting the transport if we've finished already
		
			synchronized(reconnection_LOCK) {
				
				boolean connected = false;
				Exception error = null;
		
				long abort_timeout = System.currentTimeMillis();
				
				if (address.getRequiresLinkReconnection()) {
					if (startup) {
						//time out if we haven't connected after the specified startup reconntion timeout time
						abort_timeout += address.getStartupFailureAbortTimeout();
					} else {
						//time out if we haven't connected after the specified reconnection timeout time
						abort_timeout += address.getReconnectionFailureAbortTimeout();
					}
				} else {
					//do nothing - we will timeout after one attempt which is correct behaviour
				}
				
				do {
					try {
						//try to have the transport reconnect
						transport.tryReconnect();
						connected = true;

						//TODO Could provide other services here such as compression or session level encryption
						
						//Check password
						if (address.getRequiresPassword()) {
							//get 128 random (secure) bytes
							byte[] challenge_rand = new byte[128];
							sr.nextBytes(challenge_rand);
							
							//write what our challenge random numbers are
							StreamUtils.writeBytes(transport.getOutputStream(),challenge_rand);
							
							//read the others random numbers
							byte[] response_rand = StreamUtils.readNBytes(transport.getInputStream(),1024);
							
							MessageDigest digest = MessageDigest.getInstance("MD5");
							
							//hash our password using their random numbers and send it
							//(we are authenticating ourselves here)
							digest.reset();
							digest.update(response_rand);
							digest.update(address.getPassword().getBytes("UTF8"));
							StreamUtils.writeBytes(transport.getOutputStream(),digest.digest());
							
							//read their password hashed using our random numbers
							byte[] challenge_hash = StreamUtils.readNBytes(transport.getInputStream(),1024);
							
							//check that our password hashed with our numbers = theirs
							//(we are authenticating them here)
							digest.reset();
							digest.update(challenge_rand);
							digest.update(address.getPassword().getBytes("UTF8"));
							byte[] verify_hash = digest.digest();
							
							if (!Arrays.equals(challenge_hash,verify_hash)) {
								//passwords dont match!
								throw new Exception("incorrect session password");
							}
						}
						
						//write that we are a startup
						StreamUtils.writeBoolean(transport.getOutputStream(),startup);
						transport.getOutputStream().flush();
						
						boolean rstartup = StreamUtils.readBoolean(transport.getInputStream());
						if (rstartup != startup) {
							if (startup) {
								throw new IOException("old session tried to connect");
							} else {
								IOException x = new IOException("old session must have died and been replaced");
								end(x);
								throw x;
							}
						}
						
						//socket sync checking (sanity checking that will avoid weird errors later on)
						if (startup) {
							//write our ID
							StreamUtils.writeLong(transport.getOutputStream(),session_id);
							transport.getOutputStream().flush();
		
							//get the remote session's ID
							remote_session_id = StreamUtils.readLong(transport.getInputStream());
							
							session_ids = session_id+"-"+remote_session_id;
							
						} else {
							//write our ID
							StreamUtils.writeLong(transport.getOutputStream(),session_id);
							transport.getOutputStream().flush();
							
							//this is a reconnect - check the remote session's ID
							long tmp_session_id = StreamUtils.readLong(transport.getInputStream());
							if (tmp_session_id != remote_session_id) {
								IOException failure = new IOException("Reconnected to incorrect session - sessions out of sync");
								
								failure.printStackTrace();
								
								end(failure);
								throw failure;
							}
						}
						
						//we're using acks so we need to start sending from the expected point
						if (use_acks) {
							
							if (DEBUG) System.out.println("REQUESTING THAT DATA BE SENT FROM "+reader_thread.logical_clock);
							
							//send the point that we need to get data from
							StreamUtils.writeShort(transport.getOutputStream(),reader_thread.logical_clock);
							transport.getOutputStream().flush();
							
							short requested_clock = StreamUtils.readShort(transport.getInputStream());

							if (DEBUG) System.out.println("ASKED TO SEND DATA FROM "+requested_clock);
							
							writer_thread.setOutputStream(null);
							writer_thread.setAckStream(null);
							
							if (!startup) {
								if (DEBUG) System.out.println("SETTING CLOCK IN WRITER TO "+requested_clock);
								//reconnect - need to resend from the specified clock
								writer_thread.resendFromClock(requested_clock);
								
								if (DEBUG) System.out.println("INTERRUPTING WRITER TO GET IT TO RESEND");
								writer_thread.interrupt();
							}
						}
						
						if (use_acks) {
							
							MultiplexerInputStream xin = new MultiplexerInputStream(new BufferedInputStream(transport.getInputStream()));
							MultiplexerOutputStream xout = new MultiplexerOutputStream(new BufferedOutputStream(transport.getOutputStream()));

							if (DEBUG) System.out.println("SETTING ACK STREAMS IN READER AND WRITER");
							
							reader_thread.setAckStream(xout.getOutputStream(CONTROL_STREAM_INDEX));
							writer_thread.setAckStream(xin.getInputStream(CONTROL_STREAM_INDEX));
							
							if (DEBUG) System.out.println("SETTING DATA STREAMS IN READER AND WRITER");

							//set the reader and writer streams
							reader_thread.setInputStream(xin.getInputStream(DEFAULT_STREAM_INDEX));
							writer_thread.setOutputStream(xout.getOutputStream(DEFAULT_STREAM_INDEX));
							
						} else {
							//set the reader and writer streams
//							reader_thread.setInputStream(new BufferedInputStream(transport.getInputStream()));
//							writer_thread.setOutputStream(new BufferedOutputStream(transport.getOutputStream()));
							reader_thread.setInputStream(transport.getInputStream());
							writer_thread.setOutputStream(transport.getOutputStream());
							
						}
						
						if (startup) {
							Logger.info("session connect attempt OK");
						} else {
							Logger.info("session reconnect attempt OK");
						}
					} catch (Exception e) {
						if (startup) {
							Logger.info("session connect attempt failed: "+e);
						} else {
							Logger.info("session reconnect attempt failed: "+e);
						}
						//an error occurred while connecting the transport
						connected = false;
						error = e;
						
						if (address.getRequiresLinkReconnection()) {
							//sleep for 1 second before trying again
							try {
								Thread.sleep(1000 + (long)(Math.random()*500));
							} catch (Exception x) {
							}
						}
					}
					
				//we keep trying to connect until we time out
				} while (!finished && !connected && System.currentTimeMillis() < abort_timeout);
				
				if (!connected) {
	
					//failed to connect/reconnect session
					IOException failure;
					
					if (startup) {
						if (address.getRequiresLinkReconnection()) {
							failure = new IOException("Session start failed to connect within "+address.getStartupFailureAbortTimeout()+"ms: "+error);
						} else {
							failure = new IOException("Session start failed: "+error+" (Address: "+address+")");
						}
					} else {
						if (address.getRequiresLinkReconnection()) {
							failure = new IOException("Failed to reconnect within "+address.getStartupFailureAbortTimeout()+"ms: "+error);
						} else {
							failure = new IOException("Failed to reconnect: "+error);
						}
					}
					
					end(failure);
					throw failure;
				}
				
					
			}
		}
	}

	/**
	 * Wait until we have received no data for a given time period then end this session
	 * @param ms_receive
	 */
	public void end(long ms_receive, long ms_write) {
		
		long t = System.currentTimeMillis();
		
		while (reader_thread.last_read + ms_receive > t
				|| writer_thread.last_write + ms_write > t) {
			
			long wait = 200;
			wait = Math.max(wait,(reader_thread.last_read + ms_receive)-t);
			wait = Math.max(wait,(writer_thread.last_write + ms_write)-t);
			
			Logger.info("Waiting for read thread to receive nothing for "+ms_receive+" (waiting for "+wait+")");
			try {
				Thread.sleep(wait);
			} catch (Exception e) {
			}
			
			t = System.currentTimeMillis();
		}
		
		end(null);
	}
	public void end() {
		end(null);
	}
	
	private void end(IOException e) {
		synchronized(finish_LOCK) {
			if (!finished) {
				if (e != null) {
					Logger.info("session ended: "+e,e);
//					System.out.println("SESSION ENDED:"+e);
//					e.printStackTrace();
					
					Throwable t = new Throwable("session ended by request of... (see stacktrace)");
					Logger.info("session end request: "+t,t);
//					System.out.println("SESSION ENDED BY REQUEST:"+t);
//					t.printStackTrace();
					
				}
	
				finished = true;

				if (e != null) {
					bin.setClosed(e);
					bout.setClosed(e);

					if (multiplexed && instreams != null && outstreams != null) {
						instreams.closeAll(e);
						outstreams.closeAll(e);
					}
				} else {
					bin.setClosed();
					bout.setClosed(new EOFException("session and stream has been closed"));

					if (multiplexed && instreams != null && outstreams != null) {
						instreams.closeAll();
						outstreams.closeAll(new EOFException("session and stream has been closed"));
					}
				}
				
				try {
					writer_thread.setOutputStream(null);
					writer_thread.interrupt();
				} catch (Throwable t) {
					t.printStackTrace();
				}
				try {
					reader_thread.setInputStream(null);
					reader_thread.interrupt();
				} catch (Throwable t) {
					t.printStackTrace();
				}
				transport.close();
			}
		}
	}
	
	public boolean supportsMultipleStreams() {
		return multiplexed;
	}
	
	public OutputStream getOutputStream(short n) throws IOException {
		if (multiplexed) {
			if (n >= RESERVED_STREAM_MIN && n <= RESERVED_STREAM_MAX) {
				throw new IOException ("This stream is reserved for control information");
			}

			return outstreams.getOutputStream(n);
		} else {
			if (n != DEFAULT_STREAM_INDEX) {
				throw new IOException("This session does not support multiple streams");
			}
			return bout;
		}
	}
	
	public InputStream getInputStream(short n) throws IOException {
		if (multiplexed) {
			if (n >= RESERVED_STREAM_MIN && n <= RESERVED_STREAM_MAX) {
				throw new IOException ("This stream is reserved for control information");
			}
			
			return instreams.getInputStream(n);
		} else {
			if (n != DEFAULT_STREAM_INDEX) {
				throw new IOException("This session does not support multiple streams");
			}
			return bin;
		}
	}
	
	/**
	 * Get whether this transport is alive or not
	 * @return
	 */
	public boolean isAlive() {
		return transport.isAlive();
	}
	
	/**
	 * Get the actual address this session is bound to - this will happen part way through the begin for the listener side of a session
	 * @return the actual address this session is bound to or null if the session is not yet bound
	 */
	public SessionAddress getActualAddress() {
		return transport.getActualAddress();
	}
	/*
	class ReconnectThread extends Thread {
		public ReconnectThread() {
			setName(getActualAddress()+":"+session_ids+":ReconnectionThread");
		}
		public void run() {
			while (!finished) {
				try {
					Thread.sleep(1000);
				} catch (Exception e) {
				}

				try {
					if (!transport.isAlive()) {
						if (use_acks) {
System.out.println(getActualAddress()+":"+session_ids+":RESTARTING");
							restartTransport(false);
						} else {
							end();
						}
					}
				} catch (IOException e) {
System.out.println(getActualAddress()+":"+session_ids+":RESTART FAILED "+e);
					end(e);
				} catch (Exception e) {
System.out.println(getActualAddress()+":"+session_ids+":RESTART FAILED "+e);
					end(new IOException(""+Logger.getStackTrace(e)));
				}
			}
		}
	}
		*/
	class ReaderThread extends Thread {
		long last_read = System.currentTimeMillis();
		long total_read = 0;
		
		InputStream in;
		OutputStream ack_out;
		public void setInputStream(InputStream in) {
			this.in = in;
		}
		
		public void setAckStream(OutputStream out) {
			this.ack_out = out;
		}
		
		private void readPlain() throws EOFException, Exception {
//			byte[] buf = new byte[DEFAULT_READ_BUFFER_BYTES];
			int n = 0;
			while (n != -1 && !finished) {
//				n = in.read(buf,0,buf.length);
//				if (n > 0) {
//					byte[] tmp = new byte[n];
//					System.arraycopy(buf,0,tmp,0,n);

				byte[] tmp = StreamUtils.readBytes(in);

				bin.add(tmp);
					
					total_read += tmp.length;
					last_read = System.currentTimeMillis();
//				}
			}
			
			if (n == -1) {
				throw new EOFException("end of stream");
			}
		}
		
		short logical_clock = 0;
		private void readAcked() throws EOFException, Exception {
			
			while (!finished) {
				short clock = StreamUtils.readShort(in);
				byte[] dat = StreamUtils.readBytes(in);

				if (clock == logical_clock) {
//					System.out.println("PACKETS IN SYNC "+logical_clock+"/"+clock);
					//correct packet
					bin.add(dat);
					logical_clock++;
					if (logical_clock == LOGICAL_CLOCK_MAX) {
						logical_clock = 0;
					}

					total_read += dat.length;
					last_read = System.currentTimeMillis();
				
				} else {

					if (DEBUG) System.out.println("READER IGNORING MISMATCHED PACKET "+clock);
					
					//ignore
//XXX System.out.println(session_ids+":IGNORING PACKET "+clock);					
				}
				
				//write ACKs every so often (ACK_COUNT)
				if (clock%ACK_COUNT == 0) {
					
//XXX System.out.println(session_ids+":SENDING ACK "+clock);

					if (DEBUG) System.out.println("READER WRITING ACK AT "+clock);
					
					//write an ack
					StreamUtils.writeShort(ack_out,clock);
					ack_out.flush();
					
				}
				
			}
		}
		
		public void run() {
			Exception last_error = null;
			
			boolean first = true;

			while (!finished) {
				
				try {
					if (use_acks) {
						if (first) {
//XXX System.out.println("SENDING INITIAL ACK "+0);

							//write an ack
							StreamUtils.writeShort(ack_out,(short)0);
							ack_out.flush();

							first = false;
						}
						
						readAcked();
					} else {
						readPlain();
					}
				} catch (EOFException e) {
					last_error = e;
				} catch (IOException e) {
					last_error = e;
				} catch (NullPointerException e) {
//					last_error = e;
				} catch (Exception e) {
					last_error = e;
				}

//				System.out.println(getActualAddress()+":"+session_ids+":NO AVAILABLE DATA: "+last_error);

				try {
					if (last_error != null) {
						if (use_acks) {
							Logger.warning("session "+session_ids+" restarting ("+last_error+") ("+getActualAddress()+")");
//				System.out.println(getActualAddress()+":"+session_ids+":RESTARTING ("+last_error+")");
							restartTransport(false);
						} else {
							end();
						}
					}
				} catch (IOException x) {
					Logger.warning("session "+session_ids+" restart failed ("+x+") ("+getActualAddress()+")");
//				System.out.println(getActualAddress()+":"+session_ids+":RESTART FAILED "+x);
					end(x);
				} catch (Exception x) {
					Logger.warning("session "+session_ids+" restart failed ("+x+") ("+getActualAddress()+")");
//				System.out.println(getActualAddress()+":"+session_ids+":RESTART FAILED "+x);
					end(new IOException(""+Logger.getStackTrace(x)));
				}
				/*
				if (last_error != null) {
					if (address.requires_link_reconnection) {
						try {
System.out.println(getActualAddress()+":"+session_ids+":RESTARTING: "+last_error);
//last_error.printStackTrace();
							restartTransport(false);
							last_error = null;
						} catch (Exception e) {
							//ignore - will have been reported by end()
							last_error = e;
System.out.println(getActualAddress()+":"+session_ids+":RESTART FAILED: "+last_error);
last_error.printStackTrace();
						}
						
					}		
				}
					*/
//				if (last_error != null) {
				if (last_error != null && !use_acks) {
					if (last_error instanceof IOException) {
						end((IOException)last_error);
					} else {
						end(new IOException(""+Logger.getStackTrace(last_error)));
					}
				}

				try {
					Thread.sleep(1000);
				} catch (Exception x) {
				}
			}
			
			if (DEBUG) System.out.println("READER EXITED "+finished);

			bin.setClosed();
			
//			System.out.println(getActualAddress()+":"+session_ids+":Reader thread finished - "+finished+" "+last_error);
			Logger.info("session "+session_ids+" reader thread finished ("+last_error+") ("+getActualAddress()+") ("+finished+")");
		}
	}
	class WriterThread extends Thread {
		long last_write = System.currentTimeMillis();
		long total_write = 0;
		
		OutputStream out;
		InputStream ack_in;
		
		CircularObjectBuffer resend_buffer = new CircularObjectBuffer(2 * ACK_COUNT);
		CircularShortBuffer resend_clock = new CircularShortBuffer(2 * ACK_COUNT);
//		List resend_buffer = new LinkedList();
//		List resend_clock = new LinkedList();
		Short resend_required = null;
		
		public void setOutputStream(OutputStream out) {
			this.out = out;
		}
		public void setAckStream(InputStream in) {
			this.ack_in = in;
		}
		public void resendFromClock(short rlogical_clock) {
			resend_required = new Short(rlogical_clock);
		}

		private void writePlain() throws Exception {
			while (!finished) {
				byte[] dat = bout.clearToByteArray(DEFAULT_WRITE_BUFFER_BYTES);
				if (dat.length > 0) {
					StreamUtils.writeBytes(out,dat);
//					out.write(dat,0,dat.length);
					out.flush();
					
					total_write += dat.length;
					last_write = System.currentTimeMillis();
				}
			}
		}
		
		private String printClocks() {
			StringBuffer sb = new StringBuffer();
			for (int i = 0; i < resend_clock.size(); i++) {
				if (i > 0) sb.append(",");
				sb.append(resend_clock.get(i));
			}
			return sb.toString();
		}
		
		short logical_clock = -1;
		private void writeAcked() throws Exception {
			while (!finished) {
				if (resend_required != null) {
					
					if (DEBUG) System.out.println("WRITER ASKED TO RESEND FROM "+resend_required);
					
//XXX System.out.println(session_ids+":ASKED TO RESEND FROM "+resend_required+", Buffer:"+printClocks());	
//					for (int i = resend_clock.indexOf(resend_required); i < resend_buffer.size(); i++) {
					for (int i = 0; i < resend_buffer.size(); i++) {
						byte[] dat = (byte[]) resend_buffer.get(i);
						short clock = resend_clock.get(i);
//						Short clock = (Short)resend_clock.get(i);

						if (DEBUG) System.out.println("WRITER RESENDING "+clock);
						
//						StreamUtils.writeShort(out,clock.shortValue());
						StreamUtils.writeShort(out,clock);
						StreamUtils.writeBytes(out,dat);
						out.flush();
						
						//Have we sent enough to expect an ACK?
//						if (clock.shortValue()%ACK_COUNT == ACK_OFFSET) {
/*						if (clock.shortValue()%ACK_COUNT == 0) {
System.out.println(session_ids+":WAITING FOR RESEND ACK "+clock+", Buffer:"+printClocks());	
							//read an ACK
							short rlogical_clock = StreamUtils.readShort(ack_in);
						}
*/						
					}
					resend_required = null;
					
				} else {
					byte[] dat = bout.clearToByteArray(DEFAULT_WRITE_BUFFER_BYTES);
					if (dat.length > 0) {
						logical_clock++;
						if (logical_clock == LOGICAL_CLOCK_MAX) {
							logical_clock = 0;
						}

						resend_buffer.add(dat);
//						resend_clock.add(new Short(logical_clock));
						resend_clock.add(logical_clock);
	
						
						//write the data with it's associated logical clock value
						StreamUtils.writeShort(out,(short)(logical_clock));
						StreamUtils.writeBytes(out,dat);
						out.flush();
						
						while (resend_buffer.size() > ACK_COUNT + ACK_OFFSET) {
							
							//read an ACK
//							Short rlogical_clock = new Short(StreamUtils.readShort(ack_in));
							short rlogical_clock = StreamUtils.readShort(ack_in);

//XXX System.out.println(session_ids+":GOT ACK, "+rlogical_clock+", Buffer:"+printClocks());							
							
							int chop = 1 + resend_clock.indexOf(rlogical_clock);
//							int chop = resend_clock.indexOf(rlogical_clock);
							if (chop != 0) {
								resend_clock.discard(resend_clock.size()-chop);
								resend_buffer.discard(resend_buffer.size()-chop);
//								resend_clock = resend_clock.subList(chop,resend_clock.size());
//								resend_buffer = resend_buffer.subList(chop,resend_buffer.size());
							}

//XXX System.out.println(session_ids+":PROCESSED ACK, "+rlogical_clock+", Buffer:"+printClocks());							
						}
						total_write += dat.length;
						last_write = System.currentTimeMillis();
						
/*
						//Have we sent enough to expect an ACK?
						if (logical_clock%ACK_COUNT == ACK_OFFSET) {

System.out.println(session_ids+":WAITING FOR ACK "+logical_clock+", Buffer:"+printClocks());	

							//read an ACK
							Short rlogical_clock = new Short(StreamUtils.readShort(ack_in));

System.out.println(session_ids+":GOT ACK, "+rlogical_clock+", Buffer:"+printClocks());							
							
							int chop = 1 + resend_clock.indexOf(rlogical_clock);
							resend_clock = resend_clock.subList(chop,resend_clock.size());
							resend_buffer = resend_buffer.subList(chop,resend_buffer.size());

System.out.println(session_ids+":PROCESSED ACK, "+rlogical_clock+", Buffer:"+printClocks());							
							
						}*/
					}
				}
			}
		}
		
		public void run() {
			Exception last_error = null;
			while (!finished) {
				
				try {
					if (use_acks) {
						//we need to do ACKs to deal with transport link failures
						writeAcked();
					} else {
						//we aren't dealing with transport link failures
						writePlain();
					}
				} catch (IOException e) {
					last_error = e;
				} catch (NullPointerException e) {
//					last_error = e;
				} catch (Exception e) {
					last_error = e;
				}
/*				
				if (last_error != null) {
					if (address.requires_link_reconnection) {
						try {
System.out.println(getActualAddress()+":"+session_ids+":RESTARTING: "+last_error);
//last_error.printStackTrace();
							restartTransport(false);
							last_error = null;
						} catch (Exception e) {
							last_error = e;
System.out.println(getActualAddress()+":"+session_ids+":RESTART FAILED: "+last_error);
last_error.printStackTrace();
							//ignore - will have been reported by end()
						}
						
					}					
				}
*/
				if (last_error != null && !use_acks) {
//				if (last_error != null) {
					if (last_error instanceof IOException) {
						end ((IOException)last_error);
					} else {
						end (new IOException(Logger.getStackTrace(last_error)));
					}
					
				}
				
				try {
					Thread.sleep(1000);
				} catch (Exception x) {
				}
			}

			if (DEBUG) System.out.println("WRITER EXITED "+finished);
			
//			System.out.println(getActualAddress()+":"+session_ids+":Writer thread finished - "+finished+" "+last_error);
			Logger.info("session "+session_ids+" writer thread finished ("+last_error+") ("+getActualAddress()+") ("+finished+")");
		}
	}
	
}