/*******************************************************************************
* Copyright (c) 2007 IONA Technologies PLC
* All rights reserved. This program and the accompanying materials
* are made available under the terms of the Eclipse Public License v1.0
* which accompanies this distribution, and is available at
* http://www.eclipse.org/legal/epl-v10.html
* 
* Contributors:
*     IONA Technologies PLC - initial API and implementation
*******************************************************************************/
package org.eclipse.stp.sc.jaxws.internal.wizards;

import java.util.HashSet;
import java.util.Set;
import java.util.jar.Attributes;
import java.util.jar.JarFile;
import java.util.jar.Manifest;

import org.eclipse.core.runtime.IPath;
import org.eclipse.jdt.launching.IVMInstall;
import org.eclipse.jdt.launching.JavaRuntime;
import org.eclipse.jdt.launching.LibraryLocation;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.wizard.IWizardPage;
import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.stp.common.logging.LoggingProxy;
import org.eclipse.stp.sc.common.internal.model.RuntimeCore;
import org.eclipse.stp.sc.common.internal.viewers.StpRuntimeComposite;
import org.eclipse.stp.sc.jaxws.ScJaxWsResources;
import org.eclipse.stp.sc.jaxws.wizards.StartingPointWizardPage;
import org.eclipse.stp.sc.jaxws.wizards.WebServiceProjectWizard;
import org.eclipse.stp.sc.jaxws.wizards.WsdlFirstProjectWizard;
import org.eclipse.stp.sc.jaxws.wizards.WsdlGenPage;
import org.eclipse.stp.sc.jaxws.wizards.WsdlSettingPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.wst.common.project.facet.core.IProjectFacet;
import org.eclipse.wst.common.project.facet.core.IProjectFacetVersion;
import org.eclipse.wst.common.project.facet.core.ProjectFacetsManager;
import org.eclipse.wst.common.project.facet.core.runtime.RuntimeManager;
import org.eclipse.wst.server.core.IRuntime;

/**
 * this is runtime selection page for jax-ws project.
 * user is required to select one runtime before
 * any more operations during project creation
 * @author jma
 *
 */
public class RuntimeSelectionPage extends WizardPage {

	private static final LoggingProxy LOG = LoggingProxy.getlogger(RuntimeSelectionPage.class);
	
	public StpRuntimeComposite runtimeComp;
	protected IRuntime selectedRuntime = null;
	public RuntimeSelectionPage(String pageName, String title,
			ImageDescriptor titleImage) {
		super(pageName, title, titleImage);
	}
	

	public void createControl(Composite parent) {
		Composite composite = new Composite(parent, SWT.NONE);
		
		composite.setLayoutData(GridData.FILL_BOTH);
		composite.setLayout(new GridLayout());
		
		String[] moduleTypes = new String[] {RuntimeCore.JAXWS_MODULE_TYPE};
		
		runtimeComp = new StpRuntimeComposite(composite, SWT.NONE, 
				new StpRuntimeComposite.SelectionListener() {
			public void runtimeSelected(IRuntime runtime) {
				LOG.debug("runtime selected:" + runtime.getId());
				selectedRuntime = runtime;
				setPageComplete(validatePage());
				//update wizard buttons
				getWizard().getContainer().updateButtons();
			}
		}, moduleTypes);
		
		
		runtimeComp.setLayoutData(new GridData(GridData.FILL_HORIZONTAL | GridData.FILL_VERTICAL));
		this.setControl(composite);
		
		if (runtimeComp.getRuntimeCount() == 1) {
			//only have one runtime, we will select it by default
			selectedRuntime = runtimeComp.selectRuntimeByIndex(0);
		} else {
		    setPageComplete(false);
		}

	}
	
	public int getRuntimeCount() {
		return runtimeComp.getRuntimeCount();
	}
	
	public IWizardPage getNextPage() {
		if (this.getWizard() instanceof WsdlFirstProjectWizard) {
			// update gen page control according to runtime type
			WsdlGenPage genPage = (WsdlGenPage) this.getWizard().getPage(
					WsdlFirstProjectWizard.GENATOR_PAGE);
			genPage.updateControl(selectedRuntime.getRuntimeType().getId());
			WsdlSettingPage wsdlPage = (WsdlSettingPage) this.getWizard()
					.getPage(WsdlFirstProjectWizard.WSDL_PAGE);
			return wsdlPage;
		}
		
		if(RuntimeSelectionPage.this.getWizard() instanceof WebServiceProjectWizard){
			StartingPointWizardPage startPointPage =  (StartingPointWizardPage)RuntimeSelectionPage.this.getWizard().getPage(
					WebServiceProjectWizard.STARTING_POINT_PAGE_NAME);
			startPointPage.updateControl(selectedRuntime.getRuntimeType().getId());
		}
		return super.getNextPage();
	}
	
	public boolean validatePage() {
		setMessage(null);
		setErrorMessage(null);
		if (selectedRuntime == null) {
			setErrorMessage(ScJaxWsResources.getString("runtimepage.error.noruntime"));
			return false;
		}
		//check the selected runtime again default jre version
//		return true;
		return validateRuntimeJavaFacet();
	}
	
	
	/**
	 * we have required java facetd defined in the runtime.
	 * Since the project will be created with default jre defined in eclipse.
	 * we need to validate the runtime against default jre version
	 * @return
	 */
	
	private boolean validateRuntimeJavaFacet() {
		String jreVersion = getJREVersion(JavaRuntime.getDefaultVMInstall());
	    LOG.debug("default jre version:" + jreVersion);
		String javaFacetVersion = getJavaFacetVersion(jreVersion);
		
		IProjectFacet pf = ProjectFacetsManager.getProjectFacet("jst.java");
		IProjectFacetVersion fv = pf.getVersion(javaFacetVersion);
		LOG.debug("java facet version:" + javaFacetVersion);
		
		HashSet<IProjectFacetVersion> fvs = new HashSet<IProjectFacetVersion>();
		fvs.add(fv);
		
		/**
		 * the code below doesn't work. since the runtime has two components:
		 * jax-ws and jre6. the jre6 compo is added by jst automacally.
		 * Need to figure out a workaround for this.
		 */
		Set<org.eclipse.wst.common.project.facet.core.runtime.IRuntime> runtimes = RuntimeManager.getRuntimes(fvs);
		for (org.eclipse.wst.common.project.facet.core.runtime.IRuntime runtime : runtimes) {
			if (runtime.getName().equals(selectedRuntime.getName())) {
				LOG.debug("the selected runtime has passed java facet checking");
				return true;
			}
		}
		LOG.debug("selected runtime doesn't support java " + fv);
		setErrorMessage(ScJaxWsResources.getString("runtimepage.error.javafacet") + javaFacetVersion);
		return false;
	}
	
	
	private String getJavaFacetVersion(String jreVersion) {
		return jreVersion.substring(2, 3) + ".0";
	}
	  private String getJREVersion(IVMInstall install) {
		    LibraryLocation[] libraryLocations = JavaRuntime.getLibraryLocations(install);
		    if(libraryLocations != null) {
		      for(int k = 0; k < libraryLocations.length; k++ ) {
		        IPath path = libraryLocations[k].getSystemLibraryPath();
		        String jarName = path.lastSegment();
		        if("rt.jar".equals(jarName)) {
		          try {
		            JarFile jarFile = new JarFile(path.toFile());
		            Manifest manifest = jarFile.getManifest();
		            Attributes attributes = manifest.getMainAttributes();
		            return attributes.getValue(Attributes.Name.SPECIFICATION_VERSION);
		          } catch(Exception ex) {
		            ex.printStackTrace();
		          }
		        }
		      }
		    }
		    return null;
		  }
		  

	
	public boolean canFlipToNextPage(){
		if (selectedRuntime == null) {
			return false;
		}
		return super.canFlipToNextPage();
	}
	
	public String getRuntimeType() {
		return selectedRuntime.getRuntimeType().getId();
	}
	
	public String getRuntimeID() {
		return selectedRuntime.getId();
	}

	public void selectRuntimeByIndex(int index) {
		LOG.debug("set select runtime by index:" + index);
		selectedRuntime = runtimeComp.selectRuntimeByIndex(index);
		LOG.debug("result runtime:" + selectedRuntime);
	}
}
