/**********************************************************************
 * Copyright (c) 2005 Scapa Technologies Limited and others
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * Scapa Technologies Limited - Initial API and implementation
 **********************************************************************/

package org.eclipse.stp.b2j.core.jengine.internal.utils;

import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.SimpleTimeZone;
import java.util.TimeZone;

/**
 * 
 * @author amiguel
 *
 * A runtime utility class used to perform common XSD functions
 */
public class XSDUtil {

	private static final double ONE_SECOND = 1000;
	private static final long ONE_MINUTE = 60 * (long)ONE_SECOND;
	private static final long ONE_HOUR = 60 * ONE_MINUTE;
	private static final long ONE_DAY = 24 * ONE_HOUR;
	private static final long ONE_WEEK = 7 * ONE_DAY;
	private static final long ONE_MONTH = 31 * ONE_DAY;
	private static final long ONE_YEAR = 365 * ONE_DAY;
	
	private static int charArrayIndexOf(char c, char[] cs, int cslen, int index) {
		for (int i = index; i < cslen; i++) {
			if (cs[i] == c) return i;
		}
		return -1;
	}
	
	private static int stringBufferIndexOf(char c, StringBuffer sb, int index) {
		int sblen = sb.length();
		for (int i = index; i < sblen; i++) {
			if (sb.charAt(i) == c) return i;
		}
		return -1;
	}
	
/*	public static String stripXmlText(String s) {
		return new CharStack(s).popXmlText(true,true);
	}*/
	
	public static String removeXmlComments(String s) {
		StringBuffer sb = new StringBuffer();
		char[] c = s.toCharArray();
		
		boolean skipping = false;
		
		for (int i = 0; i < c.length; i++) {
			if (i < c.length-3) {
				if (!skipping
					&& c[i] == '<'
					&& c[i+1] == '!'
					&& c[i+2] == '-'
					&& c[i+3] == '-') {
					//begin comment
					skipping = true;
					i+=3;
					continue;
				}
			}
			if (i < c.length-2) {
				if (skipping
					&& c[i] == '-'
					&& c[i+1] == '-'
					&& c[i+2] == '>') {
					//end comment
					skipping = false;
					i+=2;
					continue;
				}
			}
			if (!skipping) {
				sb.append(c[i]);
			}
		}
		return sb.toString();
	}
	
	public static String fromXMLString(String s) {
		
		char[] cs = s.toCharArray();
		int cslen = cs.length;
		
		StringBuffer ret = new StringBuffer();
		
		char cs0=0,cs1=0,cs2=0,cs3=0,cs4=0,cs5=0;
		for (int i = 0; i < cslen; i++) {
			cs0 = cs[i];
			
			if (cs0 == '&') {
				
				int len = (cslen - i);
				
				boolean resolved = false;
				
				if (len > 3) {
					cs1 = cs[i+1];
					cs2 = cs[i+2];
					cs3 = cs[i+3];

					if (cs1 == '#') {
						
						int end_parse = charArrayIndexOf(';',cs,cslen,i+1);
						if (end_parse != -1) {
							char ccode = (char)Integer.parseInt(new String(cs,i+2,end_parse-(i+2)));
							ret.append(String.valueOf(ccode));
							i = end_parse;
						}
						resolved = true;
					
					} else if (cs1 == 'l' 
								&& cs2 == 't'
								&& cs3 == ';') {
						
						ret.append('<');
						i = i+3;
						resolved = true;
	
					} else if (	cs1 == 'g' 
								&& cs2 == 't'
								&& cs3 == ';') {
	
						ret.append('>');
						i = i+3;
						resolved = true;
					}
				} 
				
				if (!resolved && len > 4) {
					cs4 = cs[i+4];

					if (	cs1 == 'a' 
								&& cs2 == 'm'
								&& cs3 == 'p'
								&& cs4 == ';') {
		
						ret.append('&');
						i = i+4;
						resolved = true;
	
					}
				}
				
				if (!resolved && len > 5) {
					cs5 = cs[i+5];
					
					if (	cs1 == 'q' 
								&& cs2 == 'u'
								&& cs3 == 'o'
								&& cs4 == 't'
								&& cs5 == ';') {
	
						ret.append('\"');
						i = i+5;
						resolved = true;
	
					} else if (	cs1 == 'a' 
								&& cs2 == 'p'
								&& cs3 == 'o'
								&& cs4 == 's'
								&& cs5 == ';') {
	
						ret.append('\'');
						i = i+5;
						resolved = true;
	
					}
				}
				
				if (!resolved) {
					ret.append(cs0);
				}
					
			} else {
				ret.append(cs0);
			}
			
		}
		return ret.toString();
	}

	public static String toXmlString(String val) {
		return toXMLString(val);
	}
	public static String toXMLString(String val) {
		//could be nillable!
		if (val == null) return val;
		
		//speed up common case at minor cost to less common cases
//		if (val.indexOf('&') == -1) return val;
		
		boolean changed = false;
		
		char[] cs = val.toCharArray();
		int cslen = cs.length;
		char c;
		StringBuffer sb = new StringBuffer();
		for (int i = 0; i < cslen; i++) {
			c = cs[i];
			if (c == '>') {
				sb.append("&gt;");
				changed = true;
			} else if (c == '<') {
				sb.append("&lt;");
				changed = true;
			} else if (c == '"') {
				sb.append("&quot;");
				changed = true;
			} else if (c == '\'') {
				sb.append("&apos;");
				changed = true;
			} else if (c == '&') {
				sb.append("&amp;");
				changed = true;
			} else {
				sb.append(c);	
			}
		}
		if (changed) {
			return sb.toString();			
		} else {
			return val;
		}
	}
	
	public static long parseDuration(String s) throws Exception {
		CharStack stack = new CharStack(s);
		boolean negate = false;
		
		char c = stack.pop();
		if (c == '-') {
			negate = true;
			c = stack.pop();
		}
		if (c != 'P') {
			throw new Exception("Error parsing duration - expected P or - ("+s+")");
		}
		long ms = 0;
		
		boolean time = false;
		if (stack.peek() == 'T') {
			time = true;
			stack.pop();
		}
		
		
		while (!stack.isEmpty()) {
			double num = stack.popNumber();
			long inum = (long)num;
			char type = stack.pop();
			if (type == 'S') {
				num *= ONE_SECOND;
				ms += (int)num;
			} else if (type == 'T') {
				time = true;
			} else if (type == 'M') {
				if (time) {
					ms += inum * ONE_MINUTE;
				} else {
					ms += inum * ONE_MONTH;
				}
			} else if (type == 'H') {
				ms += inum * ONE_HOUR;
			} else if (type == 'D') {
				ms += inum * ONE_DAY;
			} else if (type == 'Y') {
				ms += inum * ONE_YEAR;
			}
		}
		
		if (negate) {
			return -ms;
		} else {
			return ms;
		}
	}
	
	private static long appendInt(StringBuffer sb, long t, long div, char postfix) {
		long c = t / div;
		if (c > 0) {
			sb.append(c);
			sb.append(postfix);
		}
		return t - (c*div);
	}
	private static long appendDecimal(StringBuffer sb, long t, double div, char postfix) {
		double c = ((double)t) / div;
		if (c > 0) {
			sb.append(c);
			sb.append(postfix);
		}
		return t - (long)(c*div);
	}
	public static String toDuration(long t) {
		StringBuffer sb = new StringBuffer();
		
		StringBuffer sbt = new StringBuffer();
		
		if (t < 0) {
			sb.append("-");
			t = -t;
		}
		
		sb.append("P");
		t = appendInt(sb,t,ONE_YEAR,'Y');
		t = appendInt(sb,t,ONE_MONTH,'M');
		t = appendInt(sb,t,ONE_DAY,'D');
		
		t = appendInt(sbt,t,ONE_HOUR,'H');
		t = appendInt(sbt,t,ONE_MINUTE,'M');
		t = appendDecimal(sbt,t,ONE_SECOND,'S');
		
		if (sbt.length() > 0) {
			sb.append("T");
			sb.append(sbt.toString());
		}
			
		return sb.toString();
	}

///////////////////////////////////////
// XSD timezone parser
///////////////////////////////////////

	private static String toXsdTimeZone(String s) {
		int len = s.length();
		return s.substring(0,len-2) + ":" + s.substring(len-2);
	}
/*
	private static String toJavaTimeZone(String s) throws Exception {
		int len = s.length();
		if (s.endsWith("Z")) {
			return s.substring(0,len-1) + "+0000";
		} else {
			char c = s.charAt(len-6);
			if (c == '-' || c == '+') {
				return s.substring(0,len-3) + s.substring(len-2);
			} else {
				return s;
			}
		}
	}
*/
	private static TimeZone getTimeZone(String tz) {
		TimeZone timezone = null;
		
		if (tz.endsWith("Z")) {
			//Z = GMT timezone
			timezone = new SimpleTimeZone(0,"GMT+00:00");
			
		} else {
			int len = tz.length();
			char c = tz.charAt(len-6);

			
			if (c == '-') {
				String id = tz.substring(len-6);
				CharStack stack = new CharStack(tz,len-5);
				
				int rawOffset = 0;
				rawOffset -= ONE_HOUR * (int)stack.popNumber();
				stack.pop(); //pop ':'
				rawOffset -= ONE_MINUTE * (int)stack.popNumber();

				timezone = new SimpleTimeZone(rawOffset,"GMT"+id);

			} else if (c == '+') {
				String id = tz.substring(len-6);
				CharStack stack = new CharStack(tz,len-5);
				
				int rawOffset = 0;
				rawOffset += ONE_HOUR * (int)stack.popNumber();
				stack.pop(); //pop ':'
				rawOffset += ONE_MINUTE * (int)stack.popNumber();

				timezone = new SimpleTimeZone(rawOffset,"GMT"+id);
				
			} else {
				timezone = new SimpleTimeZone(0,"GMT+00:00");
				
			}
		}
		
		return timezone;
	}	

///////////////////////////////////////
// xsd:gMonthDay
///////////////////////////////////////
	private static SimpleDateFormat sdf_gMonthDay = new SimpleDateFormat("--MM-dd");
	
	private static SimpleDateFormat sdf_gMonthDayString = new SimpleDateFormat("--MM-ddZ");
	
	public static GregorianCalendar parseGMonthDay(String s) throws Exception {
		s = s.trim();
		
		GregorianCalendar cal = new GregorianCalendar(getTimeZone(s));		
		cal.setTime(sdf_gMonthDay.parse(s));
		
		return cal;
	}
	
	public static String toGMonthDay(GregorianCalendar cal) {
		SimpleDateFormat sdf = (SimpleDateFormat)sdf_gMonthDayString.clone();
		sdf.setTimeZone(cal.getTimeZone());
		return toXsdTimeZone( sdf.format(cal) );
	}
	
///////////////////////////////////////
// xsd:gDay
///////////////////////////////////////
	private static SimpleDateFormat sdf_gDay = new SimpleDateFormat("---dd");
	
	private static SimpleDateFormat sdf_gDayString = new SimpleDateFormat("---ddZ");
	
	public static GregorianCalendar parseGDay(String s) throws Exception {
		s = s.trim();
		
		GregorianCalendar cal = new GregorianCalendar(getTimeZone(s));		
		cal.setTime(sdf_gDay.parse(s));
		
		return cal;
	}
	
	public static String toGDay(GregorianCalendar cal) {
		SimpleDateFormat sdf = (SimpleDateFormat)sdf_gDayString.clone();
		sdf.setTimeZone(cal.getTimeZone());
		return toXsdTimeZone( sdf.format(cal) );
	}
	
///////////////////////////////////////
// xsd:gYearMonth
///////////////////////////////////////
	private static SimpleDateFormat sdf_gYearMonth = new SimpleDateFormat("yyyy--MM");
	
	private static SimpleDateFormat sdf_gYearMonthString = new SimpleDateFormat("yyyy-MMZ");
	
	public static GregorianCalendar parseGYearMonth(String s) throws Exception {
		s = s.trim();
		
		if (s.charAt(0) == '-') {
			//TODO ignoring preceding '-' in gYearMonth
			s = s.substring(1);
		}	

		GregorianCalendar cal = new GregorianCalendar(getTimeZone(s));		
		cal.setTime(sdf_gYearMonth.parse(s));
		
		return cal;
	}
	
	public static String toGYearMonth(GregorianCalendar cal) {
		SimpleDateFormat sdf = (SimpleDateFormat)sdf_gYearMonthString.clone();
		sdf.setTimeZone(cal.getTimeZone());
		return toXsdTimeZone( sdf.format(cal) );
	}
	
///////////////////////////////////////
// xsd:gYear
///////////////////////////////////////
	private static SimpleDateFormat sdf_gYear = new SimpleDateFormat("yyyy");
	
	private static SimpleDateFormat sdf_gYearString = new SimpleDateFormat("yyyyZ");
	
	public static GregorianCalendar parseGYear(String s) throws Exception {
		s = s.trim();
		
		if (s.charAt(0) == '-') {
			//TODO ignoring preceding '-' in gYear
			s = s.substring(1);
		}	

		GregorianCalendar cal = new GregorianCalendar(getTimeZone(s));		
		cal.setTime(sdf_gYear.parse(s));
		
		return cal;
	}
	
	public static String toGYear(GregorianCalendar cal) {
		SimpleDateFormat sdf = (SimpleDateFormat)sdf_gYearString.clone();
		sdf.setTimeZone(cal.getTimeZone());
		return toXsdTimeZone( sdf.format(cal) );
	}

///////////////////////////////////////
// xsd:gMonth
///////////////////////////////////////
	private static SimpleDateFormat sdf_gMonth = new SimpleDateFormat("--MM--");
	
	private static SimpleDateFormat sdf_gMonthString = new SimpleDateFormat("--MM--Z");
	
	public static GregorianCalendar parseGMonth(String s) throws Exception {
		s = s.trim();

		GregorianCalendar cal = new GregorianCalendar(getTimeZone(s));		
		cal.setTime(sdf_gMonth.parse(s));
		
		return cal;
	}
	
	public static String toGMonth(GregorianCalendar cal) {
		SimpleDateFormat sdf = (SimpleDateFormat)sdf_gMonthString.clone();
		sdf.setTimeZone(cal.getTimeZone());
		return toXsdTimeZone( sdf.format(cal) );
	}

///////////////////////////////////////
// xsd:time
///////////////////////////////////////
	private static SimpleDateFormat sdf_time = new SimpleDateFormat("HH:mm:ss");
	private static SimpleDateFormat sdf_timeMS = new SimpleDateFormat("HH:mm:ss.SSS");

	private static SimpleDateFormat sdf_timeString = new SimpleDateFormat("HH:mm:ss.SSSZ");

	public static GregorianCalendar parseTime(String s) throws Exception {
		s = s.trim();
		
		GregorianCalendar cal = new GregorianCalendar(getTimeZone(s));
		
		if (s.charAt(8) == '.') {
			//time has milliseconds specified
			cal.setTime(sdf_timeMS.parse(s));
		} else {
			//time has no milliseconds specified
			cal.setTime(sdf_time.parse(s));
		}

		return cal;
	}
	
	public static String toTime(GregorianCalendar cal) {
		SimpleDateFormat sdf = (SimpleDateFormat)sdf_timeString.clone();
		sdf.setTimeZone(cal.getTimeZone());
		return toXsdTimeZone( sdf.format(cal) );
	}
	
	
///////////////////////////////////////
// xsd:date
///////////////////////////////////////
	private static SimpleDateFormat sdf_date = new SimpleDateFormat("yyyy-MM-dd");

	private static SimpleDateFormat sdf_dateString = new SimpleDateFormat("yyyy-MM-ddZ");

	public static GregorianCalendar parseDate(String s) throws Exception {
		s = s.trim();

		if (s.charAt(0) == '-') {
			//TODO ignoring preceding '-' in date
			s = s.substring(1);
		}
		
		GregorianCalendar cal = new GregorianCalendar(getTimeZone(s));
		cal.setTime(sdf_date.parse(s));

		return cal;
	}
	
	public static String toDate(GregorianCalendar cal) {
		SimpleDateFormat sdf = (SimpleDateFormat)sdf_dateString.clone();
		sdf.setTimeZone(cal.getTimeZone());
		return toXsdTimeZone( sdf.format(cal) );
	}
	
///////////////////////////////////////
// xsd:dateTime
///////////////////////////////////////
	private static SimpleDateFormat sdf_dateTime = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss");
	private static SimpleDateFormat sdf_dateTimeMS = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss.SSS");

	private static SimpleDateFormat sdf_dateTimeString = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss.SSSZ");

	public static GregorianCalendar parseDateTime(String s) throws Exception {
		s = s.trim();
		
		if (s.charAt(0) == '-') {
			//TODO ignoring preceding '-' in dateTime
			s = s.substring(1);
		}
		
		GregorianCalendar cal = new GregorianCalendar(getTimeZone(s));

		if (s.charAt(19) == '.') {
			//time has milliseconds specified
			cal.setTime(sdf_dateTimeMS.parse(s));
		} else {
			//time has no milliseconds specified
			cal.setTime(sdf_dateTime.parse(s));
		}

		return cal;
	}
	
	public static String toDateTime(GregorianCalendar cal) {
		SimpleDateFormat sdf = (SimpleDateFormat)sdf_dateTimeString.clone();
		sdf.setTimeZone(cal.getTimeZone());
		return toXsdTimeZone( sdf.format(cal.getTime()) );
	}

	
///////////////////////////////////////
// TESTING
///////////////////////////////////////
	public static void main(String[] args) {
		try {
/*			
			System.out.println(toJavaTimeZone("mytimezone+07:00"));
			System.out.println(toJavaTimeZone("mytimezone-01:30"));
			System.out.println(toJavaTimeZone("mytimezone-00:00"));
			System.out.println(toJavaTimeZone("mytimezoneZ"));
			System.out.println(toJavaTimeZone("mytimezone12:15:80"));
*/			

			System.out.println(fromXMLString(fromXMLString("\"Hello &amp; World&lt;&lt;&quot;")));
			
			System.exit(0);
			System.out.println(fromXMLString("_&apos;&lt;&gt;&quot;&amp;_&apos;"));
			
			long t;
			String s;
			t = parseDuration("PT12.3S");
			System.out.println(t);
			s = toDuration(t);
			System.out.println(s);

//			System.out.println(toDateTime(parseDateTime("2004-12-01T01:02:03.456+02:00")));

			GregorianCalendar gc = new GregorianCalendar();
//			gc.setTimeInMillis(System.currentTimeMillis());
			gc.setTime(new Date(System.currentTimeMillis()));
			System.out.println(gc.getTime().getTime());
			System.out.println(parseDateTime(toDateTime(gc)).getTime().getTime());
			
			System.out.println(parseDateTime("2004-12-01T00:00:00Z").getTime());
			System.out.println(sdf_date.parse("2004-12-01").getTime());

/*			
			Date d = new Date(System.currentTimeMillis());
			System.out.println(d);
			s = toDateTime(d);
			System.out.println(s);
			d = parseDateTime(s);
			System.out.println(d);
*/			
		} catch (Exception e) {
			e.printStackTrace();
		}
	}
	
}