/*******************************************************************************
 * Copyright (c) 2003, 2005 IBM Corporation and others. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 which accompanies this distribution,
 * and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/
package org.eclipse.stp.core.internal.infrastructure.emf;

import org.eclipse.core.runtime.IStatus;
import org.eclipse.emf.common.notify.Notification;
import org.eclipse.emf.common.notify.Notifier;
import org.eclipse.emf.common.notify.impl.AdapterImpl;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.osgi.util.NLS;
import org.eclipse.stp.core.infrastructure.emf.EditModelException;

/**
 * A simple Adapter that manages read and write reference counts for any generic
 * EMF resource.
 * 
 */
public class ScribblerAdapter extends AdapterImpl {

   private EditModel       editModel          = null;

   protected static String ADAPTER_TYPE       = ScribblerAdapter.class
                                                    .getName();

   private int             readReferenceCount = 0;

   private int             editReferenceCount = 0;

   protected boolean       isNew              = true;

   protected boolean       forceRefresh;

   /**
    * 
    * @param aResource
    *           A resource that may or may not be reference counted
    * @return The ScribblerAdapter attached to the given resource or null if
    *         none exists.
    */
   public static ScribblerAdapter findAdapter(Resource aResource) {
      return (ScribblerAdapter) EcoreUtil.getAdapter(aResource.eAdapters(),
            ScribblerAdapter.ADAPTER_TYPE);
   }

   protected ScribblerAdapter(EditModel anEditModel) {
      editModel = anEditModel;
   }

   public void notifyChanged(Notification msg) {
      if (msg.getFeatureID(null) == ResourceSet.RESOURCE_SET__RESOURCES) {
         switch (msg.getEventType()) {
            case Notification.REMOVE:
            case Notification.REMOVE_MANY:
               if (msg.getOldValue() == getTarget())
                  EMFInfrastructurePlugin.log(IStatus.INFO, 0, NLS.bind(
                        Messages.Resource_0_has_been_del_, getTarget()), null);
         }

      }
   }

   public boolean isAdapterForType(Object type) {
      if (type == ADAPTER_TYPE)
         return true;
      return false;
   }

   /**
    * @return The resource that is managed by this Adapter
    */
   public Resource getResource() {
      Notifier theTarget = getTarget();
      if (theTarget instanceof Resource)
         return (Resource) theTarget;
      return null;
   }

   /**
    * 
    * Increments the read reference count.
    * 
    * @throws EditModelException
    *            If there is an error state with the reference count state
    */
   public void accessForRead() throws EditModelException {
      try {
         // if (isNew && getResource() instanceof ReferencedResource) {
         // ((ReferencedResource) getResource()).accessForRead();
         // }
         checkDeleted();
         readReferenceCount++;
      } finally {
         isNew = false;
      }
   }

   /**
    * Increments the write reference count.
    * 
    * @throws EditModelException
    *            If there is an error state with the reference count state
    */
   public void accessForWrite() throws EditModelException {
      try {
         // if (isNew && getResource() instanceof ReferencedResource)
         // ((ReferencedResource) getResource()).accessForWrite();
         checkDeleted();
         editReferenceCount++;
         Resource resource = getResource();
         if (!resource.isTrackingModification())
            resource.setTrackingModification(true);
      } finally {
         isNew = false;
      }
   }

   /*
    * Check if this resource has been removed and throw an exception if it does
    * not have a ResourceSet.
    */
   private void checkDeleted() throws EditModelException {
      Resource resource = getResource();
      if (resource.getResourceSet() == null)
         throw new EditModelException(NLS.bind(
               Messages.Resource_0_has_been_del_, resource.getURI()));
   }

   /**
    * Return the number of write accesses to this resource.
    * 
    * @return int The number of references.
    */
   private int getTotalReferenceCount() {
      return editReferenceCount + readReferenceCount;
   }

   /**
    * 
    * @return True iff the ScribblerAdapter has not yet been accessed.
    */
   public boolean isNew() {
      return isNew;
   }

   /**
    * Returns true if the resource has no write counts. This method does NOT
    * indicate if the resource should not be write-able, only that no one is
    * currently setup to write to the resource.
    * 
    * @return True iff the write count is 0.
    */
   public boolean isReadOnly() {
      return editReferenceCount <= 0;
   }

   /**
    * @return True if the resource is being used by multiple clients.
    */
   public boolean isShared() {
      return getTotalReferenceCount() > 1;
   }

   /**
    * @return True if the resource has multiple write clients
    */
   public boolean isSharedForWrite() {
      return editReferenceCount > 1;
   }

   /**
    * Decrements the read count on the resource.
    * 
    * @throws EditModelException
    *            If there is an error state with the reference count state
    * 
    */
   public void releaseFromRead() throws EditModelException {
      // if(getResource() instanceof ReferencedResource) {
      // ((ReferencedResource) getResource()).releaseFromRead();
      // }
      readReferenceCount--;
      if (readReferenceCount < 0)
         throw new EditModelException(
               "Read reference count error:  " + this.toString()); //$NON-NLS-1$
      unloadIfNecessary();
   }

   /**
    * Decrements the write count on the resource.
    * 
    * @throws EditModelException
    *            If there is an error state with the reference count state
    */
   public void releaseFromWrite() throws EditModelException {
      // if(getResource() instanceof ReferencedResource) {
      // ((ReferencedResource) getResource()).releaseFromWrite();
      // }

      editReferenceCount--;
      if (editReferenceCount < 0)
         throw new EditModelException("Write reference count error:  " + this); //$NON-NLS-1$
      unloadIfNecessary();
   }

   /**
    * 
    * @return The write reference count.
    */
   public int getEditReferenceCount() {
      return editReferenceCount;
   }

   public String toString() {
      StringBuffer buffer = new StringBuffer("ScribblerAdapter["); //$NON-NLS-1$
      buffer
            .append("ReadCnt=" + readReferenceCount + ",WriteCnt=" + editReferenceCount + ",IsNew=" + isNew + "]"); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$//$NON-NLS-4$
      return buffer.toString();
   }

   private void unloadIfNecessary() {

      if (!editModel.isDisposed() && !editModel.isReverting()) {
         Resource resource = getResource();
         if ((getTotalReferenceCount() <= 0)
               || (editReferenceCount <= 0 && resource.isModified())) {
            resource.unload();
            editModel.removeResource(resource);
         }
      }
   }
}
