/*******************************************************************************

* Copyright (c) 2007 IONA Technologies PLC

* All rights reserved. This program and the accompanying materials

* are made available under the terms of the Eclipse Public License v1.0

* which accompanies this distribution, and is available at

* http://www.eclipse.org/legal/epl-v10.html

* 

* Contributors:

*     IONA Technologies PLC - initial API and implementation

*******************************************************************************/
package org.eclipse.stp.sc.jaxws.wizards.wsdltojava;

import java.io.File;
import java.util.Hashtable;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.stp.common.logging.LoggingProxy;
import org.eclipse.stp.sc.common.internal.model.RuntimeCore;
import org.eclipse.stp.sc.common.utils.FileUtils;
import org.eclipse.stp.sc.common.workspace.WorkspaceManager;
import org.eclipse.stp.sc.jaxws.ScJaxWsPlugin;
import org.eclipse.stp.sc.jaxws.preferences.SCPreferenceConstants;
import org.eclipse.stp.sc.jaxws.runtimeprovider.IWsdlToJavaGenerator;
import org.eclipse.stp.sc.jaxws.runtimeprovider.RuntimeProviderManager;
import org.eclipse.stp.sc.jaxws.utils.MergeUtils;
import org.eclipse.stp.sc.jaxws.workspace.JaxWsWorkspaceManager;

/**
 * @author jma
 *
 */
public class WsdlToJavaGenerateAction {
    
    private static final LoggingProxy LOG = LoggingProxy.getlogger(WsdlToJavaGenerateAction.class);
    private static final String TEMP_DIR = "gentmp";
    
    /**
     * If this action is called from wsdlfrist wizard, we don't need to merge 
     */
    private boolean needMerge = true;
    
    @SuppressWarnings("unchecked")
    public synchronized void generate(String url, IProject project, Object data)
			throws CoreException {
		boolean isAutoBuild = WorkspaceManager.getWorkspaceAutoBuild();
		try {
			long startTime = System.currentTimeMillis();
			if (isAutoBuild) {
				// disable auto-build first before call generator
				LOG.debug("disable workspace autobuild before generating code");
				WorkspaceManager.setWorkspaceAutoBuild(false);
			}
			
			String runtimeType = RuntimeCore.getRuntimeType(project);
			
			IWsdlToJavaGenerator generator = RuntimeProviderManager.getInstance()
					.getWsdltoJavaGenerator(runtimeType);
			if (data != null) {
				generator.setInitializationData(null, null, data);
			}

			IPreferenceStore preferenceStore = ScJaxWsPlugin.getDefault()
					.getPreferenceStore();
			boolean mergeFlg = preferenceStore
					.getBoolean(SCPreferenceConstants.KEY_MERGE);
			if (!needMerge) {
			    mergeFlg = false;	
			}
			String oldDir = null, newDir = null;
			if (mergeFlg) {
				Hashtable genParams = (Hashtable) data;
				oldDir = (String) genParams
						.get(IWsdlToJavaGenerator.GEN_OUTPUTDIR);
				newDir = prepareMergeTmpDir(oldDir, genParams);
			}

			// check if need to merge
			generator.run(url, project);
			if (mergeFlg) {
				mergeDir(oldDir, newDir, true);
			}

			project.refreshLocal(IProject.DEPTH_INFINITE, null);
			tagGeneratedFiles(project, startTime, url);
		} catch (Throwable th) {
			th.printStackTrace();
		}
		finally {
			// enable auto-build again after call generator
			if (isAutoBuild) {
				LOG.debug("re-enable workspace autobuild after generating code");
				project.refreshLocal(IResource.DEPTH_INFINITE, null);
				WorkspaceManager.setWorkspaceAutoBuild(true);
			}
		}
	}
    
    /**
     * tag all java files within the assigned project, which create time (or update time) > startTime,
     * to indicate those files were created from the wsdl url
     * @param outputDir
     * @param startTime
     * @param url
     */
    private void tagGeneratedFiles(IContainer container, long startTime, String wsdlUrl) {
    	try {
			for (IResource res : container.members()) {
				if (!res.exists()) {
					continue;
				}
				if (res instanceof IContainer) {
					tagGeneratedFiles((IContainer) res, startTime, wsdlUrl);
				}
                if (! (res instanceof IFile)) {
                	continue;
                }
                IFile file = (IFile)res;
                if (file.getFileExtension() == null) {
                	continue;
                }
                if (!file.getFileExtension().equals("java")) {
                	continue;
                }
                if (file.getLocalTimeStamp() > startTime) {
                	file.setPersistentProperty(
                	    JaxWsWorkspaceManager.WSDL_PROPERTY, wsdlUrl);
                	LOG.debug("attach wsdl property for "+ file.getLocation().toOSString());
                }
			}
		} catch (Exception e) {
			LOG.error("Error during tag generated java file", e);
		}
    }
    
    @SuppressWarnings("unchecked")
    private String prepareMergeTmpDir(String oldDir, Hashtable genParams) {
        String mergeDir = oldDir + File.separator + TEMP_DIR;
        File file = new File(mergeDir);
        if (file.exists()) {
            file.delete();
        }
        file.mkdir();
        genParams.put(IWsdlToJavaGenerator.GEN_OUTPUTDIR, mergeDir);
        return mergeDir;
    }
    
    public void mergeDir(String oldDir, String newDir, boolean forceMerge) {
        LOG.debug("merge olddir:" + oldDir + " newDir:" + newDir);
        File newDirFile = null;
        File oldDirFile = null;
        try {
            newDirFile = new File(newDir);
            oldDirFile = new File(oldDir);
            if (!oldDirFile.exists()) {
                oldDirFile.mkdir();
            }

            Hashtable<String, File> oldFilesTab = new Hashtable<String, File>();
            if (oldDirFile.listFiles() != null) {
                for (File oldFile : oldDirFile.listFiles()) {
                    oldFilesTab.put(oldFile.getName(), oldFile);
                }
            }

            if (newDirFile.listFiles() == null) {
            	return;
            }
            for (File file : newDirFile.listFiles()) {
                String targetFile = oldDir + File.separator + file.getName();

                if (file.isDirectory()) {
                    mergeDir(targetFile, file.getAbsolutePath(), forceMerge);
                    continue;
                }
                if (oldFilesTab.containsKey(file.getName())
                		&& needMerge(file.getName()) 
                		&& forceMerge) {
                    //need to merge those two files
                    LOG.debug("merge file:" + file.getAbsolutePath() + " and "
                            + targetFile);
                    MergeUtils.mergeJavaFile(targetFile,
                            file.getAbsolutePath(), targetFile);
                } else {
                    // need to copy the new file
                    FileUtils.copyFile(file.getAbsolutePath(), oldDir);
                }
                file.delete();
            }
            newDirFile.delete();
        } catch (Exception e) {
            e.printStackTrace();
            LOG.error("error during merge dir", e);
        }
    }
    
    /**
     * we only merge Impl, Client, and Server files.
     * do not merge all generated code
     * @param fileName
     * @return
     */
    private boolean needMerge(String fileName) {
       if (fileName.endsWith("Impl.java")) {
    	   return true;
       } else if (fileName.endsWith("Client.java")) {
    	   return true;
       } else if (fileName.endsWith("Server.java")) {
    	   return true;
       }
       return false;
    }
    
    public boolean isNeedMerge() {
        return needMerge;
    }
    
    public void setNeedMerge(boolean value) {
        needMerge = value;   
    }

}
