/**********************************************************************
 * Copyright (c) 2005 Scapa Technologies Limited and others
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * Scapa Technologies Limited - Initial API and implementation
 **********************************************************************/

package org.eclipse.stp.b2j.ui.publicapi.example;

import org.eclipse.core.runtime.IPath;
import org.eclipse.stp.b2j.core.publicapi.engine.WorkbenchBPELEngine;
import org.eclipse.stp.b2j.core.publicapi.program.BPELProgram;
import org.eclipse.stp.b2j.core.publicapi.transport.session.SessionAddress;
import org.eclipse.stp.b2j.ui.publicapi.program.WorkbenchBPELProgram;

/**
 * This is a public class which demonstrates the usage of the b2j BPEL engine APIs
 * 
 * @author amiguel
 *
 */
public class WorkbenchAPIExample {

	/**
	 * This method demonstrates how to run a BPEL process on a local engine
	 * The process will run within this JVM and no external daemon process is required
	 * @param path the path of the BPEL source file
	 * @throws Exception if an error occurs while setting up or running this BPEL process
	 */
	private static void runLocalBpelProgram(IPath path) throws Exception {
		
		//Create a new BPEL program
		BPELProgram program = new WorkbenchBPELProgram(path);		

		//Create a new BPEL engine
		WorkbenchBPELEngine engine = new WorkbenchBPELEngine(program);
		
		//Run the program
		engine.runProgram();
		engine.waitForProgramCompletion();
	}
	
	/**
	 * This method demonstrates how to run a BPEL process distributed over a number of hosts
	 * The process will run spread out over the specified hosts so a b2j engine
	 * daemon must be running on all of the given hosts
	 * @param path the path to the BPEL source file
	 * @param hosts an array of hostnames or IPs to distribute the BPEL process over
	 * @throws Exception if an error occurs while setting up or running this BPEL process
	 */
	private static void runDistributedBpelProgram(IPath path, String[] hosts) throws Exception {
		
		//Create a new BPEL program
		BPELProgram program = new WorkbenchBPELProgram(path);		

		//Create an address to represent the workbench -> coordinator daemon link
		SessionAddress coord_daemon_address = new SessionAddress(
				"localhost",
				SessionAddress.TRANSPORT_PORT_ANY,
				SessionAddress.TRANSPORT_PORT_ANY,
				hosts[0],
				SessionAddress.TRANSPORT_PORT_ANY,
				SessionAddress.TRANSPORT_PORT_ANY
				);
		//HTTP, not HTTPS
		coord_daemon_address.setRequiresEncryption(false);
		//no password
		coord_daemon_address.setRequiresPassword(false);

		//Create an address to represent the workbench -> coordinator host link
		SessionAddress coord_address = new SessionAddress(
				"localhost",
				SessionAddress.TRANSPORT_PORT_ANY,
				SessionAddress.TRANSPORT_PORT_ANY,
				hosts[0],
				SessionAddress.TRANSPORT_PORT_ANY,
				SessionAddress.TRANSPORT_PORT_ANY
				);
		
		//Set the coordinator host
		program.setCoordinatorHost(coord_daemon_address,coord_address);
		
		//Add worker hosts
		for (int i = 0; i < hosts.length; i++) {

			//Create an address to represent the coordinator host -> work host [i] daemon link
			SessionAddress worker_daemon_address = new SessionAddress(
					"localhost",
					SessionAddress.TRANSPORT_PORT_ANY,
					SessionAddress.TRANSPORT_PORT_ANY,
					hosts[i],
					SessionAddress.TRANSPORT_PORT_ANY,
					SessionAddress.TRANSPORT_PORT_ANY
					);
			//HTTP, not HTTPS
			worker_daemon_address.setRequiresEncryption(false);
			//no password
			worker_daemon_address.setRequiresPassword(false);

			//Create an address to represent the coordinator host -> worker host [i] link
			SessionAddress worker_address = new SessionAddress(
					"localhost",
					SessionAddress.TRANSPORT_PORT_ANY,
					SessionAddress.TRANSPORT_PORT_ANY,
					hosts[i],
					SessionAddress.TRANSPORT_PORT_ANY,
					SessionAddress.TRANSPORT_PORT_ANY
					);
			
			//Add the worker host
			program.addWorkerHost(worker_daemon_address,worker_address);
		}
		
		//Create a new BPEL engine
		WorkbenchBPELEngine engine = new WorkbenchBPELEngine(program);
		
		//Run the program
		engine.runProgram();
		engine.waitForProgramCompletion();
	}
	
}