/**********************************************************************
 * Copyright (c) 2005 Scapa Technologies Limited and others
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * Scapa Technologies Limited - Initial API and implementation
 **********************************************************************/

package org.eclipse.stp.b2j.core.jengine.internal.compiler.bpel;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.PrintStream;
import java.net.URL;
import java.net.URLClassLoader;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Properties;
import java.util.Stack;
import java.util.jar.JarOutputStream;
import java.util.zip.ZipEntry;

import org.eclipse.stp.b2j.core.jengine.internal.api.JavaProgramFactory;
import org.eclipse.stp.b2j.core.jengine.internal.api.Program;
import org.eclipse.stp.b2j.core.jengine.internal.compiler.ScopeStack;
import org.eclipse.stp.b2j.core.jengine.internal.compiler.Switches;
import org.eclipse.stp.b2j.core.jengine.internal.compiler.TranslatorLog;
import org.eclipse.stp.b2j.core.jengine.internal.compiler.Util;
import org.eclipse.stp.b2j.core.jengine.internal.compiler.locks.LockModel;
import org.eclipse.stp.b2j.core.jengine.internal.compiler.locks.LockModelEquivalenceOptimiser;
import org.eclipse.stp.b2j.core.jengine.internal.compiler.locks.LockModelScopeOptimiser;
import org.eclipse.stp.b2j.core.jengine.internal.compiler.locks.LockSync;
import org.eclipse.stp.b2j.core.jengine.internal.compiler.wsdl.WSDLTranslator;
import org.eclipse.stp.b2j.core.jengine.internal.compiler.wsdlmap.BPELPartnerLinkRole;
import org.eclipse.stp.b2j.core.jengine.internal.compiler.wsdlmap.BPELPartnerLinkType;
import org.eclipse.stp.b2j.core.jengine.internal.compiler.wsdlmap.WSDLMap;
import org.eclipse.stp.b2j.core.jengine.internal.compiler.wsdlmap.WSDLMessage;
import org.eclipse.stp.b2j.core.jengine.internal.compiler.wsdlmap.WSDLOperation;
import org.eclipse.stp.b2j.core.jengine.internal.compiler.wsdlmap.WSDLPortType;
import org.eclipse.stp.b2j.core.jengine.internal.compiler.xmlns.NamespaceException;
import org.eclipse.stp.b2j.core.jengine.internal.compiler.xmlns.NamespaceTranslator;
import org.eclipse.stp.b2j.core.jengine.internal.compiler.xpath.XPATHTreeTranslator;
import org.eclipse.stp.b2j.core.jengine.internal.compiler.xsd.XSDTranslator;
import org.eclipse.stp.b2j.core.jengine.internal.compiler.xsdmap.XSDMap;
import org.eclipse.stp.b2j.core.jengine.internal.core.bpel.WSEndpointReference;
import org.eclipse.stp.b2j.core.jengine.internal.debug.DebugConstants;
import org.eclipse.stp.b2j.core.jengine.internal.extensions.wsdlbinding.internal.InternalBindingTranslator;
import org.eclipse.stp.b2j.core.jengine.internal.extensions.wsdlbinding.internal.XMLAccessible;
import org.eclipse.stp.b2j.core.jengine.internal.extensions.wsdlbinding.wsif.ports.DeploymentStrategyPort;
import org.eclipse.stp.b2j.core.jengine.internal.portgen.WsdlGenerator;
import org.eclipse.stp.b2j.core.jengine.internal.utils.TranslatorUtils;
import org.eclipse.stp.b2j.core.jengine.internal.utils.UIDPool;
import org.eclipse.stp.b2j.core.misc.internal.StringSplitter;
import org.eclipse.stp.b2j.core.misc.internal.XMLUtil;
import org.eclipse.stp.b2j.core.publicapi.B2jPlatform;
import org.eclipse.stp.b2j.core.publicapi.DependencyInfo;
import org.eclipse.stp.b2j.core.publicapi.JARDependency;
import org.eclipse.stp.b2j.core.publicapi.extension.wsdlbinding.WSDLBindingTranslator;
import org.eclipse.stp.b2j.core.publicapi.extension.wsdlbinding.WSDLBindingTranslatorException;
import org.eclipse.stp.b2j.core.publicapi.extension.wsdlbinding.XSDTypeTranslator;
import org.eclipse.stp.b2j.core.publicapi.importresolver.WsdlImportResolver;
import org.eclipse.stp.b2j.core.publicapi.jcompiler.CompilerUnit;
import org.eclipse.stp.b2j.core.publicapi.jcompiler.JavaCompiler;
import org.eclipse.stp.b2j.core.xml.internal.w3c.Document;
import org.eclipse.stp.b2j.core.xml.internal.w3c.DocumentBuilder;
import org.eclipse.stp.b2j.core.xml.internal.w3c.Element;
import org.eclipse.stp.b2j.core.xml.internal.w3c.Node;
import org.eclipse.stp.b2j.core.xml.internal.w3c.NodeList;
import org.eclipse.stp.b2j.core.xml.internal.w3c.Text;

/**
 * 
 * @author amiguel
 *
 * Translates a BPEL source file along with a set of WSDL and XSD files into
 * a valid engine Program
 */
public class BPELTranslator {

String DEFAULT_BPEL_FILENAME = "(Final BPEL Source)";
	
//class loader to load JAR dependancy classes
URLClassLoader deps_loader;	
	
String LOG_SOURCE = "BPEL Translator";
TranslatorLog log; 	
 
boolean debug = false;
 	
private static final String INSERT_LOCAL_VARIABLE_CREATION = "INSERT_LOCAL_VARIABLE_CREATION";
 	
String runner_imports = 
""+
"import java.util.*;\n"+
"import "+Util.jengine_package+".core.*;\n"+
"import "+Util.jengine_package+".core.api.*;\n"+
"import "+Util.jengine_package+".core.datapool.*;\n"+
"import "+Util.jengine_package+".message.Message;\n"+
//"import "+Util.jengine_package+".connector.*;\n"+
"import "+Util.jengine_package+".api.*;\n"+
"import "+Util.jengine_package+".core.bpel.*;\n"+
"import "+Util.jengine_package+".mutex.*;\n"+
"import "+Util.jengine_package+".extensions.wsdlbinding.wsif.ports.*;"+

"import "+Util.jengine_package+".utils.*;\n";
//"import "+Util.jengine_package+".utils.Program;\n\n";


String runner_template = 
"///////////////////////\n"+
"// SET BY REFLECTION //\n"+
"///////////////////////\n"+
"public "+Util.jengine_package+".core.RunnerInterface engine;\n"+
"public long runner_id;\n"+
"public Message runner_args;\n"+
"\n"+
"//////////////////////////////////////////////////\n"+
"// FOR LOCAL THREADS WHICH NEED TO SHARE MEMORY //\n"+
"//////////////////////////////////////////////////\n"+
"public void run() {\n"+
"	//for local, shared memory threading\n"+
"}\n"+
"\n"+
"//////////////////////////////////\n"+
"// ENTRY POINT FOR FIRST RUNNER //\n"+
"//////////////////////////////////\n"+
"SharedHashMap bpelThreadMap;\n"+
"SharedHashMap bpelDebugMap;\n"+
"SharedVariable bpelDebugSuspend;\n"+
"Message bpelDebugStack;\n"+
"Message bpelThreadIdList = new Message();\n"+
"int bpelThreadChildIndex = 1;\n"+
"String bpelThreadId = \"\";\n"+
"static volatile int flowHost = -1;\n"+
"public void engine_main() throws Exception {\n"+
"  //Run this compiled BPEL process\n"+
"  activity1();\n"+
"}\n\n";

//UIDPool portType_uidpool = new UIDPool();
//HashMap portType_uids = new HashMap();
UIDPool operation_uidpool = new UIDPool();
HashMap operation_uids = new HashMap();

/*
	private Integer getPortTypeUID(String portType) {
		if (!portType_uids.containsKey(portType)) {
			portType_uids.put(portType,new Integer(portType_uidpool.getUID()));
		}
		return (Integer)portType_uids.get(portType);
	}
*/
/*
	private Integer getOperationUID(String portType, String operation) {
		String key = portType+"###"+operation;
		if (!operation_uids.containsKey(key)) {
			operation_uids.put(key,new Integer(operation_uidpool.getUID()));
		}
		return (Integer)operation_uids.get(key);
	}
*/

static final int FIRST_THREAD_ID = 1;

//LockResolver scope_lock_resolver = new LockResolver();
//LockResolver assign_lock_resolver = new LockResolver();

//Debugger DBG = B2jPlugin.DBG;

XSDTypeTranslator[] codecs;
WSDLBindingTranslator[] bindings;

WSDLMap wsdlmap;
XSDMap xsdmap;

NamespaceTranslator nt;
Util compiler_util;

LockModel lock_model;

	public BPELTranslator(String program_prefix, Switches switches) {
		compiler_util = new Util(program_prefix,switches);
		nt = new NamespaceTranslator(program_prefix);

		scopes = new ScopeStack(compiler_util);
		flow_scopes = new ScopeStack(compiler_util);
	}

/////////////////////////////////////////////////////////
// BPEL PARTNER LINK STUFF
//

HashMap partnerLinks = new HashMap();

class BPELPartnerLink {
	String name;
	String qtype;
	String myRoleName;
	String partnerRoleName;
	String requiredBindingName;
	BPELPartnerLinkRole myRole;
	BPELPartnerLinkRole partnerRole;
	
	public BPELPartnerLink() {
	}
}

//
//
/////////////////////////////////////////////////////////


/////////////////////////////////////////////////////////
// BPEL ASSET LOCALISATION STUFF
// 

HashMap variable_assets = new HashMap();
HashMap semaphore_assets = new HashMap();

int bpelAssetID = 0;

class BPELAsset {
	static final int BPEL_VARIABLE = 0;
	static final int BPEL_LINK = 1;
	
	int id = 0;

	String name; //including scope
	String basename; //not including scope
	int type;
	
	String var_qtype;
	
	boolean isLocal = false;
	
	public BPELAsset(String name, String basename, String var_qtype, int type) {
		this.name = name;
		this.basename = basename;
		this.type = type;
		this.var_qtype = var_qtype;
		id = bpelAssetID++;
	}
	
	public boolean isLocal() {
		return isLocal;
	}
	
	public void setLocal(boolean b) {
		isLocal = isLocal || b;
	}
	
	public int getType() {
		return type;
	}
	
	public String getName() {
		return name;
	}
	
	public int getID() {
		return id;
	}
	
	public String getJavaRef() {
		return "bpelAssetId"+id;
	}
	
	public String getVariableQType() {
		return var_qtype;
	}
	
	ArrayList threads = new ArrayList(); //threads that reference this variable
}

public ArrayList getAllAssets() {
	ArrayList assets = new ArrayList();
	assets.addAll(variable_assets.values());
	assets.addAll(semaphore_assets.values());
	return assets;
}
public BPELAsset getOrCreateVariable(String name, String basename, String qtype) {
	BPELAsset asset = (BPELAsset)variable_assets.get(name);
	if (asset == null) {
		asset = new BPELAsset(name,basename,qtype,BPELAsset.BPEL_VARIABLE);
		variable_assets.put(name,asset);
	}
	return asset;
}
public BPELAsset getVariable(String name, String basename) {
	BPELAsset asset = (BPELAsset)variable_assets.get(name);
	return asset;
}
/**
 * Link a variable to the current thread
 * @param qname the name of the referenced variable
 */
public void addVariableRef(String basename, Node source) throws BPELTranslatorException {
	if (basename == null) return;
	if (basename.length() == 0) return;
	String name = scopes.getVariableName(basename,source);
	BPELThread thread = (BPELThread)bpel_threads.peek();
	BPELAsset asset = getVariable(name,basename);
	if (!asset.threads.contains(thread)) {
		asset.threads.add(thread);
	}
}
/**
 * Link some variables to the current thread
 * @param names the names of the referenced variables (ArrayList of Strings)
 */
public void addVariableRefs(ArrayList names, Node source) throws BPELTranslatorException {
	BPELThread thread = (BPELThread)bpel_threads.peek();
	for (int i = 0; i < names.size(); i++) {
		addVariableRef((String)names.get(i),source);
	}
}

public boolean canBeLocal(BPELAsset asset) {
	ArrayList threads = asset.threads;
	
	boolean singleThreaded = true;
	
	if (threads.size() == 1) {
		BPELThread thread = (BPELThread)threads.get(0);
		if (thread.multiple_copies) {
			//more than one copy thread accesses this asset
			singleThreaded = false;
		}
	} else if (threads.size() > 1) {
		//more than one thread accesses this asset
		singleThreaded = false;
	}
	
	if (!singleThreaded) {
		//more than one thread accesses this asset
		StringBuffer sb = new StringBuffer();
		sb.append("Asset "+asset.name+" cannot be local - ");
		for (int i = 0; i < threads.size(); i++) {
			sb.append(((BPELThread)threads.get(i)).id);
		}
		return false;
	} else {
		//only one thread accesses this asset
		return true;
	}
}

public boolean canBeConcurrentlyAccessed(BPELAsset asset) {
	ArrayList threads = asset.threads;
	
	HashMap flows = new HashMap();
	
	for (int i = 0; i < threads.size(); i++) {
		BPELThread thread = (BPELThread)threads.get(i);
		if (thread.flow != null) {
			BPELThread existing = (BPELThread)flows.get(thread.flow);
			if (existing != null) {
				//these threads are part of some similar parent flow
				//(they may run concurrently)
				
				if (!thread.hasParent(existing) && !existing.hasParent(thread)) {
					//neither of these threads is the parent of the other
					
					//this variable may be being accessed concurrently - so FALSE
					return true;
				}
			}
			flows.put(thread.flow,thread);
		}
	}
	
	return false;
}

// 
// BPEL ASSET LOCALISATION STUFF
/////////////////////////////////////////////////////////

/////////////////////////////////////////////////////////
// BPEL THREAD FLOW HIERARCHY
// 

int bpelThreadID = 0;
Stack bpel_threads = new Stack();

class BPELThread {
	int id = 0;
	boolean multiple_copies = false;
	BPELFlow flow;
	BPELState state;
	ArrayList children = new ArrayList();
	
	public BPELThread() {
		id = bpelThreadID++;
	}
	
	public BPELThread(BPELFlow flow) {
		id = bpelThreadID++;
		this.flow = flow;
		if (flow != null) {
			if (flow.thread != null) {
				flow.thread.children.add(this);
			}
		}
	}
	
	public ArrayList getChildren() {
		return children;
	}
	
	public BPELThread getParent() {
		if (flow == null) return null;
		
		return flow.thread;
	}
	
	public boolean hasParent(BPELThread thread) {
		//main thread
		if (flow == null) return false;
		
		if (flow.thread == thread) return true;
		
		return flow.thread.hasParent(thread);
	}
	
	ArrayList flows = new ArrayList();		
	
	ArrayList variableLockMethods = new ArrayList();
	ArrayList variableScopedNames = new ArrayList();
	
	public void addVariableLock(String methodName, String variableName) {
		variableLockMethods.add(methodName);
		variableScopedNames.add(variableName);
	}
}

class BPELFlow {
	BPELThread thread;
	public BPELFlow(BPELThread thread) {
		this.thread = thread;
	}
	ArrayList threads = new ArrayList();
}

// 
// END BPEL THREAD FLOW HIERARCHY
/////////////////////////////////////////////////////////

class BPELState implements CompilerUnit {
	String PREFIX = "JEngineProgram_";
	String POSTFIX = "";
	SimpleDateFormat sdf = new SimpleDateFormat("yyyyMMddHHmmssSSS");
	
	int SCOPE = 1;
	StringBuffer decl = new StringBuffer();
	StringBuffer cons = new StringBuffer();
	StringBuffer prog = new StringBuffer();

	String class_name;
	String method_name;

	ArrayList imports = new ArrayList();
	
	ArrayList substates = new ArrayList();

	HashMap marks = new HashMap();
	
	public BPELState() {
		method_name = getNextMethod();
		POSTFIX = sdf.format(new Date(System.currentTimeMillis()));
		class_name = PREFIX+POSTFIX;
	}

	public void mark(String markname) {
		marks.put(markname,new Integer(prog.length()));
	}
	
	public int getMark(String markname) {
		Integer tmp = (Integer)marks.get(markname);
		return tmp.intValue();
	}
	
	public String getPackageName() {
		//program file has no package
		return "";
	}
	
	public String getClassName() {
		return class_name;
	}

	public String getMethodName() {
		return method_name;
	}
	
	public BPELState newState() {
		BPELState state = new BPELState();
		substates.add(state);
		return state;	
	}

	private void addImport(String s) {
		imports.add(s);
	}
	private void addImports(String[] s) {
		for (int i = 0; i < s.length; i++) {
			imports.add(s[i]);
		}
	}
	
	private void appendDeclaration(String s) {
		decl.append(s).append("\n");	
	}

	private void appendConstructor(String s) {
		cons.append(s).append("\n");	
	}
	
	private void appendNoLine(String s) {
		for (int i = 0; i < SCOPE; i++) {
			prog.append("  ");	
		}
		prog.append(s);
	}

	private void appendLine(String s) {
		for (int i = 0; i < SCOPE; i++) {
			prog.append("  ");	
		}
		prog.append(s).append("\n");
	}
	
	private int getIndex() {
		return prog.length();
	}
	
	private void insertLine(String s, int index) {
		StringBuffer tmp = new StringBuffer();
		for (int i = 0; i < SCOPE; i++) {
			tmp.append("  ");	
		}
		tmp.append(s);
		prog.insert(index,tmp.toString());
	}
	
	private void insertNoLine(String s, int index) {
		StringBuffer tmp = new StringBuffer();
		for (int i = 0; i < SCOPE; i++) {
			tmp.append("  ");	
		}
		tmp.append(s);
		prog.insert(index,tmp.toString());
	}
	
	private void incrementScope() {
		SCOPE++;
	}
	private void decrementScope() {
		if (SCOPE >= 1) SCOPE--;	
	}
	
	public void importsToString(StringBuffer buf) {
		//append my imports
		for (int i = 0; i < imports.size(); i++) {
			buf.append("import ").append((String)imports.get(i)).append(";\n");
		}
		
		//append my childrens imports
		for (int i = 0; i < substates.size(); i++) {
			BPELState substate = (BPELState)substates.get(i);
			substate.importsToString(buf);	
		}
	}
	
	public void declToString(StringBuffer buf) {

		//append my declarations
		buf.append(decl.toString());

		//append my childrens declarations
		for (int i = 0; i < substates.size(); i++) {
			BPELState substate = (BPELState)substates.get(i);
			substate.declToString(buf);	
		}
	}
	
	public void consToString(StringBuffer buf) {

		//append my constructor code
		buf.append(cons.toString());

		//append my childrens constructor code
		for (int i = 0; i < substates.size(); i++) {
			BPELState substate = (BPELState)substates.get(i);
			substate.consToString(buf);	
		}
	}
	
	public void progToString(StringBuffer buf) {

		//append my logic
		buf.append("public void "+method_name+"() throws Exception {\n");
		buf.append(prog.toString());
		buf.append("}\n\n");

		//append my childrens' logic
		for (int i = 0; i < substates.size(); i++) {
			BPELState substate = (BPELState)substates.get(i);
			substate.progToString(buf);
		}
	}

	public String toString() {
		StringBuffer buf = new StringBuffer();
		
		buf.append(runner_imports);
		importsToString(buf);
		
		buf.append("//@@START_STRIP\n");
		buf.append("public class "+class_name+" {\n");
		buf.append("//@@END_STRIP\n");

		buf.append(runner_template);

		declToString(buf);
		buf.append("\n");

		buf.append("//PROGRAM INITIALISATION\n");
		buf.append("public void engine_init() throws Exception {\n");
//		buf.append("  if (false) {\n");
//		buf.append("    bpelThreadId = \""+FIRST_THREAD_ID+"\";\n");
//		buf.append("    bpelThreadIdList.append(bpelThreadId);\n");
		consToString(buf);
//		buf.append("    bpelThreadIdList.pop();\n");
//		buf.append("  }\n");
		buf.append("}\n");
		buf.append("\n");
		
		progToString(buf);
		buf.append("\n");
		buf.append("//@@START_STRIP\n");
		buf.append("}//end "+class_name+"\n");
		buf.append("//@@END_STRIP\n\n");
		
		return buf.toString();	
	}
}

ScopeStack scopes;
ScopeStack flow_scopes;

//StringBuffer decl = new StringBuffer();
//StringBuffer jprog = new StringBuffer();

int NSEQ = 1;

//int SCOPE = 1;

int METHOD = 1;

int UID = 1;
int TUID = 1;

int XPATHID = 1;

	private String getNextXPath() {
		return "xpath_expr"+XPATHID++;
	}

	private String peekNextXPath() {
		return "xpath_expr"+XPATHID;
	}

	private String getNextMethod() {
		return "activity"+METHOD++;
	}

	private String peekNextMethod() {
		return "activity"+METHOD;
	}
	
	private String getPartnerLinkName(String ncname) {
		return "bpel_plink_"+ncname;
	}

	private int nextUID() {
		return UID++;	
	}

	private int nextThreadUID() {
		return TUID++;
	}
	/*
	private String nodeContentsToJavaString(Node elem) throws BPELTranslatorException {
		StringBuffer text = new StringBuffer();

		NodeList list = elem.getChildNodes();
		for (int i = 0; i < list.getLength(); i++) {
			Node node = list.item(i);
			
			try {
				String nodeContents = XMLUtil.nodeToString(node);
				nodeContents = TranslatorUtils.toJavaString(nodeContents);
				text.append(nodeContents);
				
			} catch (Exception e) {
				throw new BPELTranslatorException(compiler_util,elem,e);
			}
		}
		
		return text.toString();
	}*/
	
	public BPELTranslator() {
	}
	
	public BPELCorrelation[] getCorrelations(Element msgActivity) {
		ArrayList bpel_correlations = new ArrayList();
		
		ArrayList correlationsList = Util.getAllElements(msgActivity);
		for (int i =0 ; i < correlationsList.size(); i++) {
			Element correlations = (Element)correlationsList.get(i);
			if (NamespaceTranslator.getName(correlations).equals("correlations")) {
				ArrayList correlationList = Util.getAllElements(correlations);
				for (int k = 0; k < correlationList.size(); k++) {
					Element correlation = (Element)correlationList.get(k);
					if (NamespaceTranslator.getName(correlation).equals("correlation")) {
						BPELCorrelation corr = new BPELCorrelation();
						corr.setncname = correlation.getAttribute("set");
						String initiate = correlation.getAttribute("initiate");
						if (initiate != null) {
							if (initiate.equalsIgnoreCase("true")) {
								corr.initiate = true;
							}
						}
						String pattern = correlation.getAttribute("pattern");
						if (pattern != null) {
							if (pattern.equals("in")) {
								corr.pattern = BPELCorrelation.BPEL_CORRELATION_IN;
							} else if (pattern.equals("out")) {
								corr.pattern = BPELCorrelation.BPEL_CORRELATION_OUT;
							} else if (pattern.equals("out-in")) {
								corr.pattern = BPELCorrelation.BPEL_CORRELATION_OUTIN;
							}
						}
					}
				}
			}
		}
		BPELCorrelation[] array = new BPELCorrelation[bpel_correlations.size()];
		bpel_correlations.toArray(array);
		return array;
	}
	
	public void translatePartnerLinks(BPELState state, Element links) throws BPELTranslatorException, NamespaceException {
		state.appendLine("//PARTNER LINKS");
		state.incrementScope();
		
		ArrayList list = Util.getAllElements(links,"partnerLink");
		for (int i = 0; i < list.size(); i++) {
			Element e = (Element)list.get(i);
			translatePartnerLink(state,e);
		}
		
		state.decrementScope();
		state.appendLine("//END PARTNER LINKS");
	}

	public void translatePartnerLink(BPELState state, Element link) throws BPELTranslatorException, NamespaceException {
		state.appendLine("//PARTNER LINK");
		
		nt.addNamespaces(link);
		
		BPELPartnerLink plink = new BPELPartnerLink();
		plink.name = link.getAttribute("name");
		plink.qtype = nt.qualify(link.getAttribute("partnerLinkType"),false);
		String myRoleName = link.getAttribute("myRole");
		String partnerRoleName = link.getAttribute("partnerRole");
		if (myRoleName == null) myRoleName = "";
		if (partnerRoleName == null) partnerRoleName = "";
		if (myRoleName.length() > 0) {
			plink.myRoleName = myRoleName;
		}
		if (partnerRoleName.length() > 0) {
			plink.partnerRoleName = partnerRoleName;
		}
		
		String useBindingName = null;
		
		Element useBinding = Util.getFirstElement(link,"useBinding");
		if (useBinding != null) {
			useBindingName = Util.getTextDirectlyUnder(useBinding);
			try {
				plink.requiredBindingName = nt.qualify(useBindingName,false);
			} catch (NamespaceException ee) {
				throw new BPELTranslatorException(compiler_util,useBinding,ee);
			}
		}
		
		partnerLinks.put(plink.name,plink);
		
//CREATEONDEMAND		state.appendDeclaration("static BPELPartnerLink "+getPartnerLinkName(plink.name)+" = new BPELPartnerLink();\n");
		state.appendLine("//Initialise the partner link");
		state.appendLine("BPELPartnerLink.setPartnerLink(\""+getPartnerLinkName(plink.name)+"\",new BPELPartnerLink());");
//		state.appendDeclaration("static BPELPartnerLink "+getPartnerLinkName(plink.name)+" = new BPELPartnerLink();\n");
//		state.appendDeclaration("//initialisation of partnerLink "+plink.name);
		
		BPELPartnerLinkType ptype = wsdlmap.getPartnerLinkType(plink.qtype);
		if (ptype == null) throw new BPELTranslatorException(compiler_util,link,"partner link type not found "+plink.qtype);
		
		BPELPartnerLinkRole myRole = null;
		BPELPartnerLinkRole partnerRole = null;
		
		if (myRoleName.length() > 0) {
			myRole = ptype.getRole(myRoleName);
			if (myRole == null) {
				throw new BPELTranslatorException(compiler_util,link,"myRole ("+myRoleName+") not found for partner link "+plink.name);
			}
			plink.myRole = myRole;
		}
		if (partnerRoleName.length() > 0) {
			partnerRole = ptype.getRole(partnerRoleName);
			if (partnerRole == null) {
				throw new BPELTranslatorException(compiler_util,link,"partnerRole ("+partnerRoleName+") not found for partner link "+plink.name);
			}
			plink.partnerRole = partnerRole;
		}
		if (plink.myRole == null && plink.partnerRole == null) throw new BPELTranslatorException(compiler_util,link,"Partner Link "+plink.name+" has no myRole and no partnerRole, must have either one or both");
		
		boolean translated = false;
		
		ArrayList exceptions = new ArrayList();
		StringBuffer errors = new StringBuffer();
		
		for (int i = 0; !translated && i < bindings.length; i++) {
			try {
				StringBuffer translation = new StringBuffer();
				if (myRole != null) {
					String pvar = "bpel_partner_link"+nextUID();

					translation.append("BPELPartnerLink "+pvar+" = BPELPartnerLink.getPartnerLink(\""+getPartnerLinkName(plink.name)+"\");\n");
//CREATEONDEMAND					translation.append(getPartnerLinkName(plink.name));
					translation.append(pvar);
					translation.append(".myRole = ");
					translation.append(bindings[i].translateDefaultEPR(nt,myRole.getQualifiedPortType(),plink.requiredBindingName));
					translation.append(";\n");
//					translation.append("BPELPartnerLink.setPartnerLink("+getPartnerLinkName(plink.name)+","+pvar+");");
				}
				if (partnerRole != null) {
					String pvar = "bpel_partner_link"+nextUID();

					translation.append("BPELPartnerLink "+pvar+" = BPELPartnerLink.getPartnerLink(\""+getPartnerLinkName(plink.name)+"\");\n");
//CREATEONDEMAND					translation.append(getPartnerLinkName(plink.name));
					translation.append(pvar);
					translation.append(".partnerRole = ");
					translation.append(bindings[i].translateDefaultEPR(nt,partnerRole.getQualifiedPortType(),plink.requiredBindingName));
					translation.append(";\n");
//					translation.append("BPELPartnerLink.setPartnerLink(\""+getPartnerLinkName(plink.name)+"\","+pvar+");");
				}
				
//				DBG.info("partner link "+plink.name+" translated by binding "+bindings[i].getClass().getName());
				log.logInfo(LOG_SOURCE,"partner link "+plink.name+" translated by binding "+bindings[i].getClass().getName());
				
				translated = true;
				
				state.appendLine(translation.toString());
/*				state.appendDeclaration("static {\n");
				state.appendDeclaration(translation.toString()+"\n");
				state.appendDeclaration("};\n");*/
				//has to be initialised by every program
//				state.appendLine(translation.toString());
				
			} catch (WSDLBindingTranslatorException e) {
				exceptions.add(e);
				errors.append("Binding "+i+" ("+bindings[i].getClass().getName()+") failed - "+e+"\n");
			}
		}
		
		if (!translated) {

//			for (int i = 0; i < exceptions.size(); i++) {
//				Exception tmp = (Exception)exceptions.get(i);
//				DBG.logVisibleError(tmp,"Binding translation failure: "+tmp.getMessage(),false);
//			}

			log.logWarning(LOG_SOURCE,"All "+bindings.length+" bindings failed to translate default EPR for partner link '"+plink.name+"' ("+compiler_util.getQNameFromQName(plink.qtype)+")");

			for (int i = 0; i < exceptions.size(); i++) {
				Exception tmp = (Exception)exceptions.get(i);
				log.logWarning(LOG_SOURCE,"Binding translation failure: "+tmp.getMessage());
			}
			
//			DBG.error("all bindings failed to translate default EPR for partner link ("+plink.name+" / "+plink.qtype+")\n"+errors+"\n");
			throw new BPELTranslatorException(compiler_util,link,"All "+bindings.length+" bindings failed to translate default EPR for partner link '"+plink.name+"' ("+compiler_util.getQNameFromQName(plink.qtype)+")");
		}
		
		nt.removeNamespaces(link);
		
	}
	
	public void translateVariables(BPELState state, Element variables, String debugVariablesId) throws BPELTranslatorException, NamespaceException {
		state.appendLine("//VARIABLES");
		state.incrementScope();
		//declaration of BPEL variables
		
		int uid = nextUID();
		
		String lock = "nonLocalScopeVariablesLock"+uid;
		String init = "nonLocalScopeVariables"+uid;
		String init_this = "this_nonLocalScopeVariables"+uid;
		String method = "initNonLocalScopeVariables"+uid;
		
		if (compiler_util.getSwitches().FLOW_GRAPH_ASSET_OPTIMISATION) {
//			state.appendDeclaration("static Object "+lock+" = new Object();");
//			state.appendDeclaration("static boolean "+init+" = false;");
//			state.appendDeclaration("static boolean "+init_this+" = false;");
			state.appendDeclaration("static Object "+lock+" = new Object();");
			//we make these NON STATIC because in the olden days each thread copy would have been in a separate
			//method and had its own one of these, but now we're mixing them we know that the thread creating
			//these will have its own ID and thus each runner needs to call it and create them.
			state.appendDeclaration("boolean "+init+" = false;");
			state.appendDeclaration("boolean "+init_this+" = false;");
			state.appendDeclaration("public void "+method+"() throws Exception {");
			state.appendDeclaration("  synchronized("+lock+") {");
			state.appendDeclaration("    if (!"+init+") {");
			state.appendDeclaration("      "+init+" = true;");
			state.appendDeclaration("      "+init_this+" = true;");
		}
		
		ArrayList list = Util.getAllElements(variables,"variable");
		for (int i = 0; i < list.size(); i++) {
			Element e = (Element)list.get(i);
			translateVariable(state,e,debugVariablesId);
		}

		if (compiler_util.getSwitches().FLOW_GRAPH_ASSET_OPTIMISATION) {
			state.appendDeclaration("    }");
			state.appendDeclaration("  }");
			state.appendDeclaration("}");
			
			state.appendLine("if (!"+init_this+") "+method+"();");
		}
		
		for (int i = 0; i < list.size(); i++) {
			Element variable = (Element)list.get(i);
			String variableName = variable.getAttribute("name");
			
			Element fromElem = Util.getFirstElement(variable,"from");
			if (fromElem != null) {
				//this variable is being assigned an initial value
	
				String ncname = NamespaceTranslator.getName(fromElem);
				String qname = fromElem.getTagName();
				
				String prefix = "";
				if (ncname.length() != qname.length()) {
					prefix = qname.substring(0,qname.length()-ncname.length());
				}
	
				//create the <to> element
				Document doc = variable.getOwnerDocument();
				
				Element toElem = doc.createElement(prefix+"to");
				toElem.setAttribute("variable",variableName);
				
				variable.appendChild(toElem);
				
				//we have just created the variable, therefore we DON'T need to translate as an assign
				//and do a lock on the variable, we can't possibly have any concurrent access at this point
	//			translateAssign();
				translateCopy(state,variable,0);
				
				variable.removeChild(toElem);
			}
		}
		
		state.decrementScope();
		state.appendLine("");
//		state.appendLine("//END VARIABLES");
	}

	public void translateVariable(BPELState state, Element variable, String debugVariablesId) throws BPELTranslatorException, NamespaceException {
		state.appendLine("//NEW VARIABLE");
		//declaration of a single process-global BPEL variable
		
		String name = variable.getAttribute("name");
		String messageType = variable.getAttribute("messageType");
		String schemaType = variable.getAttribute("type");
		String elementType = variable.getAttribute("element");
		
		if (name.length() == 0) {
			throw new BPELTranslatorException(compiler_util,variable,"no name specified for new variable: "+TranslatorUtils.toJavaString(variable));
		}
		
		String qmsg = null;
		if (messageType.length() > 0) {
			qmsg = nt.qualify(messageType,false);
			if (wsdlmap.getMessage(qmsg) == null) {
				throw new BPELTranslatorException(compiler_util,variable,"WSDL Message type "+messageType+" specified for variable but no WSDL message of that name found");
			}
		} else if (schemaType.length() > 0) {
			qmsg = nt.qualify(schemaType,false);
			if (xsdmap.getType(qmsg) == null) {
				throw new BPELTranslatorException(compiler_util,variable,"XSD type "+schemaType+" specified for variable but no XSD type of that name found");
			}
		} else if (elementType.length() > 0) {
			qmsg = nt.qualify(elementType,false);

			if (xsdmap.getRootElementTypeByQName(qmsg) == null) {
				throw new BPELTranslatorException(compiler_util,variable,"XSD Element "+elementType+" specified for variable but no XSD type mapping found for element of that name");
			}

			qmsg = xsdmap.getRootElementTypeByQName(qmsg).getQualifiedType();

		} else {
			throw new BPELTranslatorException(compiler_util,variable,"no WSDL message type (messageType), XML Schema type (type) or XML Schema element (element) specified for new variable: "+TranslatorUtils.toJavaString(variable));
		}

		scopes.addVariable(name,qmsg);

		lock_model.pushLock(compiler_util,variable,"IScopeLock:"+name);
		lock_model.pushLock(compiler_util,variable,"AssignLock:"+name);
		
		BPELAsset asset = getOrCreateVariable(scopes.getVariableName(name,variable),name,qmsg);

		if (compiler_util.getSwitches().FLOW_GRAPH_ASSET_OPTIMISATION) {
			//if this is a non-local variable, 
			//then add it to a synchronized call to the scope initialisation method
//			state.appendDeclaration("        if (!"+asset.getJavaRef()+") BPELVariable.createVariable(engine,\""+scopes.getVariable(name)+"\","+qmsg+".toEngineMessage(new "+qmsg+"()),"+asset.getJavaRef()+");");
			state.appendDeclaration("      BPELVariable.createVariable(engine,\""+scopes.getVariableName(name,variable)+"\","+qmsg+".toEngineMessage(new "+qmsg+"()),"+asset.getJavaRef()+");");
			
			//if this is a local variable then add it to the constructor so that all the programs create it 
			//(note BPELVariable is static so only one copy will actually be instantiated per host)
//			state.appendConstructor("  if ("+asset.getJavaRef()+") BPELVariable.createVariable(engine,\""+scopes.getVariable(name)+"\","+qmsg+".toEngineMessage(new "+qmsg+"()),"+asset.getJavaRef()+");");
			
		} else {
			state.appendLine("BPELVariable.createVariable(engine,\""+scopes.getVariableName(name,variable)+"\","+qmsg+".toEngineMessage(new "+qmsg+"()),"+asset.getJavaRef()+");");
			
			if (debug) {
				String msgDebugName = "bpel_variable_debug"+nextUID();
				
				state.appendLine("Message "+msgDebugName+" = new Message();");
				state.appendLine(msgDebugName+".append(\""+scopes.getVariableName(name,variable)+"\");");
				
				state.appendLine(debugVariablesId+".append("+msgDebugName+");");
			}
		}
		
	}
	/*
	public void translateCatch(BPELState state, Element ctch, String faultVariableName) throws BPELTranslatorException, NamespaceException {
		if (NamespaceTranslator.getName(ctch).equals("catchAll")) {
			state.appendLine("//CATCH ALL");

			state.appendLine("if (true) {");
		
		} else {
			String faultName = ctch.getAttribute("faultName");
			String faultVariable = ctch.getAttribute("faultVariable");
			String faultMessageType = ctch.getAttribute("faultMessageType");
			String faultElement = ctch.getAttribute("faultElement");

			if (ctch.hasAttribute("faultVariable")) {
				throw new BPELTranslatorException(compiler_util,ctch,"attribute 'faultVariable' on <catch> activity is currently not supported");
			}
			if (ctch.hasAttribute("faultMessageType")) {
				throw new BPELTranslatorException(compiler_util,ctch,"attribute 'faultMessageType' on <catch> activity is currently not supported");
			}
			if (ctch.hasAttribute("faultElement")) {
				throw new BPELTranslatorException(compiler_util,ctch,"attribute 'faultElement' on <catch> activity is currently not supported");
			}
			
			state.appendLine("//CATCH "+faultName+"");

			state.appendLine("if ("+faultVariableName+".getFaultName().equals(\""+TranslatorUtils.toJavaString(faultName)+"\")) {");
		}

		translateActivities(state,ctch);
		
		state.appendLine("}");
		state.appendLine("//END CATCH");
	}
	*/
	public void translatePreFaultHandlers(BPELState state, Element faultHandlers) throws BPELTranslatorException, NamespaceException {
		state.appendLine("//FAULT HANDLERS");
		
		state.appendLine("try {");
	}
	public void translatePostFaultHandlers(BPELState state, Element faultHandlers) throws BPELTranslatorException, NamespaceException {
		
		String faultVariableName = "bpel_fault_"+nextUID();
		
		state.appendLine("} catch (BPELFault "+faultVariableName+") {");
		
		ArrayList catches = Util.getAllElements(faultHandlers,"catch");
		Element catchAll = Util.getFirstElement(faultHandlers,"catchAll");

		boolean ifStarted = false;
		
		for (int i = 0; i < catches.size(); i++) {
			Element ctch = (Element)catches.get(i);
			
			String faultName = ctch.getAttribute("faultName");
			String faultVariable = ctch.getAttribute("faultVariable");
			String faultMessageType = ctch.getAttribute("faultMessageType");
			String faultElement = ctch.getAttribute("faultElement");

			if (ctch.hasAttribute("faultVariable")) {
				throw new BPELTranslatorException(compiler_util,ctch,"attribute 'faultVariable' on <catch> activity is currently not supported");
			}
			if (ctch.hasAttribute("faultMessageType")) {
				throw new BPELTranslatorException(compiler_util,ctch,"attribute 'faultMessageType' on <catch> activity is currently not supported");
			}
			if (ctch.hasAttribute("faultElement")) {
				throw new BPELTranslatorException(compiler_util,ctch,"attribute 'faultElement' on <catch> activity is currently not supported");
			}
			
			if (ifStarted) {
				state.appendNoLine("} else ");
			} else {
				ifStarted = true;
			}
			state.appendLine("if ("+faultVariableName+".getFaultName().equals(\""+TranslatorUtils.toJavaString(faultName)+"\")) {");
			state.appendLine("//CATCH "+faultName+"");

			translateActivities(state,ctch);
			
			state.appendLine("}");
			state.appendLine("//END CATCH");
		}
		
		if (catchAll != null) {
			
			if (ifStarted) {
				state.appendLine("} else {");
				state.appendLine("//CATCH ALL");
			} else {
				ifStarted = true;
				state.appendLine("//CATCH ALL");
				state.appendLine("if (true) {");
			}
			
			translateActivities(state,catchAll);

			state.appendLine("}");
			state.appendLine("//END CATCH");
		} else {
			if (ifStarted) {
				state.appendLine("} else {");
			}
			
			state.appendLine("throw "+faultVariableName+";");
			
			if (ifStarted) {
				state.appendLine("}");
			}
		}
		
		state.appendLine("}");
		
		state.appendLine("//END FAULT HANDLERS");
	}
	
	private String getStackTrace(Throwable t) {
		ByteArrayOutputStream os = new ByteArrayOutputStream();
		PrintStream ps = new PrintStream(os);   // printing destination
		t.printStackTrace(ps);
		return os.toString();
	}//end method
	
	public void translateReceive(BPELState state, Element receive) throws BPELTranslatorException, NamespaceException {
		state.appendLine("//RECEIVE");
		state.incrementScope();

		//BPEL receive
		
		String partnerLink = receive.getAttribute("partnerLink");
//		String portType = receive.getAttribute("portType");
		String operation = receive.getAttribute("operation");
		String inputVariable = receive.getAttribute("variable");
		String messageExchange = receive.getAttribute("messageExchange");

		if (!receive.hasAttribute("operation")) {
			throw new BPELTranslatorException(compiler_util,receive,"No 'operation' attribute found");
		}
		if (!receive.hasAttribute("partnerLink")) {
			throw new BPELTranslatorException(compiler_util,receive,"No 'partnerLink' attribute found");
		}
		
		if (messageExchange == null) {
			//default is empty message exchange
			messageExchange = "";
		}

		String createInstance = receive.getAttribute("createInstance");
		
		if (createInstance.equals("yes")) {
			throw new BPELTranslatorException(compiler_util,receive,"createInstance=\"yes\" is not supported");
		}

		addVariableRef(inputVariable,receive);
		
		BPELPartnerLink plink = (BPELPartnerLink)partnerLinks.get(partnerLink);
		if (plink == null) {
			throw new BPELTranslatorException(compiler_util,receive,"No partnerLink '"+partnerLink+"' found");
		}
		//receive means myRole
		BPELPartnerLinkRole prole = plink.myRole;
		String portType = prole.getQualifiedPortType();
		
		//
		// Get port and operation information
		//
		WSDLPortType wsdl_pt = wsdlmap.getPortType(portType); 
		if (wsdl_pt == null) {
			throw new BPELTranslatorException(compiler_util,receive,"No portType '"+portType+"' found");	
		}
		WSDLOperation wsdl_op = (WSDLOperation)wsdl_pt.name_to_operation.get(operation);
		if (wsdl_op == null) {
			throw new BPELTranslatorException(compiler_util,receive,"No operation '"+operation+"' found in portType '"+portType+"'");	
		}
		String input_type = (String)wsdl_op.message_types.get("input");

		if (inputVariable == null || inputVariable.length() == 0) {
			throw new BPELTranslatorException(compiler_util,receive,"no input variable specified for receive (port "+portType+" operation "+operation+")");
		}
		if (input_type == null) {
			throw new BPELTranslatorException(compiler_util,receive,"no input type found for receive (port "+portType+" operation "+operation+")");
		}
		String qtype = scopes.getVariableQtype(inputVariable, receive);
		if (!input_type.equals(qtype)) {
			throw new BPELTranslatorException(compiler_util,receive,"receive variable has type "+compiler_util.getQNameFromQName(qtype)+" but operation requires different WSDL message type "+compiler_util.getQNameFromQName(input_type));
		}
		
		String partnerLinkFieldName = "bpel_partner_link"+nextUID();
//CREATEONDEMAND		String partnerLinkFieldName = getPartnerLinkName(plink.name);
		//receive means myRole
		String eprFieldName = partnerLinkFieldName+".myRole";
		
		//generate IDs to hold the correlation sets
		int correlationSetID = nextUID();
		String correlationSetFieldName = "bpel_receive_correlation_"+correlationSetID;
		
		//get correlation sets element
		Element correlations = Util.getFirstElement(receive,"correlations");
	
		String inputMsgFieldName = "";
		int inputID = nextUID();
		
		if (inputVariable.length() > 0) {
			inputMsgFieldName="param"+inputID;
		}

		String invoc = null;
		
		WSDLBindingTranslator used_binding = null;

		StringBuffer problems = new StringBuffer();
		
		//try to get a standard binding to translate this invoke
		//pass in unqualified names to the binding translator
		for (int i = 0; i < bindings.length; i++) {
			try {
				invoc = bindings[i].translateReceive(nt,portType,plink.requiredBindingName,operation,messageExchange,eprFieldName,correlationSetFieldName,inputMsgFieldName);
				used_binding = bindings[i];
				
				break;
			} catch (WSDLBindingTranslatorException e) {
				problems.append(getStackTrace(e));
//				System.err.println(e);
//				e.printStackTrace();
			}	
		}
		
		if (used_binding == null) {
//DBG			DBG.error("no binding translator found for invocation.  partnerLink="+partnerLink+", operation="+operation+"\n"+problems.toString());
			throw new BPELTranslatorException(compiler_util,receive,"no binding translator found for invocation.  partnerLink="+partnerLink+", operation="+operation);
		}
		
		if (used_binding instanceof InternalBindingTranslator) {

			//populate the BPEL Partner Link field
			state.appendLine("BPELPartnerLink "+partnerLinkFieldName+" = BPELPartnerLink.getPartnerLink(\""+getPartnerLinkName(plink.name)+"\");");

			//we have an internal binding, the field names we give it map to engine Messages fetched from variables
			//if it likes, it can convert these into engine XSD types
			//fetch the input variable to a local object
			
			//create a new Message for the binding to populate
			state.appendLine("Message "+inputMsgFieldName+" = new Message();");

			if (correlations != null) {
				throw new BPELTranslatorException(compiler_util,receive,"<correlation> not supported for this binding");
			}

			//compile in the bit where the binding populates the Message
			if (invoc.endsWith("\n")) {
				state.appendNoLine(invoc);
			} else {
				state.appendLine(invoc);
			}

			//store the Message populated by the binding receive
			state.appendLine("BPELVariable.setMessage(engine,\""+scopes.getVariableName(inputVariable,receive)+"\","+inputMsgFieldName+");");
			
		} else {
			//we have a normal binding, the field names we give it map to engine XSD representations
			
			//populate the BPEL Partner Link field
			state.appendLine("BPELPartnerLink "+partnerLinkFieldName+" = BPELPartnerLink.getPartnerLink(\""+getPartnerLinkName(plink.name)+"\");");
			
			//create a new XSD type for the binding to populate
			state.appendLine(input_type+" "+inputMsgFieldName+" = new "+input_type+"();");
		
			state.appendLine("BPELCorrelationSet "+correlationSetFieldName+" = null;");
			
			//compile in the bit where the binding populates the XSD type
			if (invoc.endsWith("\n")) {
				state.appendNoLine(invoc);
			} else {
				state.appendLine(invoc);
			}

			if (correlations != null) {
				translateCorrelations(state,correlations,correlationSetFieldName,inputMsgFieldName,input_type,false,false);
			}

			//store the XSD type populated by the binding receive
			state.appendLine("BPELVariable.setMessage(engine,\""+scopes.getVariableName(inputVariable,receive)+"\","+input_type+".toEngineMessage("+inputMsgFieldName+"));");
		}
		
		state.decrementScope();
		state.appendLine("");
		state.appendLine("//END RECEIVE");
	}	
	
	public void translateReply(BPELState state, Element reply) throws BPELTranslatorException, NamespaceException {
		state.appendLine("//REPLY");
		state.incrementScope();

		//BPEL receive
		
		String partnerLink = reply.getAttribute("partnerLink");
//		String portType = reply.getAttribute("portType");
		String operation = reply.getAttribute("operation");
		String messageExchange = reply.getAttribute("messageExchange");
		String outputVariable = reply.getAttribute("variable");
		//TODO implement faultName for BPEL reply
		String faultName = reply.getAttribute("faultName");

		if (!reply.hasAttribute("operation")) {
			throw new BPELTranslatorException(compiler_util,reply,"No 'operation' attribute found");
		}
		if (!reply.hasAttribute("partnerLink")) {
			throw new BPELTranslatorException(compiler_util,reply,"No 'partnerLink' attribute found");
		}
		
		if (messageExchange == null) {
			messageExchange = "";
		}
		
		addVariableRef(outputVariable,reply);
		
		BPELPartnerLink plink = (BPELPartnerLink)partnerLinks.get(partnerLink);
		if (plink == null) {
			throw new BPELTranslatorException(compiler_util,reply,"No partnerLink '"+partnerLink+"' found");
		}
		//reply means myRole
		BPELPartnerLinkRole prole = plink.myRole;
		String portType = prole.getQualifiedPortType();
		
		//
		// Get port and operation information
		//
		WSDLPortType wsdl_pt = wsdlmap.getPortType(portType); 
		if (wsdl_pt == null) {
			throw new BPELTranslatorException(compiler_util,reply,"No portType '"+portType+"' found");	
		}
		WSDLOperation wsdl_op = (WSDLOperation)wsdl_pt.name_to_operation.get(operation);
		if (wsdl_op == null) {
			throw new BPELTranslatorException(compiler_util,reply,"No operation '"+operation+"' found in portType '"+portType+"'");	
		}
		String output_type = (String)wsdl_op.message_types.get("output");

		if (outputVariable == null || outputVariable.length() == 0) {
			throw new BPELTranslatorException(compiler_util,reply,"no input variable specified for reply  (port "+portType+" operation "+operation+")");
		}
		if (output_type == null) {
			throw new BPELTranslatorException(compiler_util,reply,"no input type found for reply (port "+portType+" operation "+operation+")");
		}
		String qtype = scopes.getVariableQtype(outputVariable, reply);
		if (!output_type.equals(qtype)) {
			throw new BPELTranslatorException(compiler_util,reply,"reply variable has type "+compiler_util.getQNameFromQName(qtype)+" but operation requires different WSDL message type "+compiler_util.getQNameFromQName(output_type));
		}
		
		String partnerLinkFieldName = "bpel_partner_link"+nextUID();
//CREATEONDEMAND		String partnerLinkFieldName = getPartnerLinkName(plink.name);
		//reply means myRole
		String eprFieldName = partnerLinkFieldName+".myRole";
		
		//generate IDs to hold the correlation sets
		int correlationSetID = nextUID();
		String correlationSetFieldName = "bpel_reply_correlation_"+correlationSetID;
		
		//get correlation sets element
		Element correlations = Util.getFirstElement(reply,"correlations");
	
	
		String outputMsgFieldName = "";
		int outputID = nextUID();
		
		if (outputVariable.length() > 0) {
			outputMsgFieldName="ret"+outputID;
		}

		String invoc = null;
		
		WSDLBindingTranslator used_binding = null;

		StringBuffer problems = new StringBuffer();
		
		//try to get a standard binding to translate this reply
		for (int i = 0; i < bindings.length; i++) {
			try {
				invoc = bindings[i].translateReply(nt,portType,plink.requiredBindingName,operation,messageExchange,eprFieldName,correlationSetFieldName,outputMsgFieldName);
				used_binding = bindings[i];
				
				break;
			} catch (WSDLBindingTranslatorException e) {
				problems.append(getStackTrace(e));
//				System.err.println(e);
//				e.printStackTrace();
			}	
		}
		
		if (used_binding == null) {
//DBG			DBG.error("no binding translator found for invocation.  partnerLink="+partnerLink+", operation="+operation+"\n"+problems.toString());
			throw new BPELTranslatorException(compiler_util,reply,"no binding translator found for invocation.  partnerLink="+partnerLink+", operation="+operation);
		}
		
		if (used_binding instanceof InternalBindingTranslator) {
			//we have an internal binding, the field names we give it map to engine Messages fetched from variables
			//if it likes, it can convert these into engine XSD types

			state.appendLine("BPELPartnerLink "+partnerLinkFieldName+" = BPELPartnerLink.getPartnerLink(\""+getPartnerLinkName(plink.name)+"\");");
			
			//fetch the Message variable for the binding to use in the reply
			state.appendLine("Message "+outputMsgFieldName+" = BPELVariable.getMessage(engine,\""+scopes.getVariableName(outputVariable,reply)+"\");");

			if (correlations != null) {
				throw new BPELTranslatorException(compiler_util,reply,"<correlation> not supported for this binding");
			}

			//compile in the bit where the binding replies using the fetched Message
			if (invoc.endsWith("\n")) {
				state.appendNoLine(invoc);
			} else {
				state.appendLine(invoc);
			}
						
		} else {
			//we have a normal binding, the field names we give it map to engine XSD representations
			
			state.appendLine("BPELPartnerLink "+partnerLinkFieldName+" = BPELPartnerLink.getPartnerLink(\""+getPartnerLinkName(plink.name)+"\");");
			
			//fetch the XSD type variable for the binding to use in the reply
			state.appendLine(output_type+" "+outputMsgFieldName+" = ("+output_type+") "+output_type+".fromEngineMessage(BPELVariable.getMessage(engine,\""+scopes.getVariableName(outputVariable,reply)+"\"));");

			state.appendLine("BPELCorrelationSet "+correlationSetFieldName+" = null;");
			
			if (correlations != null) {
				translateCorrelations(state,correlations,correlationSetFieldName,outputMsgFieldName,output_type,true,false);
			}

			//compile in the bit where the binding replies using the fetched XSD type
			if (invoc.endsWith("\n")) {
				state.appendNoLine(invoc);
			} else {
				state.appendLine(invoc);
			}
		}
		
		state.decrementScope();
		state.appendLine("");
		state.appendLine("//END REPLY");
	}
	
	public void translateCorrelations(BPELState state, Element correlations, String correlationSetFieldName, String xpathAccessibleFieldName, String xpathAccessibleType, boolean outgoing, boolean isInvocation) throws BPELTranslatorException, NamespaceException {
		ArrayList list = Util.getAllElements(correlations,"correlation");
		
		boolean merge = false;
		
		for (int i = 0; i < list.size(); i++) {
			Element correlation = (Element)list.get(i);
			String set = correlation.getAttribute("set");
			String initiate = correlation.getAttribute("initiate");
			String pattern = correlation.getAttribute("pattern");
			
			if (initiate == null) {
				initiate = "no";
			}
			if (initiate.length() == 0) {
				initiate = "no";
			}
			
			boolean translate = false;
			
			if (!isInvocation) {
				//translate this correlation
				translate = true;
				
			} else {
				if (pattern == null) {
					throw new BPELTranslatorException(compiler_util,correlation,"<correlation> on <invoke> activity must specify a pattern (in|out|out-in)");
				}
				
				if (pattern.equals("in")) {
					if (!outgoing) {
						//translate this correlation
						translate = true;
					}
				} else if (pattern.equals("out")) {
					if (outgoing) {
						//translate this correlation
						translate = true;
					}
				} else if (pattern.equals("out-in")) {
					//translate this correlation
					translate = true;
					
					if (outgoing) {
						//ok
					} else {
						//if we initiated this on the outgoing call, dont initiate it again on the incoming call
						if (initiate.equals("yes")) {
							initiate="no";
						}
					}
				} else {
					throw new BPELTranslatorException(compiler_util,correlation,"'pattern' attribute of <correlation> must specify one of 'in', 'out' or 'out-in'");
				}
			}
			
			if (translate) {
				if (xpathAccessibleFieldName.length() == 0) {
					if (isInvocation) {
						if (outgoing) {
							throw new BPELTranslatorException(compiler_util,correlations,"Correlation set with pattern=\"out\" specified but no corresponding inputVariable specified");
						} else {
							throw new BPELTranslatorException(compiler_util,correlations,"Correlation set with pattern=\"in\" specified but no corresponding outputVariable specified");
						}
					} else {
						throw new BPELTranslatorException(compiler_util,correlations,"Correlation set specified but no corresponding variable specified");
					}
				}

				if (initiate.equals("yes")) { 

					if (merge) {
						state.appendLine(correlationSetFieldName+".merge(");
					} else {
						state.appendLine(correlationSetFieldName+" = ");
					}
					
						state.appendLine("BPELCorrelationSet.createOrError(engine,\""+scopes.getCorrelationSetName(set,correlation)+"\",new "+xpathAccessibleType+"(),");
						state.incrementScope();
	
							String[] propertyNames = scopes.getCorrelationSetProperties(set,correlation);
							state.appendLine("new BPELCorrelationToken[]{");
							for (int k = 0; k < propertyNames.length; k++) {
								state.appendLine("  new BPELCorrelationToken(\""+propertyNames[k]+"\","+xpathAccessibleFieldName+"),");
							}
							state.appendLine("}");
						
						state.decrementScope();
						state.appendLine(")");
					
					if (merge) {
						state.appendLine(")");
					}
					
					state.appendLine(";");
					
				} else if (initiate.equals("no")) {
					
					if (merge) {
						state.appendLine(correlationSetFieldName+".merge(");
					} else {
						state.appendLine(correlationSetFieldName+" = ");
					}
					
						state.appendLine("BPELCorrelationSet.getOrError(engine,\""+scopes.getCorrelationSetName(set,correlation)+"\")");
					
					if (merge) {
						state.appendLine(")");
					}

					state.appendLine(";");

				} else if (initiate.equals("join")) {

					if (merge) {
						state.appendLine(correlationSetFieldName+".merge(");
					} else {
						state.appendLine(correlationSetFieldName+" = ");
					}
						
						state.appendLine("BPELCorrelationSet.getOrCreate(engine,\""+scopes.getCorrelationSetName(set,correlation)+"\",new "+xpathAccessibleType+"(),");
						state.incrementScope();
						
							String[] propertyNames = scopes.getCorrelationSetProperties(set,correlation);
							state.appendLine("new BPELCorrelationToken[]{");
							for (int k = 0; k < propertyNames.length; k++) {
								state.appendLine("  new BPELCorrelationToken(\""+propertyNames[k]+"\","+xpathAccessibleFieldName+"),");
							}
							state.appendLine("}");
	
						state.decrementScope();
						state.appendLine(")");
	
					if (merge) {
						state.appendLine(")");
					}

					state.appendLine(";");
					
				} else {
					throw new BPELTranslatorException(compiler_util,correlation,"'initiate' attribute of <correlation> must specify one of 'yes', 'no' or 'join'");
				}
				
				merge = true;
			}
		}
	}
	
	public void translateInvoke(BPELState state, Element invoke) throws BPELTranslatorException, NamespaceException {
		state.appendLine("//INVOKE");
		state.incrementScope();

		//BPEL invoke - similar to a function call
		//request-response of messages
		
		String partnerLink = invoke.getAttribute("partnerLink");
//		String portType = invoke.getAttribute("portType");
		String operation = invoke.getAttribute("operation");
		String inputVariable = invoke.getAttribute("inputVariable");
		String outputVariable = invoke.getAttribute("outputVariable");

		if (!invoke.hasAttribute("operation")) {
			throw new BPELTranslatorException(compiler_util,invoke,"No 'operation' attribute found");
		}
		if (!invoke.hasAttribute("partnerLink")) {
			throw new BPELTranslatorException(compiler_util,invoke,"No 'partnerLink' attribute found");
		}
		
		addVariableRef(inputVariable,invoke);
		addVariableRef(outputVariable,invoke);
		
		BPELPartnerLink plink = (BPELPartnerLink)partnerLinks.get(partnerLink);
		if (plink == null) {
			throw new BPELTranslatorException(compiler_util,invoke,"No partnerLink '"+partnerLink+"' found");
		}
		//invoke means partnerRole
		BPELPartnerLinkRole prole = plink.partnerRole;
		String portType = prole.getQualifiedPortType();
		
		WSDLPortType wsdl_pt = wsdlmap.getPortType(portType); 
		if (wsdl_pt == null) {
			throw new BPELTranslatorException(compiler_util,invoke,"No portType '"+portType+"' found");	
		}
		WSDLOperation wsdl_op = (WSDLOperation)wsdl_pt.name_to_operation.get(operation);
		if (wsdl_op == null) {
			throw new BPELTranslatorException(compiler_util,invoke,"No operation '"+operation+"' found in portType '"+portType+"'");	
		}
		String input_type = (String)wsdl_op.message_types.get("input");
		String output_type = (String)wsdl_op.message_types.get("output");

		if (inputVariable.length() > 0 && input_type == null) {
			throw new BPELTranslatorException(compiler_util,invoke,"inputVariable specified for invoke but no input found for that operation (port "+portType+" operation "+operation+")");
		}
		if (outputVariable.length() > 0 && output_type == null) {
			throw new BPELTranslatorException(compiler_util,invoke,"outputVariable specified for invoke but no output found for that operation (port "+portType+" operation "+operation+")");
		}
		
		if (input_type != null) {
			String qtype = scopes.getVariableQtype(inputVariable, invoke);
			if (!input_type.equals(qtype)) {
				throw new BPELTranslatorException(compiler_util,invoke,"invoke inputVariable has type "+compiler_util.getQNameFromQName(qtype)+" but operation requires different WSDL message type "+compiler_util.getQNameFromQName(input_type));
			}
		}
		if (output_type != null) {
			String qtype = scopes.getVariableQtype(outputVariable, invoke);
			if (!output_type.equals(qtype)) {
				throw new BPELTranslatorException(compiler_util,invoke,"invoke outputVariable has type "+compiler_util.getQNameFromQName(qtype)+" but operation requires different WSDL message type "+compiler_util.getQNameFromQName(output_type));
			}
		}
		
		if (wsdl_op.getType() == WSDLOperation.TYPE_REQUEST_RESPONSE) {
			if (!invoke.hasAttribute("inputVariable")) {
				throw new BPELTranslatorException(compiler_util,invoke,"no inputVariable attribute specified for WSDL request-response operation");
			}
			if (!invoke.hasAttribute("outputVariable")) {
				throw new BPELTranslatorException(compiler_util,invoke,"no outputVariable attribute specified for WSDL request-response operation");
			}
		}
		if (wsdl_op.getType() == WSDLOperation.TYPE_ONEWAY) {
			if (!invoke.hasAttribute("inputVariable")) {
				throw new BPELTranslatorException(compiler_util,invoke,"no inputVariable attribute specified for WSDL one-way operation");
			}
		}

		String partnerLinkFieldName = "bpel_partner_link"+nextUID();
//CREATEONDEMAND		String partnerLinkFieldName = getPartnerLinkName(plink.name);
		String eprFieldName = partnerLinkFieldName+".partnerRole";
		
		//generate IDs to hold the correlation sets
		int correlationSetID = nextUID();
		String correlationSetFieldNameIn = "bpel_invoke_correlation_in_"+correlationSetID;
		String correlationSetFieldNameOut = "bpel_invoke_correlation_out_"+correlationSetID;
		
		//get correlation sets element
		Element correlations = Util.getFirstElement(invoke,"correlations");
		
		
		String inputMsgFieldName = "";
		int inputID = nextUID();
		
		String outputMsgFieldName = "";
		int outputID = nextUID();

		if (inputVariable.length() > 0) {
			inputMsgFieldName="param"+inputID;
		}

		if (outputVariable.length() > 0) {
			outputMsgFieldName="ret"+outputID;
		}

		String invoc = null;
		
		WSDLBindingTranslator used_binding = null;

		ArrayList exceptions = new ArrayList();
		StringBuffer problems = new StringBuffer();
		
		//try to get a standard binding to translate this invoke
		//pass in unqualified names to the binding translator
		for (int i = 0; i < bindings.length; i++) {
			try {
				invoc = bindings[i].translateInvocation(nt,portType,plink.requiredBindingName,operation,eprFieldName,correlationSetFieldNameIn,correlationSetFieldNameOut,inputMsgFieldName,outputMsgFieldName);
				used_binding = bindings[i];
				
				break;
			} catch (WSDLBindingTranslatorException e) {
				exceptions.add(e);
				problems.append(getStackTrace(e));
//				System.err.println(e);
//				e.printStackTrace();
			}	
		}
		
		if (used_binding == null) {
			for (int i = 0; i < exceptions.size(); i++) {
				Exception tmp = (Exception)exceptions.get(i);
				log.logWarning(LOG_SOURCE,"Binding translation failure: "+tmp.getMessage());
//DBG				DBG.logVisibleError(tmp,"Binding translation failure: "+tmp.getMessage(),false);
			}
//DBG			DBG.error("no binding translator found for invocation.  partnerLink="+partnerLink+", operation="+operation+"\n"+problems.toString());
			throw new BPELTranslatorException(compiler_util,invoke,"no binding translator found for invocation (see error log for more info).  partnerLink="+partnerLink+", operation="+operation);
		}
		
		//
		// translate inline fault handlers
		//
		translatePreFaultHandlers(state,invoke);		
		
		if (used_binding instanceof InternalBindingTranslator) {
			//we have an internal binding, the field names we give it map to engine Messages fetched from variables
			//if it likes, it can convert these into engine XSD types
			
			state.appendLine("BPELPartnerLink "+partnerLinkFieldName+" = BPELPartnerLink.getPartnerLink(\""+getPartnerLinkName(plink.name)+"\");");
			
/*		
			if (outputVariable.length() > 0) {
				//invoke with return
				state.appendNoLine("BPELVariable.setMessage(engine,\""+scopes.getVariable(outputVariable)+"\",");
			}
			
			state.appendNoLine("engine.sendAndReceiveMessage(");

//			if (inputVariable.length() > 0) {
					//invoke with parameters (fetch the message from a BPELVariable)
					state.appendNoLine(""+conversationID+",BPELVariable.getMessage(engine,\""+scopes.getVariable(inputVariable)+"\")");
//			} else {
//				//invoke with no parameters (empty message)
//				state.appendNoLine(""+conversationID+",new Message(0)");
//			}

			state.appendNoLine(",");
			state.appendNoLine(conversationReturnID);
			state.appendNoLine(")");

			if (outputVariable.length() > 0) {
				//tail invoke with return
				state.appendLine(");");
			} else {
				state.appendLine(";");
			}
*/
			//fetch the input variable to a local object
			state.appendLine("Message "+inputMsgFieldName+" = BPELVariable.getMessage(engine,\""+scopes.getVariableName(inputVariable,invoke)+"\");");
			
			//prepare object to receive parameter
			if (outputVariable.length() > 0) {
				state.appendLine("Message "+outputMsgFieldName+" = null;");
			}
			
			if (correlations != null) {
				throw new BPELTranslatorException(compiler_util,invoke,"<correlation> not supported for this binding");
			}

			if (invoc.endsWith("\n")) {
				state.appendNoLine(invoc);
			} else {
				state.appendLine(invoc);
			}
			//store the received message into the appropriate BPEL variable
			if (outputVariable.length() > 0) {
				state.appendLine("BPELVariable.setMessage(engine,\""+scopes.getVariableName(outputVariable,invoke)+"\","+outputMsgFieldName+");");
			}
			
		} else {
			//we have a normal binding, the field names we give it map to engine XSD representations
			
			state.appendLine("BPELPartnerLink "+partnerLinkFieldName+" = BPELPartnerLink.getPartnerLink(\""+getPartnerLinkName(plink.name)+"\");");
			
			//
			// Marshall all exceptions into BPEL faults
			//
			state.appendLine("try {");
			
			//fetch the input variable to a local object
			state.appendLine(input_type+" "+inputMsgFieldName+" = ("+input_type+") "+input_type+".fromEngineMessage(BPELVariable.getMessage(engine,\""+scopes.getVariableName(inputVariable,invoke)+"\"));");
			
			//prepare object to receive parameter
			if (outputVariable.length() > 0) {
				state.appendLine(output_type+" "+outputMsgFieldName+" = new "+output_type+"();");
			}

			state.appendLine("BPELCorrelationSet "+correlationSetFieldNameIn+" = null;");
			state.appendLine("BPELCorrelationSet "+correlationSetFieldNameOut+" = null;");
			
			if (correlations != null) {
				translateCorrelations(state,correlations,correlationSetFieldNameOut,inputMsgFieldName,input_type,true,true);
			}
			
			if (invoc.endsWith("\n")) {
				state.appendNoLine(invoc);
			} else {
				state.appendLine(invoc);
			}

			if (correlations != null) {
				translateCorrelations(state,correlations,correlationSetFieldNameIn,outputMsgFieldName,output_type,false,true);
			}
			
			//store the received message into the appropriate BPEL variable
			if (outputVariable.length() > 0) {
				state.appendLine("BPELVariable.setMessage(engine,\""+scopes.getVariableName(outputVariable,invoke)+"\","+output_type+".toEngineMessage("+outputMsgFieldName+"));");
			}
			
			//
			// Marshall all exceptions into BPEL faults and rethrow BPELFaults
			//
			state.appendLine("} catch (BPELFault t) {");
			state.appendLine("  throw t;");
			state.appendLine("} catch (Throwable t) {");
			state.appendLine("  throw new BPELFault(t);");
			state.appendLine("}");
		}
		
		//
		// translate inline fault handlers
		//
		translatePostFaultHandlers(state,invoke);		
		
		state.decrementScope();
		state.appendLine("");
	}
	
	
	
	public void translateFlow(BPELState state, Element flow) throws BPELTranslatorException, NamespaceException {
		
		lock_model.pushNewThreads(compiler_util,flow);
		
		//the isolated thing here can be ignored
		flow_scopes.pushScope("bpelThreadIdList.get("+(bpel_threads.size()-1)+")",false,flow);
		
		state.appendLine("//FLOW");
		state.incrementScope();
		//parallel execution of N sequences
		
		BPELThread currthread = (BPELThread)bpel_threads.peek();
		BPELFlow bpflow = new BPELFlow(currthread);
		currthread.flows.add(bpflow);
		
		//expect some link stuff here for synchronisation
		//they will just be something like Mutexes that multiple sequences have access to

		ArrayList list = Util.getAllElements(flow);
		for (int i = 0; i < list.size(); i++) {
			Element links = (Element)list.get(i);
			if (NamespaceTranslator.getName(links).equals("links")) {
				translateLinks(state,links);	
			}	
		}		
		
		int flowID = nextUID();
		
		state.appendLine("Message flowMessage"+flowID+" = new Message();");
		state.appendLine("");
		state.appendLine("//launch a new engine runner for each of the activities in this flow");
		state.appendLine("String[] hosts"+flowID+" = engine.getHostsArray();");
		
		state.appendLine("Message args"+flowID+" = new Message();");
		state.appendLine("Message callstack"+flowID+" = ((Runner)engine).getStackContents();");
		state.appendLine("args"+flowID+".append(callstack"+flowID+");");
		state.appendLine("args"+flowID+".append(bpelThreadIdList);");
		state.appendLine("int threadId"+flowID+" = 1;");

		state.appendLine("Object deploymentStrategy"+flowID+" = "+DeploymentStrategyPort.class.getName()+".getDeploymentStrategy();");
		state.appendLine(DeploymentStrategyPort.class.getName()+".setDeploymentStrategy("+DeploymentStrategyPort.class.getName()+".DEPLOYMENT_STRATEGY_EVEN_SPREAD);");
		
//		ArrayList list = Util.getElements(flow);
		for (int i = 0; i < list.size(); i++) {

			Element elem = (Element)list.get(i);

			boolean threadcopies = false;
			
			String threadcount = "1";
			
			if (elem.hasAttribute("threadcount")) {
				try {
					XPATHTreeTranslator resolver = new XPATHTreeTranslator();
					String xpath_method = getNextXPath();
					String expr = resolver.resolveToNumber(xpath_method,elem.getAttribute("threadcount"),compiler_util.getSwitches(),nt,scopes,flow_scopes,wsdlmap,elem);
					state.appendDeclaration(expr);
					threadcount = xpath_method+"()";
				} catch (Exception e) {
					throw new BPELTranslatorException(compiler_util,elem,e);
				}
				
				//could be any number, but could be more than 1
				threadcopies = true;
			}

			//
			//create a new state for the subactivity
			//
			// STATE = CURRENT THREAD (PARENT THREAD)
			// NEWSTATE = NEW THREAD (CHILD THREAD)
			//
			BPELState newstate = state.newState();

			state.appendLine("int flow_loop_maximum_"+flowID+"_"+i+" = (int)"+threadcount+";");
			
			state.appendLine("for (int i"+flowID+" = 0; i"+flowID+" < flow_loop_maximum_"+flowID+"_"+i+"; i"+flowID+"++) {");

			//Append the thread index (1 - N)
			state.appendLine("  args"+flowID+".append(threadId"+flowID+");");
			
			//append the child BPEL thread ID
			state.appendLine("  String childBpelThreadId = bpelThreadId + \":\"+(threadId"+flowID+"++);");
			state.appendLine("  bpelThreadIdList.append(childBpelThreadId);");

			//launch a thread to run this activity
//using host indexes now, not hostnames			state.appendLine("  flowMessage"+flowID+".appendAll(engine.launchRunner(1,\""+newstate.getMethodName()+"\",hosts"+flowID+"[(++flowHost)%hosts"+flowID+".length],args"+flowID+"));");
			state.appendLine("if (deploymentStrategy"+flowID+" == "+DeploymentStrategyPort.class.getName()+".DEPLOYMENT_STRATEGY_EVEN_SPREAD) { ");
			//deploy threads evenly across all hosts
			state.appendLine("  flowMessage"+flowID+".appendAll(engine.launchRunner(1,\""+newstate.getMethodName()+"\",((++flowHost)%hosts"+flowID+".length),args"+flowID+"));");
			state.appendLine("} else if (deploymentStrategy"+flowID+" == "+DeploymentStrategyPort.class.getName()+".DEPLOYMENT_STRATEGY_SPECIFIC_HOST) { ");
			state.appendLine("  flowMessage"+flowID+".appendAll(engine.launchRunner(1,\""+newstate.getMethodName()+"\","+DeploymentStrategyPort.class.getName()+".getSpecificHostIndex(),args"+flowID+"));");
			state.appendLine("} else {");
			//deploy threads on first host
			state.appendLine("  flowMessage"+flowID+".appendAll(engine.launchRunner(1,\""+newstate.getMethodName()+"\",0,args"+flowID+"));");
			state.appendLine("}");

			//remove the thread index
			state.appendLine("  args"+flowID+".pop();");
			
			//remove the child BPEL thread ID
			state.appendLine("  bpelThreadIdList.pop();");
			
			state.appendLine("}");

			//have the new activity grab the callstack
			newstate.appendLine("    Message flow_stack"+flowID+" = (Message)runner_args.get(0);");
			newstate.appendLine("    ((Runner)engine).setStackContents(flow_stack"+flowID+");");
			
			//have the new activity get the id list
			newstate.appendLine("    bpelThreadIdList = (Message)runner_args.get(1);");
			newstate.appendLine("    bpelThreadChildIndex = ((Number)runner_args.get(2)).intValue();");
			newstate.appendLine("    bpelThreadId = (String)bpelThreadIdList.get(bpelThreadIdList.length()-1);");
			if (debug) {
				newstate.appendLine("    bpelThreadMap = engine.getHashMap(\""+DebugConstants.BPEL_THREADS_MAP+"\");");
				newstate.appendLine("    bpelThreadMap.put(bpelThreadId,bpelThreadId);");
				newstate.appendLine("    bpelDebugMap = engine.getHashMap(\""+DebugConstants.BPEL_DEBUG_MAP+"\");");
				newstate.appendLine("    bpelDebugSuspend = engine.getVariable(\""+DebugConstants.BPEL_BREAKPOINT_VAR+"\");");

				newstate.appendLine("    bpelDebugStack = new Message("+DebugConstants.STACKTRACE_THREAD_NAME+");");
				newstate.appendLine("    bpelDebugStack.append(bpelThreadId);");
				newstate.appendLine("    engine.ipushStack(bpelDebugStack);");
			}
			
			newstate.appendLine("    engine.ipushStack(\"BPEL **new thread - "+newstate.getMethodName()+" "+TranslatorUtils.toJavaString(compiler_util.getBpelLineNumberString(flow))+"**\");");
			if (debug) {
				newstate.appendLine("    try {");
			}
			
			BPELThread thread = new BPELThread(bpflow);
			thread.multiple_copies = threadcopies;
			thread.state = newstate;
			bpflow.threads.add(thread);
			bpel_threads.push(thread);

			lock_model.pushNewThread(compiler_util,flow,"Thread_"+nextUID());
			
			newstate.appendDeclaration("boolean localVariableInit"+thread.id+" = false;");
			newstate.appendLine("if (!localVariableInit"+thread.id+") {");
			newstate.mark(INSERT_LOCAL_VARIABLE_CREATION);
			newstate.appendLine("}");
			
			//translate the activity for this thread
			translateActivity(newstate,(Element)list.get(i),i+1);

			if (debug) {
				newstate.appendLine("    } catch (Exception bpelRunnerThreadException) {");
				newstate.appendLine("      bpelDebugMap.put(bpelThreadId+\"_TERMINATED\",new Integer(1));");
				newstate.appendLine("      throw bpelRunnerThreadException;");
				newstate.appendLine("    }");
			}

			lock_model.popThread(compiler_util,flow);
			
			if (threadcopies) {
				//may have duplicate copies of this thread
				lock_model.duplicatePreviousThread(compiler_util,flow);
			}
			
			bpel_threads.pop();

		}
		
		state.appendLine("");
		state.appendLine("args"+flowID+" = null;");
		state.appendLine("callstack"+flowID+" = null;");
		state.appendLine("//wait for each of those activities to finish");
		state.appendLine("for (int i = 0; i < flowMessage"+flowID+".length(); i++) {");
		state.appendLine("  Long tmp = (Long)flowMessage"+flowID+".get(i);");
		state.appendLine("  engine.joinRunner(tmp);");
		state.appendLine("}");

		state.decrementScope();		
		state.appendLine("//END FLOW");

		flow_scopes.popScope(flow);
		
		lock_model.popThreads(compiler_util,flow);

	}
	
	public void translateFrom(BPELState state, Element from, String totypeClass, String totypeField) throws BPELTranslatorException, NamespaceException {
		//totypeClass and totypeField describe the object we're supposed to be generating
		//in the normal case, they will both just hold the static classname of the object
		//  e.g. <xsdpackage>._string
		//so that we can use this class as a return type and also use its static methods to 
		//convert

		boolean fieldPassing = (totypeField != null);
		if (!fieldPassing) totypeField = totypeClass;
		
		if (from.hasAttribute("variable")) {
			String fromvariable = from.getAttribute("variable");
			String qtype = scopes.getVariableQtype(fromvariable,from);

			addVariableRef(fromvariable,from);

			if (from.hasAttribute("part")) {
				String frompart = from.getAttribute("part");
				
				WSDLMessage wsdlmessage = wsdlmap.getMessage(qtype);
				if (wsdlmessage == null) {
					throw new BPELTranslatorException(compiler_util,from,"WSDL Message "+qtype+" not found");
				}
				String parttype = wsdlmessage.getPartType(frompart);
				if (parttype == null) {
					throw new BPELTranslatorException(compiler_util,from,"WSDL Message part type "+frompart+" not found (message type "+qtype+")");
				}

				Element fromQuery = Util.getFirstElement(from,"query");
				
				if (fromQuery != null) {

					throw new BPELTranslatorException(compiler_util,from,"<query> element under <assign><copy> has been removed in a more recent version of the BPEL specification");
					
					/*
					//TODO is it correct that when I assign one string element to another the tag names are copied over too?
					//what would a query from assign into anything else? just the same
					//would assign into a variable part for example
					//could it be that I need to keep the old tagnames?
					//
					//perhaps in the unique case of the <to> assign with <query> it should set the tag names
					//to the original?  so that only the contrents are carried around?
					
					String query = Util.getTextDirectlyUnder(fromQuery);
					query = "$"+fromvariable+"."+frompart+query;

					try {
						XPATHTreeTranslator resolver = new XPATHTreeTranslator();
						String xpath_method = getNextXPath();
						String expr = resolver.resolveToNodeSet(xpath_method,query,compiler_util.getSwitches(),nt,scopes,flow_scopes,wsdlmap,from);
						state.appendDeclaration(expr);
		
						state.appendNoLine("("+totypeClass+")"+xpath_method+"().get(0).getValue()");
		
						addVariableRefs(resolver.getReferencedVariables(),from);
						
					} catch (Exception e) {
						throw new BPELTranslatorException(compiler_util,from,e);
					}*/
					
				} else {
				
					//when copying parts, we are assuming that minOccurs = 1 and maxOccurrs = 1, so we copy [0] only
					state.appendNoLine("(("+qtype+") "+qtype+".fromEngineMessage(");
					state.appendNoLine("BPELVariable.getMessage(engine,\""+scopes.getVariableName(fromvariable,from)+"\")");
					state.appendNoLine("))."+NamespaceTranslator.getElement(frompart)+"[0]");
				}

			} else if (from.hasAttribute("property")) {
				//from property assignment
				String property = from.getAttribute("property");
				String qpropName = nt.qualify(property,false);
				
				//get the nodeset for the property and then get the value of the first (and only) element
				state.appendNoLine("(("+qtype+") "+qtype+".fromEngineMessage(");
				state.appendNoLine("BPELVariable.getMessage(engine,\""+scopes.getVariableName(fromvariable,from)+"\")");
				state.appendNoLine(")).getBpelProperty(\""+qpropName+"\").get(0).getValue()");

			} else {
				//Message to Message assignment can just take an engine message and push an engine message
				state.appendNoLine("(("+qtype+") "+qtype+".fromEngineMessage(");
				state.appendNoLine("BPELVariable.getMessage(engine,\""+scopes.getVariableName(fromvariable,from)+"\")");
				state.appendNoLine("))");
			}

		} else if (from.hasAttribute("partnerLink")) {
			
			String partnerLink = from.getAttribute("partnerLink");
			String role = from.getAttribute("endpointReference");

			String plinkName = getPartnerLinkName(partnerLink);
			
			if (role == null) {
				throw new BPELTranslatorException(compiler_util,from,"copy from partner link must specify endpointReference attribute");
			}
			
			if (role.equals("myRole")) {
//CREATEONDEMAND				state.appendNoLine(plinkName+".myRole");
				state.appendNoLine("BPELPartnerLink.getPartnerLink(\""+plinkName+"\").myRole");
			} else if (role.equals("partnerRole")) {
//CREATEONDEMAND				state.appendNoLine(plinkName+".partnerRole");
				state.appendNoLine("BPELPartnerLink.getPartnerLink(\""+plinkName+"\").partnerRole");
			} else {
				throw new BPELTranslatorException(compiler_util,from,"endpointReference attribute int partner link assignment must be either 'myRole' or 'partnerRole'");
			}

		} else if (from.hasAttribute("expression")) {
			
			throw new BPELTranslatorException(compiler_util,from,"The 'expression' attribute on the <from> part of the <assign><copy> activity was removed in an updated version of the BPEL specification");
			
		} else {

			Element literal_elem = Util.getFirstElement(from,"literal");
			if (literal_elem == null) {

				Element expression_elem = Util.getFirstElement(from,"expression");
				if (expression_elem != null) {
					throw new BPELTranslatorException(compiler_util,from,"<expression> element under <assign>..<from> was removed in a later version of the BPEL specification");
				}
				
				String expression = Util.getTextDirectlyUnder(from);

				try {
					XPATHTreeTranslator resolver = new XPATHTreeTranslator();
					String xpath_method = getNextXPath();
					String expr;
					if (fieldPassing) {
						expr = resolver.resolveToTypeUsingInstance(xpath_method,expression,compiler_util.getSwitches(),nt,totypeClass,totypeField,scopes,flow_scopes,wsdlmap,from);
					} else {
						expr = resolver.resolveToType(xpath_method,expression,compiler_util.getSwitches(),nt,totypeClass,scopes,flow_scopes,wsdlmap,from);
					}
					state.appendDeclaration(expr);

					if (fieldPassing) {
						state.appendNoLine(xpath_method+"("+totypeField+")");
					} else {
						state.appendNoLine(xpath_method+"()");
					}

					addVariableRefs(resolver.getReferencedVariables(),from);
					
				} catch (Exception e) {
					throw new BPELTranslatorException(compiler_util,from,e);
				}
				
				
			} else {
				
				//constant value

				StringBuffer text = new StringBuffer();
	
				NodeList list = literal_elem.getChildNodes();
								
				for (int i = 0; i < list.getLength(); i++) {
					Node node = list.item(i);
					
					if (node.getNodeType() == Node.COMMENT_NODE) {
						//ignore
					} else {
					
						try {
							String nodeContents = XMLUtil.nodeToString(node);
							nodeContents = TranslatorUtils.toJavaString(nodeContents);
							text.append(nodeContents);
							
						} catch (Exception e) {
							throw new BPELTranslatorException(compiler_util,from,e);
						}
					}
				}
	
				if (totypeClass == null) {
					throw new BPELTranslatorException(compiler_util,from,"cannot assign an XSD type to an entire WSDL message, need to assign to a part of a message");
				}
	
				if (compiler_util.getSwitches().ASSIGN_CONSTANTS_CACHING) {
					int constID = nextUID();
					String assignConstant = "assignConstant"+constID;
					
					state.appendDeclaration("static "+totypeClass+" "+assignConstant+" = null;");
					
					
					if (fieldPassing) {
						state.appendDeclaration("public "+totypeClass+" getAssignConstant"+constID+"("+totypeClass+" "+totypeField+") throws Exception {");
					} else {
						state.appendDeclaration("public "+totypeClass+" getAssignConstant"+constID+"() throws Exception {");
					}
					state.appendDeclaration("  if ("+assignConstant+" == null) {");
					if (fieldPassing) {
						state.appendDeclaration("    "+assignConstant+" = ("+totypeClass+") "+totypeField+".ifromXML(\""+text.toString().trim()+"\");");
					} else {
						state.appendDeclaration("    "+assignConstant+" = ("+totypeClass+") "+totypeField+".fromXML(\""+text.toString().trim()+"\");");
					}
					state.appendDeclaration("  }");
					state.appendDeclaration("  return "+assignConstant+";");
					state.appendDeclaration("}\n");
			
					if (fieldPassing) {
						state.appendNoLine("getAssignConstant"+constID+"("+totypeField+")");
					} else {
						state.appendNoLine("getAssignConstant"+constID+"()");
					}
				} else {
					if (fieldPassing) {
						state.appendNoLine("("+totypeClass+") "+totypeField+".ifromXML(\""+text+"\")");
					} else {
						state.appendNoLine("("+totypeClass+") "+totypeField+".fromXML(\""+text+"\")");
					}
				}
			}
		}

	}
	
/*	public static void main(String[] args) {
		System.out.println(getVariableAndPartOnly("$variable.part"));
		System.out.println(getVariableAndPartOnly("$variable"));
	}*/

	private static String[] getVariableAndPartOnly(Element expression) {
		String query = Util.getTextDirectlyUnder(expression).trim();
		return getVariableAndPartOnly(query);
	}
	private static String[] getVariableAndPartOnly(String query) {
		boolean error = false;
		
		String var = null;
		String part = null;
		
		int i = 0;
		
		char[] cs = query.toCharArray();
		for ( ; i < cs.length; i++) {
			if (part != null) {
				//we are reading the part name
				
				if (Character.isJavaLetterOrDigit(cs[i])) {
					part = part+cs[i];
				} else {
					//not part of the part name, and not whitespace (we trimmed)
					error = true;
					break;
				}
			} else if (var != null) {
				//we are reading the var name
				
				if (Character.isJavaLetterOrDigit(cs[i])) {
					var = var+cs[i];
				} else if (cs[i]=='.') {
					//begin the part
					part = "";
				} else {
					//not part of the variable name, not a part separator '.', and not whitespace (we trimmed)
					error = true;
					break;
				}
			} else if (cs[i] == '$') {
				//we are at the beginning and we found a $, begin the variable
				var = "";
				
			} else {
				error = true;
				break;
				
			}
		}
		
		if (error) {
			return null;
		} else {
			return new String[]{var,part};
		}
	}
	
	public void translateCopy(BPELState state, Element copy, int index) throws BPELTranslatorException, NamespaceException {
		state.appendLine("//COPY");
		state.incrementScope();

		Element from = Util.getFirstElement(copy,"from");
		Element to = Util.getFirstElement(copy,"to");
		
		if (from == null) {
			throw new BPELTranslatorException(compiler_util,copy,"Expected <from> element but found none");	
		}
		if (to == null) {
			throw new BPELTranslatorException(compiler_util,copy,"Expected <to> element but found none");	
		}

		if (compiler_util.getSwitches().APPEND_BPEL_TO_CALLSTACK) {
			state.appendLine("engine.ipushStack(\"BPEL "+TranslatorUtils.toJavaString(copy)+" (elem"+index+") "+TranslatorUtils.toJavaString(compiler_util.getBpelLineNumberString(copy))+" "+TranslatorUtils.toJavaString(from)+"->"+TranslatorUtils.toJavaString(to)+"\");");
//			state.appendLine("engine.ipushStack(\"BPEL "+TranslatorUtils.toJavaString(copy)+" (elem"+index+")\");");
//			state.appendLine("engine.ipushStack(\"BPEL   + "+TranslatorUtils.toJavaString(from)+"\");");
//			state.appendLine("engine.ipushStack(\"BPEL   + "+TranslatorUtils.toJavaString(to)+"\");");
		}
		
		//Could munge stuff like $a and $a.b in here to use the proper attributes variable and part?
		//we've already appended it to the callstack so why not?
		String[] fromparts = getVariableAndPartOnly(from);
		String[] toparts = getVariableAndPartOnly(to);
		if (!from.hasAttribute("variable")
			&& !from.hasAttribute("part")
			&& !from.hasAttribute("property")
			&& !from.hasAttribute("partnerLink")
			&& !to.hasAttribute("variable")
			&& !to.hasAttribute("part")
			&& !to.hasAttribute("property")
			&& !to.hasAttribute("partnerLink")) {

			if (fromparts != null && toparts != null) {
				if ((fromparts[1] != null && toparts[1] != null) 
					|| (fromparts[1] == null && toparts[1] == null)) {

					{
						String[] varpart = getVariableAndPartOnly(from);
						if (varpart != null) {
							if (varpart[0] != null) {
								from.setAttribute("variable", varpart[0]);
								
								log.logWarning(LOG_SOURCE, "Munged <from> element at "+compiler_util.getBpelLineNumberString(from)+", adding variable "+varpart[0]);
								
								if (varpart[1] != null) {
									from.setAttribute("part", varpart[1]);
			
									log.logWarning(LOG_SOURCE, "Munged <from> element at "+compiler_util.getBpelLineNumberString(from)+", adding variable part "+varpart[1]);
								}
							}
						} else {
							
							log.logWarning(LOG_SOURCE, "Couldn't munge <from> element at "+compiler_util.getBpelLineNumberString(from));
						}
					}

					{
						String[] varpart = getVariableAndPartOnly(to);
						if (varpart != null) {
							if (varpart[0] != null) {
								to.setAttribute("variable", varpart[0]);
								
								log.logWarning(LOG_SOURCE, "Munged <to> element at "+compiler_util.getBpelLineNumberString(to)+", adding variable "+varpart[0]);
								
								if (varpart[1] != null) {
									to.setAttribute("part", varpart[1]);
			
									log.logWarning(LOG_SOURCE, "Munged <to> element at "+compiler_util.getBpelLineNumberString(to)+", adding variable part "+varpart[1]);
								}
							}
						} else {
							
							log.logWarning(LOG_SOURCE, "Couldn't munge <to> element at "+compiler_util.getBpelLineNumberString(to));
						}
					}
					
				}
			}

		}
		
		
		if (to.hasAttribute("variable")) {
			String tovariable = to.getAttribute("variable");

			String qtype = scopes.getVariableQtype(tovariable,copy);

			addVariableRef(tovariable,to);

			if (qtype == null) throw new BPELTranslatorException(compiler_util,copy,"BPEL <copy>: could not find type for variable "+tovariable);
			
			Element toQuery = Util.getFirstElement(to,"query");
			
			if (toQuery != null) {
				
				throw new BPELTranslatorException(compiler_util,copy,"<query> element under <assign><copy> has been removed in an up to date version of the BPEL specification");
				
				/*
				int nset = nextUID();
				
				String query = Util.getTextDirectlyUnder(toQuery);
				if (to.hasAttribute("part")) {
					String topart = to.getAttribute("part");
					query = "$"+tovariable+"/"+topart+query;
				} else {
					query = "$"+tovariable+query;
				}

				state.appendNoLine("XNodeSet assignSet"+nset+" = ");
				
				try {
					XPATHTreeTranslator resolver = new XPATHTreeTranslator();
					String xpath_method = getNextXPath();
					String expr = resolver.resolveToNodeSet(xpath_method,query,compiler_util.getSwitches(),nt,scopes,flow_scopes,wsdlmap,to);
					state.appendDeclaration(expr);
	
					state.appendNoLine(xpath_method+"()");
	
					addVariableRefs(resolver.getReferencedVariables(),from);
					
				} catch (Exception e) {
					throw new BPELTranslatorException(compiler_util,from,e);
				}
				state.appendLine(";");
				
				//should be a nodeset of size 1
				state.appendLine("XNode assignNode"+nset+" = assignSet"+nset+".get(0);");

//				NamespaceTranslator tmpNT = compiler_util.createNamespaceTranslator();
				
				//we will cast into an XMLAccessible interface
				String qanyType = XMLAccessible.class.getName();
				
				state.appendLine(qanyType+" assignTo"+nset+" = ("+qanyType+")assignNode"+nset+".getValue();");

				state.appendNoLine("assignNode"+nset+".setValueInContainer(");
				
				translateFrom(state,from,qanyType,"assignTo"+nset);
				state.appendLine(");");
				
				//store the modified variable

				//get the root XNode which should corresponsd to the WSDL message fetched by the XPATH?
				state.appendLine("while (assignNode"+nset+".getParent() != null) assignNode"+nset+" = assignNode"+nset+".getParent();");
				//set the variable based on this modified message
				state.appendLine("BPELVariable.setMessage(engine,\""+scopes.getVariableName(tovariable,to)+"\", "+qtype+".toEngineMessage(assignNode"+nset+".getValue()));");
				*/
				
			} else if (to.hasAttribute("part")) {
				String topart = to.getAttribute("part");

				int msg = nextUID();
				
				//get the variable
				state.appendLine(qtype+" assign"+msg+" = ("+qtype+") "+qtype+".fromEngineMessage(BPELVariable.getMessage(engine,\""+scopes.getVariableName(tovariable,to)+"\"));");

				//do the assignment
				
				WSDLMessage wsdlmsg = wsdlmap.getMessage(qtype);
				
				if (wsdlmsg == null) throw new BPELTranslatorException(compiler_util,copy,"BPEL <copy>: Could not find WSDL Message "+qtype);
			
				String parttype = (String)wsdlmsg.part_to_xsdtype.get(topart);

//should this be in or not?
				if (parttype == null) throw new BPELTranslatorException(compiler_util,copy,"XSD part "+topart+" not found in WSDL message "+qtype+")");
				
				state.appendLine("assign"+msg+"."+NamespaceTranslator.getElement(topart)+" = new "+parttype+"[1];");
				state.appendNoLine("assign"+msg+"."+NamespaceTranslator.getElement(topart)+"[0] = ");
				translateFrom(state,from,parttype,null);
				state.appendLine(";");
				
				//store the modified variable
				state.appendLine("BPELVariable.setMessage(engine,\""+scopes.getVariableName(tovariable,to)+"\", "+qtype+".toEngineMessage(assign"+msg+"));");

			} else if (to.hasAttribute("property")) {
				String toproperty = to.getAttribute("property");
				String qpropName = nt.qualify(toproperty,false);
				
				int msg = nextUID();
				int nset = nextUID();

				//get the variable
				state.appendLine(qtype+" assign"+msg+" = ("+qtype+") "+qtype+".fromEngineMessage(BPELVariable.getMessage(engine,\""+scopes.getVariableName(tovariable,to)+"\"));");
				
				//get the nodeset for the query assignment
				state.appendNoLine("XNodeSet assignSet"+nset+" = ");
				state.appendNoLine("assign"+msg+".getBpelProperty(\""+qpropName+"\")");
				state.appendLine(";");

				//get the node to assign to
				state.appendLine("XNode assignNode"+nset+" = assignSet"+nset+".get(0);");
				
//				NamespaceTranslator tmpNT = compiler_util.createNamespaceTranslator();
				
				//we will cast into an XMLAccessible interface
				String qanyType = XMLAccessible.class.getName();

				//the actual instance to read the value into
				state.appendLine(qanyType+" assignTo"+nset+" = ("+qanyType+")assignNode"+nset+".getValue();");

				//the assignment node container to assign the value into
				state.appendNoLine("assignNode"+nset+".setValueInContainer(");
				
				translateFrom(state,from,qanyType,"assignTo"+nset);
				state.appendLine(");");
				
				//store the modified variable

				//get the root XNode which should corresponsd to the WSDL message fetched by the XPATH?
				state.appendLine("while (assignNode"+nset+".getParent() != null) assignNode"+nset+" = assignNode"+nset+".getParent();");
				//set the variable based on this modified message
				state.appendLine("BPELVariable.setMessage(engine,\""+scopes.getVariableName(tovariable,to)+"\", "+qtype+".toEngineMessage(assignNode"+nset+".getValue()));");
				
			} else {

				//assign whole WSDL message
				state.appendNoLine("BPELVariable.setMessage(engine,\""+scopes.getVariableName(tovariable,to)+"\","+qtype+".toEngineMessage(");

				//translateFrom(state,from,null,null,null);
				
				translateFrom(state,from,qtype,null);

				state.appendLine("));");
			}

		} else if (to.hasAttribute("partnerLink")) {
			
			String partnerLink = to.getAttribute("partnerLink");
			String plinkName = getPartnerLinkName(partnerLink);
			
//CREATEONDEMAND			state.appendNoLine(plinkName+".partnerRole = ");
			state.appendNoLine("BPELPartnerLink.getPartnerLink(\""+plinkName+"\").partnerRole = ");
			translateFrom(state,from,WSEndpointReference.class.getName(),null);
			state.appendLine(";");
			
			//TODO we should instead of this actually get a reference to the partner link and
			//find out whether we're supposed to assign to the myRole or partnerRole
//CREATEONDEMAND			state.appendNoLine(plinkName+".myRole = "+plinkName+".partnerRole;");
			state.appendLine("BPELPartnerLink.getPartnerLink(\""+plinkName+"\").myRole = BPELPartnerLink.getPartnerLink(\""+plinkName+"\").partnerRole;"); 

		} else {
			
			int nset = nextUID();
			
			String query = Util.getTextDirectlyUnder(to);
			
			StringBuffer variablebuf = new StringBuffer();
			
			int varStart = query.indexOf('$');
			
			if (varStart == -1) {
				throw new BPELTranslatorException(compiler_util,to,"could not find variable name under <assign>..<to> element");
			}
			
			for (int k = varStart+1; k < query.length(); k++) {
				char c = query.charAt(k);
				if (c == '/'
					|| c == '.'
					|| c == '\\'
					|| Character.isWhitespace(c)) {
					break;
				} else {
					variablebuf.append(c);
				}
			}
			
			String tovariable = variablebuf.toString();
/*
			String topart = null;
			
			int dotindex = tovariable.indexOf('.');
			if (dotindex != -1) {
				topart = tovariable.substring(dotindex+1);
				tovariable = tovariable.substring(0,dotindex);
			}*/
						
			String qtype = scopes.getVariableQtype(tovariable,copy);

			addVariableRef(tovariable,to);

			if (qtype == null) throw new BPELTranslatorException(compiler_util,copy,"BPEL <copy>: could not find type for variable "+tovariable);			

			state.appendNoLine("XNodeSet assignSet"+nset+" = ");
			
			try {
				XPATHTreeTranslator resolver = new XPATHTreeTranslator();
				String xpath_method = getNextXPath();
				String expr = resolver.resolveToNodeSet(xpath_method,query,compiler_util.getSwitches(),nt,scopes,flow_scopes,wsdlmap,to);
				state.appendDeclaration(expr);

				state.appendNoLine(xpath_method+"()");

				addVariableRefs(resolver.getReferencedVariables(),from);
				
			} catch (Exception e) {
				throw new BPELTranslatorException(compiler_util,from,e);
			}
			state.appendLine(";");
			
			//should be a nodeset of size 1
			state.appendLine("XNode assignNode"+nset+" = assignSet"+nset+".get(0);");

//			NamespaceTranslator tmpNT = compiler_util.createNamespaceTranslator();
			
			//we will cast into an XMLAccessible interface
			String qanyType = XMLAccessible.class.getName();
			
			state.appendLine(qanyType+" assignTo"+nset+" = ("+qanyType+")assignNode"+nset+".getValue();");

			state.appendNoLine("assignNode"+nset+".setValueInContainer(");
			
			translateFrom(state,from,qanyType,"assignTo"+nset);
			state.appendNoLine(",true"); //ignore if there is no container - probably a top level variable set
			state.appendLine(");");
			
			//store the modified variable

			//get the root XNode which should corresponsd to the WSDL message fetched by the XPATH?
			state.appendLine("while (assignNode"+nset+".getParent() != null) assignNode"+nset+" = assignNode"+nset+".getParent();");
			
			//set the variable based on this modified message
			state.appendLine("BPELVariable.setMessage(engine,\""+scopes.getVariableName(tovariable,to)+"\", "+qtype+".toEngineMessage(assignNode"+nset+".getValue()));");
			
		}

		if (compiler_util.getSwitches().APPEND_BPEL_TO_CALLSTACK) {
			state.appendLine("engine.ipopStack();");
//			state.appendLine("engine.ipopStack();");
//			state.appendLine("engine.ipopStack();");
		}
		state.decrementScope();
		state.appendLine("");
	}
	
	public void translateAssign(BPELState state, Element assign) throws BPELTranslatorException, NamespaceException{
		state.appendLine("//ASSIGN");
		state.incrementScope();

		int isolatedScopeIndex = pushScope(state,assign,true,true);
		
		ArrayList list = Util.getAllElements(assign);
		for (int i = 0; i < list.size(); i++) {
			Element copy = (Element)list.get(i);
			if (NamespaceTranslator.getName(copy).equals("copy")) {
				translateCopy(state,copy,i);
			} else {
//DBG				DBG.warning("unrecognised tag name compiling assign \""+copy.getTagName()+"\"");	
				log.logWarning(LOG_SOURCE,"unrecognised tag name compiling assign \""+copy.getTagName()+"\"");
			}
		}
		
		popScope(state,assign,isolatedScopeIndex,true);
		
		state.decrementScope();		
		state.appendLine("//END ASSIGN");
		state.appendLine("");
	}

	public void translateForDuration(BPELState state, String for_duration, Node source) throws BPELTranslatorException {
		try {
			String dtype = nt.qualify(XSDTypeTranslator.XSD_TYPE_DURATION,false);

			XPATHTreeTranslator resolver = new XPATHTreeTranslator();
			String xpath_method = getNextXPath();
			String expr = resolver.resolveToDurationInMillis(xpath_method,for_duration,compiler_util.getSwitches(),nt,scopes,flow_scopes,wsdlmap,source);
			state.appendDeclaration(expr);

//			System.out.println("XPATH expression = "+expr);

//			state.appendNoLine("Math.max(1, (("+dtype+") "+dtype+".fromXML(");
			state.appendNoLine(xpath_method+"()");
//			state.appendNoLine(")).INTERNAL_VALUE.longValue() )");
			
			addVariableRefs(resolver.getReferencedVariables(),source);
			
		} catch (Exception e) {
//			e.printStackTrace();
			throw new BPELTranslatorException(compiler_util,source,e);
		}
	}

	public void translateUntilDeadline(BPELState state, String until_deadline, Node source) throws BPELTranslatorException {
		try {
			String dtype = nt.qualify(XSDTypeTranslator.XSD_TYPE_DATETIME,false);

			XPATHTreeTranslator resolver = new XPATHTreeTranslator();
			String xpath_method = getNextXPath();
			String expr = resolver.resolveToDateOrDateTimeInMillis(xpath_method,until_deadline,compiler_util.getSwitches(),nt,scopes,flow_scopes,wsdlmap,source);
			state.appendDeclaration(expr);

//			System.out.println("XPATH expression = "+expr);

			state.appendNoLine("Math.max(1, ");
			state.appendNoLine(xpath_method+"()-System.currentTimeMillis()");
			state.appendNoLine(" )");
			
			addVariableRefs(resolver.getReferencedVariables(),source);
			
		} catch (Exception e) {
//			e.printStackTrace();
			throw new BPELTranslatorException(compiler_util,source,e);
		}
	}

	public void translateWait(BPELState state, Element wt) throws BPELTranslatorException {
		state.appendLine("//WAIT");
		
		String waitID = "sleepTime"+nextUID();

		Element forElem = Util.getFirstElement(wt,"for");
		Element untilElem = Util.getFirstElement(wt,"until");
		
		if (forElem != null) {
			String for_duration = Util.getTextDirectlyUnder(forElem);

			state.appendNoLine("long "+waitID+" = ");
			translateForDuration(state,for_duration,forElem);
			state.appendLine(";");
			
		} else if (untilElem != null) {
			String until_deadline = Util.getTextDirectlyUnder(untilElem);

			state.appendNoLine("long "+waitID+" = ");
			translateUntilDeadline(state,until_deadline,untilElem);
			state.appendLine(";");

		} else {
			throw new BPELTranslatorException(compiler_util,wt,"<wait> must have either <for> or <until> element");
		}
		state.appendLine("if ("+waitID+" > 1) Thread.sleep("+waitID+");");
		state.appendLine("");
	}

	public void translateEmpty(BPELState state, Element assign) throws BPELTranslatorException, NamespaceException {
		state.appendLine("//(EMPTY)");
	}

	public void translatePick(BPELState state, Element pick) throws BPELTranslatorException, NamespaceException {
		state.appendLine("//PICK");

		int pickID = nextUID();
		
		//not used - we dont use start activities
		String createInstance = pick.getAttribute("createInstance");
		
		int onMessageCount = 0;
		int onAlarmCount = 0;
		
		ArrayList unsorted_incoming = Util.getAllElements(pick);
		ArrayList incoming = new ArrayList();
		for (int i = 0; i < unsorted_incoming.size(); i++) {
			Element inmsg = (Element)unsorted_incoming.get(i);
			String tagname = NamespaceTranslator.getName(inmsg);
			if (tagname.equals("onMessage")) {
				onMessageCount++;
				incoming.add(inmsg);
			}
		}
		for (int i = 0; i < unsorted_incoming.size(); i++) {
			Element inmsg = (Element)unsorted_incoming.get(i);
			String tagname = NamespaceTranslator.getName(inmsg);
			if (tagname.equals("onAlarm")) {
				onAlarmCount++;
				incoming.add(inmsg);
			}
		}
		
		String inputsId = "pickResults"+pickID+"_";
		
		String timeoutId = "timeoutId"+pickID;
		String timeoutIndexId = "timeoutIndexId"+pickID;
		
		if (onAlarmCount > 0) {
			state.appendLine("long "+timeoutId+";");
		} else {
			state.appendLine("long "+timeoutId+";");
		}
		state.appendLine("int "+timeoutIndexId+";");
		
		ArrayList bindingsList = new ArrayList();
		for (int i = 0; i < bindings.length; i++) {
			bindingsList.add(bindings[i]);
		}
		
		ArrayList problems = new ArrayList();
		
		ArrayList pick_portTypes = new ArrayList();
		ArrayList pick_operations = new ArrayList();
		ArrayList pick_inputtypes = new ArrayList();
		ArrayList pick_variables = new ArrayList();
		
		ArrayList alarmIds = new ArrayList();
		ArrayList alarmIndexes = new ArrayList();
		
		ArrayList alarmRepeats = new ArrayList();
		
		//
		// pick onAlarm
		// Compile in the absolute finish times for each alarm (may later be incremented by repeatEvery calls)
		//
		for (int i = 0; i < incoming.size(); i++) {
			Element inmsg = (Element)incoming.get(i);
			String tagname = NamespaceTranslator.getName(inmsg);

			nt.addNamespaces(inmsg);
			
			if (tagname.equals("onAlarm")) {
				
				Element forElem = Util.getFirstElement(inmsg,"for");
				Element untilElem = Util.getFirstElement(inmsg,"until");
				Element repeatElem = Util.getFirstElement(inmsg,"repeatEvery");
				
				alarmRepeats.add(repeatElem);
				
				String alarmId = "pick_timeout"+pickID+"_"+i;
//				String alarmIncId = null;
				
				alarmIds.add(alarmId);
				alarmIndexes.add(new Integer(i));
				
				//remember that if the duration compiles to 0 then it should be turned into 1
				//so that java doesn't block forever

				if (forElem != null) {
					String for_duration = Util.getTextDirectlyUnder(forElem);
					
					state.appendLine("//calculate absolute timeout time for onMessage "+i+" (for)");
					state.appendNoLine("long "+alarmId+" = System.currentTimeMillis()+");
					translateForDuration(state,for_duration,forElem);
					state.appendLine(";");
					
				} else if (untilElem != null) {
					String until_deadline = Util.getTextDirectlyUnder(untilElem);
					
					state.appendLine("//calculate absolute timeout time for onMessage "+i+" (until)");
					state.appendNoLine("long "+alarmId+" = System.currentTimeMillis()+");
					translateUntilDeadline(state,until_deadline,untilElem);
					state.appendLine(";");

				} else if (repeatElem != null) {
					
					throw new BPELTranslatorException(compiler_util,repeatElem,"<repeatEvery> in <pick> has been removed from BPEL spec");
/*					
					String repeat_every = Util.getTextDirectlyUnder(repeatElem);

					state.appendLine("//calculate absolute timeout time for onMessage "+i+" (repeatEvery with no for or until)");
					state.appendNoLine("long "+alarmId+" = System.currentTimeMillis()+");
					translateForDuration(state,repeat_every,repeatElem);
					state.appendLine(";");
*/			
				} else {
					throw new BPELTranslatorException(compiler_util,inmsg,"<onAlarm> in pick must have one of <for>, <until> or <repeatEvery>");
				}
					
			} else {
				alarmRepeats.add(null);
			}
			nt.removeNamespaces(inmsg);
		}

		//
		// Binding translate pick START
		//
		for (int i = 0; i < bindingsList.size(); i++) {
			WSDLBindingTranslator binding = (WSDLBindingTranslator)bindingsList.get(i);
			try {
				binding.translatePickStart(onMessageCount,onAlarmCount,timeoutId);
			} catch (WSDLBindingTranslatorException e) {
				//this binding cannot translate this pick
				bindingsList.remove(i--);
				problems.add(e);
			}
		}
		
		//
		// Compile in the while loop for repeatEvery alarms
		//
		String loopId = "pickLoop"+pickID;
		
		state.appendLine("boolean "+loopId+" = true;");
		state.appendLine("while ("+loopId+") {");
		state.incrementScope();
		state.appendLine(loopId+" = false;");
		
		//
		// Figure out the shortest timeout value based on the absolute timeout values
		//
		if (onAlarmCount > 0) {
			state.appendLine("//reset timeout values (ready for finding minimum onAlarm timeout)");
			state.appendLine(""+timeoutId+" = Long.MAX_VALUE;");
			state.appendLine(""+timeoutIndexId+" = 0;");

			for (int i = 0; i < alarmIds.size(); i++) {
				String alarmId = (String)alarmIds.get(i);
				Integer alarmIndex = (Integer)alarmIndexes.get(i);

				state.appendLine("if ("+alarmId+" < "+timeoutId+") {");
				state.incrementScope();
				
				state.appendLine(timeoutId+" = "+alarmId+";");
				state.appendLine(timeoutIndexId+" = "+alarmIndex+";");
				
				state.decrementScope();
				state.appendLine("}");
			}
			state.appendLine("//convert timeout value to relative");
			state.appendLine(timeoutId+" = Math.max(1,"+timeoutId+"-System.currentTimeMillis());");
			
		} else {
			state.appendLine("//reset timeout values (0 means no timeout - no onAlarms)");
			state.appendLine(timeoutId+" = 0;");
			state.appendLine(timeoutIndexId+" = 0;");
		}

		ArrayList correlationElements = new ArrayList();
		ArrayList correlationFieldNames = new ArrayList();
		
		//
		// pick onMessage
		// compile in binding translations for picking different messages
		//
		for (int i = 0; i < incoming.size(); i++) {
			Element inmsg = (Element)incoming.get(i);
			String tagname = NamespaceTranslator.getName(inmsg);

			nt.addNamespaces(inmsg);
			
			if (tagname.equals("onMessage")) {
				String partnerLink = inmsg.getAttribute("partnerLink"); 
				String operation = inmsg.getAttribute("operation");
				String messageExchange = inmsg.getAttribute("messageExchange");
				String variable = inmsg.getAttribute("variable");

				if (!inmsg.hasAttribute("operation")) {
					throw new BPELTranslatorException(compiler_util,inmsg,"No 'operation' attribute found");
				}
				if (!inmsg.hasAttribute("partnerLink")) {
					throw new BPELTranslatorException(compiler_util,inmsg,"No 'partnerLink' attribute found");
				}
				
				if (messageExchange == null) {
					messageExchange = "";
				}

				BPELPartnerLink plink = (BPELPartnerLink)partnerLinks.get(partnerLink);
				if (plink == null) {
					throw new BPELTranslatorException(compiler_util,inmsg,"No partnerLink '"+partnerLink+"' found");
				}
				//pick (receive) means myRole
				BPELPartnerLinkRole prole = plink.myRole;
				String portType = prole.getQualifiedPortType();
				
				//
				// Get port and operation information
				//
				WSDLPortType wsdl_pt = wsdlmap.getPortType(portType); 
				if (wsdl_pt == null) {
					throw new BPELTranslatorException(compiler_util,inmsg,"No portType '"+portType+"' found");	
				}
				WSDLOperation wsdl_op = (WSDLOperation)wsdl_pt.name_to_operation.get(operation);
				if (wsdl_op == null) {
					throw new BPELTranslatorException(compiler_util,inmsg,"No operation '"+operation+"' found in portType '"+portType+"'");	
				}
				String input_type = (String)wsdl_op.message_types.get("input");

				if (variable == null || variable.length() == 0) {
					throw new BPELTranslatorException(compiler_util,inmsg,"no input variable specified for pick onMessage (port "+portType+" operation "+operation+")");
				}
				if (input_type == null) {
					throw new BPELTranslatorException(compiler_util,inmsg,"no input type found for pick onMessage (port "+portType+" operation "+operation+")");
				}
				String qtype = scopes.getVariableQtype(variable, inmsg);
				if (!input_type.equals(qtype)) {
					throw new BPELTranslatorException(compiler_util,inmsg,"pikc onMessage variable has type "+compiler_util.getQNameFromQName(qtype)+" but operation requires different WSDL message type "+compiler_util.getQNameFromQName(input_type));
				}
				
				pick_portTypes.add(wsdl_pt);
				pick_operations.add(wsdl_op);
				pick_inputtypes.add(input_type);
				pick_variables.add(scopes.getVariableName(variable,inmsg));
				
//CREATEONDEMAND				String partnerLinkFieldName = getPartnerLinkName(plink.name);
				String partnerLinkFieldName = "bpel_partner_link"+nextUID();
				//pick (receive) means myRole
				String eprFieldName = partnerLinkFieldName+".myRole";
				
				//Fetch the partner link right here
				state.appendLine("BPELPartnerLink "+partnerLinkFieldName+" = BPELPartnerLink.getPartnerLink(\""+getPartnerLinkName(plink.name)+"\");");
				
				//generate IDs to hold the correlation sets
				int correlationSetID = nextUID();
				String correlationSetFieldName = "bpel_pick_correlation_"+correlationSetID;
				
				//get correlation sets element
				Element correlations = Util.getFirstElement(inmsg,"correlations");
				
				correlationFieldNames.add(correlationSetFieldName);
				correlationElements.add(correlations);

				for (int b = 0; b < bindingsList.size(); b++) {
					WSDLBindingTranslator binding = (WSDLBindingTranslator)bindingsList.get(b);
					try {
						if (binding instanceof InternalBindingTranslator) {
							//internal bindings get Messages
							binding.translatePickOnMessage(nt,portType,plink.requiredBindingName,operation,messageExchange,eprFieldName,correlationSetFieldName,""+inputsId+i);
						} else { 
							//non-internal bindings get XSD types
							binding.translatePickOnMessage(nt,portType,plink.requiredBindingName,operation,messageExchange,eprFieldName,correlationSetFieldName,""+inputsId+i);
						}
					} catch (WSDLBindingTranslatorException e) {
						//this binding cannot translate this pick
						bindingsList.remove(b--);
						problems.add(e);
					}
				}
				
			}
			nt.removeNamespaces(inmsg);
		}
		
		//
		// pick FINISH
		//
		String indexFieldName="pickparam"+pickID;
		String pickInvoc = null;
		WSDLBindingTranslator used_binding = null;
		
		for (int b = 0; b < bindingsList.size(); b++) {
			used_binding = (WSDLBindingTranslator)bindingsList.get(b);
			try {
				pickInvoc = used_binding.translatePickFinish(indexFieldName);
				break;
			} catch (WSDLBindingTranslatorException e) {
				//this binding cannot translate this pick
				bindingsList.remove(b--);
				problems.add(e);
			}
		}

		//
		// pick compilation into program
		//
		try {
			if (used_binding != null) {

				if (used_binding instanceof InternalBindingTranslator) {
					//we have an internal binding, the field names we give it map to engine Messages fetched from variables
					//if it likes, it can convert these into engine XSD types
					//fetch the input variable to a local object
					
					//set up the input fields array as the binding requires
					//(create a new Message for the binding to populate)
					for (int i = 0; i < onMessageCount; i++) {
//						String input_type = (String)pick_inputtypes.get(i);

						state.appendLine("Message "+inputsId+i+" = new Message();\n");
							
						if (correlationElements.get(i) != null) {
							throw new BPELTranslatorException(compiler_util,pick,"<correlation> not supported for this binding");
						}
					}
					
					//compile in the index field
					state.appendLine("int "+indexFieldName+" = 0;");
					
					//compile in the bit where the binding populates the Message
					if (pickInvoc.endsWith("\n")) {
						state.appendNoLine(pickInvoc);
					} else {
						state.appendLine(pickInvoc);
					}
		
					//check which we onMessage or onAlarm we received
					for (int i = 0; i < incoming.size(); i++) {

						Element pickElem = (Element)incoming.get(i);
						
						if (i > 0) {
							state.appendNoLine("} else ");
						}
						if (i < onMessageCount) {
							state.appendLine("if ("+indexFieldName+" == "+i+") {");
							
							state.incrementScope();
							state.appendLine("  engine.ipushStack(\"BPEL "+TranslatorUtils.toJavaString(pickElem)+" "+TranslatorUtils.toJavaString(compiler_util.getBpelLineNumberString(pickElem))+"\");");
				
							//first N indexes are always messages
							state.appendLine("BPELVariable.setMessage(engine,\""+pick_variables.get(i)+"\","+inputsId+i+");");
							
							state.appendLine("  //onMessage "+i+" activities");

							//compile the activities under this element
							translateActivities(state,pickElem);
				
							state.appendLine("  engine.ipopStack();");
							state.decrementScope();
							
						} else {
							state.appendLine("if ("+timeoutIndexId+" == "+i+") {");
							
							state.incrementScope();
							state.appendLine("  engine.ipushStack(\"BPEL "+TranslatorUtils.toJavaString(pickElem)+" "+TranslatorUtils.toJavaString(compiler_util.getBpelLineNumberString(pickElem))+"\");");
				
							//remaining N indexes are always timeouts
							
							state.appendLine("  //onAlarm "+i+" activities");
							
							//compile the activities under this element
							translateActivities(state,pickElem);
							
							Element repeatElem = (Element)alarmRepeats.get(i);
							if (repeatElem != null) {
								String repeat_every = Util.getTextDirectlyUnder(repeatElem);
								String alarmId = "pick_timeout"+pickID+"_"+i;
								
								state.appendNoLine(alarmId+" += ");
								translateForDuration(state,repeat_every,repeatElem);
								state.appendLine(";");

								state.appendLine(loopId+" = true; //repeatEvery");
							}

							state.appendLine("  engine.ipopStack();");
							state.decrementScope();
							
						}
						
						//tail the IF
						if (i == (incoming.size()-1)) {
							state.appendLine("}");
						}
					}
					
				} else {
					//we have a normal binding, the field names we give it map to engine XSD representations
					
					//set up the input fields array as the binding requires
					//(create a new XSD type for the binding to populate)
					for (int i = 0; i < onMessageCount; i++) {
						String input_type = (String)pick_inputtypes.get(i);
						state.appendLine(input_type+" "+inputsId+i+" = new "+input_type+"();");

						//generate a correlation set stub, it may not be populated but we pass it 
						//into the binding so the variable needs to exist even if its null
						state.appendLine("BPELCorrelationSet "+correlationFieldNames.get(i)+" = null;");
					}
					

					//compile in the index field
					state.appendLine("int "+indexFieldName+" = 0;");
					
					//compile in the bit where the binding populates the XSD type
					if (pickInvoc.endsWith("\n")) {
						state.appendNoLine(pickInvoc);
					} else {
						state.appendLine(pickInvoc);
					}
		
					//check which we onMessage or onAlarm we received
					for (int i = 0; i < incoming.size(); i++) {

						Element pickElem = (Element)incoming.get(i);

						if (i > 0) {
							state.appendNoLine("} else ");
						}
						if (i < onMessageCount) {
							state.appendLine("if ("+indexFieldName+" == "+i+") {");
							
							if (correlationElements.get(i) != null) {
								translateCorrelations(state,(Element)correlationElements.get(i),(String)correlationFieldNames.get(i),inputsId+i,(String)pick_inputtypes.get(i),false,false);
							}

							//store the XSD type populated by the binding receive
							state.appendLine("BPELVariable.setMessage(engine,\""+pick_variables.get(i)+"\","+pick_inputtypes.get(i)+".toEngineMessage("+inputsId+i+"));");
							
							//compile the activities under this onMessage
							state.appendLine("  //onMessage "+i+" activities");
	
							state.incrementScope();
							state.appendLine("  engine.ipushStack(\"BPEL "+TranslatorUtils.toJavaString(compiler_util.getBpelLineNumberString(pickElem))+" "+TranslatorUtils.toJavaString(pickElem)+"\");");
				
							translateActivities(state,pickElem);
				
							state.appendLine("  engine.ipopStack();");
							state.decrementScope();
						} else {
							state.appendLine("if ("+timeoutIndexId+" == "+i+") {");
							
							state.incrementScope();
							state.appendLine("  engine.ipushStack(\"BPEL "+TranslatorUtils.toJavaString(compiler_util.getBpelLineNumberString(pickElem))+" "+TranslatorUtils.toJavaString(pickElem)+"\");");
				
							//remaining N indexes are always timeouts
							
							state.appendLine("  //onAlarm "+i+" activities");
							
							//compile the activities under this element
							translateActivities(state,pickElem);
							
							Element repeatElem = (Element)alarmRepeats.get(i);
							if (repeatElem != null) {
								String repeat_every = Util.getTextDirectlyUnder(repeatElem);
								String alarmId = "pick_timeout"+pickID+"_"+i;
								
								state.appendNoLine(alarmId+" += ");
								translateForDuration(state,repeat_every,repeatElem);
								state.appendLine(";");

								state.appendLine(loopId+" = true; //repeatEvery");
							}

							state.appendLine("  engine.ipopStack();");
							state.decrementScope();
							
						}
						
						//tail the IF
						if (i == (incoming.size()-1)) {
							state.appendLine("}");
						}
					}

				}
				
			} else {
				throw new BPELTranslatorException(compiler_util,pick,"no single binding translator could translate entire pick, see error log for more details");
			}
		} catch (BPELTranslatorException x) {
			for (int i = 0; i < problems.size(); i++) {
				Exception e = (Exception)problems.get(i);
//				DBG.logVisibleError(e,"Binding Translation Failed:"+e.getMessage(),false);
				log.logWarning(LOG_SOURCE,"Binding Translation Failed:"+e.getMessage());
			}
			throw x;
		}

		state.decrementScope();
		state.appendLine("}//end while");
		
		
//		state.appendLine("  engine.ipopStack();");
	}

	public void translateCondition(BPELState state, Element cond) throws BPELTranslatorException, NamespaceException {
		//compile in an xpath method
		String condition = Util.getTextDirectlyUnder(cond);

		//translate XPATH here
		try {
			XPATHTreeTranslator resolver = new XPATHTreeTranslator();
			String xpath_method = getNextXPath();
			String expr = resolver.resolveToBoolean(xpath_method,condition,compiler_util.getSwitches(),nt,scopes,flow_scopes,wsdlmap,cond);
			state.appendDeclaration(expr);
			
			state.appendNoLine(xpath_method+"()");

			addVariableRefs(resolver.getReferencedVariables(),cond);
			
		} catch (Exception e) {
			throw new BPELTranslatorException(compiler_util,cond,e);
		}
	}
	
	public void translateIf(BPELState state, Element iff) throws BPELTranslatorException, NamespaceException {
		state.appendLine("//IF");

		Element firstcondition = Util.getFirstElement(iff,"condition");
		
		ArrayList cases = new ArrayList();

		if (Util.getFirstElement(iff,"otherwise") != null) {
			log.logWarning(LOG_SOURCE,"<otherwise> element from original <switch> activity has changed to <else>");
		}
		if (Util.getFirstElement(iff,"case") != null) {
			log.logWarning(LOG_SOURCE,"<case> element from original <switch> activity has changed to <then> and <elseif>");
		}
		
		Element then = Util.getFirstElement(iff,"then");
		if (then == null) {
			throw new BPELTranslatorException(compiler_util,iff,"expected <then> element under <if> activity");
		}
		then.appendChild(firstcondition);
		cases.add(then);
		
		cases.addAll(Util.getAllElements(iff,"elseif"));
		
		Element otherwise = Util.getFirstElement(iff,"else");

		for (int i = 0; i < cases.size(); i++) {
			Element cse = (Element)cases.get(i);
			
			if (i == 0) {
				state.appendNoLine("if (");
			} else {
				state.appendNoLine("} else if (");
			}
			
			Element condition = Util.getFirstElement(cse,"condition");
			if (condition == null) throw new BPELTranslatorException(compiler_util,cse,"no condition found under switch case");
			
			translateCondition(state,condition);
			
			state.appendLine(") {");

			//do Dead Path Elimination for all other cases
			for (int k = 0; k < cases.size(); k++) {
				Element alternate = (Element)cases.get(k);
				if (k != i) {
					//do DPE for every case other than this case
					translateDeadPathElimination(state,alternate);
				}
			}
			
			//append stuff under case
			translateActivities(state,cse);
			
		}
		
		if (otherwise != null) {
			state.appendLine("} else {");

			//do Dead Path Elimination for all other cases
			for (int k = 0; k < cases.size(); k++) {
				Element alternate = (Element)cases.get(k);
				
				//do DPE for every case
				translateDeadPathElimination(state,alternate);
			}
			
			
			//append stuff under otherwise
			translateActivities(state,otherwise);
			
			state.appendLine("}");
		} else {
			state.appendLine("}");
		}
		
	}

	public void translateRepeatUntil(BPELState state, Element rpet) throws BPELTranslatorException , NamespaceException{
		state.appendLine("//REPEAT UNTIL");

		Element condition = Util.getFirstElement(rpet,"condition");

		state.appendNoLine("do {");

		translateActivities(state,rpet);

		state.appendLine("} while (");

		if (condition == null) throw new BPELTranslatorException(compiler_util,rpet,"no condition found under repeatUntil");
		translateCondition(state,condition);

		state.appendLine(");");

		state.appendLine("//END REPEAT UNTIL");
	}
	
	public void translateForEach(BPELState state, Element foreach) throws BPELTranslatorException, NamespaceException {
		state.appendLine("//FOREACH");
		
		String counterName = foreach.getAttribute("counterName");
		String parallel = foreach.getAttribute("parallel");
		
		if (counterName == null) {
			throw new BPELTranslatorException(compiler_util,foreach,"'counterName' attribute required on <forEach> activity");
		}
		if (parallel == null) {
			throw new BPELTranslatorException(compiler_util,foreach,"'parallel' attribute required on <forEach> activity");
		}
		
		boolean isParallel = parallel.equals("yes");
		
		if (!isParallel && !parallel.equals("no")) {
			throw new BPELTranslatorException(compiler_util,foreach,"'parallel' attribute on <forEach> activity must be 'yes' or 'no'");
		}

		Element startValueElem = Util.getFirstElement(foreach,"startCounterValue");
		if (startValueElem == null) {
			throw new BPELTranslatorException(compiler_util,foreach,"<forEach> activity must contain a <startCounterValue> element");
		}
		String startCounterValue = Util.getTextDirectlyUnder(startValueElem);
		
		Element finalValueElem = Util.getFirstElement(foreach,"finalCounterValue");
		if (finalValueElem == null) {
			throw new BPELTranslatorException(compiler_util,foreach,"<forEach> activity must contain a <finalCounterValue> element");
		}
		String finalCounterValue = Util.getTextDirectlyUnder(finalValueElem);
		
		//get the BPEL namespace from the forEach node
		String tagname = foreach.getTagName();
		String nsbpel = tagname.substring(0,tagname.indexOf(':')+1);
		
		Document doc = foreach.getOwnerDocument();

		Element subElem;
		Element childrenElem;
		
		if (isParallel) {
			
			//flow
			Element flowElem = doc.createElement(nsbpel+"flow");

			//scope
			Element scopeElem = doc.createElement(nsbpel+"scope");
			scopeElem.setAttribute("isolated","no");
			scopeElem.setAttribute("threadcount","1 + ( "+finalCounterValue+" ) - ( "+startCounterValue+" )");
			
			//variable
			Element variablesElem = doc.createElement(nsbpel+"variables");
			
			Element variableElem = doc.createElement(nsbpel+"variable");
			variableElem.setAttribute("xmlns:xsd",NamespaceTranslator.NAMESPACE_XSD);
			variableElem.setAttribute("name",counterName);
			variableElem.setAttribute("type","xsd:int");
			
			//value assignment
			Element assignElem = doc.createElement(nsbpel+"assign");
			
			Element copyElem = doc.createElement(nsbpel+"copy");
			
			Element fromElem = doc.createElement(nsbpel+"from");
			
			Text valueText = doc.createTextNode("( "+startCounterValue+" ) + getThreadChildIndex() - 1");//start value
			
			Element toElem = doc.createElement(nsbpel+"to");
			toElem.setAttribute("variable",counterName);
			
			flowElem.appendChild(scopeElem);
			scopeElem.appendChild(variablesElem);
			variablesElem.appendChild(variableElem);
			scopeElem.appendChild(assignElem);
			assignElem.appendChild(copyElem);
			copyElem.appendChild(fromElem);
			fromElem.appendChild(valueText);
			copyElem.appendChild(toElem);
			
			subElem = flowElem;
			childrenElem = scopeElem;
			
		} else {

			//scope
			Element scopeElem = doc.createElement(nsbpel+"scope");
			scopeElem.setAttribute("isolated","no");
			
			//variable creation
			Element variablesElem = doc.createElement(nsbpel+"variables");
			
			Element variableElem = doc.createElement(nsbpel+"variable");
			variableElem.setAttribute("xmlns:xsd",NamespaceTranslator.NAMESPACE_XSD);
			variableElem.setAttribute("name",counterName);
			variableElem.setAttribute("type","xsd:int");
			
			Element maxVariableElem = doc.createElement(nsbpel+"variable");
			maxVariableElem.setAttribute("xmlns:xsd",NamespaceTranslator.NAMESPACE_XSD);
			maxVariableElem.setAttribute("name",counterName+"_bpel_forEach_generated_maximum");
			maxVariableElem.setAttribute("type","xsd:int");

			//initial value assign
			Element assignElem = doc.createElement(nsbpel+"assign");
			
			Element copyElem = doc.createElement(nsbpel+"copy");
			
			Element fromElem = doc.createElement(nsbpel+"from");
			
			Text startText = doc.createTextNode("("+startCounterValue+") - 1");//start value
			
			Element toElem = doc.createElement(nsbpel+"to");
			toElem.setAttribute("variable",counterName);
			
			//max value assign
			Element maxCopyElem = doc.createElement(nsbpel+"copy");
			
			Element maxFromElem = doc.createElement(nsbpel+"from");
			
			Text maxText = doc.createTextNode(finalCounterValue);//max value
			
			Element maxToElem = doc.createElement(nsbpel+"to");
			maxToElem.setAttribute("variable",counterName+"_bpel_forEach_generated_maximum");
			
			//while loop
			Element whileElem = doc.createElement(nsbpel+"while");
			
			Element conditionElem = doc.createElement(nsbpel+"condition");
			
//			Text conditionText = doc.createTextNode("$"+counterName+" < ( "+finalCounterValue+" )");
			Text conditionText = doc.createTextNode("$"+counterName+" < $"+counterName+"_bpel_forEach_generated_maximum");
			
			//increment assign
			Element assigniElem = doc.createElement(nsbpel+"assign");
			
			Element copyiElem = doc.createElement(nsbpel+"copy");
			
			Element fromiElem = doc.createElement(nsbpel+"from");
			Text fromiText = doc.createTextNode("$"+counterName+" + 1");
			
			Element toiElem = doc.createElement(nsbpel+"to");
			toiElem.setAttribute("variable",counterName);
			
			//counter variable
			scopeElem.appendChild(variablesElem);
			variablesElem.appendChild(variableElem);
			variablesElem.appendChild(maxVariableElem);

			//assign initial value
			scopeElem.appendChild(assignElem);
			assignElem.appendChild(copyElem);
			copyElem.appendChild(fromElem);
			copyElem.appendChild(toElem);
			fromElem.appendChild(startText);
			
			//assign max value
			assignElem.appendChild(maxCopyElem);
			maxCopyElem.appendChild(maxFromElem);
			maxCopyElem.appendChild(maxToElem);
			maxFromElem.appendChild(maxText);
			
			//serial while loop
			scopeElem.appendChild(whileElem);
			whileElem.appendChild(conditionElem);
			conditionElem.appendChild(conditionText);
			
			//increment assign
			whileElem.appendChild(assigniElem);
			assigniElem.appendChild(copyiElem);
			copyiElem.appendChild(fromiElem);
			copyiElem.appendChild(toiElem);
			fromiElem.appendChild(fromiText);
			
			subElem = scopeElem;
			childrenElem = whileElem;

		}

		//switch the children under this forEach activity
		NodeList childrenList = foreach.getChildNodes();
		ArrayList children = new ArrayList();
		for (int i = 0; i < childrenList.getLength(); i++) {
			children.add(childrenList.item(i));
		}

		for (int i = 0; i < children.size(); i++) {
			foreach.removeChild((Node)children.get(i));
		}
		for (int i = 0; i < children.size(); i++) {
			childrenElem.appendChild((Node)children.get(i));
		}
		foreach.appendChild(subElem);

//		try {
//			String TEMP = XMLUtil.nodeToString(subElem);
//			System.out.println("FOREACH TRANSLATION");
//			System.out.println(TEMP);
//		} catch (Exception e) {
//			e.printStackTrace();
//		}
		
		//translate the generated forEach activities
		translateActivities(state,foreach);

		foreach.removeChild(subElem);
		//switch them back (no real reason to do this but just in case)
		for (int i = 0; i < children.size(); i++) {
			childrenElem.removeChild((Node)children.get(i));
		}
		for (int i = 0; i < children.size(); i++) {
			foreach.appendChild((Node)children.get(i));
		}

		
		state.appendLine("//END FOREACH");
	}
	
	public void translateWhile(BPELState state, Element whle) throws BPELTranslatorException , NamespaceException{
		state.appendLine("//WHILE");

		Element condition = Util.getFirstElement(whle,"condition");

		state.appendNoLine("while (");
		
		if (condition == null) throw new BPELTranslatorException(compiler_util,whle,"no condition found under while");
		translateCondition(state,condition);
		
		state.appendLine(") {");

		translateActivities(state,whle);

		state.appendLine("}");
		state.appendLine("//END WHILE");
	}
	
	public void translateSequence(BPELState state, Element sequence) throws BPELTranslatorException, NamespaceException {
		state.appendLine("//SEQUENCE");
		
		translateActivities(state,sequence);

		state.appendLine("//END SEQUENCE");
	}
	
	public void translateTargets(BPELState state, Element targets) throws BPELTranslatorException {
		Element join = Util.getFirstElement(targets,"joinCondition");
		
		if (join != null) {
			//explicit join condition
			String joinExpression = Util.getTextDirectlyUnder(join);
			
			try {
			
				XPATHTreeTranslator resolver = new XPATHTreeTranslator();
				String xpath_method = getNextXPath();
				String expr = resolver.resolveLinksToBoolean(xpath_method,joinExpression,compiler_util.getSwitches(),nt,scopes,flow_scopes,wsdlmap,join);
				state.appendDeclaration(expr);

				state.appendLine("if ("+xpath_method+"()) { //begin link scope");
				
			} catch (Exception e) {
				throw new BPELTranslatorException(compiler_util,join,e);
			}

		} else {
			//implicit join condition
			
			boolean found = false;
			
			ArrayList list = Util.getAllElements(targets);
			for (int i = 0; i < list.size(); i++) {
				Element standard = (Element)list.get(i);
				String tagname = NamespaceTranslator.getName(standard);
				if (tagname.equals("target")) {
					
					if (!found) {
						found = true;
						state.appendLine("//LINK - TARGET (Message Wait)");
						state.incrementScope();
						state.appendNoLine("if (");
					} else {
						state.appendNoLine("&&");
					}
					
					String linkName = standard.getAttribute("linkName");
					state.appendNoLine("BPELLink.incoming(engine,\""+flow_scopes.getLinkName(linkName,join)+"\")");
				}
			}

			if (found) {
				state.appendLine(") { //begin link scope");
			}
		}
	}
	
	public void translateSources(BPELState state, Element sources) throws BPELTranslatorException {
		boolean found = false;
		
		ArrayList list = Util.getAllElements(sources);
		for (int i = 0; i < list.size(); i++) {
			Element standard = (Element)list.get(i);
			String tagname = NamespaceTranslator.getName(standard);
			if (tagname.equals("source")) {
				if (!found) {
					found = true;
					state.appendLine("//LINK - SOURCE (Message Send)");
					state.incrementScope();
				}
				
				String linkName = standard.getAttribute("linkName");
				Element transition = Util.getFirstElement(standard,"transitionCondition");
				
				if (transition != null) {
					String transitionExpression = Util.getTextDirectlyUnder(transition);
					
					try {
					
						XPATHTreeTranslator resolver = new XPATHTreeTranslator();
//System.out.println(transitionExpression);						
						String xpath_method = getNextXPath();
						String expr = resolver.resolveToBoolean(xpath_method,transitionExpression,compiler_util.getSwitches(),nt,scopes,flow_scopes,wsdlmap,transition);
						state.appendDeclaration(expr);
		
						state.appendLine("BPELLink.outgoing(engine,\""+flow_scopes.getLinkName(linkName,transition)+"\","+xpath_method+"());");
						
					} catch (Exception e) {
						throw new BPELTranslatorException(compiler_util,transition,e);
					}
				} else {
					state.appendLine("BPELLink.outgoing(engine,\""+flow_scopes.getLinkName(linkName,transition)+"\",true);");
				}
				
			}
		}

		if (found) {
			state.decrementScope();
			state.appendLine("");
		}
	}
	
	public void translatePreEffectiveStandard(BPELState state, Element standards) throws BPELTranslatorException {
		//standard elements that need to be compiled in BEFORE an activity
		
		Element targets = Util.getFirstElement(standards,"targets");
		if (targets != null) {
			translateTargets(state,targets);
		}
		
		state.decrementScope();
		state.appendLine("");
	} 
	
	public String[] getDeclaredLinks(Element flow) throws BPELTranslatorException {
		Element links_elem = Util.getFirstElement(flow,"links");
		ArrayList links = Util.getAllElements(links_elem,"link");
		
		String[] linkNames = new String[links.size()];
		for (int i = 0; i < links.size(); i++) {
			Element link = (Element)links.get(i);
			linkNames[i] = link.getAttribute("name");
		}
		
		return linkNames;
	}
	
	public void translateDeadPathElimination(BPELState state, Element dpe) throws BPELTranslatorException {

		// - search through this entire tree
		// - if we find a link source then we may need to do an BPELLink.outgoing(false)
		// - if we find any link declarations then any sources to those links beneath can be ignored

		state.appendLine("//DEAD PATH ELIMINATION");
		r_translateDeadPathElimination(state,dpe,new ArrayList());
		
	}
	
	public void r_translateDeadPathElimination(BPELState state, Element dpe, ArrayList ignored_links) throws BPELTranslatorException {
		// recursively traverse each element and search for link declarations beneath it.
		// if it has declarations add them to an arraylist of ignored links, do depth first on the remaining elements
		// and pop the ignored links off when we're done

		String tagname = NamespaceTranslator.getName(dpe);

		//check this activity
		if (tagname.equals("sources")) {
			ArrayList sources = Util.getAllElements(dpe,"source");
			for (int i = 0; i < sources.size(); i++) {
				Element source = (Element)sources.get(i);
				String linkName = source.getAttribute("linkName");
				
				boolean ignored = false;
				
				for (int k = 0; k < ignored_links.size(); k++) {
					if (linkName.equals(ignored_links.get(k))) {
						ignored = true;
					}
				}
				
				if (!ignored) {
					//compile in DPE outgoing link -> negative
					state.appendLine("BPELLink.outgoing(engine,\""+flow_scopes.getLinkName(linkName,source)+"\",false);");
				}
			}
			
		} else {
		
			//add any links declared in this activity to the ignored list
			//NOTE - this must come before we check the children because the links declared under this flow
			//       would be used by the children activities
			String[] links = new String[0];
			if (tagname.equals("flow")) {
				links = getDeclaredLinks(dpe);
			}

			//Add the ignored links from this activity
			for (int i = 0; i < links.length; i++) {
				ignored_links.add(links[i]);
			}
			
			//check all children
			ArrayList children = Util.getAllElements(dpe);
			for (int i = 0; i < children.size(); i++) {
				Element child = (Element)children.get(i);
				r_translateDeadPathElimination(state,child,ignored_links);
			}
			
			//Remove the ignored links from this activity
			for (int i = 0; i < links.length; i++) {
				ignored_links.remove(ignored_links.size()-1);
			}
		}
	}
	
	public void translatePostEffectiveStandard(BPELState state, Element standards) throws BPELTranslatorException {
		//standard elements that need to be compiled in AFTER an activity

		Element targets = Util.getFirstElement(standards,"targets");
		Element sources = Util.getFirstElement(standards,"sources");
		
		if (sources != null) {
			translateSources(state,sources);
		}
		
		if (targets != null) {
			state.appendLine("} else {");
			
			translateDeadPathElimination(state,standards);

			state.appendLine("}//end link scope");
		}
		
	}
	
	public void translateLink(BPELState state, Element link) throws BPELTranslatorException {
		String name = link.getAttribute("name");
		
		//add this link to the flow's <links> scope
		flow_scopes.addLink(name);
	}
	
	public void translateLinks(BPELState state, Element links) throws BPELTranslatorException, NamespaceException {
		state.appendLine("//LINKS");
		state.incrementScope();		
		
		ArrayList list = Util.getAllElements(links);
		for (int i = 0; i < list.size(); i++) {
			translateLink(state,(Element)list.get(i));
		}
		
		state.decrementScope();		
		state.appendLine("//END LINKS");
	}

	public void translateCorrelationSet(BPELState state, Element corrset) throws BPELTranslatorException {
		nt.addNamespaces(corrset);
		
		String name = corrset.getAttribute("name");
		String propertiesList = corrset.getAttribute("properties");
		
		if (propertiesList == null) {
			throw new BPELTranslatorException(compiler_util,corrset,"<correlationSet> element must specify a 'properties' attribute");
		}
		String[] properties = Util.split(propertiesList);
		
		for (int i = 0; i < properties.length; i++) {
			try {
				properties[i] = nt.qualify(properties[i],false);
			} catch (NamespaceException e) {
				throw new BPELTranslatorException(compiler_util,corrset,"Failed to translate namespace for property "+properties[i]+" in correlation set definition");
			}
		}
		
		scopes.addCorrelationSet(name,properties);
		
		//whenever we enter this scope we must set the correlation set to uninitialised
		//so that faults can be thrown on attempts to access an uninitialised correlation set
		state.appendLine("BPELCorrelationSet.setUninitialised(engine,\""+scopes.getCorrelationSetName(name,corrset)+"\");\n");
		nt.removeNamespaces(corrset);
	}
	
	public void translateCorrelationSets(BPELState state, Element corrsets) throws BPELTranslatorException, NamespaceException {
		state.appendLine("//CORRELATION SETS");
		state.incrementScope();		
		
		ArrayList list = Util.getAllElements(corrsets);
		for (int i = 0; i < list.size(); i++) {
			translateCorrelationSet(state,(Element)list.get(i));
		}
		
		state.decrementScope();		
		state.appendLine("//END CORRELATION SETS");
	}
	
	public void translateExit(BPELState state, Element xit) throws BPELTranslatorException, NamespaceException {
		state.appendLine("//EXIT");
		state.appendLine("engine.terminate();");
	}
	
	private int pushScope(BPELState state, Element source, boolean isolated, boolean ignoreParentIsolation) throws BPELTranslatorException, NamespaceException {
		int isolatedLockIndex = state.getIndex();

		BPELThread currthread = (BPELThread)bpel_threads.peek();

		scopes.pushScope("bpelThreadIdList.get("+(bpel_threads.size()-1)+")",isolated,source,ignoreParentIsolation);

		return isolatedLockIndex;
	}
	
	private void popScope(BPELState state, Element source, int isolatedLockIndex, boolean isAssign) throws BPELTranslatorException, NamespaceException {

		translateLockUsedVariables(state,isolatedLockIndex,isAssign,source);
		scopes.popScope(source);
	}
	
	public void translateScope(BPELState state, Element scope) throws BPELTranslatorException, NamespaceException {
		
		String isolated = scope.getAttribute("isolated");
		
		if (isolated.length() > 0) {
			if (!isolated.equals("yes") && !isolated.equals("no")) {
				log.logWarning(LOG_SOURCE,"Attribute 'isolated' on scope should be 'yes' or 'no' (case sensitive)");
			}
		}

		String debugVariablesId = "bpel_debug_variables_id"+nextUID();
		if (debug) {
			state.appendDeclaration("Message "+debugVariablesId+";");
			state.appendLine("    "+debugVariablesId+" = new Message("+DebugConstants.STACKTRACE_VARIABLES+");");
			state.appendLine("    engine.ipushStack("+debugVariablesId+");");
		}

		int isolatedLockIndex = pushScope(state,scope,isolated.equals("yes"),false);
		
		state.appendLine("//SCOPE");

		Element tmp;
		tmp = Util.getFirstElement(scope,"variables");
		if (tmp != null) translateVariables(state,tmp,debugVariablesId);
		state.appendLine("");
		tmp = Util.getFirstElement(scope,"correlationSets");
		if (tmp != null) translateCorrelationSets(state,tmp);
		state.appendLine("");
		Element faultHandlers = Util.getFirstElement(scope,"faultHandlers");
		if (faultHandlers != null) translatePreFaultHandlers(state,faultHandlers);
		state.appendLine("");
		//TODO translate Compensation Handlers
//		tmp = XMLConfigUtil.getFirstElement(children,"compensationHandlers")
//		if (tmp != null) translateCompensationHandlers(state,tmp);
//		state.appendLine("");
		//TODO translate Event Handlers
//		tmp = XMLConfigUtil.getFirstElement(children,"eventHandlers");
//		if (tmp != null) translateEventHandlers(state,tmp);
//		state.appendLine("");
		//TODO translate Termination Handlers
//		tmp = XMLConfigUtil.getFirstElement(children,"terminationHandlers");
//		if (tmp != null) translateTerminationHandlers(state,tmp);
//		state.appendLine("");
		
		translateActivities(state, scope);

		popScope(state,scope,isolatedLockIndex,false);
		
		if (faultHandlers != null) translatePostFaultHandlers(state,faultHandlers);
		
		if (debug) {
			state.appendLine("    engine.ipopStack();");
		}
		
	}
	
	private void translateLockUsedVariables(BPELState state, int prefix_index, boolean isAssign, Node source) throws BPELTranslatorException, NamespaceException {
		String[] names = scopes.getIsolatedVariableNames(source);
		
		if (names.length > 0) {
			Arrays.sort(names);
			
			String type = "SCOPE";
			if (isAssign) type = "ASSIGN";
			
			log.logInfo(LOG_SOURCE,"Variable Locking Order ("+type+") "+compiler_util.getBpelLineNumberString(source)+":");
		}
		
		StringBuffer lockBuffer = new StringBuffer();
		StringBuffer releaseBuffer = new StringBuffer();
		
		//
		// Translate automatic release of all locks when faults are thrown
		//
		lockBuffer.append("try {\n");
		
		for (int i = 0; i < names.length; i++) {
			String scopedName = scopes.getVariableName(names[i],source);
			
			if (isAssign) {
				scopedName = "ALOCK:\"+\""+scopedName;
			} else {
				scopedName = "SLOCK:\"+\""+scopedName;
			}
			
			log.logInfo(LOG_SOURCE,"  "+(1+i)+" - variable '"+names[i]+"' (\""+scopedName+"\")");
			
			BPELThread thread = (BPELThread)bpel_threads.peek();
			
			ArrayList parentlist = new ArrayList();
			
			boolean isMultipleThreads = thread.multiple_copies;
			
			BPELThread tmpthread = thread.getParent();
			while (tmpthread != null) {
				parentlist.add(""+tmpthread.id);
				if (tmpthread.multiple_copies) {
					isMultipleThreads = true;
				}
				
				tmpthread = tmpthread.getParent();
			}
			
			String[] parents = new String[parentlist.size()];
			parentlist.toArray(parents);

/*NEW LOCK RESOLVER			
			if (isAssign) {
				assign_lock_resolver.addLock(parents,""+thread.id,scopedName,false);
				if (isMultipleThreads) {
					assign_lock_resolver.addLock(parents,""+thread.id+"_duplicate",scopedName,true);
				}
			} else {
				scope_lock_resolver.addLock(parents,""+thread.id,scopedName,false);
				if (isMultipleThreads) {
					scope_lock_resolver.addLock(parents,""+thread.id+"_duplicate",scopedName,true);
				}
			}
			*/
//need to add to LockResolver here.  We are in scope so we need to know if the current thread is duplicated via
//a 'threadcount' attribute at any point.  We could potentially make this a simple boolean as all translation
//will happen in a single chain
//
//either that or do we have something to represent the current thread?
//		
//we need to call a declared method here and then generate the methods at the end of the translation
//
//when we generate the methods we only add a body if that variable is supposed to be locked.

			if (!compiler_util.getSwitches().LOCK_RESOLVING) {
				lockBuffer.append("BPELVariable.lockVariable(engine,\""+scopedName+"\");\n");
//				state.appendLine("BPELVariable.releaseVariable(engine,\""+scopedName+"\");");
				releaseBuffer.append("BPELVariable.releaseVariable(engine,\""+scopedName+"\");\n");
			} else {
				String methodName = "bpelVariableLock_"+nextUID();

				if (isAssign) {
					//assign
					lock_model.pushSync(compiler_util,source,"AssignLock:"+names[i],methodName,scopes.getScopeLogicalStart(),scopes.getScopeLogicalEnd());
				} else {
					//isolated scope
					lock_model.pushSync(compiler_util,source,"IScopeLock:"+names[i],methodName,scopes.getScopeLogicalStart(),scopes.getScopeLogicalEnd());
				}
				
				thread.addVariableLock(methodName,scopedName);
				lockBuffer.append(methodName+"_LOCK(\""+scopedName+"\");\n");
//				state.appendLine(methodName+"_RELEASE(\""+scopedName+"\");\n");
				releaseBuffer.append(methodName+"_RELEASE(\""+scopedName+"\");\n");
			}
		}

		//
		// Translate automatic release of all locks when faults are thrown
		//
		state.appendLine(releaseBuffer.toString());

		String tmpFaultName = "bpel_temporary_fault_"+nextUID();
		state.appendLine("} catch (BPELFault "+tmpFaultName+") {");

		state.appendLine(releaseBuffer.toString());
		
		state.appendLine("  throw "+tmpFaultName+";");
		state.appendLine("}");
		
		//insert the locking code
		state.insertLine(lockBuffer.toString(),prefix_index);

	}
	
	private void translateActivities(BPELState state, Element parent) throws BPELTranslatorException, NamespaceException {
		state.incrementScope();
		ArrayList list = Util.getAllElements(parent);
		for (int i = 0; i < list.size(); i++) {
			BPELState newstate = state.newState();
			state.appendLine(newstate.getMethodName()+"();");
			translateActivity(newstate,(Element)list.get(i),i+1);
//We start hitting the 63353 bytes per method code limit if we don't generate new states
//			translateActivity(state,(Element)list.get(i),i+1);
		}
		state.decrementScope();
	}
	
	private String getVariableFrom(String query, Element source) throws BPELTranslatorException, NamespaceException {
		StringBuffer variablebuf = new StringBuffer();
		
		int varStart = query.indexOf('$');
		
		if (varStart == -1) {
			throw new BPELTranslatorException(compiler_util,source,"could not find variable name in assignment query");
		}
		
		for (int k = varStart+1; k < query.length(); k++) {
			char c = query.charAt(k);
			if (c == '/'
				|| c == '.'
				|| c == '\\'
				|| Character.isWhitespace(c)) {
				break;
			} else {
				variablebuf.append(c);
			}
		}
		
		String tovariable = variablebuf.toString();

		return tovariable;
	}

	public void translateJavaImport(BPELState state, Element jimport) throws BPELTranslatorException, NamespaceException {
		String pkg = jimport.getAttribute("package");
		
		if (pkg == null) {
			throw new BPELTranslatorException(compiler_util,jimport,"<java_import> element without 'package' attribute");
		}
		if (pkg.trim().length()==0) {
			throw new BPELTranslatorException(compiler_util,jimport,"<java_import> element with empty 'package' attribute");
		}
		
		runner_imports = runner_imports+"\nimport "+pkg+";";
	}
	
	public void translateJavaDebug(BPELState state, Element jdebug) throws BPELTranslatorException, NamespaceException {
		try {
			
			state.appendLine("//JAVA DEBUG");
			
			String vars = jdebug.getAttribute("variables");
			
			String[] variables = new String[0];
			
			if (vars == null) vars = "";
			if (vars.length() > 0) {
				variables = Util.split(vars);
			}

//			NamespaceTranslator tmpNT = compiler_util.createNamespaceTranslator();

			if (variables.length > 0) {
				state.appendLine("{");
			}
			
			for (int i = 0; i < variables.length; i++) {
				//fetch all the requested variables into the expected field names
				
				String toType = getNextXPath();
				String toString = getNextXPath();
				String toObj = getNextXPath();
				
				XPATHTreeTranslator translator = new XPATHTreeTranslator();
				state.appendDeclaration(translator.resolveToXmlString(toType,variables[i],compiler_util.getSwitches(),nt,scopes,flow_scopes,wsdlmap,jdebug));
				addVariableRefs(translator.getReferencedVariables(),jdebug);
				
				translator = new XPATHTreeTranslator();
				state.appendDeclaration(translator.resolveToString(toString,variables[i],compiler_util.getSwitches(),nt,scopes,flow_scopes,wsdlmap,jdebug));
				addVariableRefs(translator.getReferencedVariables(),jdebug);
				
				translator = new XPATHTreeTranslator();
				state.appendDeclaration(translator.resolveToNodeSet(toObj,variables[i],compiler_util.getSwitches(),nt,scopes,flow_scopes,wsdlmap,jdebug));
				addVariableRefs(translator.getReferencedVariables(),jdebug);
				
				state.appendLine("String xpath_var"+i+" = \"\"+"+toString+"();");
				state.appendLine("String xml_var"+i+";");
				state.appendLine("XNode obj_var"+i+";\n");
				state.appendLine("String ret_var"+i+" = null;");
				state.appendLine("try {\n");
				state.appendLine("	xml_var"+i+" = \"\"+"+toType+"();");
				state.appendLine("} catch (Exception e) {\n");
				state.appendLine("	xml_var"+i+" = xpath_var"+i+";");
				state.appendLine("}\n");
				state.appendLine("try {\n");
				state.appendLine("	obj_var"+i+" = "+toObj+"().get(0);\n");
				state.appendLine("} catch (Exception e) {\n");
				state.appendLine("	obj_var"+i+" = null;");
				state.appendLine("}\n");
			}
			
			NodeList list = jdebug.getChildNodes();
			for (int i = 0; i < list.getLength(); i++) {
				Node node = list.item(i);
				if (node.getNodeType() == Node.TEXT_NODE) {
					state.appendLine(""+node.getNodeValue());
				}
			}
			
			if (variables.length > 0) {
				for (int i = 0; i < variables.length; i++) {
					String tovariable = getVariableFrom(variables[i],jdebug);
					String qtype = scopes.getVariableQtype(tovariable,jdebug);
					addVariableRef(tovariable,jdebug);
					if (qtype == null) throw new BPELTranslatorException(compiler_util,jdebug,"BPEL <java_debug>: could not find type for variable "+tovariable);			

					state.appendLine("if (ret_var"+i+" != null) {\n");

					//get it to recreate itself based on the contents of the XML
					state.appendLine("obj_var"+i+".setValueInContainer( ((org.eclipse.stp.b2j.core.jengine.internal.extensions.wsdlbinding.internal.XMLAccessible)obj_var"+i+".getValue()).ifromXML(ret_var"+i+"), true);\n");
					
					//get the root XNode which should corresponsd to the WSDL message fetched by the XPATH?
					state.appendLine("while (obj_var"+i+".getParent() != null) obj_var"+i+" = obj_var"+i+".getParent();");
					
					//set the variable based on this modified message
					state.appendLine("BPELVariable.setMessage(engine,\""+scopes.getVariableName(tovariable,jdebug)+"\", "+qtype+".toEngineMessage(obj_var"+i+".getValue()));");
					
					
					state.appendLine("}\n");
				}
				
				state.appendLine("}");
			}
			state.appendLine("//END JAVA DEBUG");
			
		} catch (Exception e) {
			throw new BPELTranslatorException(compiler_util,jdebug,e);
		}
			
	}
	
	public void translateActivity(BPELState newstate, Element activity, int index) throws BPELTranslatorException, NamespaceException {

		nt.addNamespaces(activity);

		if (compiler_util.getSwitches().APPEND_BPEL_TO_CALLSTACK) {
			newstate.appendLine("engine.ipushStack(\"BPEL "+TranslatorUtils.toJavaString(activity)+" (elem"+index+") "+TranslatorUtils.toJavaString(compiler_util.getBpelLineNumberString(activity))+"\");");
		}
		
		if (debug) {
			newstate.appendLine("bpelDebugSuspend.varFetch();");
			newstate.appendLine("while (bpelDebugSuspend.getValueInt() == 1) {");
			newstate.appendLine("  try {");
			newstate.appendLine("    Thread.sleep(2500);");
			newstate.appendLine("  } catch (Exception bpelDebugSuspendInterrupt) {}");
			newstate.appendLine("  bpelDebugSuspend.varFetch();");
			newstate.appendLine("}//end while");
		}

		translatePreEffectiveStandard(newstate,activity);

		String tagname = NamespaceTranslator.getName(activity);
		
		if (tagname.equals("receive")) {
			translateReceive(newstate,activity);	
		} else if (tagname.equals("reply")) {
			translateReply(newstate,activity);	
		} else if (tagname.equals("invoke")) {
			translateInvoke(newstate,activity);
		} else if (tagname.equals("assign")) {
			translateAssign(newstate,activity);
		} else if (tagname.equals("throw")) {
			throw new BPELTranslatorException(compiler_util,activity,"<throw> activity is not currently supported");
		} else if (tagname.equals("exit")) {
			translateExit(newstate,activity);
		} else if (tagname.equals("wait")) {
			translateWait(newstate,activity);
		} else if (tagname.equals("empty")) {
			translateEmpty(newstate,activity);
		} else if (tagname.equals("sequence")) {
			translateSequence(newstate,activity);
		} else if (tagname.equals("if")) {
			translateIf(newstate,activity);
		} else if (tagname.equals("switch")) {
			throw new BPELTranslatorException(compiler_util,activity,"<switch> activity has been renamed in a later BPEL spec to <if>");
//			translateSwitch(newstate,activity);
		} else if (tagname.equals("while")) {
			translateWhile(newstate,activity);
		} else if (tagname.equals("forEach")) {
			translateForEach(newstate,activity);
		} else if (tagname.equals("pick")) {
			translatePick(newstate,activity);
		} else if (tagname.equals("flow")) {
			translateFlow(newstate,activity);
		} else if (tagname.equals("scope")) {
			translateScope(newstate,activity);
		} else if (tagname.equals("compensate")) {
			//???
		} else if (tagname.equals("java_import")) {
			translateJavaImport(newstate,activity);
		} else if (tagname.equals("java_debug")) {
			translateJavaDebug(newstate,activity);
		} else if (tagname.equals("for")) {
			//ignore - part of a subexpression
		} else if (tagname.equals("until")) {
			//ignore - part of a subexpression
		} else if (tagname.equals("condition")) {
			//ignore - part of a subexpression
		} else if (tagname.equals("repeatEvery")) {
			//ignore - part of a subexpression
		} else if (tagname.equals("variables")) {
			//ignore - part of a subexpression
		} else if (tagname.equals("variable")) {
			//ignore - part of a subexpression
		} else if (tagname.equals("terminate")) {
//			log.logWarning(LOG_SOURCE,"<terminate> activity has been renamed in a later BPEL spec to <exit> - this terminate activity will be ignored");
//			translateExit(newstate,activity);
			throw new BPELTranslatorException(compiler_util,activity,"<terminate> activity has been renamed in a later BPEL spec to <exit>");
		} else if (tagname.equals("import")) {
			//ignore - import tag
		} else if (tagname.equals("startCounterValue")) {
			//ignore - part of a subexpression
		} else if (tagname.equals("finalCounterValue")) {
			//ignore - part of a subexpression
		} else if (tagname.equals("partnerLinks")) {
			//ignore - part of a subexpression
		} else {
//DBG			DBG.warning("unrecognised element compiling activity \""+tagname+"\"");
			log.logWarning(LOG_SOURCE,"unrecognised element compiling activity \""+tagname+"\"");
		}

		translatePostEffectiveStandard(newstate,activity);

		if (compiler_util.getSwitches().APPEND_BPEL_TO_CALLSTACK) {
			newstate.appendLine("engine.ipopStack();");
		}

		nt.removeNamespaces(activity);
		
	}
	
	public void translateProcess(BPELState state, Element process) throws BPELTranslatorException, NamespaceException {
		String name = process.getAttribute("name");

		lock_model = new LockModel(name);
		
		//create the main flow
		BPELThread thread = new BPELThread();
		thread.state = state;
		bpel_threads.push(thread);
		
		nt.addNamespaces(process);

//		NodeList children = process.getChildNodes();

		Element tmp;

		state.appendLine("    engine.ipushStack(\"BPEL **new thread - bpel process "+TranslatorUtils.toJavaString(compiler_util.getBpelLineNumberString(process))+"**\");");
		if (debug) {
			state.appendLine("    try {");
		}

		state.appendLine("    bpelThreadId = \""+FIRST_THREAD_ID+"\";");
		state.appendLine("    bpelThreadIdList.append(bpelThreadId);");

		if (debug) {
			state.appendLine("    bpelThreadMap = engine.newHashMap(\""+DebugConstants.BPEL_THREADS_MAP+"\");");
			state.appendLine("    bpelThreadMap.put(bpelThreadId,bpelThreadId);");
			state.appendLine("    bpelDebugMap = engine.newHashMap(\""+DebugConstants.BPEL_DEBUG_MAP+"\");");
			state.appendLine("    bpelDebugSuspend = engine.newVariable(\""+DebugConstants.BPEL_BREAKPOINT_VAR+"\",SharedVariable.INTEGER,true);");
			state.appendLine("    bpelDebugSuspend.setValue((int)0);");
			state.appendLine("    bpelDebugSuspend.varStore();");
			
			state.appendLine("    bpelDebugStack = new Message("+DebugConstants.STACKTRACE_THREAD_NAME+");");
			state.appendLine("    bpelDebugStack.append(bpelThreadId);");
			state.appendLine("    engine.ipushStack(bpelDebugStack);");
		}
		
		state.appendDeclaration("boolean localVariableInit"+thread.id+" = false;");
		state.appendLine("if (!localVariableInit"+thread.id+") {");
		state.mark(INSERT_LOCAL_VARIABLE_CREATION);
		state.appendLine("}");
		
		if (compiler_util.getSwitches().SINGLE_HOST_ASSET_OPTIMISATION) {
			state.appendConstructor("  //BPEL ASSET DEPLOYMENT");
			state.appendConstructor("  checkForAssetLocalisation();");
		}

/*		//translate imports
		ArrayList imports = Util.getAllElements(process,"import");
		for (int i = 0; i < imports.size(); i++) {
			tmp = (Element)imports.get(i);
			String import_namespace = tmp.getAttribute("namespace");
			String import_location = tmp.getAttribute("location");
			String import_type = tmp.getAttribute("importType");
			//TODO translate import
		}
	*/	
		
		//translate Partner Links
		tmp = Util.getFirstElement(process,"partnerLinks");
		if (tmp != null) translatePartnerLinks(state,tmp);
		state.appendLine("");
		
		translateScope(state, process);

		if (debug) {
			state.appendLine("    } catch (Exception bpelRunnerThreadException) {");
			state.appendLine("      bpelDebugMap.put(bpelThreadId+\"_TERMINATED\",new Integer(1));");
			state.appendLine("      throw bpelRunnerThreadException;");
			state.appendLine("    }");
		}
		state.appendLine("    engine.ipopStack();");
		
		//go through each asset and convert it to local if possible
		ArrayList assets = getAllAssets();
		for (int i = 0; i < assets.size(); i++) {
			BPELAsset asset = (BPELAsset)assets.get(i);
			if (canBeLocal(asset)) {
				asset.setLocal(true);
			}
			String atype;
			if (asset.getType() == BPELAsset.BPEL_LINK) {
				atype = "(Synchronisation Link)";
			} else {
				atype = "(Variable)";
			}
			state.appendDeclaration("//"+asset.getJavaRef()+" - "+asset.getName()+" "+atype+"");
			if (compiler_util.getSwitches().FLOW_GRAPH_ASSET_OPTIMISATION) {
				state.appendDeclaration("static boolean "+asset.getJavaRef()+" = "+asset.isLocal()+";");
				
				if (asset.isLocal()) {
					ArrayList threads = asset.threads;
					if (threads.size() == 0) {
//DBG 						DBG.info("asset "+asset.basename+" is never accessed");
						log.logInfo(LOG_SOURCE,"asset "+asset.basename+" is never accessed");
					} else if (threads.size() == 1) {
						BPELThread refthread = (BPELThread)threads.get(0);
						BPELState refstate = refthread.state;
						
						//note that we done need synchronisation or anything since by definition of it being local
						//there will only be 1 thread accessing it at once

						//need to compile in creation of the asset to this threads BPELState
						int mark = refstate.getMark(INSERT_LOCAL_VARIABLE_CREATION);

						String qmsg = asset.getVariableQType();

						refstate.prog.insert(mark,"  if ("+asset.getJavaRef()+") BPELVariable.createVariable(engine,\""+asset.name+"\","+qmsg+".toEngineMessage(new "+qmsg+"()),"+asset.getJavaRef()+");\n");
						
					} else {
						log.logWarning(LOG_SOURCE,"asset is local but accessing thread pool size is != 1 ("+threads.size()+")");
//DBG						DBG.error("asset is local but accessing thread pool size is != 1 ("+threads.size()+")");
					}
				}
				
			} else {
				state.appendDeclaration("static boolean "+asset.getJavaRef()+" = false;");
			}
		}
		if (compiler_util.getSwitches().SINGLE_HOST_ASSET_OPTIMISATION) {
			state.appendDeclaration("");
			state.appendDeclaration("//ASSET DEPLOYMENT SPECIAL CASE");
			state.appendDeclaration("public void checkForAssetLocalisation() throws Exception {");
			state.appendDeclaration("  if (engine.getHostsArray().length == 1) {");
			state.appendDeclaration("    //we are running on a single host - everything can be local");
			for (int i = 0; i < assets.size(); i++) {
				BPELAsset asset = (BPELAsset)assets.get(i);
				state.appendDeclaration("    "+asset.getJavaRef()+" = true;");
			}
			state.appendDeclaration("  }");
			state.appendDeclaration("}");
		}
		
		if (compiler_util.getSwitches().LOCK_RESOLVING) {

			/*NEW LOCK RESOLVER			
			assign_lock_resolver.resolve();
			scope_lock_resolver.resolve();
			*/
			
			//for each thread, iterate through the list of required locks and build the relevant method
			//as a declaration
			
//			log.logWarning(LOG_SOURCE,lock_model.toString());
//			System.out.println(lock_model);
			new LockModelScopeOptimiser().optimise(lock_model);
			new LockModelEquivalenceOptimiser().optimise(lock_model);

			log.logInfo(LOG_SOURCE,"Variable lock analysis:");
			log.logInfo(LOG_SOURCE,lock_model.toString());
			log.logInfo(LOG_SOURCE,"Variable lock analysis (enabled locks only):");
			log.logInfo(LOG_SOURCE,lock_model.toStringNoDisabled());
//			System.out.println(lock_model);
			
			buildLockMethods(thread,state);
		}
		
		nt.removeNamespaces(process);
	}

	private void buildLockMethods(BPELThread thread, BPELState state) {
		ArrayList methods = thread.variableLockMethods;
		ArrayList names = thread.variableScopedNames;
		
		System.out.println("Building lock methods for "+thread.id+" ("+names.size()+")");

		for (int i = 0; i < names.size(); i++) {
			String scopedName = (String)names.get(i);
			String method = (String)methods.get(i);

			LockSync sync = lock_model.getSyncForMethodAlias(method);
			
			state.appendDeclaration("public void "+method+"_LOCK(String name) throws Exception {\n");
//			if (assign_lock_resolver.mustLock(scopedName) || scope_lock_resolver.mustLock(scopedName)) {
			if (sync.isEnabled()) {
				state.appendDeclaration("  BPELVariable.lockVariable(engine,name);\n");
//				log.logWarning(LOG_SOURCE,"  Lock REQUIRED "+scopedName);
			} else {
				state.appendDeclaration("  //No lock required (optimisation)\n");
//				log.logWarning(LOG_SOURCE,"  Lock ignored "+scopedName);
			}
			state.appendDeclaration("}\n\n");
			state.appendDeclaration("public void "+method+"_RELEASE(String name) throws Exception {\n");
//			if (assign_lock_resolver.mustLock(scopedName) || scope_lock_resolver.mustLock(scopedName)) {
			if (sync.isEnabled()) {
				state.appendDeclaration("  BPELVariable.releaseVariable(engine,name);\n");
			} else {
				state.appendDeclaration("  //No lock required (optimisation)\n");
			}
			state.appendDeclaration("}\n\n");
		}
		
		ArrayList children = thread.getChildren();
		for (int i = 0; i < children.size(); i++) {
			BPELThread child = (BPELThread)children.get(i);
			buildLockMethods(child,state);
		}
	}
	
	public BPELState getJava(String bpel, Document doc) throws Exception {

		Element process_element = doc.getDocumentElement();
		if (!NamespaceTranslator.getName(process_element).equals("process")) {
			throw new BPELTranslatorException(compiler_util,process_element,"Expected top level <process> element but found "+process_element.getTagName());
		}
		
		BPELState program_state = new BPELState();
		translateProcess(program_state,process_element);

		//append any declarations necessary for each binding
		for (int i = 0; i < bindings.length; i++) {
			program_state.appendDeclaration(bindings[i].getDeclarations());
			program_state.addImports(bindings[i].getImports());
		}
		
		return program_state;
	}
	
	private XSDTypeTranslator[] getCodecs(WSDLBindingTranslator[] bindings) throws BPELTranslatorException {
		HashMap codec_classes = new HashMap();
		ArrayList codecs = new ArrayList();
		
//		codecs.add(new MessageCodec());
		
		for (int i = 0; i < bindings.length; i++) {
			XSDTypeTranslator[] bcodecs = bindings[i].getCodecs();
			if (bcodecs == null) throw new BPELTranslatorException(compiler_util,null,"binding "+bindings[i].getClass().getName()+" returned null codec array");
			for (int k = 0; k < bcodecs.length; k++) {
				Class codec_class = bcodecs[k].getClass();
				if (codec_classes.get(codec_class) == null) {
					codec_classes.put(codec_class,"");
					codecs.add(bcodecs[k]);
//DBG					DBG.info("added XSD Codec "+bcodecs[k].getClass().getName());
					log.logInfo(LOG_SOURCE,"added XSD Codec "+bcodecs[k].getClass().getName());
				}
			}	
		}
		
		XSDTypeTranslator[] ret = new XSDTypeTranslator[codecs.size()];
		for (int i = 0; i < ret.length; i++) {
			ret[i] = (XSDTypeTranslator)codecs.get(i);
		}		
	
		return ret;	
	}

	public CompilerUnit[] getSources(String bpel, Document bpel_doc, String wsdl[], String xsd[], WSDLBindingTranslator[] bindings, String[] xsd_files, String[] wsdl_files) throws Exception {
		
		//get the bindings
		this.bindings = bindings;

		//create the WSDL and XSD maps
		xsdmap = new XSDMap(compiler_util,log,xsd,wsdl);
		wsdlmap = new WSDLMap(compiler_util,log,wsdl);

		//initialise the bindings
		for (int i = 0; i < bindings.length; i++) {
			//DBG DBG.info("initialising binding "+bindings[i].getClass().getName());
			log.logInfo(LOG_SOURCE,"initialising binding "+bindings[i].getClass().getName());
			bindings[i].init(compiler_util,log,xsdmap,wsdlmap,deps_loader);
		}

		//get the codecs from the bindings
		//NOTE: bindings MUST be initialised first
		this.codecs = getCodecs(bindings);

		//initialise the codecs
		for (int i = 0; i < codecs.length; i++) {
			//DBG DBG.info("initialising codec "+codecs[i].getClass().getName());
			log.logInfo(LOG_SOURCE,"initialising xsd type translator "+codecs[i].getClass().getName());
			codecs[i].init(compiler_util,xsdmap,wsdlmap,deps_loader);
		}
		
		//running the WSDLTranslator will populate the bindings with the extra information they need
		// to translate ports and invocations
		WSDLTranslator wsdlt = new WSDLTranslator(compiler_util,log,xsdmap,wsdlmap);
		CompilerUnit[] wsdljava = wsdlt.getDefinitionSources(wsdl,bindings,codecs,wsdl_files);

		XSDTranslator xsdt = new XSDTranslator(compiler_util,log,xsdmap,wsdlmap);
		CompilerUnit[] xsdjava = xsdt.getDefinitionSources(xsd,wsdl,codecs,xsd_files,wsdl_files);

		CompilerUnit java = getJava(bpel,bpel_doc);

		CompilerUnit[] all = new CompilerUnit[wsdljava.length + xsdjava.length + 1];
	
		//BPEL java source
		all[0] = java;
		//WSDL java sources
		System.arraycopy(wsdljava,0,all,1,wsdljava.length);
		//XSD java sources
		System.arraycopy(xsdjava,0,all,wsdljava.length+1,xsdjava.length);

		return all;
	}

	private static String resolveToAbsoluteBaseURI(String base_uri, String location_uri) throws Exception {
//		URI cd = new URI(base_uri);
//		URI uri = new URI(location_uri);
		String uri = location_uri;
		
//		if (!uri.isAbsolute()) {
		if (uri.indexOf(':') == -1) {
			uri = base_uri + "/" + location_uri;
			//URI is not absolute (according to definition of java.net.URI)
//			uri = new URI(base_uri.toString() + "/" + uri.toString());
		}

		String resolvedUri = uri.toString();
		
		String parentUri = resolvedUri.substring( 0, Math.max(resolvedUri.lastIndexOf('/')+1,resolvedUri.lastIndexOf('\\')+1) );
		
		return parentUri;
	}
	private static String resolveToAbsoluteURI(String base_uri, String location_uri) throws Exception {
		String uri = location_uri;
		
		if (uri.indexOf(':') == -1) {
			uri = base_uri + "/" + location_uri;
			//URI is not absolute (according to definition of java.net.URI)
		}

		String resolvedUri = uri.toString();
	
		return resolvedUri;
//		String parentUri = resolvedUri.substring( 0, Math.max(resolvedUri.lastIndexOf('/')+1,resolvedUri.lastIndexOf('\\')+1) );
//		return parentUri;
	}
	
	private String getFromLocationURI(Element tmp, WsdlImportResolver iresolver, String base_uri, String location_uri) throws Exception {
		
		System.out.println("LOCATION "+base_uri+" "+location_uri);
		
//		URI cd = new File(".").toURI();
//		URI cd = new URI(base_uri);
//		URI uri = new URI(location_uri);
		String uri = location_uri;

		//resolve relative URIs against the provided base URI (location of the BPEL source file)
	
		if (uri.indexOf(':') == -1) {
			uri = base_uri + "/" + location_uri;
//		if (!uri.isAbsolute()) {
//			uri = new URI(base_uri.toString() + "/" + uri.toString());
//			uri = cd.resolve(uri);
		}
		
		//if we use URLs all the time then we can potentially run BPEL files from the web
		
		try {
//			String contents = ImportFetcher.fetch(uri.toURL());
//			String contents = iresolver.fetchUrl(uri.toURL(),true,true);
			String contents = iresolver.fetchUrl(new URL(uri),true,true);

			XMLUtil.documentFromString(contents);
			
			return contents;
//		} catch (SAXParseException e) {
//			log.logWarning(LOG_SOURCE, "Invalid XML document imported from '"+uri+"'");
//			throw e;
		} catch (Exception e) {
			log.logWarning(LOG_SOURCE, "Failed to resolve import URI '"+uri+"'");
			if (tmp != null) {
				throw new BPELTranslatorException(compiler_util,tmp,e);
			} else {
				throw e;
			}
		}
	}
	
	private void checkImportType(String imp_type) throws Exception {
		if (imp_type.equals(NamespaceTranslator.NAMESPACE_WSDL)
				|| imp_type.equals(NamespaceTranslator.NAMESPACE_XSD)) {
			//ok
		} else {
			log.logWarning(LOG_SOURCE,"Cannot import unknown import type '"+imp_type+"'");
			throw new Exception("Cannot import unknown import type '"+imp_type+"'");
		}
	}
	
	public void getAllImports(WsdlImportResolver iresolver, String bpel, String imp_type, String loc_uri, String base_uri, ArrayList files, ArrayList file_uris, ArrayList file_locs) throws Exception {
		
		DocumentBuilder builder = XMLUtil.getDocumentBuilder();
		Document doc = builder.parse(new ByteArrayInputStream(bpel.getBytes()),loc_uri);
		
		Element process = doc.getDocumentElement();
		if (!NamespaceTranslator.getName(process).equals("process")
			&& !NamespaceTranslator.getName(process).equals("definitions")) {
			throw new BPELTranslatorException(compiler_util,process,"Expected top level <process> or <definitions> element but found "+process.getTagName());
		}

		NamespaceTranslator nt = compiler_util.createNamespaceTranslator();
		
		nt.addNamespaces(process);
		//translate imports
		ArrayList imports = Util.getAllElements(process,"import");
		
		for (int i = 0; i < imports.size(); i++) {
			Element tmp = (Element)imports.get(i);
			String import_namespace = tmp.getAttribute("namespace");
			String import_location = tmp.getAttribute("location");

			log.logInfo(LOG_SOURCE,loc_uri+" imports -> "+import_location);
			
			if (NamespaceTranslator.getName(process).equals("process")) {
				String import_type = tmp.getAttribute("importType");
			
				checkImportType(import_type);
				
				//TODO do something with the namespace attribute?
			
				if (import_type.equals(imp_type)) {
//					System.out.println("Importing from \""+import_location+"\"");
					String content_uri = resolveToAbsoluteBaseURI(base_uri,import_location);
					String content = getFromLocationURI(tmp,iresolver,base_uri,import_location);
					String content_loc = resolveToAbsoluteURI(base_uri,import_location);
					
					if (!files.contains(content)) {
						files.add(content);
						file_uris.add(content_uri);
						file_locs.add(content_loc);
					}
				}
			} else if (NamespaceTranslator.getName(process).equals("definitions")) {

				if (imp_type.equals(NamespaceTranslator.NAMESPACE_WSDL) && import_location.toLowerCase().endsWith("wsdl")) {
					//we want imported WSDL files, and it's a WSDL file
					String content_uri = resolveToAbsoluteBaseURI(base_uri,import_location);
					String content = getFromLocationURI(tmp,iresolver,base_uri,import_location);
					String content_loc = resolveToAbsoluteURI(base_uri,import_location);
					
					if (!files.contains(content)) {
						files.add(content);
						file_uris.add(content_uri);
						file_locs.add(content_loc);
					}
					
				} else if (imp_type.equals(NamespaceTranslator.NAMESPACE_XSD) && import_location.toLowerCase().endsWith("xsd")) {
					//we want imported XSD files, and it's an XSD file
					String content_uri = resolveToAbsoluteBaseURI(base_uri,import_location);
					String content = getFromLocationURI(tmp,iresolver,base_uri,import_location);
					String content_loc = resolveToAbsoluteURI(base_uri,import_location);
					
					if (!files.contains(content)) {
						files.add(content);
						file_uris.add(content_uri);
						file_locs.add(content_loc);
					}
				}
			}
		}
		nt.removeNamespaces(process);
	}
	
	public void getXsdImports(WsdlImportResolver iresolver, String bpel, String loc_uri, String base_uri, ArrayList files, ArrayList file_uris, ArrayList file_locs) throws Exception {
		getAllImports(iresolver,bpel,NamespaceTranslator.NAMESPACE_XSD,loc_uri,base_uri,files,file_uris,file_locs);
	}
	public void getWsdlImports(WsdlImportResolver iresolver, String bpel, String loc_uri, String base_uri, ArrayList files, ArrayList file_uris, ArrayList file_locs) throws Exception {
		getAllImports(iresolver,bpel,NamespaceTranslator.NAMESPACE_WSDL,loc_uri,base_uri,files,file_uris,file_locs);
	}
	
	public Program getProgram(boolean debug, String bpel, String base_uri, JavaCompiler jcompiler, WsdlImportResolver iresolver, JARDependency[] deps, WSDLBindingTranslator[] bindings, TranslatorLog log, boolean validateOnly) throws Exception {
		this.debug = debug;
		this.log = log;
		
		DocumentBuilder builder = XMLUtil.getDocumentBuilder();
		Document bpel_doc = builder.parse(new ByteArrayInputStream(bpel.getBytes()),DEFAULT_BPEL_FILENAME);
		
		compiler_util.setBpelSource(bpel,bpel_doc);
		
		log.logInfo(LOG_SOURCE,deps.length+" JAR dependencies for Program compilation");

		URL[] dep_urls = new URL[deps.length];
		for (int i = 0; i < deps.length; i++) {
			dep_urls[i] = new File(deps[i].getFilePath()).toURL();
			log.logInfo(LOG_SOURCE,"Dependancy ClassLoader URL "+(i+1)+": "+dep_urls[i]);
		}
		
		deps_loader = new URLClassLoader(dep_urls);
		
		//add XSD and WSDL imports from the BPEL file
		ArrayList xsd = new ArrayList();
		ArrayList xsd_uris = new ArrayList();
		ArrayList xsd_locs = new ArrayList();
		ArrayList wsdl = new ArrayList();
		ArrayList wsdl_uris = new ArrayList();
		ArrayList wsdl_locs = new ArrayList();
		
		getXsdImports(iresolver,bpel,DEFAULT_BPEL_FILENAME,base_uri,xsd,xsd_uris,xsd_locs);
		getWsdlImports(iresolver,bpel,DEFAULT_BPEL_FILENAME,base_uri,wsdl,wsdl_uris,wsdl_locs);

		//add XSD and WSDL imports from the WSDL and XSD files
		int xsd_length, wsdl_length;
		do {
			xsd_length = xsd.size();
			wsdl_length = wsdl.size();
			for (int i = 0; i < wsdl_length; i++) {
				getXsdImports(iresolver,(String)wsdl.get(i),(String)wsdl_locs.get(i),(String)wsdl_uris.get(i),xsd,xsd_uris,xsd_locs);
				getWsdlImports(iresolver,(String)wsdl.get(i),(String)wsdl_locs.get(i),(String)wsdl_uris.get(i),wsdl,wsdl_uris,wsdl_locs);
			}
			
//			break;
		} while (xsd_length != xsd.size() || wsdl_length != wsdl.size());
		
//		log.logInfo(LOG_SOURCE, wsdl.size()+" WSDL files");
//		log.logInfo(LOG_SOURCE, xsd.size()+" XSD files");
		
		for (int i = 0; i < wsdl_locs.size(); i++) {
			log.logInfo(LOG_SOURCE, "WSDL Import: "+wsdl_locs.get(i));
		}
		for (int i = 0; i < xsd_locs.size(); i++) {
			log.logInfo(LOG_SOURCE, "XSD Import: "+xsd_locs.get(i));
		}
		
		//Since auto java ports wont ever use imported WSDL files we can just add them at the end.
		{
			//we can also ignore WSDL_URIS
			
			DependencyInfo[] alldeps = B2jPlatform.getAllDependencyInfo();
			
			for (int i = 0; i < alldeps.length; i++) {
				DependencyInfo dep = alldeps[i];
				Properties[] ports = dep.getAutoJavaPorts();
				for (int k = 0; k < ports.length; k++) {
					Properties port = ports[k];
					String clazz = port.getProperty("Class");
					String methods = port.getProperty("Methods").trim();
					if (clazz != null && clazz.length() > 0) {
						log.logInfo(LOG_SOURCE,"Auto-generating WSDL for "+clazz);
//System.out.println("Generating WSDL for "+clazz);
						
						ArrayList allMethods = new ArrayList();
						
						if (methods != null) {
							if (methods.length() != 0) {
								String[] tmp = StringSplitter.split(methods," ");
								for (int v = 0; v < tmp.length; v++) {
//System.out.println("Generating WSDL for "+clazz+"."+tmp[v]);
									allMethods.add(tmp[v].trim());
								}
							}
						}
						
//System.out.println("Generating WSDL now");

						WsdlGenerator gen = new WsdlGenerator(clazz,allMethods,deps_loader);
//System.out.println("Generator ran OK");
						
						String s = gen.getWsdl();
						
						if (s == null) {
							throw new Exception("WSDL auto-generation failed for "+clazz);
						}
						
						try {
							FileOutputStream fout = new FileOutputStream("C:\\temp\\ALLSRC_"+clazz+".wsdl");
							fout.write(s.getBytes());
							fout.close();
						} catch (Exception e) {
						}
						
						wsdl.add(s);
//						wsdl_uris.add(".");
					}
				}
			}
			
		}

		System.out.println(wsdl.size()+" WSDL files");
		
//		DBG.info((xsd.length-xsdold.length)+" Extra XSD imports from XSD and WSDL files");
//		DBG.info((wsdl.length-wsdlold.length)+" Extra WSDL imports from XSD and WSDL files");
		
		String[] xsd_sources = new String[xsd.size()];
		xsd.toArray(xsd_sources);
		String[] wsdl_sources = new String[wsdl.size()];
		wsdl.toArray(wsdl_sources);
		
		String[] xsd_files = new String[xsd_locs.size()];
		xsd_locs.toArray(xsd_files);
		String[] wsdl_files = new String[wsdl_locs.size()];
		wsdl_locs.toArray(wsdl_files);
		
		CompilerUnit[] all = getSources(bpel,bpel_doc,wsdl_sources,xsd_sources,bindings,xsd_files,wsdl_files);
		
		//
		// Validate only - dont return a fully compiled program
		//
		if (validateOnly) return null;

		log.logInfo(LOG_SOURCE, "Compiling generated Java");

		long jt = System.currentTimeMillis();
		
		jcompiler.setMainSourceFile(all[0]);

		//STATICALLY synchronize on this whole bit
		
		//clear the 'bin' class directory
		
		//go through the extra sources here and see if they have changed based on our 'src' directory

		//if they HAVE NOT changed then fetch them out of our 'bincache'
		//								add them to a package-dependant temporary JAR file
		//                              AND stick them in the 'bin' directory
		
		//if they HAVE changed then just do nothing right now
		
		//surely there is a better way to do this?  we could do this in two stages - one where we 
		//cache the compilation (HARD) and another where we cache the actual classes (EASY)
		
		//to cache the classes all we have to do is not use program 'extra classes' and instead
		//build package-based dependency JARs
		
		CompilerUnit[] extra = new CompilerUnit[all.length-1];
		System.arraycopy(all,1,extra,0,extra.length);
		jcompiler.addExtraSources(extra);

		jcompiler.compile();

		jt = System.currentTimeMillis()-jt;

		log.logInfo(LOG_SOURCE, "Java compilation complete ("+(jt/1000)+"s)");
		
		byte[] theclass = jcompiler.getMainSourceClass();
		byte[][] classes = jcompiler.getExtraSourceClasses();
		String[] pkgs = jcompiler.getExtraSourceClassesPackages();
		
		//Generate a JAR for each distinct extra class package - to facilitate caching
		HashMap pkgbouts = new HashMap();
		HashMap pkgjars = new HashMap();
		for (int i = 0; i < pkgs.length; i++) {
			String fullname = pkgs[i];
			String fullpkg = fullname.substring(0,fullname.lastIndexOf('/')+1);
			
			JarOutputStream jout = (JarOutputStream)pkgjars.get(fullpkg);
			if (jout == null) {
				
				log.logInfo(LOG_SOURCE, "Creating dependency JAR for package "+fullpkg);

				ByteArrayOutputStream bout = new ByteArrayOutputStream();
				pkgbouts.put(fullpkg,bout);
				jout = new JarOutputStream(bout);
				pkgjars.put(fullpkg,jout);
			}
			jout.putNextEntry(new ZipEntry(fullname));
			jout.write(classes[i]);
			jout.closeEntry();
		}
		
		ArrayList extradeps = new ArrayList();
		Iterator it = pkgjars.keySet().iterator();
		while (it.hasNext()) {
			String pkg = (String)it.next();
			
			JarOutputStream jout = (JarOutputStream)pkgjars.get(pkg);
			jout.close();
			
			ByteArrayOutputStream bout = (ByteArrayOutputStream)pkgbouts.get(pkg);
			
			JARDependency jardep = new JARDependency(bout.toByteArray(),pkg+".jar");
			
			pkgjars.put(pkg,null);	//facilitate GC
			pkgbouts.put(pkg,null);	//facilitate GC
			
			extradeps.add(jardep);
		}
		
		//build a new JARDependency array
		JARDependency[] findeps = new JARDependency[deps.length + extradeps.size()];
		System.arraycopy(deps,0,findeps,0,deps.length);
		for (int i = 0; i < extradeps.size(); i++) {
			findeps[deps.length + i] = (JARDependency)extradeps.get(i); 
		}
		
		//remove the extra classes - they are now all part of the JAR dependencies
		classes = new byte[0][];

		//create a new Program object
		Program program = JavaProgramFactory.createEngineProgramFromClassFiles(theclass,classes,findeps);
//		Program program = JavaProgramFactory.createEngineProgramFromClassFiles(theclass,classes,deps);

//		System.out.println("FINAL BPEL TO RUN:\n"+bpel);
		
		
		return program;
	}
}