/**********************************************************************
 * Copyright (c) 2005 Scapa Technologies Limited and others
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * Scapa Technologies Limited - Initial API and implementation
 **********************************************************************/

package org.eclipse.stp.b2j.core.jengine.internal.compiler.xpath;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;

/**
 * 
 * @author amiguel
 * 
 * This class has become far less useful with the advent of runtime-determined scopes
 * (caused by the multiple-copy flow stuff)
 * 
 * This class takes multiple XPATH expressions and matches subexpressions in them.
 * It then moves these subexpressions into root expressions and replaces the 
 * original subexpressions with references.  This means that with caching and even
 * more so with static expression functions we never do more work than we absolutely
 * need to because of changed variables.
 * 
 *
 * When all XPATH expressions have been built into trees, we can match subtrees.
 * These subtrees can then be merged into expressions in their own right, referenced
 * by the earlier expressions.
 * 
 * At this point we could feasibly make the expressions and their caching static and synchronized,
 * meaning calculations will only occur once for each host, rather than each thread.
 * This should be made into an optional switch though (however, the saving in space and cpu time
 * is likely to outweigh the cost of the synchronisation)
 * 
 * One effect of this is that, as XPATH has no power function, it will diminish the cost of doing
 * powers as multiple functions
 * 
 * Also, when we have these trees we can could create another algorithm which caches 
 * 
 * Proposed tree matching algorithm (1):
 * - we create one tree with E roots, where E is the number of expressions
 * - the tree is traversed depth first and each variable is given a pseudo ID from 0-N
 * - the tree is traversed depth first and each node in the tree is given a hashcode based on its children
 * 		- if a node is a variable then it has the hashcode of the variable name
 * - the tree is traversed depth first and each node is added to a map, indexed by its hashcode
 * - the tree is traversed breadth first and each node is checked in the map for a duplicate count. Note that
 *   each check must be checked more thoroughly in case the hashcoding has failed.  The effect of searching for
 *   duplicates across the tree in breadth first fashion is to create subexpressions as large as possible and not
 *   to create subexpressions for all elements of two matching subtrees
 * - if a node is found to be duplicated then the original node is moved to the root of the tree and all duplicates
 *   are set to be 'references' to that root node.   If a node is found to be duplicated then its children are ignored.
 * - The above steps are then repeated until no changes are found
 * - we then only have to compile in all the expressions.  Each root node compiles in to a function (including
 *   root nodes that have been made into 'references' - they may be referenced by the BPEL and must be compiled
 *   in as proxies to their references.  NOTE, a further optimisation here would be to keep track of nodes that
 *   are genuinely referenced from the BPEL and to only keep those as root node references.
 * 
 * 
 * One general difficulty in all this is deciding what subexpression functions return and how they modify the
 * stack and return object.  However, the previous sentence contains the key.  The root functions called from the 
 * BPEL need to be marked as such and they need to create a context stack and a return object.  Subexpressions
 * are then passed the context stack for modification (or something) and always return an object which nsRet is
 * set to.
 * 
 * This will work but it is not going to be easy to implement and its not worth it at the moment.
 * 
 */
public class XPATHTreeMatcher implements Comparator {

XPathNode root;
ArrayList list;
HashMap map;
	
	private void addToMap(XPathNode node) {
		for (int i = 0; i < node.getChildCount(); i++) {
			addToMap(node.getChild(i));
		}
		String hcode = node.getHashCode();

		ArrayList countlist = (ArrayList)map.get(hcode);
		if (countlist == null) {
			countlist = new ArrayList();
			map.put(hcode,countlist);
		}
		
		countlist.add(node);
		list.add(node);
	}
	
	public void transform(XPathNode node) {
		
		//transform depth first
//		for (int i = 0; i < children.size(); i++) {
//			transform((XPathNode)children.get(i));
//		}
		
		root = node;
//		map = new HashMap();
		
		//get a hashcode for each node
		for (int i = 0; i < node.getChildCount(); i++) {
			node.getChild(i).recalculate();
		}		
		
		boolean first = true;
		
		boolean changed = true;
		
		while (changed) {
			
			changed = false;
			
			list = new ArrayList();
			map = new HashMap();
			
			//add each to the map
			for (int i = 0; i < node.getChildCount(); i++) {
				addToMap(node.getChild(i));
			}
	
			//sort them by hashcode (has the effect of sorting them largest expression first)
			Collections.sort(list,this);

			if (first) {
				for (int i = 0; i < list.size(); i++) {
					System.out.println(((XPathNode)list.get(i)).getHashCode());
				}
				first = false;
			}	
			
			String previous = "";
			
			//
			for (int i = 0; i < list.size(); i++) {
				XPathNode nextnode = (XPathNode)list.get(i);
				String next = nextnode.getHashCode();
				if (nextnode.getType() != XPathNode.TYPE_NODE_REFERENCE
					&& nextnode.getType() != XPathNode.TYPE_ROOTS_CONTAINER) {
					
					if (previous.equals(next)) {
						ArrayList dups = (ArrayList)map.get(previous);

System.out.println("FOUND DUPLICATE NODES - "+dups.size());					
System.out.println(nextnode);					
						
						XPathNode ref = new XPathNode(XPathNode.TYPE_NODE_REFERENCE);
						ref.ref = nextnode;
						
						for (int k = 0; k < dups.size(); k++) {
							XPathNode dup = (XPathNode)dups.get(k);
							dup.getParent().replaceChild(dup,ref);
						}
						
						root.addChild(nextnode);
						
						changed = true;
						break;
					}
/*					
					if (previous.equals(next)) {
						
						ArrayList
						
try {						
	System.out.println("FOUND DUPLICATE");						
	Thread.sleep(1000);
} catch (Exception e) {
}
						//we've found a duplicate
						XPathNode ref = new XPathNode(XPathNode.TYPE_NODE_REFERENCE);
						ref.addChild(nextnode);
						
						nextnode.getParent().replaceChild(nextnode,ref);
						
						XPathNode prevnode = (XPathNode)list.get(i-1);
						prevnode.getParent().replaceChild(prevnode,ref);

						root.addChild(nextnode);
						
						nextnode = (XPathNode)list.get(++i);
						next = nextnode.getHashCode();
						while (next.equals(previous)) {
							
							nextnode.getParent().replaceChild(nextnode,ref);
							
							nextnode = (XPathNode)list.get(++i);
							next = nextnode.getHashCode();
						}
	
						changed = true;
						break;
					}
*/					
				}
				previous = next;
			}
		}
	}
	
	public int compare(Object o1, Object o2) {
		XPathNode n1 = (XPathNode)o1;
		XPathNode n2 = (XPathNode)o2;
		
		return n1.getHashCode().compareTo(n2.getHashCode());
	}
}