/**********************************************************************
 * Copyright (c) 2005 Scapa Technologies Limited and others
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * Scapa Technologies Limited - Initial API and implementation
 **********************************************************************/

package org.eclipse.stp.b2j.core.jengine.internal.core.datapool;


import org.eclipse.stp.b2j.core.jengine.internal.core.SubControllerInterface;
import org.eclipse.stp.b2j.core.jengine.internal.core.VariableTypeException;
import org.eclipse.stp.b2j.core.jengine.internal.message.Message;

/**
 * 
 * @author amiguel
 * 
 * Part of the generic runtime engine API, implements a Variable which seamlessly
 * shares data across any part of the engine.
 */
public class SharedVariable {

public static final int NONE = -1;
public static final int INTEGER = 0;
public static final int LONG = 1;
public static final int FLOAT = 2;
public static final int DOUBLE = 3;
public static final int STRING = 4;
public static final int BYTES = 5;
public static final int MESSAGE = 6;

Object value = null;

SubControllerInterface controller;
int id;
String name;

int type = NONE;
	
	public SharedVariable(SubControllerInterface controller, String name, int id, int type, Object value) {
		this.controller = controller;
		this.name = name;
		this.id = id;
		this.type = type;
		this.value = value;
	}
	
	public boolean isDirty() throws Exception {
		if (controller == null) return false;
		return controller.isVariableDirty(id);
	}
	
	public void varStore() throws Exception {
		if (controller == null) return;
		controller.storeVariable(id,type,value);		
	}
	
	public void varFetch() throws Exception {
		if (controller == null) return;
		value = controller.fetchVariable(id,type);
	}

	public void setValue(int i) throws ClassCastException {
		setValue(new Integer(i));
	}

	public void setValue(long i) throws ClassCastException {
		setValue(new Long(i));
	}

	public void setValue(float i) throws ClassCastException {
		setValue(new Float(i));
	}

	public void setValue(double i) throws ClassCastException {
		setValue(new Double(i));
	}

	public void setValue(Object newval) throws ClassCastException {
		switch (type) {
			case INTEGER:
				value = new Integer( ((Number)newval).intValue() );
				break;
			case STRING:
				value = (String)newval;
				break;
			case BYTES:
				value = (byte[])newval;
				break;
			case MESSAGE:
				value = (Message)newval;
				break;
			case LONG:
				value = new Long( ((Number)newval).longValue() );
				break;
			case DOUBLE:
				value = new Double( ((Number)newval).doubleValue() );
				break;
			case FLOAT:
				value = new Float( ((Number)newval).floatValue() );
				break;
		}
	}

	public Object getValue() {
		return value;
	}
	
	public int getValueInt() throws VariableTypeException {
		if (type == STRING ||
			type == BYTES) {
			throw new VariableTypeException("attempted to get number value when variable is not a number");
		}
		return ((Number)value).intValue();
	}
	public long getValueLong() throws VariableTypeException {
		if (type == STRING ||
			type == BYTES) {
			throw new VariableTypeException("attempted to get number value when variable is not a number");
		}
		return ((Number)value).longValue();
	}
	public float getValueFloat() throws VariableTypeException {
		if (type == STRING ||
			type == BYTES) {
			throw new VariableTypeException("attempted to get number value when variable is not a number");
		}
		return ((Number)value).floatValue();
	}
	public double getValueDouble() throws VariableTypeException {
		if (type == STRING ||
			type == BYTES) {
			throw new VariableTypeException("attempted to get number value when variable is not a number");
		}
		return ((Number)value).doubleValue();
	}
	public String getValueString() throws VariableTypeException {
		if (type != STRING) {
			throw new VariableTypeException("attempted to get string value when variable is not a string");
		}
		return (String)value;
	}
	public byte[] getValueBytes() throws VariableTypeException {
		if (type != BYTES) {
			throw new VariableTypeException("attempted to get bytes value when variable is not a bytes");
		}
		return (byte[])value;
	}
	public Message getValueMessage() throws VariableTypeException {
		if (type != MESSAGE) { 
			throw new VariableTypeException("attempted to get Message value when variable is not a Message");
		}
		return (Message)value;
	}
		
	public String toString() {
		return String.valueOf(value);
	}
		
}