/**********************************************************************
 * Copyright (c) 2005 Scapa Technologies Limited and others
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * Scapa Technologies Limited - Initial API and implementation
 **********************************************************************/

package org.eclipse.stp.b2j.core.jengine.internal.multiplex;


import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.OutputStream;

/**
 * @author amiguel
 *
 * An OutputStream implementation that can be written to and cleared
 * to a static byte array at any time.
 */
public class ByteArrayOutBuffer extends OutputStream {
	
public static final int INFINITE_BUFFER = -1;
public static final int DEFAULT_BUFFER = 400000;

Object buffer_LOCK = new Object();
ByteArrayOutputStream bout = new ByteArrayOutputStream();
byte[] cache = null;
boolean waiting_for_data = false;
boolean waiting_for_space = false;

IOException ioexception = null;

int max_buffer = DEFAULT_BUFFER;

	public ByteArrayOutBuffer() {
	}
	public ByteArrayOutBuffer(int max_buffer) {
		this.max_buffer = max_buffer;
	}

	public int size() {
		return bout.size();
	}
	
	public byte[] clearToByteArray(int max) throws IOException, InterruptedException {
		synchronized(buffer_LOCK) {

			if (cache == null) {
				cache = clearToByteArray();
			}
			
			//we have data in the cache
			
			if (max >= cache.length) {
				//we can return the whole cache
				byte[] tmp = cache;
				cache = null;
				return tmp;
				
			} else {
				//we can return the first N bytes of the cache
				byte[] tmp = new byte[max];
				byte[] ncache = new byte[cache.length-max];

				System.arraycopy(cache,0,tmp,0,max);
				System.arraycopy(cache,max,ncache,0,cache.length-max);
				cache = ncache;
				
				return tmp;
			}
		}
	}
	
	public byte[] clearToByteArray() throws IOException, InterruptedException {
		synchronized(buffer_LOCK) {
			while (bout.size() == 0) {
				if (ioexception != null) {
					throw ioexception;
				}

				waiting_for_data = true;
				buffer_LOCK.wait();
				waiting_for_data = false;
				if (ioexception != null) {
					throw ioexception;
				}
			}
			
			byte[] tmp = bout.toByteArray();
			bout.reset();
			
			if (waiting_for_space) {
				buffer_LOCK.notifyAll();
			}
			
			return tmp;
		}
	}
	
	public void setClosed(IOException err) {
		ioexception = err;
		synchronized(buffer_LOCK) {
			buffer_LOCK.notifyAll();
		}
	}
	
	public void close() {
		setClosed(new IOException("stream closed"));
	} 
	public void flush() {
	} 
	public void write(byte[] b) throws IOException {
		synchronized(buffer_LOCK) {
			if (ioexception != null) {
				throw ioexception;
			}
			
			while (b.length + size() > max_buffer 
					&& size() != 0
					&& max_buffer >= 0) {
				waiting_for_space = true;
				try {
					buffer_LOCK.wait();
				} catch (InterruptedException e) {}
				waiting_for_space = false;
//				if (ioexception != null) {
//					throw ioexception;
//				}
			}
			
			bout.write(b,0,b.length);
			if (waiting_for_data) {
				buffer_LOCK.notify();
			}
		}
	} 
	public void write(byte[] b, int off, int len) throws IOException {
		synchronized(buffer_LOCK) {
			if (ioexception != null) {
				throw ioexception;
			}
			
			while (b.length + size() > max_buffer 
					&& size() != 0
					&& max_buffer >= 0) {
				
				waiting_for_space = true;
				try {
					buffer_LOCK.wait();
				} catch (InterruptedException e) {}
				waiting_for_space = false;
//				if (ioexception != null) {
//					throw ioexception;
//				}
			}
			
			bout.write(b,off,len);
			if (waiting_for_data) {
				buffer_LOCK.notify();
			}
		}
	} 
	public void write(int b) throws IOException {
		synchronized(buffer_LOCK) {
			if (ioexception != null) {
				throw ioexception;
			}
			
			while (1 + size() > max_buffer 
					&& size() != 0
					&& max_buffer >= 0) {
				
				waiting_for_space = true;
				try {
					buffer_LOCK.wait();
				} catch (InterruptedException e) {}
				waiting_for_space = false;
//				if (ioexception != null) {
//					throw ioexception;
//				}
			}
			
			bout.write(b);
			if (waiting_for_data) {
				buffer_LOCK.notify();
			}
		}
	}  

}