/*******************************************************************************
 * Copyright (c) 2003, 2005 IBM Corporation and others. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 which accompanies this distribution,
 * and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/
package org.eclipse.stp.core.internal.infrastructure.emf;

import java.io.IOException;
import java.util.Arrays;
import java.util.Collections;
import java.util.EventObject;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import org.eclipse.core.commands.operations.DefaultOperationHistory;
import org.eclipse.core.commands.operations.IOperationHistory;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.emf.common.command.CommandStackListener;
import org.eclipse.emf.common.notify.Notification;
import org.eclipse.emf.common.notify.impl.AdapterImpl;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.osgi.util.NLS;
import org.eclipse.stp.core.infrastructure.assertion.Assert;
import org.eclipse.stp.core.infrastructure.emf.EditModelDisposedException;
import org.eclipse.stp.core.infrastructure.emf.EditModelEvent;
import org.eclipse.stp.core.infrastructure.emf.EditModelException;
import org.eclipse.stp.core.infrastructure.emf.IEditModel;
import org.eclipse.stp.core.infrastructure.emf.IEditModelErrorHandler;
import org.eclipse.stp.core.infrastructure.emf.IEditModelListener;
import org.eclipse.stp.core.infrastructure.emf.IEditModelScribbler;
import org.eclipse.stp.core.infrastructure.emf.IScribblerDomain;
import org.eclipse.stp.core.infrastructure.emf.WorkbenchResourceHelper;
import org.eclipse.stp.core.internal.infrastructure.emfworkbench.EMFWorkbenchContext;
import org.eclipse.wst.common.internal.emf.resource.CompatibilityXMIResource;
import org.eclipse.wst.common.internal.emf.resource.ReferencedResource;
import org.eclipse.wst.common.internal.emf.utilities.ExtendedEcoreUtil;

/**
 * The following class is the implementation of IEditModel.
 * 
 * <p>
 * Clients are not allowed to subclass this class
 * </p>
 */
public final class EditModel implements IEditModel, CommandStackListener {

   public static final int           OPEN_FOR_BUSINESS = 0;

   public static final int           DISPOSED          = 2;

   private final String              editModelID;

   private int                       state;

   private final ListenerList        listeners         = new ListenerList();

   private final Set                 resources         = new HashSet();

   private final ScribblerTracker    scribblerTracker;

   private IEditModelErrorHandler    errorHandler;

   private final EMFWorkbenchContext emfContext;

   private IOperationHistory         operationHistory;

   private ResourceAdapter           resourceAdapter   = new ResourceAdapter();

   private boolean                   reverting;

   protected class ResourceAdapter extends AdapterImpl {

      /**
       * @return True only if the type supplied is this Edit Model's
       *         resoureAdapter
       */
      public boolean isAdapterForType(Object type) {
         return type == resourceAdapter;
      }

      public void notifyChanged(Notification notification) {
         if (notification.getEventType() == Notification.SET
               && notification.getFeatureID(null) == Resource.RESOURCE__IS_LOADED) {
            resourceIsLoadedChanged((Resource) notification.getNotifier(),
                  notification.getOldBooleanValue(), notification
                        .getNewBooleanValue());
         }

         if (notification.getEventType() == Notification.SET
               && notification.getFeatureID(null) == Resource.RESOURCE__IS_MODIFIED) {
            resourceIsModified((Resource) notification.getNotifier(),
                  notification.getOldBooleanValue(), notification
                        .getNewBooleanValue());
         }
      }

      private void resourceIsLoadedChanged(Resource aResource,
            boolean wasLoaded, boolean isLoaded) {

         if (isLoaded)
            adaptForTracking(aResource);
         if (/* !isReverting && */hasListeners()) {
            int eventCode = isLoaded ? EditModelEvent.LOADED_RESOURCE
                  : EditModelEvent.UNLOADED_RESOURCE;
            EditModelEvent evt = new EditModelEvent(eventCode, Collections
                  .singletonList(aResource));
            notifyListeners(evt);
         }
      }

      private void resourceIsModified(Resource aResource, boolean wasModified,
            boolean isModified) {
         if (isModified && !wasModified && hasListeners()) {
            notifyListeners(new EditModelEvent(EditModelEvent.DIRTY,
                  Collections.singletonList(aResource)));
         }

      }
   }

   /**
    * @param anEditModelID
    *           The label to use to uniquely identify the IEditModel
    * @param context
    *           The context that will be used to load resources for this
    *           IEditModel
    */
   public EditModel(String anEditModelID, EMFWorkbenchContext context) {
      editModelID = anEditModelID;
      emfContext = context;
      state = OPEN_FOR_BUSINESS;
      scribblerTracker = new ScribblerTracker(anEditModelID);
      errorHandler = LoggerErrorHandler.INSTANCE;
      processLoadedResources();
   }

   /**
    * @see IEditModel#addListener(IEditModelListener)
    */
   public void addListener(IEditModelListener aListener)
         throws EditModelException {
      complainIfDisposed();
      listeners.add(aListener);
   }

   /**
    * @see IEditModel#removeListener(IEditModelListener)
    */
   public void removeListener(IEditModelListener aListener) {
      listeners.remove(aListener);
   }

   /**
    * @see IEditModel#isInterrestedInResource(Resource)
    */
   public boolean isInterrestedInResource(Resource aResource) {
      if (isDisposed())
         return false;
      IScribblerDomain[] domains = scribblerTracker.getScribblerDomains();
      for (int i = 0; i < domains.length; i++) {
         try {
            if (domains[i].isContained(aResource))
               return true;
         } catch (RuntimeException e) {
         }
      }
      return false;
   }

   /**
    * @see IEditModel#isDisposed()
    */
   public boolean isDisposed() {
      return state == DISPOSED;
   }

   /**
    * @see IEditModel#createScribbler(IScribblerDomain[], boolean)
    */
   public IEditModelScribbler createScribbler(IScribblerDomain[] theDomains,
         boolean makeReadOnly) throws EditModelException {
      complainIfDisposed();
      IEditModelScribbler scribbler = new EditModelScribbler(this, theDomains,
            makeReadOnly);
      addScribbler(scribbler);

      return scribbler;
   }

   /**
    * @see IEditModel#getEmfContext()
    */
   public EMFWorkbenchContext getEmfContext() throws EditModelException {
      complainIfDisposed();
      return emfContext;
   }

   /**
    * @see IEditModel#getOperationHistory()
    */
   public IOperationHistory getOperationHistory() throws EditModelException {
      complainIfDisposed();
      if (operationHistory == null)
         operationHistory = new DefaultOperationHistory();
      return operationHistory;
   }

   /**
    * @see IEditModel#setErrorHandler(IEditModelErrorHandler)
    */
   public void setErrorHandler(IEditModelErrorHandler anErrorHandler) {
      errorHandler = anErrorHandler != null ? anErrorHandler
            : LoggerErrorHandler.INSTANCE;
   }

   /**
    * @see IEditModel#getEditModelLabel()
    */
   public String getEditModelLabel() {
      return editModelID;
   }

   /* ------- BEGIN NON-API METHODS ----------- */

   public void commandStackChanged(EventObject event) {
   }

   public IEditModelErrorHandler getErrorHandler() {
      return errorHandler;
   }

   public int getReferenceCount() {
      return scribblerTracker.getSize();
   }

   public void dispose() {

      synchronized (this) {
         if (isDisposed())
            return;
         state = DISPOSED;
         EditModelEvent event = new EditModelEvent(EditModelEvent.PRE_DISPOSE);
         notifyListeners(event);
         synchronized (listeners) {
            Object[] theListeners = listeners.getListeners();
            listeners.remove(theListeners);
         }
      }
   }

   public final ListenerList getListeners() {
      return listeners;
   }

   public boolean isReverting() {
      return reverting;
   }

   public boolean isManagingResource(Resource resource) {
      return resources.contains(resource);
   }

   public void resourceChanged(EditModelEvent anEvent) {
      int code = anEvent.getEventCode();
      Resource resource;
      List changedResources;
      switch (code) {
         case EditModelEvent.REMOVED_RESOURCE:
            changedResources = anEvent.getChangedResources();
            for (int i = 0; i < changedResources.size(); i++) {
               resource = (Resource) changedResources.get(i);
               if ((resource != null) && (isInterrestedInResource(resource))) {
                  // ScribblerAdapter adapter = (ScribblerAdapter)
                  // EcoreUtil.getAdapter(resource
                  // .eAdapters(), ScribblerAdapter.ADAPTER_TYPE);
                  // resource.eAdapters().remove(adapter);
                  resources.remove(resource);
               }
            }
            break;
         case EditModelEvent.ADDED_RESOURCE:
            changedResources = anEvent.getChangedResources();
            for (int i = 0; i < changedResources.size(); i++) {
               resource = (Resource) changedResources.get(i);
               if ((resource != null) && (isInterrestedInResource(resource)))
                  enableResourceTracking(resource);
            }
            break;
      }
      if (hasListeners())
         notifyListeners(anEvent);
   }

   /* package */void processLoadedResources() {
      try {
         ResourceSet[] sets = getEmfContext().getResourceSets();
         for (int setIndx = 0; setIndx < sets.length; setIndx++) {
            List loaded = sets[setIndx].getResources();
            if (!loaded.isEmpty()) {
               int size = loaded.size();
               Resource resource;
               for (int i = 0; i < size; i++) {
                  resource = (Resource) loaded.get(i);
                  Assert.isNotNull(resource);
                  if (!resources.contains(resource)
                        && isInterrestedInResource(resource))
                     enableResourceTracking(resource);
               }
            }
         }

      } catch (EditModelException e) {
         EMFInfrastructurePlugin.log(EMFInfrastructurePlugin.createErrorStatus(
               0, e.getLocalizedMessage(), e));
      }
   }

   /* package */void enableResourceTracking(Resource aResource) {

      Assert.isNotNull(aResource);

      if (!resources.contains(aResource)) {
         adaptForTracking(aResource);
         if (aResource instanceof ReferencedResource) {
            // We need a better way to pass this through the save options
            // instead.
            // We also need to make this dynamic based on the project target
            ((ReferencedResource) aResource)
                  .setFormat(CompatibilityXMIResource.FORMAT_MOF5);
         } else if (aResource instanceof CompatibilityXMIResource) {
            ((CompatibilityXMIResource) aResource)
                  .setFormat(CompatibilityXMIResource.FORMAT_MOF5);
         }

         resources.add(aResource);
         ResourceAdapter rAdapter = (ResourceAdapter) EcoreUtil.getAdapter(
               aResource.eAdapters(), resourceAdapter);
         if (rAdapter == null)
            aResource.eAdapters().add(resourceAdapter);
      }
   }

   /**
    * The EditModelScribbler implementation manages its own removal, so this
    * method is exposed for them to remove themselves when necessary.
    * 
    * @param scribbler
    *           The scribbler that will no longer be tracked by this IEditModel
    */
   /* package */void removeScribbler(IEditModelScribbler scribbler) {
      scribblerTracker.release(scribbler);
   }

   /* package */boolean isShared(IEditModelScribbler aScribbler) {

      IEditModelScribbler[] scribblers = (IEditModelScribbler[]) scribblerTracker
            .getScribblers().toArray(
                  new IEditModelScribbler[scribblerTracker.getScribblers()
                        .size()]);
      for (int i = 0; i < scribblers.length; i++) {
         if (!scribblers[i].isReadOnly() && aScribbler != scribblers[i]
               && scribblers[i].matches(aScribbler))
            return true;
      }
      return false;
   }

   /**
    * 
    * @return A set of {@link Resource}s.
    */
   public Set getResources() {
      return resources;
   }

   /* package */Resource getResource(URI aUri, boolean localOnly,
         boolean requestLoad) throws EditModelException {

      complainIfDisposed();
      if (aUri == null || aUri.segmentCount() == 0)
         return null;

      IFile platformFile = null;
      if (ResourceUtilities.isPlatformResourceURI(aUri)) {
         platformFile = ResourceUtilities.getPlatformFile(aUri);
         if (platformFile == null)
            return null;
         if (platformFile.exists()) {
            try {
               platformFile.refreshLocal(IResource.DEPTH_ONE, null);
            } catch (CoreException e1) {
               // TODO Auto-generated catch block
               e1.printStackTrace();
            }
         }
      }
      Resource foundResource = null;
      synchronized (this) {

         try {
            synchronized (resources) {
               Resource resource = null;
               for (Iterator resourcesItr = resources.iterator(); resourcesItr
                     .hasNext()
                     && foundResource == null;) {
                  resource = (Resource) resourcesItr.next();
                  if (ExtendedEcoreUtil.endsWith(resource.getURI(), aUri))
                     foundResource = resource;
               }
               /*
                * Do not search the Resource sets if the resource is available
                * in the EditModel's list of resource references.
                */
               if (foundResource == null) {
                  ResourceSet[] sets = getEmfContext().getResourceSets();
                  for (int i = 0; i < sets.length && foundResource == null; i++) {
                     List loadedResources = sets[i].getResources();
                     for (Iterator loadedResourcesItr = loadedResources
                           .iterator(); loadedResourcesItr.hasNext()
                           && foundResource == null;) {
                        resource = (Resource) loadedResourcesItr.next();
                        if (ExtendedEcoreUtil.endsWith(resource.getURI(), aUri))
                           foundResource = resource;
                     }
                  }
               }
               if (localOnly) {
                  if (foundResource != null)
                     enableResourceTracking(foundResource);
                  return foundResource;
               }

               if (foundResource == null)
                  foundResource = getResourceSet().getResource(aUri, false);

               if (foundResource == null)
                  foundResource = getResourceSet().createResource(aUri);

               if (foundResource != null)
                  enableResourceTracking(foundResource);

               if (requestLoad
                     && (null != foundResource && !foundResource.isLoaded())) {
                  if (platformFile == null || platformFile.exists()) {
                     foundResource.load(Collections.EMPTY_MAP); // reload it
                  }
               }
            }
         } catch (RuntimeException e) {
            errorHandler.handleLoadFailed(aUri, e);
         } catch (IOException e) {
            errorHandler.handleLoadFailed(aUri, e);
         }
      }
      return foundResource;
   }

   /* package */void removeResource(Resource resource) {
      resources.remove(resource);
      getResourceSet().getResources().remove(resource);
   }

   /* package */void handleDelete(Resource aResource) {

      synchronized (this) {

      }
   }

   /* package */void handleRevert(Resource[] theResources,
         IProgressMonitor monitor) {

      synchronized (this) {
         reverting = true;

         try {
            if (monitor == null)
               monitor = new NullProgressMonitor();
            monitor.beginTask(Messages.Reverting_contents_of_editor,
                  theResources.length + 2);
            monitor.worked(1);
            for (int i = 0; i < theResources.length; i++) {
               monitor.subTask(NLS.bind(Messages.Reverting_resource,
                     theResources[i].getURI()));
               try {

                  if (theResources[i].isModified()) {
                     theResources[i].unload();
                     theResources[i].load(Collections.EMPTY_MAP);
                  }

               } catch (IOException e) {
                  errorHandler.handleRevertFailed(theResources[i], e);
               }
               monitor.worked(1);
            }

            monitor.subTask(Messages.Notifying_interested_lis_);
            if (listeners.size() > 0) {
               EditModelEvent event = new EditModelEvent(EditModelEvent.REVERT,
                     Arrays.asList(theResources));
               notifyListeners(event);
            }
            monitor.worked(1);
            monitor.subTask(Messages.Done_reverting_contents_of_editor);
            monitor.done();
         } finally {
            reverting = false;
         }
      }
   }

   /* package */void handleSave(Resource[] theResources,
         IProgressMonitor monitor) {

      synchronized (this) {

         if (monitor == null)
            monitor = new NullProgressMonitor();
         monitor.beginTask(Messages.Saving_contents_of_editor,
               theResources.length + 2);
         monitor.worked(1);
         for (int i = 0; i < theResources.length; i++) {
            monitor.subTask(NLS.bind(Messages.Saving_resource, theResources[i]
                  .getURI()));
            try {
               if (theResources[i].isLoaded() && theResources[i].isModified())
                  theResources[i].save(Collections.EMPTY_MAP);

            } catch (IOException e) {
               errorHandler.handleSaveFailed(theResources[i], e);
            }
            monitor.worked(1);
         }

         monitor.subTask(Messages.Notifying_interested_lis_);
         if (listeners.size() > 0) {
            EditModelEvent event = new EditModelEvent(EditModelEvent.SAVE,
                  Arrays.asList(theResources));
            notifyListeners(event);
         }

         monitor.worked(1);
         monitor.subTask(Messages.Done_saving_contents_of_editor);
         monitor.done();
      }
   }

   /* package */void handleDelete(Resource[] resourcesToDelete,
         IProgressMonitor monitor) throws EditModelException {

      // how should we handle this case when someone else is using the resource?
      for (int i = 0; i < resourcesToDelete.length; i++) {

         removeResource(resourcesToDelete[i]);
         if (resourcesToDelete[i].isLoaded())
            resourcesToDelete[i].unload();

         try {
            IFile ifile = WorkbenchResourceHelper.getFile(resourcesToDelete[i]);
            if (ifile.exists())
               ifile.delete(true, true, monitor);
         } catch (CoreException e) {
            EMFInfrastructurePlugin.logError(0, e.getLocalizedMessage(), e);
            throw new EditModelException(e.getLocalizedMessage());
         }
      }
   }

   /* package */UnreleasedScribblerException[] computeUnreleasedScribblers() {
      return scribblerTracker.computeUnreleasedScribblers();
   }

   /**
    * Exposed only for testing purposes.
    * 
    * @param event
    */
   private void notifyListeners(EditModelEvent event) {
      Object[] notificationList = listeners.getListeners();
      for (int i = 0; i < notificationList.length; i++)
         ((IEditModelListener) notificationList[i]).editModelChanged(this,
               event);
   }

   /**
    * Return the Resource Set, if available. If the underlying Edit Model has
    * been disposed, then this method will return null.
    * 
    * @return org.eclipse.emf.ecore.resource.ResourceSet
    */
   private ResourceSet getResourceSet() {
      try {
         if (getEmfContext() != null)
            return getEmfContext().getResourceSet();
      } catch (EditModelException e) {
         EMFInfrastructurePlugin.log(EMFInfrastructurePlugin.createErrorStatus(
               0, e.getLocalizedMessage(), e));
      }
      return null;
   }

   /**
    * Add the necessary adapter for tracking, if not already present. Returns
    * the adapter that was found or added.
    * 
    * @param aResource
    *           The resource that the edit model will enable reference counting
    *           for
    * @return The tracking adapter that was added or removed
    */
   private ScribblerAdapter adaptForTracking(Resource aResource) {
      ScribblerAdapter adapter = null;
      synchronized (aResource) {
         adapter = ScribblerAdapter.findAdapter(aResource);
         if (adapter == null) {
            adapter = new ScribblerAdapter(this);
            aResource.eAdapters().add(adapter);
         }
      }
      return adapter;
   }

   /**
    * 
    * @throws EditModelException
    *            If state == DISPOSED
    */
   private void complainIfDisposed() throws EditModelException {
      if (state == DISPOSED)
         throw new EditModelDisposedException();
   }

   /**
    * The accessibility of this method is restricted to the IEditModel to ensure
    * that no one tries to add a scribbler to this IEditModel that the
    * IEditModel did not create.
    * 
    * @param scribbler
    *           The scribbler to begin tracking for this edit model
    */
   private void addScribbler(IEditModelScribbler scribbler) {
      scribblerTracker.track(scribbler);
   }

   private boolean hasListeners() {
      return listeners.size() > 0;
   }

}
