/*******************************************************************************
 * Copyright (c) 2003 IBM Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.stp.core.internal.introspection;

import java.util.HashMap;
import java.util.Map;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.ISafeRunnable;
import org.eclipse.core.runtime.Platform;
import org.eclipse.emf.common.util.URI;
import org.eclipse.stp.core.infrastructure.assertion.Assert;
import org.eclipse.stp.core.internal.Messages;
import org.eclipse.stp.core.internal.RegistryReader;
import org.eclipse.stp.core.internal.STPCorePlugin;
import org.eclipse.stp.core.introspection.IComponentTypeIntrospector;
import org.eclipse.stp.core.introspection.IShareableComponentTypeFactory;
import org.eclipse.stp.core.sca.ComponentType;
import org.eclipse.stp.core.sca.Implementation;
import org.eclipse.stp.core.sca.SCAFactory;
import org.eclipse.stp.core.sca.UnknownImplementation;

/**
 * This registry consumes the 'componentTypeIntrospector' extension point.
 * 
 * <p>
 * The introspectors are searched first by scheme and then by extension. If an
 * introspector is registered for both the scheme and the extension, the scheme
 * is given priority.
 * </p>
 * 
 * <p>
 * Clients should not require access to the registry (indeed it is not API).
 * Clients should refer to
 * {@link org.eclipse.stp.core.introspection.IModelIntrospectionManager}.
 * </p>
 * 
 * @since 1.0
 * 
 */
public class ComponentTypeIntrospectorRegistry extends RegistryReader {

   private static final String                      COMPONENT_TYPE_INTROSPECTOR = "componentTypeIntrospector"; //$NON-NLS-1$

   private static final String                      ATT_EXTENSION               = "extension";                //$NON-NLS-1$ 

   private static final String                      ATT_CLASS                   = "class";                    //$NON-NLS-1$

   private static final String                      ATT_IMPLEMENTATION_TYPE     = "implementationElementType"; //$NON-NLS-1$

   private static final String                      ATT_URI_FACTORY_CLASS       = "shareableURIFactoryClass"; //$NON-NLS-1$

   /* the singleton instance */
   private static ComponentTypeIntrospectorRegistry instance;

   private final Map                                introspectionExtensions     = new HashMap();

   private final Map                                substitutionGroups          = new HashMap();

   private final Map                                shareableFactories          = new HashMap();

   /**
    * Create the registry for the STPCorePlugin.PLUGIN_ID and
    * COMPONENT_TYPE_INTROSPECTOR extension point.
    * 
    */
   public ComponentTypeIntrospectorRegistry() {
      super(Platform.getExtensionRegistry(), STPCorePlugin.PLUGIN_ID,
            COMPONENT_TYPE_INTROSPECTOR);
   }

   /**
    * 
    * @return An initialized singletone instance of the registry.
    */
   public static ComponentTypeIntrospectorRegistry getInstance() {

      if (instance != null)
         return instance;
      // we just need an object to sync on since we are
      // in a static block
      synchronized (COMPONENT_TYPE_INTROSPECTOR) {
         if (instance == null) {
            instance = new ComponentTypeIntrospectorRegistry();
            instance.readRegistry();
         }
      }
      return instance;

   }

   /**
    * 
    * @param implementation
    *           The implementation URI that may have a registered introspector
    *           available.
    * 
    * @return True if and only if the scheme or extension of the implementation
    *         URI has a registered introspector.
    */
   public boolean hasIntrospector(URI implementation) {

      return (introspectionExtensions.containsKey(implementation
            .fileExtension()));
   }

   public URI getShareableURI(Implementation anImplementation) {
      IShareableComponentTypeFactory factory = getShareableFactory(anImplementation);
      if (factory != null) {
         IFile shareableFile = factory
               .findShareableFile(ModelIntrospectionManager
                     .getStructuralFeature(anImplementation), anImplementation);
         if (shareableFile != null)
            return URI
                  .createURI("comptype:" + shareableFile.getFullPath().toString()); //$NON-NLS-1$
      }
      return null;
   }

   /**
    * Creates an IComponentTypeIntrospector for the implementation URI. The
    * introspectors are searched first by scheme and then by extension. If an
    * introspector is registered for both the scheme and the extension, the
    * scheme is given priority.
    * 
    * @param aComponentType
    *           The ComponentType to add the bind (e.g. add the Introspector
    *           to).
    * @param implementation
    *           The implementation URI of the relevant ComponentType
    * @return A valid introspector for the implementation URI
    */
   public IComponentTypeIntrospector createAndBindIntrospector(
         ComponentType aComponentType, URI implementation) {
      IConfigurationElement configuration = (IConfigurationElement) introspectionExtensions
            .get(implementation.fileExtension());
      if (configuration != null)
         try {
            IComponentTypeIntrospector introspector = (IComponentTypeIntrospector) configuration
                  .createExecutableExtension(ATT_CLASS);
            UnknownImplementation unknownImplemenation = SCAFactory.eINSTANCE
                  .createUnknownImplementation();
            unknownImplemenation.setUri(implementation.toString());
            aComponentType.getEObject().eAdapters().add(introspector);
            introspector.init(unknownImplemenation);
            return introspector;
         } catch (CoreException e) {
            STPCorePlugin.logError(0, Messages.Could_not_create_class, e);
         }
      return null;
   }

   /**
    * Creates an IComponentTypeIntrospector for the implementation URI. The
    * introspectors are searched first by scheme and then by extension. If an
    * introspector is registered for both the scheme and the extension, the
    * scheme is given priority.
    * 
    * @param implementation
    *           The implementation URI of the relevant ComponentType
    * @return A valid introspector for the implementation URI
    */
   public IComponentTypeIntrospector createIntrospector(
         Implementation implementation) {
      Assert.isNotNull(implementation.getEObject().eContainer());

      String serializedSubstitutionGroupElement = ModelIntrospectionManager
            .getSubstitutionGroupName(implementation);
      IConfigurationElement configuration = (IConfigurationElement) substitutionGroups
            .get(serializedSubstitutionGroupElement);
      if (configuration != null)
         try {
            IComponentTypeIntrospector introspector = (IComponentTypeIntrospector) configuration
                  .createExecutableExtension(ATT_CLASS);
            introspector.init(implementation);
            return introspector;
         } catch (CoreException e) {
            STPCorePlugin.logError(0, Messages.Could_not_create_class, e);
         }
      return null;
   }

   protected boolean readElement(IConfigurationElement element) {
      if (element.getAttribute(ATT_EXTENSION) != null)
         introspectionExtensions.put(element.getAttribute(ATT_EXTENSION),
               element);
      if (element.getAttribute(ATT_IMPLEMENTATION_TYPE) != null)
         substitutionGroups.put(element.getAttribute(ATT_IMPLEMENTATION_TYPE),
               element);
      return true;
   }

   /**
    * This method assumes that the Implementation is contained so that it can
    * check the FeatureMap of the containing model element. Otherwise, the
    * method checks for the FeatureAdapter on the Implementation. If neither of
    * these attempts succeeds, then this method will fail miserably (e.g. return
    * null).
    * 
    * 
    * @param implementation
    *           The implementation which has a feature that can be used to
    *           search the registry.
    * @return The correct IShareableComponentTypeFactory or null if none is
    *         specified.
    */
   public IShareableComponentTypeFactory getShareableFactory(
         Implementation implementation) {
      return getShareableFactory(ModelIntrospectionManager
            .getSubstitutionGroupName(implementation));
   }

   private IShareableComponentTypeFactory getShareableFactory(
         final String serializedElement) {

      IShareableComponentTypeFactory factory = (IShareableComponentTypeFactory) shareableFactories
            .get(serializedElement);
      if (factory != null)
         return factory;
      synchronized (shareableFactories) {

         ISafeRunnable runnable = new ISafeRunnable() {
            public void run() throws Exception {
               IConfigurationElement element = (IConfigurationElement) substitutionGroups
                     .get(serializedElement);
               if (element != null) {
                  IShareableComponentTypeFactory factory = (IShareableComponentTypeFactory) element
                        .createExecutableExtension(ATT_URI_FACTORY_CLASS);
                  shareableFactories.put(serializedElement, factory);
               }
            }

            public void handleException(Throwable exception) {

            }
         };
         Platform.run(runnable);
      }

      return (IShareableComponentTypeFactory) shareableFactories
            .get(serializedElement);
   }
}
