/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.stp.core.internal.introspection.query;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.emf.ecore.util.ExtendedMetaData;
import org.eclipse.stp.core.introspection.IDynamicModelIntrospector;
import org.eclipse.stp.core.sca.SCAObject;

public class CombinedQuery implements IModelQuery {

   private final Set     children          = new HashSet();

   private Map           untouchedElements = null;

   private CombinedQuery parent;

   private int           type              = IDynamicModelIntrospector.BATCH;

   public CombinedQuery() {
   }

   public CombinedQuery(int aType) {
      type = aType;
   }

   public void setParent(CombinedQuery newParent) {
      parent = newParent;
   }

   public int getType() {
      return type;
   }

   public void addChildQuery(IModelQuery moreSpecificQuery) {
      if (moreSpecificQuery != null) {
         children.add(moreSpecificQuery);
         ((CombinedQuery) moreSpecificQuery).setParent(this);
      }
   }

   public Set getChildren() {
      return Collections.unmodifiableSet(children);
   }

   public List matches(List theSearchSet) {
      if (theSearchSet.isEmpty())
         return Collections.EMPTY_LIST;

      EObject currentObject = null;
      List result = new ArrayList();
      for (int i = 0; i < theSearchSet.size(); i++) {
         currentObject = (EObject) theSearchSet.get(i);
         if (matches(currentObject))
            result.add(currentObject);
      }
      if (result.isEmpty())
         return Collections.EMPTY_LIST;
      return result;
   }

   protected final boolean matchesChildren(EObject anObject) {
      IModelQuery childQuery = null;
      for (Iterator childrenItr = children.iterator(); childrenItr.hasNext();) {
         childQuery = (IModelQuery) childrenItr.next();
         if (!childQuery.matches(anObject))
            return false;
      }
      return true;
   }

   public boolean matches(EObject theObject) {
      return matchesChildren(theObject);
   }

   public boolean applyDelta(EObject anObject) {
      calculateUntouchedElements(anObject);
      return applyChildrenDelta(anObject);
   }

   public boolean applyChildrenDelta(EObject anObject) {

      IModelQuery childQuery = null;
      for (Iterator childrenItr = children.iterator(); childrenItr.hasNext();) {
         childQuery = (IModelQuery) childrenItr.next();
         if (!childQuery.applyDelta(anObject))
            return false;
      }
      return true;
   }

   public String toString() {
      StringBuffer toString = new StringBuffer(""); //$NON-NLS-1$ 
      IModelQuery childQuery = null;
      for (Iterator childrenItr = children.iterator(); childrenItr.hasNext();) {
         childQuery = (IModelQuery) childrenItr.next();
         toString.append(childQuery.toString()).append("\n"); //$NON-NLS-1$
      }
      return toString.toString();
   }

   protected final Map getUntouchedElements(EObject anObject) {

      calculateUntouchedElements(anObject);
      return untouchedElements;
   }

   /**
    * @param anObject
    */
   public void calculateUntouchedElements(EObject anObject) {
      if (untouchedElements == null) {
         untouchedElements = new HashMap();
         List contents = anObject.eContents();
         EObject eObj = null;
         for (int i = 0; i < contents.size(); i++) {
            eObj = (EObject) contents.get(i);
            EStructuralFeature feature = eObj.eContainmentFeature();
            EStructuralFeature affiliation = ExtendedMetaData.INSTANCE
                  .getAffiliation(feature);
            if (affiliation != null)
               feature = affiliation;
            List list = (List) untouchedElements.get(feature.getName());
            if (list == null) {
               list = new ArrayList();
               untouchedElements.put(feature.getName(), list);
            }
            list.add(eObj);
         }
      }
   }

   protected final List getUntouchedElements(EObject anObject,
         EStructuralFeature aFeature) {
      EStructuralFeature affiliation = ExtendedMetaData.INSTANCE
            .getAffiliation(aFeature);
      if (affiliation != null)
         aFeature = affiliation;
      return (List) getUntouchedElements(anObject).get(aFeature.getName());
   }

   protected final List getParentUntouchedElements(EObject anObject,
         EStructuralFeature aFeature) {
      if (parent != null)
         return parent.getUntouchedElements(anObject, aFeature);
      return Collections.EMPTY_LIST;
   }

   public void removeUntouched(SCAObject anObject, EStructuralFeature feature) {
      List untouched = getUntouchedElements(anObject.getEObject(), feature);
      if (untouched != null) {
         EObject eObj = null;
         for (int i = 0; i < untouched.size(); i++) {
            eObj = (EObject) untouched.get(i);
            EcoreUtil.remove(eObj);
         }
      }
   }

}
