/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.stp.core.internal.introspection.query;

import java.util.List;

import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EFactory;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.emf.ecore.util.FeatureMapUtil;
import org.eclipse.stp.core.infrastructure.assertion.Assert;
import org.eclipse.stp.core.introspection.FeatureAdapter;
import org.eclipse.stp.core.introspection.IDynamicModelIntrospector;
import org.eclipse.stp.core.sca.SCAObject;

public class StructuralFeatureExpression extends CombinedQuery implements
      IModelQuery {

   private EStructuralFeature expectedSubstitutionGroupFeature;

   private EClass             expectedType;

   private EStructuralFeature expectedFeature;

   public StructuralFeatureExpression(EStructuralFeature theExpectedFeature,
         EClass theExpectedType,
         EStructuralFeature theSubstitutionGroupFeature, int theChangeType) {
      super(theChangeType);
      Assert.isTrue(!FeatureMapUtil.isFeatureMap(theExpectedFeature));
      expectedFeature = theExpectedFeature;
      expectedType = theExpectedType;
      expectedSubstitutionGroupFeature = theSubstitutionGroupFeature;
   }

   public boolean matches(EObject anObject) {
      if (expectedFeature.isMany()) {
         List elements = (List) anObject.eGet(expectedFeature);
         for (int i = 0; i < elements.size(); i++) {
            EObject eChild = (EObject) elements.get(i);
            if (eChild.eClass().equals(expectedType)) {
               if (matchesChildren(eChild))
                  return true;
            }
         }
      } else {
         EObject eChild = (EObject) anObject.eGet(expectedFeature);
         if (eChild != null && eChild.eClass().equals(expectedType)) {
            if (matchesChildren(eChild))
               return true;
         }
      }
      return false;
   }

   public boolean applyDelta(EObject anObject) {

      switch (getType()) {
         case IDynamicModelIntrospector.BATCH: {
            if (expectedFeature.isMany()) {
               List elements = (List) anObject.eGet(expectedFeature);
               // look for a matching child that matches the subquery
               for (int i = 0; i < elements.size(); i++) {
                  EObject eChild = (EObject) elements.get(i);
                  if (eChild.eClass().equals(expectedType)) {
                     if (matchesChildren(eChild)) {
                        if (applyChildrenDelta(eChild)) {
                           List untouchedElements = getParentUntouchedElements(
                                 anObject, expectedFeature);
                           untouchedElements.remove(eChild);
                           return true;
                        }
                        return false;
                     }
                  }
               }
               // none was found, so create the matching child.
               EObject eChild = createType();
               if (applyChildrenDelta(eChild)) {
                  elements.add(eChild);
                  return true;
               }
               return false;
            }
            // else ...
            EObject eChild = (EObject) anObject.eGet(expectedFeature);
            if (eChild != null && eChild.eClass().equals(expectedType)
                  && matchesChildren(eChild)) {
               if (applyChildrenDelta(eChild)) {
                  List untouchedElements = getParentUntouchedElements(anObject,
                        expectedFeature);
                  untouchedElements.remove(eChild);
                  return true;
               }
               return false;
            }
            eChild = createType();
            if (applyChildrenDelta(eChild)) {
               anObject.eSet(expectedFeature, eChild);
               return true;
            }
            return false;
         }
         case IDynamicModelIntrospector.INCREMENTAL_ADD: {
            if (expectedFeature.isMany()) {
               List elements = (List) anObject.eGet(expectedFeature);
               // we automatically create and add the new element
               EObject eChild = createType();
               if (applyChildrenDelta(eChild)) {
                  elements.add(eChild);
                  return true;
               }
               return false;
            }
            // else ...
            EObject eChild = (EObject) anObject.eGet(expectedFeature);
            eChild = createType();
            if (applyChildrenDelta(eChild)) {
               anObject.eSet(expectedFeature, eChild);
               return true;
            }
            return false;
         }
         case IDynamicModelIntrospector.INCREMENTAL_UPDATE: {
            if (expectedFeature.isMany()) {
               List elements = (List) anObject.eGet(expectedFeature);
               // look for a matching child that matches the subquery
               for (int i = 0; i < elements.size(); i++) {
                  EObject eChild = (EObject) elements.get(i);
                  if (eChild != null && eChild.eClass().equals(expectedType)
                        && matchesChildren(eChild))
                     return applyChildrenDelta(eChild);
               }
               // none was found, we do nothing
               return false;
            }
            // else ...
            EObject eChild = (EObject) anObject.eGet(expectedFeature);
            if (eChild != null && eChild.eClass().equals(expectedType)
                  && matchesChildren(eChild))
               return applyChildrenDelta(eChild);
            return false;
         }
         case IDynamicModelIntrospector.INCREMENTAL_REMOVE: {
            if (expectedFeature.isMany()) {
               List elements = (List) anObject.eGet(expectedFeature);
               // look for a matching child that matches the subquery
               for (int i = 0; i < elements.size(); i++) {
                  EObject eChild = (EObject) elements.get(i);
                  if (eChild.eClass().equals(expectedType)) {
                     if (matchesChildren(eChild)) {
                        EcoreUtil.remove(eChild);
                        return true;
                     }
                  }
               }
               // none was found, so create the matching child.
               return false;
            }
            // else ...
            EObject eChild = (EObject) anObject.eGet(expectedFeature);
            if (eChild != null && eChild.eClass().equals(expectedType)
                  && matchesChildren(eChild)) {
               EcoreUtil.remove(eChild);
               return true;
            }
            return false;
         }
         default:
            return false;
      }
   }

   private EObject createType() {
      // we didn't find a matching element, so create one
      EFactory eFactory = expectedType.getEPackage().getEFactoryInstance();
      EObject eChild = eFactory.create(expectedType);
      if (expectedSubstitutionGroupFeature != null)
         FeatureAdapter.setFeature((SCAObject) eChild,
               expectedSubstitutionGroupFeature);
      return eChild;
   }

   public String toString() {
      StringBuffer toString = new StringBuffer(" has feature ").append(expectedFeature.getName()) //$NON-NLS-1$
            .append(" of type ").append(expectedType.getName()); //$NON-NLS-1$
      return toString
            .append("\n[").append(super.toString()).append("]").toString(); //$NON-NLS-1$ //$NON-NLS-2$
   }

}
