/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/

package org.eclipse.stp.core.sca.util;

import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IResourceProxy;
import org.eclipse.core.resources.IResourceProxyVisitor;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.stp.core.infrastructure.emf.DefaultScribblerDomain;
import org.eclipse.stp.core.infrastructure.emf.EditModelException;
import org.eclipse.stp.core.infrastructure.emf.IEditModelFactory;
import org.eclipse.stp.core.infrastructure.emf.IEditModelScribbler;
import org.eclipse.stp.core.infrastructure.emf.IResourceDescriptor;
import org.eclipse.stp.core.sca.SCACoreRoot;
import org.eclipse.stp.core.sca.SCAObject;

public class ModelFileIndex implements IResourceProxyVisitor {

   private static final IResourceDescriptor[] NO_DESCRIPTORS = new IResourceDescriptor[0];

   private static final SCAObject[]           NO_MODELS      = new SCAObject[0];

   private final Set                          composites     = new HashSet();

   private final Set                          componentTypes = new HashSet();

   private final Set                          modules        = new HashSet();

   private final Set                          subsystems     = new HashSet();

   private final IProject                     project;

   private IEditModelScribbler                scribbler;

   public ModelFileIndex(IProject theProject) {
      project = theProject;

      try {
         project.accept(this, IResource.NONE);
      } catch (CoreException e) {
         // TODO Auto-generated catch block
         e.printStackTrace();
      }
   }

   public synchronized void addModelFile(IFile file) {
      if (ModelFileIndexer.COMPOSITE_EXT.equals(file.getFileExtension()))
         composites.add(file);
      if (ModelFileIndexer.COMPONENT_TYPE_EXT.equals(file.getFileExtension()))
         componentTypes.add(file);
      if (ModelFileIndexer.MODULE_EXT.equals(file.getFileExtension()))
         modules.add(file);
      if (ModelFileIndexer.SUBSYSTEM_EXT.equals(file.getFileExtension()))
         subsystems.add(file);
   }

   public synchronized SCAObject[] getModelObjects() {

      if (getScribbler() == null)
         return NO_MODELS;

      Set models = new HashSet();

      IFile file = null;
      SCACoreRoot root = null;
      Resource resource = null;
      for (Iterator compositesIterator = composites.iterator(); compositesIterator
            .hasNext();) {
         file = (IFile) compositesIterator.next();
         resource = scribbler.getResource(file.getFullPath());
         if (resource.getContents().size() > 0) {
            root = (SCACoreRoot) resource.getContents().get(0);
            if (root.getComposite() != null)
               models.add(root.getComposite());
         }
      }
      for (Iterator componentTypeIterator = componentTypes.iterator(); componentTypeIterator
            .hasNext();) {
         file = (IFile) componentTypeIterator.next();
         resource = scribbler.getResource(file.getFullPath());
         if (resource.getContents().size() > 0) {
            root = (SCACoreRoot) resource.getContents().get(0);
            if (root.getComponentType() != null)
               models.add(root.getComponentType());
         }
      }
      for (Iterator modulesIterator = modules.iterator(); modulesIterator
            .hasNext();) {
         file = (IFile) modulesIterator.next();
         resource = scribbler.getResource(file.getFullPath());
         if (resource.getContents().size() > 0) {
            root = (SCACoreRoot) resource.getContents().get(0);
            if (root.getModule() != null)
               models.add(root.getModule());
         }
      }
      for (Iterator subsystemsIterator = subsystems.iterator(); subsystemsIterator
            .hasNext();) {
         file = (IFile) subsystemsIterator.next();
         resource = scribbler.getResource(file.getFullPath());
         if (resource.getContents().size() > 0) {
            root = (SCACoreRoot) resource.getContents().get(0);
            if (root.getSubsystem() != null)
               models.add(root.getSubsystem());
         }
      }
      return (SCAObject[]) models.toArray(new SCAObject[models.size()]);
   }

   public IProject getProject() {
      return project;
   }

   protected IEditModelScribbler getScribbler() {
      if (scribbler == null) {
         try {
            scribbler = IEditModelFactory.eINSTANCE.createScribblerForRead(
                  project, new DefaultScribblerDomain() {
                     public String getEditModelLabel() {
                        return project.getName() + ".index"; //$NON-NLS-1$
                     }

                     public IResourceDescriptor[] getResourceDescriptors() {
                        return NO_DESCRIPTORS;
                     }
                  });
         } catch (EditModelException e) {
            e.printStackTrace();
         }
      }
      return scribbler;
   }

   public void dispose() {
      if (scribbler == null) {
         try {
            scribbler.close(null);
         } catch (EditModelException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
         }
      }
   }

   public boolean visit(IResourceProxy proxy) throws CoreException {
      String name = proxy.getName();
      if (name.endsWith(ModelFileIndexer.COMPOSITE_EXT))
         addResource(proxy.requestResource());
      else if (name.endsWith(ModelFileIndexer.COMPONENT_TYPE_EXT))
         addResource(proxy.requestResource());
      else if (name.endsWith(ModelFileIndexer.MODULE_EXT))
         addResource(proxy.requestResource());
      else if (name.endsWith(ModelFileIndexer.SUBSYSTEM_EXT))
         addResource(proxy.requestResource());
      return true;
   }

   private void addResource(IResource resource) {
      if (resource.getType() == IResource.FILE)
         addModelFile((IFile) resource);
   }
}
