/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.stp.core.saf.handler;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.stp.core.saf.ISAFManager;
import org.eclipse.stp.core.saf.IUIContext;
import org.eclipse.stp.core.saf.exception.InvalidOperationException;
import org.eclipse.stp.core.saf.exception.SAFException;
import org.eclipse.stp.core.sca.Component;
import org.eclipse.stp.core.sca.SCAObject;
import org.eclipse.wst.common.frameworks.datamodel.IDataModelOperation;

/**
 * This interface must be implemented by all component handlers. Most handlers
 * should extend the abstract class
 * {@link org.eclipse.stp.core.saf.handler.AbstractComponentHandler AbstractComponentHandler}
 * which provides default behavior for a number of methods. 
 */
public interface IComponentHandler extends IHandler {

	// ================
	// create component
	// ================

	/**
	 * Creates a component of the kind this handler is specialized for.
	 * 
	 * @return the new component
	 * @exception SAFException
	 *                when a component cannot be created
	 */
	public Component createComponent() throws SAFException;

	// ==================
	// createComponentFor
	// ==================
	
	/**
	 * Queries if an unimplemented component of the kind supported by this
	 * handler can be created. Some component kinds may not be authored in a
	 * top-down scenario. In these cases, this method will return <code>false</code>. If this
	 * method returns <code>true</code>, the handler's
	 * {@link #createComponent() createComponent()} method may be called to
	 * create a component.
	 * 
	 * @return <code>true</code> if the handler can create an unimplemented
	 *         <code>Component</code> object; <code>false</code> otherwise
	 */
	public boolean canCreateComponent();
	
	/**
	 * Queries if a component of the kind supported by this handler can be
	 * created for the specified implementation file. If this method returns
	 * <code>true</code>, the handler's
	 * {@link #createComponentFor(Object,IContainer,IUIContext)} method may be
	 * called to create a component for the file.
	 * <p>
	 * This method should be quick to execute, because it may be called to populate a
	 * context menu or a user dialog.
	 * 
	 * @param implementationObj
	 *            an object representing the implementation for a component
	 * @return <code>true</code> if the handler can create an
	 *         <code>Component</code> object from the specified file;
	 *         <code>false</code> otherwise
	 */
	public boolean canCreateComponentFor(Object implementationObj);

	/**
	 * Creates a component with an implementation based on the specified
	 * implementation model object. Optionally, this method may create other
	 * objects required to support the new component (such as entry points,
	 * other components, etc.). This method is not expected to associate a
	 * resource with the returned component or any of the supporting objects.
	 * 
	 * @param implementationObj
	 *            the object representing the implementation for the component
	 * @param container
	 *            the IContainer that holds the module, subsystem, composite, etc.
	 *            that will use the resulting component. The handler could use
	 *            the container's location to generate relative file paths that
	 *            it stores in the component (to reference the implementation
	 *            file, for example) or use the location as a place for additional
	 *            files that need to be generated.
	 * @param callback
	 *            callback object that the handler can use to obtain additional
	 *            information needed to create the component. Callers
	 *            can pass in a null value.
	 * @return an array of {@link SCAObject} containing the new component and
	 *         any other SCA objects generated by the handler to
	 *         support the new component
	 * @exception SAFException
	 *                an unexpected exception occurred
	 * @exception InvalidOperationException
	 *                a component cannot be created for the given implementation
	 *                object
	 * @exception InterruptedException
	 *                the caller did not provide a callback object needed by the
	 *                handler, or the callback request was canceled or aborted
	 */
	public SCAObject[] createComponentFor(Object implementationObj,
			IContainer container, IUIContext callback)
			throws SAFException, InterruptedException, InvalidOperationException;

	/**
	 * Queries if implementation files can be created for components created by
	 * this component handler. A caller can use this method to distinguish
	 * fully functional component handlers from those that do not provide
	 * support for generating implementations for the components they create.
	 * 
	 * @return <code>true</code> if support is provided for creating
	 *         implementations; <code>false</code> otherwise
	 */
	public boolean canCreateImplementation();

	/**
	 * Creates an implementation file or files for the specified component
	 * based on the definition of the component. The handler is expected to
	 * update the implementation object in the component to reference the
	 * generated file or files, if necessary. The implementation file or
	 * files will be regenerated and replaced. User changes made to these
	 * files will be not be preserved.
	 * <p>
	 * This method should be aware that the implementation files it generates
	 * may replace some already being managed in a team environment.
	 * 
	 * @param theComponent
	 *            the component whose implementation is to be generated
	 * @param container
	 *            the IContainer that holds the module, subsystem, composite, etc.
	 *            that contains the given component. The handler could use the
	 *            container's location to generate relative file paths that it
	 *            stores in the component (to reference the implementation file,
	 *            for example) or use the location as a place for additional files
	 *            that need to be generated.
	 * @param callback
	 *            callback object that the handler can use to obtain additional
	 *            information needed to create the implementation files. 
	 *            Callers may pass in a null value.
	 * @return an IDataModelOperation instance that, once executed by the
	 *         caller, creates the new implementation
	 * @exception SAFException
	 *                an unexpected exception occurred
	 * @exception InvalidOperationException
	 *                an implementation cannot be created for the given
	 *                component kind
	 * @exception InterruptedException
	 *                the caller did not provide a callback object needed by the
	 *                handler, or the callback request was canceled or aborted.
	 */
	public IDataModelOperation createImplementationFor(
			Component theComponent, IContainer container,
			IUIContext callback) throws SAFException,
			InterruptedException, InvalidOperationException;

	/**
	 * Returns an array of implementation files associated with the given
	 * component. The "primary" file should be the first element in the array.
	 * 
	 * @param theComponent
	 *            the component whose implementation files are returned
	 * @return an array of the associated implementation files
	 * @exception SAFException
	 *                An error occurred finding the implementation files.
	 */
	public IFile[] getImplementationFor(Component theComponent)
			throws SAFException;

	/**
	 * This method opens a registered editor for the given component
	 * implementation type.
	 * 
	 * @param theComponent
	 *            the component whose implementation is to be opened
	 * @throws SAFException
	 *             An unexpected exception occurred.
	 * @exception InvalidOperationException
	 *                The component's implementation cannot be opened.
	 */
	public void open(Component theComponent)
			throws SAFException, InvalidOperationException;

	/**
	 * Queries if the component handler provides support for setting the
	 * implementation of a component using the method setImplementation(). This
	 * method should be called before calling setImplementation(). A caller can
	 * use this method to determine if the corresponding component handler
	 * implements the method
	 * {@link #setImplementation(Component, IFile, IUIContext) setImplementation()}.
	 * If this method returns false, neither method should be called.
	 * 
	 * @return <code>true</code> if the handler supports the calling of setImplementation();
	 *         <code>false</code> otherwise
	 */
	public boolean canSetImplementation();

	/**
	 * Sets the implementation of the component to the specified model object.
	 * Generally, existing interfaces and references will be preserved. No
	 * clean-up is done on existing implementation files.
	 * 
	 * @param component
	 *            the component being modified
	 * @param implementationObject
	 *            the object to be used to implement the specified component
	 * @param callback
	 *            callback object that the component handler can use to obtain
	 *            additional information needed to create the component; can be
	 *            left null, although some calls may fail
	 * @return an IDataModelOperation instance that, once executed by the
	 *         caller, sets the implementation
	 * @exception SAFException
	 *                An unexpected exception occurred.
	 * @exception InvalidOperationException
	 *                The implementation cannot be set for the given component.
	 * @exception InterruptedException
	 *                A callback method throws an exception, or
	 *                the user cancels the callback conversation.
	 */
	public IDataModelOperation setImplementation(Component component,
			Object implementationObject, IUIContext callback)
			throws SAFException, InterruptedException, InvalidOperationException;

	/**
	 * Queries if a component of the kind supported by this handler supports
	 * interfaces of the kind specified by the given interface handler in its
	 * services or references.
	 * 
	 * @param interfaceHandler
	 *            the interface handler that will be used to manipulate the
	 *            interface. The implementation of this method can obtain the
	 *            interface kind by invoking {@link IInterfaceHandler#getKind()}
	 *            when performing the evaluation.
	 * @param service
	 *            flag indicating if the query applies to the services of the
	 *            component or its references. The values
	 *            {@link ISAFManager#SERVICE} and
	 *            {@link ISAFManager#REFERENCE} can be used for improved
	 *            code readability.
	 * @return <code>true</code> if the component supports the specified use
	 *         of the interface kind; <code>false</code> otherwise
	 * @throws SAFException
	 *             An unexpected exception occurred in the handler.
	 */
	public boolean isInterfaceKindSupported(IInterfaceHandler interfaceHandler,
			boolean service) throws SAFException;
	
	// ======================
	// Service/Reference Support
	// ======================
	
	/**
	 * Queries if a reference can be added by the client to a component of the
	 * supported kind.
	 * 
	 * @return <code>true</code> if the client is allowed to add a reference
	 *         object; <code>false</code> otherwise
	 */
	public boolean canAddReference();
	
	/**
	 * Queries if a service can be added by the client to a component of the
	 * supported kind.
	 * 
	 * @return <code>true</code> if the client is allowed to add a service
	 *         object; <code>false</code> otherwise
	 */
	public boolean canAddService();

	/**
	 * Queries if an interface can be set by the client in an existing service
	 * or reference contained by a component of the supported kind.
	 * 
	 * @return <code>true</code> if the client is allowed to set the interface
	 *         on a given service or reference object; <code>false</code>
	 *         otherwise
	 */
	public boolean canSetInterface();
	
	/**
	 * Sets the component to be one of the specified kind. Generally, existing
	 * interfaces and references will be preserved. The component must not
	 * already be typed (that is, it must contain an untyped
	 * {@link org.eclipse.stp.core.sca.AbstractImplementation AbstractImplementation}
	 * or no implementation at all). Call {@link #resetKind(Component) }
	 * beforehand, if necessary. If the kind cannot be set for the given
	 * component (perhaps due to unsupported interface kinds already set), an
	 * error message will be displayed through the given callback object.
	 * 
	 * @param theComponent
	 *            the component being modified
	 * @return an IDataModelOperation instance that, once executed by the
	 *         caller, sets the component kind
	 * @exception SAFException
	 *                No handler for the specified kind is found, or the
	 *                type is not supported.
	 * @exception InvalidOperationException
	 *                The component kind cannot be set for the given
	 *                component.
	 */
	public IDataModelOperation setKind(Component theComponent)
			throws SAFException, InvalidOperationException;
	
	/**
	 * Changes the component to be unclassified by removing the implementation
	 * of the component or by creating an untyped
	 * {@link org.eclipse.stp.core.sca.AbstractImplementation AbstractImplementation}.
	 * Generally, existing interfaces and references will be preserved. No
	 * clean-up is done on existing implementation files. If the component is
	 * already unclassified, this method performs no work.
	 * 
	 * @param theComponent
	 *            the component being modified
	 * @return an {@link IDataModelOperation} instance that, once executed by the
	 *         caller, resets the component kind
	 * @exception SAFException
	 *                No handler for the specified kind is found, or the
	 *                handler throws an exception.
   	 * @exception InvalidOperationException
	 *                The component kind cannot be reset for the given
	 *                component. 
	 */
	public IDataModelOperation resetKind(Component theComponent)
			throws SAFException, InvalidOperationException;
		
}
