/*******************************************************************************
* Copyright (c) 2006 IONA Technologies PLC
* All rights reserved. This program and the accompanying materials
* are made available under the terms of the Eclipse Public License v1.0
* which accompanies this distribution, and is available at
* http://www.eclipse.org/legal/epl-v10.html
* 
* Contributors:
*     IONA Technologies PLC - initial API and implementation
*******************************************************************************/
package org.eclipse.stp.sc.jaxws.deploy.wtp;

import java.util.HashSet;
import java.util.Set;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.SubProgressMonitor;
import org.eclipse.jface.window.Window;
import org.eclipse.jst.j2ee.web.project.facet.IWebFacetInstallDataModelProperties;
import org.eclipse.jst.j2ee.web.project.facet.WebFacetInstallDataModelProvider;
import org.eclipse.stp.common.logging.LoggingProxy;
import org.eclipse.stp.sc.jaxws.CeltixKitUtils;
import org.eclipse.stp.sc.jaxws.ScJaxWsResources;
import org.eclipse.stp.sc.jaxws.deploy.IDeployDelegate;
import org.eclipse.stp.sc.jaxws.workspace.JaxWsWorkspaceManager;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.wst.common.componentcore.datamodel.FacetInstallDataModelProvider;
import org.eclipse.wst.common.componentcore.datamodel.properties.IFacetDataModelProperties;
import org.eclipse.wst.common.frameworks.datamodel.DataModelFactory;
import org.eclipse.wst.common.frameworks.datamodel.IDataModel;
import org.eclipse.wst.common.project.facet.core.IFacetedProject;
import org.eclipse.wst.common.project.facet.core.IProjectFacet;
import org.eclipse.wst.common.project.facet.core.IProjectFacetVersion;
import org.eclipse.wst.common.project.facet.core.ProjectFacetsManager;
import org.eclipse.wst.common.project.facet.core.runtime.IRuntime;
import org.eclipse.wst.common.project.facet.core.runtime.RuntimeManager;
import org.eclipse.wst.server.core.IModule;
import org.eclipse.wst.server.core.IServer;
import org.eclipse.wst.server.core.IServerWorkingCopy;
import org.eclipse.wst.server.core.ServerCore;
import org.eclipse.wst.server.core.ServerUtil;



public class WTPDeployer implements IDeployDelegate {

    public static final String FACET_JST_WEB = "jst.web";
    public static final String FACET_JST_WEB_VERSION = "2.3";
    public static final String FACET_JST_JAVA = "jst.java";
    public static final String FACET_JST_JAVA_VERSION = "5.0";
    
    private static final LoggingProxy LOG = LoggingProxy.getlogger(WTPDeployer.class);

    private static final String FACET_NATURE = "org.eclipse.wst.common.project.facet.core.nature";
    private static final String WTP_NATURE = "org.eclipse.wst.common.modulecore.ModuleCoreNature";
    
    
    IFacetedProject facetedProject;

    public void deployWebService(IProject project,
                                 IFile wsdlFile,
                                 IProgressMonitor monitor) throws Exception {

        monitor.subTask(ScJaxWsResources.getString("wtpdeploy.monitor.createproject"));
        setupWTPProject(project);
        monitor.worked(10);

        monitor.subTask(ScJaxWsResources.getString("wtpdeploy.monitor.assemble"));

        SubProgressMonitor assembleMonitor = new SubProgressMonitor(monitor, 60);
        assembleWebContent(project, wsdlFile, assembleMonitor);
        monitor.worked(60);

        monitor.subTask(ScJaxWsResources.getString("wtpdeploy.monitor.setupruntime"));
        setupProjectFacets(project, monitor);
        monitor.worked(30);
        String url = WtpUtils.getServerUrl(wsdlFile);
        LOG.info("wsdl url on server:" + url);
    }

    private void addfacets(IFacetedProject facetProj,
        Set<IProjectFacetVersion> facets) {
        Set<IFacetedProject.Action> actions = new HashSet<IFacetedProject.Action>();

        for (IProjectFacetVersion facetVersion : facets) {
            if (facetProj.hasProjectFacet(facetVersion.getProjectFacet())) {
                continue;
            }

            IDataModel dm;

            if (facetVersion.getProjectFacet().getId().equals(FACET_JST_JAVA)) {
                dm = DataModelFactory.createDataModel(new FacetInstallDataModelProvider());
                dm.setProperty(IFacetDataModelProperties.FACET_ID, facetVersion.getProjectFacet().getId());
                dm.setProperty(IFacetDataModelProperties.FACET_PROJECT_NAME,
                               facetProj.getProject().getName());
                dm.setProperty(IFacetDataModelProperties.FACET_VERSION_STR,
                               facetVersion.getVersionString()); //$NON-NLS-1$
            } else { //jst.web
                dm = DataModelFactory.createDataModel(new WebFacetInstallDataModelProvider());
                dm.setProperty(IWebFacetInstallDataModelProperties.CONTEXT_ROOT, 
                               facetProj.getProject().getName());
                dm.setProperty(IWebFacetInstallDataModelProperties.SOURCE_FOLDER, 
                               facetProj.getProject().getName());
            }

            actions.add(new IFacetedProject.Action(IFacetedProject.Action.Type.INSTALL, facetVersion, dm));
        }

        try {
            if (!actions.isEmpty()) {
                facetProj.modify(actions, null);
            }
        } catch (CoreException e) {
            LOG.error("error during add project facets", e);
        }
    }

    @SuppressWarnings("unchecked")
    protected void setupProjectFacets(IProject project, IProgressMonitor monitor)
        throws CoreException {
        LOG.debug("add jst.web and jst.java facet to project");
        //IFacetedProject facetedProject = ProjectFacetsManager.create(project);
        facetedProject = ProjectFacetsManager.create(project);

        IProjectFacet webFacet = ProjectFacetsManager.getProjectFacet(FACET_JST_WEB);
        IProjectFacet javaFacet = ProjectFacetsManager.getProjectFacet(FACET_JST_JAVA);
        IProjectFacetVersion webFacetVersion = webFacet.getVersion(FACET_JST_WEB_VERSION);
        IProjectFacetVersion javaFacetVersion = javaFacet.getVersion(FACET_JST_JAVA_VERSION);

        Set<IProjectFacetVersion> defaultFacets = new HashSet<IProjectFacetVersion>();

        if (!facetedProject.hasProjectFacet(javaFacet)) {
            //add jst.java
            defaultFacets.add(javaFacetVersion);
        }

        if (!facetedProject.hasProjectFacet(webFacet)) {
            //add jst.web
            defaultFacets.add(webFacetVersion);
        }

        addfacets(facetedProject, defaultFacets);

        Set<IRuntime> runtimes = RuntimeManager.getRuntimes();

        Set<IRuntime> webRuntimes = new HashSet<IRuntime>();
        for (IRuntime runtime : runtimes) {
            if (runtime.supports(webFacet) && runtime.supports(webFacetVersion)
                && runtime.supports(javaFacet) && runtime.supports(javaFacetVersion)) {
                webRuntimes.add(runtime);
            }
        }
        
        if (webRuntimes.size() == 0) {
            LOG.info("did not set target runtime since no runtime has been installed");
            return;
        }

        //refresh the probject before set runtime
        project.refreshLocal(IResource.DEPTH_INFINITE, null);
        try {
            facetedProject.setTargetedRuntimes(webRuntimes, monitor);
        } catch (Exception ee) {
            LOG.debug(ee.toString());
        }
        
        facetedProject.setTargetedRuntimes(webRuntimes, monitor);

        Display.getDefault().asyncExec(new Runnable() {
                public void run() {
                    try {
                        Shell shell = new Shell(Display.getDefault());
                        RuntimeSelectionDialog dialog = new RuntimeSelectionDialog(shell);

                        if (dialog.open() == Window.OK) {
                            IRuntime runtime = dialog.getSelectedRuntime();
                            facetedProject.setPrimaryRuntime(runtime, null);
                            setupServer(runtime.getName(), facetedProject.getProject());
                        }

                    } catch (Exception e) {
                        LOG.error(e);
                    }
                }

                /**
                 * deploy the project onto server.
                * @throws CoreException
                 */
                private void setupServer(String runtimeName, IProject project) throws CoreException {

                    IModule module = ServerUtil.getModule(project);
                    IServer[] servers = ServerCore.getServers();

                    for (IServer server : servers) {
                        if (server.getRuntime().getName().equals(runtimeName)) {
                            if (ServerUtil.containsModule(server, module, null)) {
                                LOG.info("project already deployed on server" + server);
                                return; //already deployed to server
                            }

                            IModule[] add = new IModule[] {module};
                            LOG.info("deploy project:"
                                     + project.getName()
                                     + " to server:"
                                     + server.getName());

                            IServerWorkingCopy workingCopy = server.createWorkingCopy();
                            ServerUtil.modifyModules(workingCopy, add, null, null);
                            workingCopy.save(true, null);
                            ServerCore.setDefaultServer(module, server, null);

                            return;
                        }
                    }
                }
            });
    }

    
    protected void setupWTPProject(IProject project) {
        try {
            //add project nature.
            JaxWsWorkspaceManager.addProjectNature(project, FACET_NATURE);
            JaxWsWorkspaceManager.addProjectNature(project, WTP_NATURE);
        } catch (Exception e) {
            LOG.error("error during add wtp nature", e);
        }
    }

    protected void assembleWebContent(IProject project, IFile wsdlFile, IProgressMonitor monitor) {
        try {
            monitor.beginTask(ScJaxWsResources.getString(
                    "wtpdeploy.monitor.copywebxml"), 5);

            // copy web.xml file
            IFile webXmlFile = JaxWsWorkspaceManager.getWebXmlFile(project);

            if (!webXmlFile.exists()) {
                String webXmlFilePath = CeltixKitUtils.getWebXmlPath();
                JaxWsWorkspaceManager.copyFile(webXmlFilePath,
                                          webXmlFile,
                                          JaxWsWorkspaceManager.COPY_NO_OVERWRITE,
                                          monitor);
            }

            monitor.worked(5);

            // copy celtix-servlet.xml
            monitor.beginTask(ScJaxWsResources.getString("wtpdeploy.monitor.copyservletxml"), 5);

            IFolder webInfFolder = JaxWsWorkspaceManager.getWebInfFolder(project);
            IFile configFile = JaxWsWorkspaceManager.getServletConfigFile(project);
            JaxWsWorkspaceManager.copyFile(configFile, webInfFolder, monitor);
            monitor.worked(5);

            // copy wsdl and schema files
            monitor.beginTask(ScJaxWsResources.getString(
                    "wtpdeploy.monitor.copywsdl"), 10);

            IContainer wsdlFolder = JaxWsWorkspaceManager.getWSDLFolder(project);
            HashSet<String> extensions = new HashSet<String>();
            extensions.add("wsdl");
            JaxWsWorkspaceManager.copyFilesInFolder(wsdlFolder,
                JaxWsWorkspaceManager.getWebWSDLFolder(project), extensions, monitor);
            monitor.worked(10);

            monitor.beginTask(ScJaxWsResources.getString(
                    "wtpdeploy.monitor.copyjars"), 80);

            //copy lib files
            String libPath = CeltixKitUtils.getLibPath();
            JaxWsWorkspaceManager.copyFilesInFolder(libPath,
                                               JaxWsWorkspaceManager.getWebLibFolder(project),
                                               ".jar",
                                               JaxWsWorkspaceManager.COPY_NO_OVERWRITE,
                                               monitor);
            monitor.worked(80);
        } catch (Exception e) {
            LOG.error("error during assemble web content", e);
        }
    }
}
