/*******************************************************************************
* Copyright (c) 2006 IONA Technologies PLC
* All rights reserved. This program and the accompanying materials
* are made available under the terms of the Eclipse Public License v1.0
* which accompanies this distribution, and is available at
* http://www.eclipse.org/legal/epl-v10.html
* 
* Contributors:
*     IONA Technologies PLC - initial API and implementation
*******************************************************************************/
package org.eclipse.stp.sc.jaxws.runtimeprovider;

import java.util.List;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.Path;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.stp.common.logging.LoggingProxy;
import org.eclipse.stp.sc.common.workspace.WorkspaceManager;
import org.eclipse.stp.sc.jaxws.properties.ScJaxWsPropertyConstants;

/**
 * handles the kit related resources for a given type of runtime provider.
 * See the "org.eclipse.stp.sc.jaxws.runtimeProvider" extension point 
 */
public class RuntimeKitProcessor implements IRuntimeKitProcessor {
    
    private static final LoggingProxy LOG = LoggingProxy.getlogger(RuntimeKitProcessor.class);
    
    // used as key to store, in the project persistant properties, the name of the substitution VAR used by
    // the project 
//    private static final String RUNTIME_VAR_NAME_PROP = "runtimePathVarName";
//    private static final QualifiedName VAR_NAME_PROP_KEY = new QualifiedName(ScJaxWsPlugin.PLUGIN_ID,
//                                                                             RUNTIME_VAR_NAME_PROP);
    
    private IRuntimeKitProcessorTemplate template;
    private IPath kitRootDirectory;

    public RuntimeKitProcessor(IRuntimeKitProcessorTemplate aTemplate, IPath aKitRootDirectory) {
        template = aTemplate;
        kitRootDirectory = aKitRootDirectory;
    }

    /* (non-Javadoc)
     * @see org.eclipse.stp.sc.jaxws.runtimeprovider.IRuntimeKitProcessor#getDefaultBuildPathEntries()
     */
    public List<IPath> getDefaultBuildPathEntries(IProject project){
        return template.getDefaultBuildPathEntries(project);
    }
    
    /* (non-Javadoc)
     * @see org.eclipse.stp.sc.jaxws.runtimeprovider.IRuntimeKitProcessor#getKitRootDirectory()
     */
    public IPath getInstallationDirectory(){
        return (kitRootDirectory != null 
                && kitRootDirectory.toFile().exists()
                && kitRootDirectory.toFile().isDirectory())
            ? kitRootDirectory 
            : getDefaultInstallationDirectory();
    }
    
    public IPath getKitRootDirectory(IProject prj) {
    	IPath kitPath = null;
    	String kitVar = null;
		try {
			kitVar = prj.getPersistentProperty(
					ScJaxWsPropertyConstants.PROPERTY_KEY_RUNTIME_ROOT);
		} catch (CoreException e) {
			LOG.debug("Exception getting project kit path", e);
		}
    	if (kitVar != null && (!kitVar.trim().equals(""))) {
    		kitPath = new Path(kitVar);
    	} else {
    		kitPath = getInstallationDirectory();
    	}
    	return kitPath;
    }

    /* (non-Javadoc)
     * @see org.eclipse.stp.sc.jaxws.runtimeprovider.IRuntimeKitProcessor#getDefaultKitRootDirectory()
     */
    public IPath getDefaultInstallationDirectory(){
        return template.getDefaultInstallDirectory();
    }
    
    /* (non-Javadoc)
     * @see org.eclipse.stp.sc.jaxws.runtimeprovider.IRuntimeKitProcessor#getDefaultSubstitutionVarName()
     */
    public String getSubstitutionVarName(){
        return template.getDefaultSubstitutionVarName();
    }

    /* (non-Javadoc)
     * @see org.eclipse.stp.sc.jaxws.runtimeprovider.IRuntimeKitProcessor#getDefaultSubstitutionVarName()
     */
    public String getDefaultSubstitutionVarName(){
        return template.getDefaultSubstitutionVarName();
    }

    /* (non-Javadoc)
     * @see org.eclipse.stp.sc.jaxws.runtimeprovider.IRuntimeKitProcessor#validateBuildPathEntries(java.util.List)
     */
    public List<IPath> validateBuildPathEntries(List<IPath> proposedBuildPath){
        return template.validateBuildPathEntries(proposedBuildPath);

    }
    
    /* (non-Javadoc)
     * @see org.eclipse.stp.sc.jaxws.runtimeprovider.IRuntimeKitProcessor#completeJavaProjectBuildPath(org.eclipse.jdt.core.IJavaProject)
     */
    public void completeJavaProjectBuildPath(IJavaProject jProject) throws CoreException {
        String projectSubstitutionVarName = getKitRootDirectory(jProject.getProject()).toOSString();
        WorkspaceManager.addEntriesToClassPath(jProject, 
        		WorkspaceManager.convertFilesToPathEnries(
        				getDefaultBuildPathEntries(jProject.getProject()),
                            projectSubstitutionVarName));
    }

    public void clearJavaProjectBuildPath(IJavaProject jProject) throws CoreException {
        String varName = getKitRootDirectory(jProject.getProject()).toOSString();
        if(varName == null){
        	return;
        }
        WorkspaceManager.removeEntriesFromClassPath(jProject, varName);       
    } 

    public boolean isInstallationDirectoryValid(IPath installationPath) {
        return template.isInstallDirectoryValid(installationPath);
    }
}