/*******************************************************************************
* Copyright (c) 2006 IONA Technologies PLC
* All rights reserved. This program and the accompanying materials
* are made available under the terms of the Eclipse Public License v1.0
* which accompanies this distribution, and is available at
* http://www.eclipse.org/legal/epl-v10.html
* 
* Contributors:
*     IONA Technologies PLC - initial API and implementation
*******************************************************************************/
package org.eclipse.stp.sc.jaxws.workspace;


import java.util.ArrayList;
import java.util.List;

import org.eclipse.core.internal.resources.Folder;
import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IProjectDescription;
import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.QualifiedName;
import org.eclipse.jdt.core.IClasspathEntry;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.JavaCore;
import org.eclipse.jdt.core.JavaModelException;
import org.eclipse.jdt.launching.JavaRuntime;
import org.eclipse.stp.common.logging.LoggingProxy;
import org.eclipse.stp.sc.common.workspace.WorkspaceManager;
import org.eclipse.stp.sc.jaxws.ScJaxWsResources;
import org.eclipse.stp.sc.jaxws.properties.ScJaxWsPropertyConstants;
import org.eclipse.stp.sc.jaxws.runtimeprovider.IJavaToWsdlGenerator;
import org.eclipse.stp.sc.jaxws.runtimeprovider.IProjectDirTemplate;
import org.eclipse.stp.sc.jaxws.runtimeprovider.RuntimeProviderManager;
import org.eclipse.stp.sc.jaxws.utils.ScJDTUtils;

public class JaxWsWorkspaceManager extends WorkspaceManager{
    public static final String WEB_DEPLOY_FOLDER = "WebContent";
    public static final String SERVLET_CONFIG_FILENAME = "cxf-servlet.xml";
    public static final String WEB_XML_FILENAME = "web.xml";
    public static final String SPRING_CONFIG_FILENAME = "spring.xml";
    public static final String WEB_INF_DIR = "WEB-INF";
    public static final String WEB_WSDL_DIR = "wsdl";
    public static final String WEB_CLASS_DIR = "classes";
    public static final String WEB_LIB_DIR = "lib";
    public static final boolean COPY_NO_OVERWRITE = false;
    public static final boolean COPY_OVERWRITE = true;
    
    private static final String CXF_GENERATOR_CLSNAME = 
    	"org.eclipse.stp.sc.cxf.generators.CeltixJavaToWsdlGenerator";
    
    public static final QualifiedName INTERFACE_PROPERTY = 
    	new QualifiedName("org.eclipse.stp", "interface");
    
    public static final QualifiedName CLASS_PROPERTY = 
    	new QualifiedName("org.eclipse.stp", "class");
    
    public static final QualifiedName WSDL_PROPERTY = 
    	new QualifiedName("org.eclipse.stp", "wsdl");
    
    private static final LoggingProxy LOG = LoggingProxy.getlogger(JaxWsWorkspaceManager.class);

    protected JaxWsWorkspaceManager() {
    }

    /**
     * create an jaxws project
     * @param monitor
     * @param projectName
     * @param locationPath
     * @return
     */
    public static IProject createProject(IProgressMonitor monitor,
        String projectName, IPath locationPath, String projectMode) {
        monitor = (monitor != null) ? monitor : new NullProgressMonitor();
        monitor.beginTask(ScJaxWsResources.getString("jaxws.project.creation.progress.proj"), 50);

        try {
            IWorkspaceRoot root = ResourcesPlugin.getWorkspace().getRoot();
            monitor.worked(10);
            monitor.subTask(ScJaxWsResources.getString("jaxws.project.creation.progress.dir"));
            monitor.worked(20);

            IProject theProject = root.getProject(projectName);

            IProjectDescription description = ResourcesPlugin.getWorkspace()
                                                             .newProjectDescription(theProject
                    .getName());

            if ((locationPath != null)
                    && !Platform.getLocation().equals(locationPath)) {
                description.setLocation(locationPath);
            }

            // create and open the initial project
            theProject.create(description, monitor);
            theProject.open(monitor);

            // First setup the Java Nature
            if (projectMode.equals(ScJaxWsPropertyConstants.PROJECT_MODE_JAVAFIRST)) {
            	setupJavaNature(theProject);
            	monitor.worked(30);
            	ScNature.addToProject(theProject, projectMode);
            	monitor.worked(35);
            } else {
            	//for wsdl first, we only setup nature if it is cxf integration
            	IJavaToWsdlGenerator generator = RuntimeProviderManager.getInstance().getJavaToWsdlGenerator(theProject);
                if (generator != null) {
                	if (generator.getClass().getName().equals(
                			CXF_GENERATOR_CLSNAME)) {
                		setupJavaNature(theProject);
                    	monitor.worked(30);
                    	ScNature.addToProject(theProject, projectMode);
                    	monitor.worked(35);
                	}
                }
            }
            
            monitor.worked(40);

            return theProject;
        } catch (Exception e) {
            LOG.error("Error in creating project. ", e);
            return null;
        } finally {
            monitor.done();
        }
    }
    
    
    public static void setupProjectNature(IProject project, String projectMode) throws CoreException {
    	if (!project.hasNature(JavaCore.NATURE_ID)) {
    	    setupJavaNature(project);
    	}
    	ScNature.addToProject(project, projectMode);
    }

    public static void createWebDeployFolders(IProject hostProject,
        IProgressMonitor monitor) {
        try {
            IFolder deployFolder = getWebDeployFolder(hostProject);
            createFolder(deployFolder, monitor);

            IFolder webInfFolder = deployFolder.getFolder(WEB_INF_DIR);
            createFolder(webInfFolder, monitor);

            IFolder metaInfFolder = deployFolder.getFolder(WorkspaceManager.META_INF_DIR);
            createFolder(metaInfFolder, monitor);

            IFolder wsdlFolder = webInfFolder.getFolder(WEB_WSDL_DIR);
            createFolder(wsdlFolder, monitor);

            IFolder libFolder = webInfFolder.getFolder(WEB_LIB_DIR);
            createFolder(libFolder, monitor);

            IFolder clsFolder = webInfFolder.getFolder(WEB_CLASS_DIR);
            createFolder(clsFolder, monitor);
        } catch (Exception e) {
            LOG.error(e);
        }
    }



    public static IFolder getWebDeployFolder(IProject hostProject) {
        return hostProject.getFolder(WEB_DEPLOY_FOLDER);
    }

    public static IFolder getWebInfFolder(IProject hostProject) {
        return getWebDeployFolder(hostProject).getFolder(WEB_INF_DIR);
    }

    public static IFolder getWebWSDLFolder(IProject hostProject) {
        return getWebDeployFolder(hostProject).getFolder(WEB_INF_DIR)
                   .getFolder(WSDL_FOLDER);
    }

    public static IFolder getWebClsFolder(IProject hostProject) {
        return getWebDeployFolder(hostProject).getFolder(WEB_INF_DIR)
                   .getFolder(WEB_CLASS_DIR);
    }

    public static IFolder getWebLibFolder(IProject hostProject) {
        return getWebDeployFolder(hostProject).getFolder(WEB_INF_DIR)
                   .getFolder(WEB_LIB_DIR);
    }

    public static IFile getServletConfigFile(IProject project) {
        IContainer container = getWSDLFolder(project);
        
        return ScJDTUtils.getFileFromContainer(container, SERVLET_CONFIG_FILENAME);
    }
    
    public static IFile getSpringConfigFile(IProject project) { 
    	IContainer container = getWSDLFolder(project);

        return ScJDTUtils.getFileFromContainer(container, SPRING_CONFIG_FILENAME);
    }

    public static IFile getWebXmlFile(IProject project) {
        IFolder folder = getWebInfFolder(project);

        return folder.getFile(WEB_XML_FILENAME);
    }
    
    
    public static IContainer getWSDLFolder(IProject hostProject) {
        IProjectDirTemplate temp = RuntimeProviderManager.getInstance().getProjectDirTemplate(hostProject);
        if (temp == null) {
        	return hostProject.getFolder(WSDL_FOLDER);
        }
        if(temp.getWsdlDir(hostProject) == null){
        	return hostProject;
        }
        return temp.getWsdlDir(hostProject);
    }

    public static IFolder getSrcFolder(IProject hostProject) {
    	IProjectDirTemplate temp = RuntimeProviderManager.getInstance().getProjectDirTemplate(hostProject);
    	if (temp == null) {
    		return hostProject.getFolder(SRC_FOLDER);
    	}
        return temp.getSrcDir(hostProject);
    }

    public static IFolder getClassesFolder(IProject hostProject) {
    	IProjectDirTemplate temp = RuntimeProviderManager.getInstance().getProjectDirTemplate(hostProject);
    	if (temp == null) {
    		return hostProject.getFolder(CLASSES_FOLDER);
    	}
        return temp.getClassFolder(hostProject);
    }

    public static IContainer getWSDLGenFolder(IPath path, IProject project) throws Exception {
 
    	IContainer wsdlRoot = getWSDLFolder(project);
        if (!wsdlRoot.exists() && wsdlRoot instanceof IFolder) {
            ((IFolder)wsdlRoot).create(false, true, null);
        }
 
        return wsdlRoot;
	}
    
    /**
     * This method sets up the Java nature on a project.
     *
     * @param project
     */
    public static void setupJavaNature(IProject project)
        throws CoreException {
        addProjectNature(project, JavaCore.NATURE_ID);

        // create default src and bin folders
        IFolder srcFolder = getSrcFolder(project);

        if (!srcFolder.exists()) {
            srcFolder.create(false, true, null);
        }

        IFolder classesFolder = getClassesFolder(project);

        if (!classesFolder.exists()) {
            classesFolder.create(false, true, null);
        }

        // add classpath to java project
        IJavaProject javaProject = JavaCore.create(project);
        setupJavaNatureClasspath(javaProject);
        
        // change default output location to classes folder
        javaProject.setOutputLocation(getClassesFolder(project).
            getFullPath(), null);
    }
    
    
    /**
     * This method sets up the classpath that is needed to build the java code
     * that is produced by the java generator.
     *
     * @param codeGenHelper
     * @param javaProj
     * @throws CfgException
     * @throws JavaModelException
     */
    protected static void setupJavaNatureClasspath(IJavaProject javaProj)
        throws JavaModelException {
        // Initialize entries            
        List<IClasspathEntry> libraryEntries = new ArrayList<IClasspathEntry>();

        // Add system JRE 
        IClasspathEntry jreEntry = JavaRuntime.getJREVariableEntry();
        libraryEntries.add(jreEntry);

        // Add default entry for src folder in project
        IClasspathEntry srccpe = JavaCore.newSourceEntry(getSrcFolder(
                    javaProj.getProject()).getFullPath());
        libraryEntries.add(srccpe);

        // Set projects classpath
        javaProj.setRawClasspath(libraryEntries.toArray(
                (IClasspathEntry[])new IClasspathEntry[0]), null);
    }
  
}
