/*******************************************************************************

* Copyright (c) 2007 IONA Technologies PLC

* All rights reserved. This program and the accompanying materials

* are made available under the terms of the Eclipse Public License v1.0

* which accompanies this distribution, and is available at

* http://www.eclipse.org/legal/epl-v10.html

* 

* Contributors:

*     IONA Technologies PLC - initial API and implementation

*******************************************************************************/
package org.eclipse.stp.sc.sca.scdl;

import java.io.File;
import java.util.Hashtable;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import org.eclipse.stp.common.logging.LoggingProxy;
import org.eclipse.stp.sc.common.validator.XMLUtils;
import org.eclipse.stp.sc.sca.ScScaResources;
import org.w3c.dom.Attr;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Text;

/**
 * This class is used to generate default.scdl file for 
 * service creation component project
 * @author jma
 */
public class ScdlBuilder {

	private static final LoggingProxy LOG = LoggingProxy.getlogger(ScdlBuilder.class);
	private static final String DEFAULT_SCDL_NAME = "default.scdl";
	private static final String TAG_SERVICE = "service";
	private static final String TAG_INTERFACE_JAVA = "interface.java";
	private static final String TAG_INTERFACE_WS = "interface.wsdl";
	private static final String TAG_BINDING_RMI = "binding.rmi";
	private static final String TAG_BINDING_WS = "binding.ws";
	private static final String TAG_COMPONENT = "component";
	private static final String TAG_IMPLEMENTATION_JAVA = "implementation.java";
	private static final String TAG_COMPOSITE = "composite";
	
	private static final String TAG_XMLNS = "xmlns";
	private static final String TAG_RMI = "rmi";
	private static final String TAG_WSDLI = "wsdli";
	private static final String TAG_IMPORT_SDO = "import.sdo";
	private static final String TAG_DBSDO = "dbsdo";

	private static final String ATTR_NAME = "name";
	private static final String ATTR_INTERFACE = "interface";
	private static final String ATTR_REFERENCE = "reference";
	private static final String ATTR_CLASS = "class";
	
	private static final String ATTR_WSDLLOCATION = "wsdlLocation";
	private static final String ATTR_WSDLINTERFACE = "wsdl.interface";
	
	private static final String SEPERATOR = "#";
	private static final String ATTR_LOCATION = "location";
	
	public enum Language {
		JAVA,
		CPP,
		RUBY,
		PHP
	}
	
	public enum InfType {
		JAVA,
		WSDL
	}
	
	public enum BindingType {
		RMI,
		WS
	}
	
	
	String fileName;
	String filePath;
	String fileFullPath;
	
	String compName;
	Language compLan;
	String compClass;
	
	String serviceName;
	String interfaceName;
	InfType interfaceType;
	
	BindingType bindingType;
	Hashtable<String, String> bindingAttrs;
	
	String compositeName;
	
    String url; //for WS interface and binding
	public ScdlBuilder() {
		fileName = DEFAULT_SCDL_NAME;
		filePath = ".";
	}
	
	
	
	public boolean createScdlFile() {
		try {
			DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
			factory.setNamespaceAware(true);
			DocumentBuilder builder = factory.newDocumentBuilder();
			Document document = builder.newDocument();
			Element elem = createCompositeElem(document);
			createSDO(document, elem);
			createComponentElem(document, elem);
			createServiceElem(document, elem);
			String outputFile = fileFullPath;
			if (outputFile == null) {
			    outputFile = filePath + File.separator + fileName;
			}
			LOG.debug("generate scdl file at:" +outputFile);
			XMLUtils.saveDocument(document, outputFile);
		} catch (ParserConfigurationException e) {
		    LOG.error("error during create scdl file", e);
		    return false;
        }
	
		return true;
	}
	
	private void createSDO(Document document, Element parent) {
		if (bindingType == null) {
			return;
		}
		if (bindingType != BindingType.WS) {
			return;
		}
		Element elem = document.createElementNS(ScScaResources.getString("namespace.dbsdo"), TAG_IMPORT_SDO);
		elem.setPrefix(TAG_DBSDO);
		elem.setAttribute(TAG_XMLNS + ":" + TAG_DBSDO, 
		ScScaResources.getString("namespace.rmi"));
		elem.setAttribute(ATTR_LOCATION, 
				bindingAttrs.get(ATTR_LOCATION));
		parent.appendChild(elem);
		
	}
	
	private void createServiceElem(Document document, Element parent) {
		if (serviceName == null || serviceName.equals("")) {
			//no service has been defined
			return;
		}
		Element elem = document.createElement(TAG_SERVICE);
		elem.setAttribute(ATTR_NAME, serviceName);
		parent.appendChild(elem);
		//create interface node
		if (interfaceName != null) {
			createInterface(document, elem);
		}
		
		//create binding node
		if (bindingType != null) {
			createBinding(document, elem);
		}
			
		//create reference node
		Element refElem = document.createElement(ATTR_REFERENCE);
		Text textElem = document.createTextNode("");
		textElem.setNodeValue(compName);
		refElem.appendChild(textElem);
		elem.appendChild(refElem);
	}
	
	private void createInterface(Document document, Element parent) {
		if (interfaceType == InfType.JAVA) {
			Element infElem = document.createElement(TAG_INTERFACE_JAVA);
			infElem.setAttribute(ATTR_INTERFACE, interfaceName);
			parent.appendChild(infElem);
		} else if (interfaceType == InfType.WSDL) {
			//only use the simple name as interface name for ws interface
			if (interfaceName.indexOf(".")>0) {
				interfaceName = interfaceName.substring(
						interfaceName.lastIndexOf(".") + 1,
						interfaceName.length());
			}
			
			Element infElem = document.createElement(TAG_INTERFACE_WS);
			String wsInterface = url + SEPERATOR + ATTR_WSDLINTERFACE + "(" + interfaceName + ")";
			infElem.setAttribute(ATTR_INTERFACE, wsInterface);
			
			
			String wsdliNS = ScScaResources.getString("namespace.wsdli");
			String wsdlLocStr = url + " " + bindingAttrs.get(ATTR_LOCATION);
			Attr attr = document.createAttributeNS(wsdliNS, ATTR_WSDLLOCATION);
			attr.setPrefix(TAG_WSDLI);
			attr.setValue(wsdlLocStr);
			infElem.setAttributeNode(attr);
			
			parent.appendChild(infElem);
		}
	}
	
	private void createBinding(Document document, Element parent) {
	    if (bindingType == BindingType.RMI) {
	    	Element bindElem = document.createElementNS(
					ScScaResources.getString("namespace.rmi"),
					TAG_BINDING_RMI);
			bindElem.setPrefix(TAG_RMI);
			addAllAttributes(bindElem, bindingAttrs);
			parent.appendChild(bindElem);
	    } else if (bindingType == BindingType.WS) {
	    	//web service binding
	    	Element bindElem;
			bindElem = document.createElement(TAG_BINDING_WS);
			addAllAttributes(bindElem, bindingAttrs);
			parent.appendChild(bindElem);
	    }
	}
	
	private void addAllAttributes(Element elem, Hashtable<String, String> attrTable) {
		for (String attr : attrTable.keySet()) {
			elem.setAttribute(attr, attrTable.get(attr));
		}
	}
	
	private void createComponentElem(Document document, Element parent) {
		Element elem = document.createElement(TAG_COMPONENT);
		elem.setAttribute(ATTR_NAME, compName);
		if (compLan == Language.JAVA) {
            Element implElem = document.createElement(TAG_IMPLEMENTATION_JAVA);
            implElem.setAttribute(ATTR_CLASS, compClass);
            elem.appendChild(implElem);
		}
		parent.appendChild(elem);
	}
	
	private Element createCompositeElem(Document document) {
		Element elem = document.createElement(TAG_COMPOSITE);
		elem.setAttribute(TAG_XMLNS, ScScaResources.getString("namespace.sca"));
		if (bindingType != null && bindingType == BindingType.RMI) {
			elem.setAttribute(TAG_XMLNS + ":" + TAG_RMI, 
					ScScaResources.getString("namespace.rmi"));
		} else if (bindingType != null && bindingType == BindingType.WS) {
			elem.setAttribute(TAG_XMLNS + ":" + TAG_WSDLI, 
					ScScaResources.getString("namespace.wsdli"));
		}
		elem.setAttribute(ATTR_NAME, compositeName);
		document.appendChild(elem);
		return elem;
	}



	public Hashtable<String, String> getBindingAttrs() {
		return bindingAttrs;
	}



	public void setBindingAttrs(Hashtable<String, String> bindingAttrs) {
		this.bindingAttrs = bindingAttrs;
	}



	public BindingType getBindingType() {
		return bindingType;
	}



	public void setBindingType(BindingType bindingType) {
		this.bindingType = bindingType;
	}



	public String getCompClass() {
		return compClass;
	}



	public void setCompClass(String compClass) {
		this.compClass = compClass;
	}



	public Language getCompLan() {
		return compLan;
	}



	public void setCompLan(Language compLan) {
		this.compLan = compLan;
	}



	public String getCompName() {
		return compName;
	}



	public void setCompName(String compName) {
		this.compName = compName;
	}



	public String getCompositeName() {
		return compositeName;
	}



	public void setCompositeName(String compositeName) {
		this.compositeName = compositeName;
	}



	public String getFileName() {
		return fileName;
	}



	public void setFileName(String fileName) {
		this.fileName = fileName;
	}



	public String getFilePath() {
		return filePath;
	}



	public void setFilePath(String filePath) {
		this.filePath = filePath;
	}

	public String getFileFullPath() {
		return fileFullPath;
	}



	public void setFileFullPath(String filePath) {
		this.fileFullPath = filePath;
	}


	public String getInterfaceName() {
		return interfaceName;
	}



	public void setInterfaceName(String interfaceName) {
		this.interfaceName = interfaceName;
	}



	public InfType getInterfaceType() {
		return interfaceType;
	}



	public void setInterfaceType(InfType interfaceType) {
		this.interfaceType = interfaceType;
	}



	public String getServiceName() {
		return serviceName;
	}



	public void setServiceName(String serviceName) {
		this.serviceName = serviceName;
	}
	
	public String getUrl() {
		return url;
	}



	public void setUrl(String value) {
		this.url = value;
	}
}
