/*******************************************************************************
* Copyright (c) 2006 IONA Technologies PLC
* All rights reserved. This program and the accompanying materials
* are made available under the terms of the Eclipse Public License v1.0
* which accompanies this distribution, and is available at
* http://www.eclipse.org/legal/epl-v10.html
* 
* Contributors:
*     IONA Technologies PLC - initial API and implementation
*******************************************************************************/
package org.eclipse.stp.sc.sca.java.workspace;


import java.io.File;
import java.util.ArrayList;
import java.util.List;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IProjectDescription;
import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.Platform;
import org.eclipse.jdt.core.IClasspathEntry;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.JavaCore;
import org.eclipse.jdt.core.JavaModelException;
import org.eclipse.jdt.launching.JavaRuntime;
import org.eclipse.stp.common.logging.LoggingProxy;
import org.eclipse.stp.sc.common.workspace.WorkspaceManager;
import org.eclipse.stp.sc.sca.java.ScaJavaResources;
import org.eclipse.stp.sc.sca.workspace.ScaNature;

public class ScaWorkspaceManager extends WorkspaceManager{
	
	public static final String SCDL_FILE_NAME = "default.scdl";
	public static final String SCA_PATH = "sca";
	public static final String SCDL_FILE_PATH = META_INF_DIR + File.separator + SCA_PATH;
    
    private static final LoggingProxy LOG = LoggingProxy.getlogger(ScaWorkspaceManager.class);

    protected ScaWorkspaceManager() {
    }
    
    public static IFolder getJarOutputFolder(IProject hostProject) {
    	return getBinFolder(hostProject);
    }
    
    public static IFolder getScdlFileFolder(IProject hostProject) {
        return hostProject.getFolder(SCDL_FILE_PATH);	
    }
    
    /**
     * return the name as projectName + _sca.jar.
     * @todo. let user the configure the output jar in preference or project property
     * @param hostProject
     * @return
     */
    public static String getJarOutputName(IProject hostProject) {
    	return hostProject.getName() + "_sca.jar";
    }
    
    public static IFile getJarOutputFile(IProject hostProject) {
    	IFolder jarFolder = getJarOutputFolder(hostProject);
    	return jarFolder.getFile(getJarOutputName(hostProject));
    }

    /**
     * create an sca.java project
     * @param monitor
     * @param projectName
     * @param locationPath
     * @return
     */
    public static IProject createProject(IProgressMonitor monitor,
        String projectName, IPath locationPath) {
        monitor = (monitor != null) ? monitor : new NullProgressMonitor();
        monitor.beginTask(ScaJavaResources.getString("sca.java.project.creation.progress.proj"), 50);

        try {
            IWorkspaceRoot root = ResourcesPlugin.getWorkspace().getRoot();
            monitor.worked(10);
            monitor.subTask(ScaJavaResources.getString("sca.java.project.creation.progress.dir"));
            monitor.worked(20);

            IProject theProject = root.getProject(projectName);

            IProjectDescription description = ResourcesPlugin.getWorkspace()
                                                             .newProjectDescription(theProject
                    .getName());

            if ((locationPath != null)
                    && !Platform.getLocation().equals(locationPath)) {
                description.setLocation(locationPath);
            }
            
            

            // create and open the initial project
            theProject.create(description, monitor);
            theProject.open(monitor);

            createProjectDirs(theProject, monitor);
            // First setup the Java Nature
            setupJavaNature(theProject);
            monitor.worked(30);

            // now setup the SC nature
            ScaNature scaNature = new ScaNature();
            scaNature.addToProject(theProject);
            ScaJavaNature scaJavaNature = new ScaJavaNature();
            scaJavaNature.addToProject(theProject);
            monitor.worked(35);
            monitor.worked(40);

            
            return theProject;
        } catch (Exception e) {
            LOG.error("Error in creating project. ", e);
            return null;
        } finally {
            monitor.done();
        }
    }

    private static void createProjectDirs(IProject proj, IProgressMonitor monitor) throws CoreException {
    	IFolder binFolder = getBinFolder(proj);
        createFolder(binFolder, monitor);
        
        IFolder metaInfFolder = getMetaInfFolder(proj);
        createFolder(metaInfFolder, monitor);
        
        IFolder scaFolder = metaInfFolder.getFolder(SCA_PATH);
        createFolder(scaFolder, monitor);
        
        IFolder wsdlFolder = getWSDLFolder(proj);
        createFolder(wsdlFolder, monitor);

    }

    public static String getDefaultWSServiceName(String portType) {
    	return portType + "Service";
    }
    
    public static String getDefaultWSPortName(String portType) {
    	return portType + "ServiceSoap11Port";
    }
    
    public static String getDefaultWSUrl(String portType) {
    	return DEFAULT_URL + portType + "Service";
    }
    
    public static String getDefaultWSUrl() {
    	return DEFAULT_URL;
    }
    
    
    /**
     * This method sets up the Java nature on a project.
     *
     * @param project
     */
    protected static void setupJavaNature(IProject project)
        throws CoreException {
        addProjectNature(project, JavaCore.NATURE_ID);

        // create default src and bin folders
        IFolder srcFolder = getSrcFolder(project);

        if (!srcFolder.exists()) {
            srcFolder.create(false, true, null);
        }

        IFolder classesFolder = getClassesFolder(project);

        if (!classesFolder.exists()) {
            classesFolder.create(false, true, null);
        }

        // add classpath to java project
        IJavaProject javaProject = JavaCore.create(project);
        setupJavaNatureClasspath(javaProject);
        
        // change default output location to classes folder
        javaProject.setOutputLocation(getClassesFolder(project).
            getFullPath(), null);
    }
    
    
    /**
     * This method sets up the classpath that is needed to build the java code
     * that is produced by the java generator.
     *
     * @param codeGenHelper
     * @param javaProj
     * @throws CfgException
     * @throws JavaModelException
     */
    protected static void setupJavaNatureClasspath(IJavaProject javaProj)
        throws JavaModelException {
        // Initialize entries            
        List<IClasspathEntry> libraryEntries = new ArrayList<IClasspathEntry>();

        // Add system JRE 
        IClasspathEntry jreEntry = JavaRuntime.getJREVariableEntry();
        libraryEntries.add(jreEntry);

        // Add default entry for src folder in project
        IClasspathEntry srccpe = JavaCore.newSourceEntry(getSrcFolder(
                    javaProj.getProject()).getFullPath());
        libraryEntries.add(srccpe);

        // Set projects classpath
        javaProj.setRawClasspath(libraryEntries.toArray(
                (IClasspathEntry[])new IClasspathEntry[0]), null);
    }
    
    public static IFolder getWSDLFolder(IProject hostProject) {
        return hostProject.getFolder(WSDL_FOLDER);
    }

    public static IFolder getSrcFolder(IProject hostProject) {
    	return hostProject.getFolder(SRC_FOLDER);
    }

    public static IFolder getClassesFolder(IProject hostProject) {
    	return hostProject.getFolder(CLASSES_FOLDER);
    }
    
    private static final String DEFAULT_URL = "http://localhost:8080/service/";
}
