/*******************************************************************************
* Copyright (c) 2006 IONA Technologies PLC
* All rights reserved. This program and the accompanying materials
* are made available under the terms of the Eclipse Public License v1.0
* which accompanies this distribution, and is available at
* http://www.eclipse.org/legal/epl-v10.html
* 
* Contributors:
*     IONA Technologies PLC - initial API and implementation
*******************************************************************************/
package org.eclipse.stp.soas.deploy.tomcat.deploydriver;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;

import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.datatools.connectivity.IConnectionProfile;
import org.eclipse.stp.sc.common.utils.FileUtils;
import org.eclipse.stp.soas.deploy.core.DeployCorePlugin;
import org.eclipse.stp.soas.deploy.core.DeploySessionBase;
import org.eclipse.stp.soas.deploy.core.FilePackageOutputDescriptor;
import org.eclipse.stp.soas.deploy.core.IPackageOutputDescriptor;
import org.eclipse.stp.soas.deploy.core.IVersion;
import org.eclipse.stp.soas.deploy.core.Version;
import org.eclipse.stp.soas.deploy.tomcat.cp.TomcatConnection;


public class TomcatDeploySession extends DeploySessionBase {

	private static final String CONNECTION_FACTORY_ID = "Tomcat.Container";
	
	private TomcatConnection mConnection;

	
	public TomcatDeploySession(IConnectionProfile profile) {
		super(profile);
	}
	
	public IStatus[] preDeployCheck(IPackageOutputDescriptor descriptor) {
		openConnection();
		if (mConnection == null || mConnection.getConnectException() != null) {
			
			Status status = new Status(IStatus.ERROR, DeployCorePlugin.getDefault().getBundle().getSymbolicName(), -1,
					DeployCorePlugin.getDefault().getResourceString(
							"DeployCheck.ERROR.CannotConnectToServer", //$NON-NLS-1$
							new Object[] { descriptor.getLogicalName(),
									getConnectionProfile().getName()}),
					mConnection.getConnectException());
			return new IStatus[] { status};
		}

		File deployDir = (File) mConnection.getRawConnection();
		if (!deployDir.exists()) {
			Status status = new Status(
					IStatus.ERROR,
					DeployCorePlugin.getDefault().getBundle()
							.getSymbolicName(),
					-1,
					"Deploy directory "
							+ deployDir.getAbsolutePath()
							+ " doesn't exists.",

					null);
			return new IStatus[] { status};
		}

        File deployPackage = new File(deployDir, descriptor.getLogicalName());
		if (deployPackage.exists()) {
			Status status = new Status(
					IStatus.WARNING,
					DeployCorePlugin.getDefault().getBundle()
							.getSymbolicName(),
					-1,
					"Package "
							+ descriptor.getLogicalName()
							+ " exists on server.  Existing package will be overwritten.",

					null);
			return new IStatus[] { status};
		}
		return new IStatus[0];
	}


	@Override
	public void deploy(IPackageOutputDescriptor descriptor) throws Exception {
		
        FilePackageOutputDescriptor desc = (FilePackageOutputDescriptor)descriptor;
		openConnection();
		File server = (File) mConnection.getRawConnection();
		File newPackage = new File(server, descriptor.getLogicalName());
		newPackage.createNewFile();
		
		
		FileChannel rbc = null;
		FileChannel wbc = null;
		try {
			ByteBuffer bb = ByteBuffer.allocate(0x2000);
			rbc = new FileInputStream(desc.getFile()).getChannel();
			wbc = new FileOutputStream(newPackage).getChannel();
			while (rbc.read(bb) > 0) {
				bb.flip();
				wbc.write(bb);
				bb.clear();
			}
		}
		finally {
			if (rbc != null) {
				try {
					rbc.close();
				}
				catch (IOException e) {
				}
			}
			if (wbc != null) {
				try {
					wbc.close();
				}
				catch (IOException e) {
				}
			}
		}

		
	}

	public void dispose() {
		if (mConnection != null) {
			mConnection.close();
			mConnection = null;
		}
	}

	public IVersion getServerVersion() {
		return new Version(5,5,0,new String());
	}
	
	private void openConnection() {
		if (mConnection == null) {
			mConnection = (TomcatConnection)getConnectionProfile().createConnection(
	        		CONNECTION_FACTORY_ID);
		}
	}

	@Override
	public void undeploy(IPackageOutputDescriptor descriptor) throws Exception {		
		openConnection();
		
		boolean pkgDeleted = true;
		boolean FolderDeleted = true;
		File server = (File) mConnection.getRawConnection();
		File newPackage = new File(server, descriptor.getLogicalName());
		if(newPackage.exists()){
			pkgDeleted = false;
			pkgDeleted = newPackage.delete();
		}
		String expandFolderName = getFileNameWithoutExt(newPackage, '.');
		File expandFolder = new File(server, expandFolderName);
		if(expandFolder.exists()){
			FolderDeleted = false;
			FolderDeleted = FileUtils.deleteFile(expandFolder);
		}
		
		if(!pkgDeleted || !FolderDeleted){
			throw new Exception("Unable to undeploy application " + expandFolderName);
		}
		 
		
	}

	private static String getFileNameWithoutExt(File file, char split) {
		String fullName = file.getName();
		int i = fullName.indexOf(split);
		int leg = fullName.length();
		return (i > 0 ? (i + 1) == leg ? "" : fullName.substring(0, i - 1) : "");
	}

	@Override
	public IStatus[] preUndeployCheck(IPackageOutputDescriptor descriptor) {
		// TODO Auto-generated method stub
		return null;
	}

	public IStatus[] preUnDeployCheck(IPackageOutputDescriptor descriptor) {
		// TODO Auto-generated method stub
		return null;
	}   

}
