/*******************************************************************************
 * Copyright (c) 2004-2006 Sybase, Inc.
 * 
 * All rights reserved. This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License v1.0 which
 * accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Sybase, Inc. - initial API and implementation
 ******************************************************************************/
package org.eclipse.stp.soas.deploy.core;

import java.io.ByteArrayInputStream;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IMarker;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IResourceDelta;
import org.eclipse.core.resources.IncrementalProjectBuilder;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.datatools.connectivity.IConnectionProfile;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.resource.impl.ResourceSetImpl;
import org.eclipse.emf.ecore.xmi.XMLResource;
import org.eclipse.stp.soas.deploy.models.deployfile.DeployConfiguration;
import org.eclipse.stp.soas.deploy.models.deployfile.DeployPackage;
import org.eclipse.stp.soas.deploy.models.deployfile.DeployServer;
import org.eclipse.stp.soas.deploy.models.deployfile.Root;
import org.eclipse.stp.soas.internal.deploy.core.cp.StpProfileManager;
import org.eclipse.stp.soas.internal.deploy.ui.wizards.DeployFileNewWizard;

/**
 * @author rcernich
 * 
 * Created on Dec 1, 2004
 */
public class DeploymentDefinitionBuilder extends IncrementalProjectBuilder {

	public static final String BUILDER_ID = DeployCorePlugin.getDefault()
			.getBundle().getSymbolicName()
			+ ".deploymentDefinitionBuilder"; //$NON-NLS-1$
	public static final String MARKER_TYPE = DeployCorePlugin.getDefault()
			.getBundle().getSymbolicName()
			+ ".deploymentDefinitionProblemMarker"; //$NON-NLS-1$
	public static final String MARKER_ATTR_PROBLEM_ELEMENT_URL = "problemElementURL"; //$NON-NLS-1$

	/**
	 *  
	 */
	public DeploymentDefinitionBuilder() {
		super();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.core.internal.events.InternalBuilder#build(int,
	 *      java.util.Map, org.eclipse.core.runtime.IProgressMonitor)
	 */
	protected IProject[] build(int kind, Map args, IProgressMonitor monitor)
			throws CoreException {
		if (kind == IncrementalProjectBuilder.FULL_BUILD) {
			buildAll();
		}
		else {
			build(getDelta(getProject()));
		}
		return new IProject[0];
	}

	protected boolean isInteresting(IResource resource) {
		int resType = resource.getType();
		return resType == IResource.FOLDER
				|| resType == IResource.PROJECT
				|| resType == IResource.ROOT
				|| (resType == IResource.FILE && DeployFileNewWizard.DEPLOY_FILE_EXTENSION
						.equals(resource.getFileExtension()));
	}

	private void buildAll() throws CoreException {
		buildContainer(getProject());
	}

	private void buildContainer(IContainer container) throws CoreException {
		IResource[] children = container.members();
		for (int index = 0, count = children.length; index < count; ++index) {
			if (children[index].getType() == IResource.FILE) {
				if (isInteresting(children[index])) {
					fileChanged((IFile) children[index]);
				}
			}
			else {
				buildContainer((IContainer) children[index]);
			}
		}
	}

	private void build(IResourceDelta resDelta) throws CoreException {
		if (resDelta != null && isInteresting(resDelta.getResource())) {
			switch (resDelta.getKind()) {
			case IResourceDelta.ADDED:
			case IResourceDelta.CHANGED:
				resourceChanged(resDelta);
				break;
			case IResourceDelta.REMOVED:
				resourceRemoved(resDelta);
				break;
			case IResourceDelta.ADDED_PHANTOM:
			case IResourceDelta.REMOVED_PHANTOM:
			default:
				break;
			}
		}
	}

	private void resourceChanged(IResourceDelta resDelta) throws CoreException {
		switch (resDelta.getResource().getType()) {
		case IResource.FILE:
			// Build this resource
			fileChanged((IFile) resDelta.getResource());
			break;
		case IResource.FOLDER:
		case IResource.PROJECT:
		case IResource.ROOT:
			IResourceDelta[] children = resDelta.getAffectedChildren();
			for (int index = 0, count = children.length; index < count; ++index) {
				build(children[index]);
			}
			break;
		default:
			break;
		}
	}

	private void resourceRemoved(IResourceDelta resDelta) throws CoreException {
		switch (resDelta.getResource().getType()) {
		case IResource.FILE:
			// Build this resource
			fileRemoved((IFile) resDelta.getResource());
			break;
		case IResource.FOLDER:
		case IResource.PROJECT:
		case IResource.ROOT:
			IResourceDelta[] children = resDelta.getAffectedChildren();
			for (int index = 0, count = children.length; index < count; ++index) {
				build(children[index]);
			}
			break;
		default:
			break;
		}
	}

	private void fileChanged(IFile file) throws CoreException {
		ResourceSet rs = null;
		try {
			// Erase any existing markers
			file.deleteMarkers(MARKER_TYPE, true, IResource.DEPTH_ZERO);

			// Load the resource
			rs = new ResourceSetWithOptions();
			Resource resource = rs.getResource(URI
					.createPlatformResourceURI(file.getFullPath().toString()),
					true);

			// Log any errors that occurred during loading
			for (Iterator it = resource.getErrors().iterator(); it.hasNext();) {
				Resource.Diagnostic diag = (Resource.Diagnostic) it.next();
				IMarker marker = file.createMarker(MARKER_TYPE);
				marker.setAttribute(IMarker.SEVERITY, IMarker.SEVERITY_ERROR);
				marker.setAttribute(IMarker.LOCATION, file.getFullPath()
						.toString());
				marker.setAttribute(IMarker.LINE_NUMBER, diag.getLine());
				marker.setAttribute(MARKER_ATTR_PROBLEM_ELEMENT_URL, diag
						.getLocation());
				marker.setAttribute(IMarker.MESSAGE, diag.getMessage());
			}

			// Log any warnings that occurred during loading
			for (Iterator it = resource.getWarnings().iterator(); it.hasNext();) {
				Resource.Diagnostic diag = (Resource.Diagnostic) it.next();
				IMarker marker = file.createMarker(MARKER_TYPE);
				marker.setAttribute(IMarker.SEVERITY, IMarker.SEVERITY_WARNING);
				marker.setAttribute(IMarker.LOCATION, file.getFullPath()
						.toString());
				marker.setAttribute(IMarker.LINE_NUMBER, diag.getLine());
				marker.setAttribute(MARKER_ATTR_PROBLEM_ELEMENT_URL, diag
						.getLocation());
				marker.setAttribute(IMarker.MESSAGE, diag.getMessage());
			}

			// Custom validation
			validateDefinition(file, resource);
		}
		catch (RuntimeException e) {
			e.printStackTrace();
			IMarker marker = file.createMarker(MARKER_TYPE);
			marker.setAttribute(IMarker.SEVERITY, IMarker.SEVERITY_ERROR);
			marker
					.setAttribute(IMarker.LOCATION, file.getFullPath()
							.toString());
			marker.setAttribute(IMarker.MESSAGE, e.getMessage());
		}
		finally {
			if (rs != null) {
				for (Iterator it = rs.getResources().iterator(); it.hasNext();) {
					Resource res = (Resource) it.next();
					if (res.isLoaded()) {
						res.unload();
					}
				}
				rs.getResources().clear();
			}
		}
	}

	private void fileRemoved(IFile file) throws CoreException {
	}

	protected void validateDefinition(IFile file, Resource resource)
			throws CoreException {
		if (resource.getContents().size() == 0
				|| !(resource.getContents().get(0) instanceof Root)) {
			IMarker marker = file.createMarker(MARKER_TYPE);
			marker.setAttribute(IMarker.SEVERITY, IMarker.SEVERITY_ERROR);
			marker
					.setAttribute(IMarker.LOCATION, file.getFullPath()
							.toString());
			marker.setAttribute(IMarker.MESSAGE, DeployCorePlugin
					.getDefault().getResourceString(
							"Builder.InvalidDefinitionError")); //$NON-NLS-1$
			return;
		}

		Root root = (Root) resource.getContents().get(0);
		for (Iterator it = root.getServer().iterator(); it.hasNext();) {
			validateServer(file, (DeployServer) it.next());
		}

		for (Iterator it = root.getPackage().iterator(); it.hasNext();) {
			validatePackage(file, (DeployPackage) it.next());
		}
	}

	protected void validateServer(IFile file, DeployServer server)
			throws CoreException {
		if (StpProfileManager.getInstance().getProfileByName(
				server.getProfileName()) == null) {
			IMarker marker = file.createMarker(MARKER_TYPE);
			marker.setAttribute(IMarker.SEVERITY, IMarker.SEVERITY_ERROR);
			marker
					.setAttribute(IMarker.LOCATION, file.getFullPath()
							.toString());
			marker.setAttribute(IMarker.MESSAGE, DeployCorePlugin
					.getDefault().getResourceString(
							"Builder.InvalidServerError", //$NON-NLS-1$
							new Object[] { server.getProfileName()}));
			marker.setAttribute(MARKER_ATTR_PROBLEM_ELEMENT_URL, server
					.eResource().getURIFragment(server));
		}

		if (server.getTargetingConfiguration().size() == 0) {
			IMarker marker = file.createMarker(MARKER_TYPE);
			marker.setAttribute(IMarker.SEVERITY, IMarker.SEVERITY_WARNING);
			marker
					.setAttribute(IMarker.LOCATION, file.getFullPath()
							.toString());
			marker.setAttribute(IMarker.MESSAGE, DeployCorePlugin
					.getDefault().getResourceString(
							"Builder.UnusedServerWarning", //$NON-NLS-1$
							new Object[] { server.getProfileName()}));
			marker.setAttribute(MARKER_ATTR_PROBLEM_ELEMENT_URL, server
					.eResource().getURIFragment(server));
		}
	}

	protected void validatePackage(IFile file, DeployPackage pkg)
			throws CoreException {
		if (pkg.getTargetConfiguration().size() == 0) {
			IMarker marker = file.createMarker(MARKER_TYPE);
			marker.setAttribute(IMarker.SEVERITY, IMarker.SEVERITY_ERROR);
			marker
					.setAttribute(IMarker.LOCATION, file.getFullPath()
							.toString());
			marker.setAttribute(IMarker.MESSAGE, DeployCorePlugin
					.getDefault().getResourceString(
							"Builder.UntargetedPackageError", //$NON-NLS-1$
							new Object[] { pkg.getPackageFile()}));
			marker.setAttribute(MARKER_ATTR_PROBLEM_ELEMENT_URL, pkg
					.eResource().getURIFragment(pkg));
			return;
		}

		for (Iterator it = pkg.getTargetConfiguration().iterator(); it
				.hasNext();) {
			validateConfiguration(file, (DeployConfiguration) it.next());
		}
	}

	protected void validateConfiguration(IFile file,
			DeployConfiguration configuration) throws CoreException {
		if (configuration.getTargetServer() == null) {
			IMarker marker = file.createMarker(MARKER_TYPE);
			marker.setAttribute(IMarker.SEVERITY, IMarker.SEVERITY_ERROR);
			marker
					.setAttribute(IMarker.LOCATION, file.getFullPath()
							.toString());
			marker.setAttribute(IMarker.MESSAGE, DeployCorePlugin
					.getDefault().getResourceString(
							"Builder.UspecifiedTargetServerError", //$NON-NLS-1$
							new Object[] { configuration.getSourcePackage()
									.getPackageFile()}));
			marker.setAttribute(MARKER_ATTR_PROBLEM_ELEMENT_URL, configuration
					.eResource().getURIFragment(configuration));
		}

		IPackage ip = Utilities.adaptToIPackage(configuration);
		if (ip == null) {
			IMarker marker = file.createMarker(MARKER_TYPE);
			marker.setAttribute(IMarker.SEVERITY, IMarker.SEVERITY_ERROR);
			marker
					.setAttribute(IMarker.LOCATION, file.getFullPath()
							.toString());
			marker.setAttribute(IMarker.MESSAGE, DeployCorePlugin
					.getDefault().getResourceString(
							"Builder.CouldNotAdaptPackageError", //$NON-NLS-1$
							new Object[] {
									configuration.getSourcePackage()
											.getPackageFile(),
									configuration.getTargetServer()
											.getProfileName()}));
			marker.setAttribute(MARKER_ATTR_PROBLEM_ELEMENT_URL, configuration
					.eResource().getURIFragment(configuration));
			return;
		}

		if (ip instanceof ILogicalPackage) {
			IStatus[] validationStatus;
			ILogicalPackage ilp = (ILogicalPackage) ip;
			IPackageConfiguration packageConfig = null;
			if (ip instanceof IConfigurablePackage) {
				IPackageConfigurationManager packageConfigManager = ((IConfigurablePackageExtension) ilp
						.getExtension()).getPackageConfigurationManager();
				byte[] configOverride = configuration.getConfigOverride();
				if (configOverride != null && configOverride.length != 0) {
					try {
						packageConfig = packageConfigManager
								.createPackageConfiguration(
										(IConfigurablePackage) ilp,
										new ByteArrayInputStream(configOverride));
					}
					catch (Exception e) {
						IMarker marker = file.createMarker(MARKER_TYPE);
						marker.setAttribute(IMarker.SEVERITY,
								IMarker.SEVERITY_ERROR);
						marker.setAttribute(IMarker.LOCATION, file
								.getFullPath().toString());
						marker
								.setAttribute(
										IMarker.MESSAGE,
										DeployCorePlugin
												.getDefault()
												.getResourceString(
														"Builder.ConfigOverrideParseError", //$NON-NLS-1$
														new Object[] {
																configuration
																		.getSourcePackage()
																		.getPackageFile(),
																configuration
																		.getTargetServer()
																		.getProfileName(),
																e
																		.getLocalizedMessage()}));
						marker.setAttribute(MARKER_ATTR_PROBLEM_ELEMENT_URL,
								configuration.eResource().getURIFragment(
										configuration));
						return;
					}
				}
			}

			IPackageConstructorExtension ipce = getPackageConstructor(ilp,
					configuration);
			if (ipce == null) {
				validationStatus = new IStatus[0];
			}
			else {
				validationStatus = ipce.validatePackage(ilp, packageConfig,
						null);
			}

			for (Iterator it = Arrays.asList(validationStatus).iterator(); it
					.hasNext();) {
				IStatus status = (IStatus) it.next();
				IMarker marker = file.createMarker(MARKER_TYPE);
				marker.setAttribute(IMarker.SEVERITY,
						convertStatusSeverityToMarkerSeverity(status
								.getSeverity()));
				marker.setAttribute(IMarker.LOCATION, file.getFullPath()
						.toString());
				marker.setAttribute(IMarker.MESSAGE, DeployCorePlugin
						.getDefault().getResourceString(
								"Builder.PackageConfigurationError", //$NON-NLS-1$
								new Object[] {
										configuration.getSourcePackage()
												.getPackageFile(),
										configuration.getTargetServer()
												.getProfileName(),
										status.getMessage()}));
				marker
						.setAttribute(MARKER_ATTR_PROBLEM_ELEMENT_URL,
								configuration.eResource().getURIFragment(
										configuration));
			}
		}
	}

	private int convertStatusSeverityToMarkerSeverity(int statusSeverity) {
		int markerSeverity;
		switch (statusSeverity) {
		case IStatus.ERROR:
			markerSeverity = IMarker.SEVERITY_ERROR;
			break;
		case IStatus.WARNING:
			markerSeverity = IMarker.SEVERITY_WARNING;
			break;
		case IStatus.INFO:
		default:
			markerSeverity = IMarker.SEVERITY_INFO;
			break;
		}
		return markerSeverity;
	}

	private IPackageConstructorExtension getPackageConstructor(
			ILogicalPackage pkg, DeployConfiguration deployConfig) {
		IConnectionProfile profile = StpProfileManager.getInstance()
				.getProfileByName(
						deployConfig.getTargetServer().getProfileName());
		if (profile == null) {
			return null;
		}

		IDeployDriverExtension driver = DeploymentExtensionManager
				.getInstance().getDeployDriver(pkg, profile.getProviderId());
		if (driver == null) {
			return null;
		}

		return driver.getPackageConstructor(pkg.getTechnologyType());
	}

	private static class ResourceSetWithOptions extends ResourceSetImpl {

		public ResourceSetWithOptions() {
			super();
			loadOptions = new HashMap();
			loadOptions.put(XMLResource.OPTION_PROCESS_DANGLING_HREF,
					XMLResource.OPTION_PROCESS_DANGLING_HREF_RECORD);
		}
	}

}