/**
 ******************************************************************************
 * Copyright (c) 2006, Intalio Inc.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Intalio Inc. - initial API and implementation
 *******************************************************************************
 *
 * Date         Author             Changes
 * Jun 18, 2007      Antoine Toulme     Created
 */
package org.eclipse.stp.bpmn.export;

import java.io.File;
import java.util.LinkedList;
import java.util.List;

import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Path;
import org.eclipse.gmf.runtime.common.ui.util.WindowUtil;
import org.eclipse.gmf.runtime.diagram.ui.image.ImageFileFormat;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.DirectoryDialog;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.dialogs.WizardExportResourcesPage;

/**
 * The page used to hold the destination fields. Does the essential of the work.
 * 
 * @author <a href="mailto:atoulme@intalio.com">Antoine Toulm�</a>
 * @author <a href="http://www.intalio.com">&copy; Intalio, Inc.</a>
 */
public class ExportImagePage extends WizardExportResourcesPage implements IExportImagePage {

    private Text folderText;
    private boolean folderValid;
    private String folder = "";
    private boolean fileNameValid = false;
    private Text fileNameText;
    private String fileName = "image";
    private Combo imageFormatCombo;
    private ImageFileFormat imageFormat = ImageFileFormat.JPG;

    public ExportImagePage(IStructuredSelection sel) {
        super("Select the diagram to export", sel);
        setTitle("Export a BPMN diagram as an image");
        setDescription("Choose a file and a format to export the " +
                "diagram as an image");
        
    }

    /**
     * utility method to create a composite widget
     * @param parent the parent widget
     * @param columns the number of columns in the grid layout for the new
     * composite.
     * @return the new composite widget
     */
    private Composite createComposite(Composite parent, int columns) {
        Composite composite = new Composite(parent, SWT.NONE);

        GridLayout gridLayout = new GridLayout();
        gridLayout.numColumns = columns;
        gridLayout.marginHeight = 0;
        gridLayout.marginWidth = 0;
        gridLayout.makeColumnsEqualWidth = false;

        GridData data =
            new GridData(
                GridData.HORIZONTAL_ALIGN_FILL | GridData.FILL_HORIZONTAL);

        composite.setLayoutData(data);
        composite.setLayout(gridLayout);

        return composite;
    }

    /**
     * Creates the Destination Group, the text representing the
     * folder to export to and the file name for the image.
     */
    @Override
    protected void createDestinationGroup(Composite parent) {
        Composite composite = createComposite(parent, 3);
        Label folderLabel = new Label(composite,SWT.NONE);
        folderLabel.setText("Export folder");
        folderText = new Text(composite, SWT.BORDER);
        folderText.setText(folder == null ? ""  : folder);
        folderText.addModifyListener(new ModifyListener() {
            public void modifyText(ModifyEvent e) {
                validateFolderText();
            }
        });
        GridData gridData = new GridData(GridData.FILL_HORIZONTAL);
        gridData.widthHint = 250;
        folderText.setLayoutData(gridData);

        Button button = new Button(composite, SWT.PUSH);
        button.setText("Browse");
        button.setLayoutData(WindowUtil.makeButtonData(button));
        button.addSelectionListener(new SelectionAdapter() {
            public void widgetSelected(SelectionEvent event) {
                handleBrowseButtonPressed();
            }
        });
        Label fileNameLabel = new Label(composite,SWT.NONE);
        fileNameLabel.setText("Image file name");
        fileNameText = new Text(composite, SWT.BORDER);
        fileNameText.setText(fileName + "." + imageFormat.getName().toLowerCase());
        fileNameText.addModifyListener(new ModifyListener() {
            public void modifyText(ModifyEvent e) {
                validateFileNameText();
            }
        });
        gridData =
            new GridData(
                GridData.HORIZONTAL_ALIGN_FILL | GridData.GRAB_HORIZONTAL);
        gridData.widthHint = 250;
        fileNameText.setLayoutData(gridData);
    }

    /* (non-Javadoc)
     * @see org.eclipse.stp.bpmn.export.IExportImagePage#handleEvent(org.eclipse.swt.widgets.Event)
     */
    public void handleEvent(Event event) {
    }
    
    
    @SuppressWarnings("unchecked")
    @Override
    protected List getTypesToExport() {
        List types = new LinkedList();
        types.add("bpmn_diagram");
        return types;
    }
    
    /* (non-Javadoc)
     * @see org.eclipse.stp.bpmn.export.IExportImagePage#getSelectedResources()
     */
    @SuppressWarnings("unchecked")
    @Override
    public List getSelectedResources() {
        return super.getSelectedResources();
    }
    /**
     * The options group contains the image fomat combo box.
     */
    @Override
    protected void createOptionsGroupButtons(Group optionsGroup) {
        Composite composite = createComposite(optionsGroup, 2);
        Label label = new Label(composite,SWT.NONE);
        label.setText("Image format");
        imageFormatCombo = new Combo(composite, SWT.DROP_DOWN | SWT.READ_ONLY);
        imageFormatCombo.setItems(getImageFormatItems());
        imageFormatCombo.setText(imageFormat.getName());
        imageFormatCombo.addSelectionListener(new SelectionAdapter() {
            public void widgetSelected(SelectionEvent e) {
                imageFormat =
                    ImageFileFormat.resolveImageFormat(
                        imageFormatCombo.getSelectionIndex());
                
                // update filename to reflect new format
                fileNameText.setText(fileName + "." + imageFormat.getName().toLowerCase());
                validateFileNameText();
            }
        });
        GridData gridData =
            new GridData(
                GridData.HORIZONTAL_ALIGN_FILL | GridData.FILL_HORIZONTAL);
        gridData.widthHint = 250;
        imageFormatCombo.setLayoutData(gridData);
    }
    
    /**
     * get the supported image formats from the enumerated type.
     * @return array of supported image formats.
     */
    private String[] getImageFormatItems() {
        String[] items = new String[ImageFileFormat.VALUES.length];
        for (int i = 0; i < ImageFileFormat.VALUES.length; i++) {
            items[i] = ImageFileFormat.VALUES[i].getName();
        }
        return items;
    }
    /**
     * handle a browse button pressed selection.
     */
    private void handleBrowseButtonPressed() {
        DirectoryDialog dialog =
            new DirectoryDialog(Display.getCurrent().getActiveShell());
        dialog.setMessage("Export image");
        dialog.setText("Choose a folder to hold the exported image");

        String dirName = folderText.getText();
        if (!dirName.equals("")) {
            File path = new File(dirName);
            if (path.exists())
                dialog.setFilterPath(new Path(dirName).toOSString());
        }

        String selectedDirectory = dialog.open();
        if (selectedDirectory != null) {
            folderText.setText(selectedDirectory);
        }
    }
    
    /**
     * validate the folder text field.
     */
    private void validateFolderText() {

        if (folderText.getText().equals("")) {
            setMessage(null);
            folderValid = false;
            return;
        }

        IPath path = new Path("");
        if (!path.isValidPath(folderText.getText())) {
            setErrorMessage("The folder is invalid");
            folderValid = false;
            return;
        }

        File file = new File(folderText.getText());
        if (!file.exists()) {
            setErrorMessage("The folder does not exist.");
            folderValid = false;
            return;
        }

        folderValid = true;
        folder = folderText.getText();
        if (fileNameValid) {
            setMessage(null);
        } else {
            validateFileNameText();
        }
    }
    
    /**
     * validate the file name text field.
     */
    private void validateFileNameText() {
        IStatus nameStatus =
            ResourcesPlugin.getWorkspace().validateName(
                fileNameText.getText(),
                IResource.FILE);

        if (!nameStatus.isOK()) {
            setErrorMessage(nameStatus.getMessage());
            fileNameValid = false;
            return;
        }

        fileNameValid = true;
        
        IPath filePath = (new Path(fileNameText.getText())).removeFileExtension();
        fileName = filePath.toString();
        if (folderValid) {
            setMessage(null);
        } else {
            validateFolderText();
        }
    }
    
    /**
     * Make sure that there is only one resource selected,
     * with a bpmn_diagram extension.
     */
    @Override
    protected boolean validateSourceGroup() {
        if (getSelectedResources().isEmpty()) {
            setErrorMessage("You need to select one .bpmn_diagram file");
            return false;
        }
        for (Object res : getSelectedResources()) {
            String ext = ((IResource) res).getFileExtension();
            if (!"bpmn_diagram".equals(ext)) {
                setErrorMessage("Please select a .bpmn_diagram file.");
                return false;
            }
        }
        
        if (getSelectedResources().size() > 1) {
            setErrorMessage("Cannot select more than " +
                    "one diagram to export at a time.");
            return false;
        }
        setMessage(null);
        return true;
    }
    /* (non-Javadoc)
     * @see org.eclipse.stp.bpmn.export.IExportImagePage#getDestinationPath()
     */
    public IPath getDestinationPath() {
        IPath path = new Path(folder);
        String fnWithExtension = fileName +  "." + 
            imageFormat.getName().toLowerCase();
        
        return path.append(fnWithExtension);
    }

    /* (non-Javadoc)
     * @see org.eclipse.stp.bpmn.export.IExportImagePage#getImageFormat()
     */
    public ImageFileFormat getImageFormat() {
        return imageFormat;
    }
    
}