/**********************************************************************
 * Copyright (c) 2005 Scapa Technologies Limited and others
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * Scapa Technologies Limited - Initial API and implementation
 **********************************************************************/

package org.eclipse.stp.b2j.core.jengine.internal.miniengine;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.lang.reflect.Field;
import java.net.URL;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;

import org.eclipse.stp.b2j.core.jengine.internal.api.EngineFactory;
import org.eclipse.stp.b2j.core.jengine.internal.api.Program;
import org.eclipse.stp.b2j.core.jengine.internal.core.PrintHandler;
import org.eclipse.stp.b2j.core.jengine.internal.core.Runner;
import org.eclipse.stp.b2j.core.jengine.internal.core.SubControllerInterface;
import org.eclipse.stp.b2j.core.jengine.internal.core.api.ControllerInterface;
import org.eclipse.stp.b2j.core.jengine.internal.core.api.DaemonInterface;
import org.eclipse.stp.b2j.core.jengine.internal.core.api.TraceListener;
import org.eclipse.stp.b2j.core.jengine.internal.core.datapool.SharedHashMap;
import org.eclipse.stp.b2j.core.jengine.internal.core.datapool.SharedVariable;
import org.eclipse.stp.b2j.core.jengine.internal.core.sync.SharedBarrier;
import org.eclipse.stp.b2j.core.jengine.internal.core.sync.SharedMutex;
import org.eclipse.stp.b2j.core.jengine.internal.core.sync.SharedSemaphore;
import org.eclipse.stp.b2j.core.jengine.internal.message.Message;
import org.eclipse.stp.b2j.core.jengine.internal.mutex.MultiQueuedBlockingMap;
import org.eclipse.stp.b2j.core.jengine.internal.mutex.ObjectBuffer;
//import org.eclipse.stp.b2j.core.jengine.internal.mutex.QueuedSemaphore;
import org.eclipse.stp.b2j.core.jengine.internal.mutex.UnqueuedSemaphore;
import org.eclipse.stp.b2j.core.jengine.internal.utils.ID;
import org.eclipse.stp.b2j.core.jengine.internal.utils.Logger;
import org.eclipse.stp.b2j.core.jengine.internal.utils.StreamUtils;
import org.eclipse.stp.b2j.core.publicapi.JARDependency;

import java.util.LinkedList;

/**
 * 
 * @author amiguel
 * 
 * The local engine main Controller implementation, see the plugin architecture docs for more info.
 */
public class MiniController implements ControllerInterface, SubControllerInterface {

TraceListener tracelistener;
Object tracelisteners_LOCK = new Object();
ArrayList tracelisteners = new ArrayList();
long clock_offset = 0;

Program program;
Class program_class;

Object IDS_LOCK = new Object();
int IDS = 0;
	private int getNextId() {
		int tmp;
		synchronized(IDS_LOCK) {
			tmp = IDS++;
		}
		return tmp;
	}

int controller_id = 0;
	
Object variables_LOCK = new Object();
HashMap variables = new HashMap();
HashMap variable_values = new HashMap();
ArrayList variable_names = new ArrayList();

Object maps_LOCK = new Object();
HashMap maps = new HashMap();
HashMap maps_shared = new HashMap();

Object semaphores_LOCK = new Object();
HashMap semaphores = new HashMap();
HashMap semaphore_values = new HashMap();

Object mutex_LOCK = new Object();
HashMap mutexes = new HashMap();

Object barrier_LOCK = new Object();
HashMap barriers = new HashMap();

Object runners_LOCK = new Object();
//ArrayList runners = new ArrayList();
private int RUNNERS_ID = 1;
LinkedList activeRunners = new LinkedList();
RunnerThreadGroup runner_group = new RunnerThreadGroup();

Object localmaps_LOCK = new Object();
HashMap localmaps = new HashMap();

MultiQueuedBlockingMap conversations = new MultiQueuedBlockingMap();

ObjectBuffer trace_buffer = new ObjectBuffer();
ObjectBuffer print_buffer = new ObjectBuffer();
ObjectBuffer debug_buffer = new ObjectBuffer();

TraceThread tracethread;
PrintThread printthread;
DebugThread debugthread;


	public MiniController(TraceListener tl, JARDependency[] deps) {
		this.tracelistener = tl;
		//TODO we are currently ignoring JAR dependencies in here because the engine will not use any of the engine extension points

		tracethread = new TraceThread();
		printthread = new PrintThread();
		debugthread = new DebugThread();
		
		tracethread.start();
		printthread.start();
		debugthread.start();
	}
	
	public void addTraceListener(TraceListener listener) {
		synchronized(tracelisteners_LOCK) {
			tracelisteners.add(listener);
		}
	}
	public void removeTraceListener(TraceListener listener) {
		synchronized(tracelisteners_LOCK) {
			tracelisteners.remove(listener);
		}
	}
	
	public void setHeadless(boolean headless) {
	}
	
	public void setLogLevel(boolean error, boolean warning, boolean info) throws Exception {
		Logger.PRINT_ERROR = error;
		Logger.PRINT_WARNING = warning;
		Logger.PRINT_INFO = info;
	}

	public long setProgram(Program p) throws Exception {
		JARDependency[] deps = p.getDependencies();
		
		clock_offset = System.currentTimeMillis();
		
		URL[] dep_urls = new URL[deps.length];
		for (int i = 0; i < deps.length; i++) {
			dep_urls[i] = new File(deps[i].getFilePath()).toURL();
		}
		
		program = p;
		program_class = program.getProgramClass(dep_urls,MiniController.class.getClassLoader());
		
		return clock_offset;
	}
//	public Message launchRunner(int count, String method, String host) throws Exception;
//	public Message launchRunner(int count, String method, String host, List args) throws Exception;

//	public void joinRunner(Long remote_id) throws Exception;
	public void closeConnection() {
		//TODO implement
	}

	public void terminate() throws Exception {
		//TODO implement
		
		debug("Aborting...\n");
		
		synchronized(runners_LOCK) {
			for (int i = 0; i < activeRunners.size(); i++) {
				Runner runner = (Runner)activeRunners.get(i);
				try {
					runner.stop();
				} catch (Throwable t) {
				}
				
				//TODO proper way to do this would be to have runners check often if they should stop
				//     and then interrupt them every so often until they returned
			}
		}
	}

	public long getClock() {
		return System.currentTimeMillis() - clock_offset;
	}
	public String getClientHost() throws Exception {
		return "localhost";
	}
	public String getHost() throws Exception {
		return "localhost";
	}
	public String[] getHosts() throws Exception {
		return new String[]{"localhost"};
	}

	public String[] getVariableNames() throws Exception {
		synchronized(variables_LOCK) {
			String[] varnames = new String[variable_names.size()];
			variable_names.toArray(varnames);
			return varnames;
		}
	}

	public HashMap getEngineLocalStorageMap(String name) {
		synchronized(localmaps_LOCK) {
			HashMap map = (HashMap)localmaps.get(name);
			if (map == null) {
				map = new HashMap();
				localmaps.put(name,map);
			}
			return map;
		}
	}

	PrintHandler ph;
	public void setPrintHandler(PrintHandler ph) throws Exception {
		this.ph = ph;
	}
	
	public void trace(Message m) throws Exception {
//		tracelistener.trace((Message)m.clone());
		trace_buffer.add(m);
	}
	public void print(String msg) throws Exception {
//System.out.println(msg);
		if (ph != null) {
			print_buffer.add(msg);
		} else {
//		tracelistener.print(msg);
			ph.print(msg);
		}
	}
	public void debug(String msg) throws Exception {
//System.out.println(msg);
		if (ph != null) {
			ph.debug(msg);
		} else {
			debug_buffer.add(msg);
		}
//		tracelistener.debug(msg);
	}
	
	public void notifyRunnerDeath(long id, Runner runner) throws Exception {
		
		synchronized(runners_LOCK) {
			activeRunners.remove(runner);
		}
		
		//Doesn't need to do anything
		
		//note that the strictly proper way to do this is to do a wait on something in joinRunner and then
		//signal that something in notifyRunnerDeath.  But this way is much more simple and fast
	}
	public void joinRunner(Long rid) throws Exception {
		long tmp = rid.longValue();
		
//		int index = ID.RUNNER_INDEX(rid.longValue());
		Runner runner = null;
		synchronized(runners_LOCK) {
			//search for the runner to wait for
			for (int i = 0; i < activeRunners.size(); i++) {
				Runner tmprunner = (Runner)activeRunners.get(i);
				if (tmp == tmprunner.getRunnerId()) {
					runner = tmprunner;
					break;
				}
			}
		}
		if (runner == null) {
			return;
		}
		runner.join();
	}

	public void sendMessage(String conversation, Message m) throws Exception {
		conversations.put(conversation,(Message)m.clone());
	}

	public Message sendAndReceiveMessage(String conversation, Message msg, String conversationReturn) throws Exception {
		conversations.put(conversation,(Message)msg.clone());
		Message m = (Message)conversations.get(new String[]{conversationReturn},0);
		m.pop(); //remove the conversation from the end
		return m;
	}
	public Message sendAndReceiveMessage(String conversation, Message msg, Message conversationReturns) throws Exception {
		conversations.put(conversation,(Message)msg.clone());
		
		String[] rets = new String[conversationReturns.length()];
		for (int i = 0; i < rets.length; i++) {
			rets[i] = (String)conversationReturns.get(i);
		}
		
		return (Message)conversations.get(rets,0);
	}
	
	public Message receiveMessage(String conversation) throws Exception {
		Message m = (Message)conversations.get(new String[]{conversation},0);
		m.pop(); //remove the conversation from the end
		return m;
	}
	public Message receiveMessage(String conversation, long timeoutMS) throws Exception {
		try {
			Message m = (Message)conversations.get(new String[]{conversation},timeoutMS);
			m.pop(); //remove the conversation from the end
			return m;
		} catch (InterruptedException e) {
			return null;
		}
	}
	public Message receiveMessage(Message conversationReturns) throws Exception {

		String[] rets = new String[conversationReturns.length()];
		for (int i = 0; i < rets.length; i++) {
			rets[i] = (String)conversationReturns.get(i);
		}

		return (Message)conversations.get(rets,0);
	}
	public Message receiveMessage(Message conversationReturns, long timeoutMS) throws Exception {
		
		String[] rets = new String[conversationReturns.length()];
		for (int i = 0; i < rets.length; i++) {
			rets[i] = (String)conversationReturns.get(i);
		}

		try {
			return (Message)conversations.get(rets,timeoutMS);
		} catch (InterruptedException e) {
			return null;
		}
	}

	public Message launchRunner(int count, String method, int host_index) throws Exception {
		return (Message)launchRunner(count,method,new Message())[0];
	}
	public Message launchRunner(int count, String method, int host_index, List args) throws Exception {
		Message margs = new Message();
		for (int i = 0; i < args.size(); i++) {
			margs.append((String)args.get(i));
		}
		return (Message)launchRunner(count,method,margs)[0];
	}
	public Message launchRunner(int count, String method, int host_index, Message args) throws Exception {
		return (Message)launchRunner(count,method,(Message)args.clone())[0];
	}
	public Message launchRunner(int count, String method, int host_index, String[] args) throws Exception {
		Message margs = new Message();
		for (int i = 0; i < args.length; i++) {
			margs.append((String)args[i]);
		}
		return (Message)launchRunner(count,method,margs)[0];
	}
	public ArrayList launchRunnerLocal(int count, String method, ArrayList args) throws Exception {
		Message margs = new Message();
		for (int i = 0; i < args.size(); i++) {
			margs.append((String)args.get(i));
		}
		return (ArrayList)launchRunner(count,method,margs)[1];
	}

	private Object[] launchRunner(int count, String method, Message args) throws Exception {
		Message launched = new Message();
		ArrayList rthreads = new ArrayList();
		
		for (int i = 0; i < count; i++) {

			synchronized(runners_LOCK) {
				
				long rid = ID.CREATE_ID(controller_id,RUNNERS_ID++);
				String rid_hex = ID.ID_HEX(rid);
	
				Object program_object = program_class.newInstance();

				Field runner_id = program_class.getDeclaredField("runner_id");
				runner_id.setAccessible(true); //just in case
				runner_id.setLong(program_object,rid);
	
				Field runner_args = program_class.getDeclaredField("runner_args");
				runner_args.setAccessible(true); //just in case
				runner_args.set(program_object,args.clone());
	
	//			Field runner_id_hex = program_class.getDeclaredField("runner_id_hex");
	//			runner_id_hex.setAccessible(true); //just in case
	//			runner_id_hex.set(program_object,rid_hex);
	
				Runner r = new Runner(runner_group,"JEngine (mini) Runner Thread (ID: "+ID.ID_HEX(rid)+")",this,rid,program_class,program_object,method);
				activeRunners.add(r);
	/*			
				synchronized(wait_directory_LOCK) {
					RunnerLock rlock = new RunnerLock();
					wait_directory.put(new Long(rid),rlock);
				}
	*/			
				r.start();
				
				//add this runners ID to the launched list
				launched.append(rid);
				rthreads.add(r);
			
			}
		}
		return new Object[]{launched,rthreads};
	}
	
	
	public SharedSemaphore newSemaphore(String name, int initial) throws Exception {
		//atomic get or create
		synchronized(semaphores_LOCK) {
			SharedSemaphore s = getSemaphore(name);
			if (s == null) {
				//new semaphore
				int sid = getNextId();
				
				s = new SharedSemaphore(this,name,sid);

				semaphores.put(new Integer(sid),s);
				semaphores.put(name,s);
				semaphore_values.put(new Integer(sid),new UnqueuedSemaphore(initial));
				
				s.semSignal(initial);
			}
			return s;
		}
	}
	public SharedBarrier newBarrier(String name, int size) throws Exception {
		synchronized(barrier_LOCK) {
			SharedBarrier b = getBarrier(name);
			if (b == null) {
				//new barrier
				b = new SharedBarrier(newSemaphore(name+"%mini_BARRIER_WAITING",0),newSemaphore(name+"%mini_BARRIER_BARRIER",0),size);
				
				barriers.put(name,b);
			}
			return b;
		}
	}
	public SharedMutex newMutex(String name) throws Exception {
		synchronized(mutex_LOCK) {
			SharedMutex m = getMutex(name);
			if (m == null) {
				//new mutex
				m = new SharedMutex(newSemaphore(name+"%mini_MUTEX",1));
				
				mutexes.put(name,m);
			}
			return m;
		}
	}

	public SharedSemaphore getSemaphore(String name) throws Exception {
		synchronized(semaphores_LOCK) {
			return (SharedSemaphore)semaphores.get(name);
		}
	}
	public SharedBarrier getBarrier(String name) throws Exception {
		synchronized(barrier_LOCK) {
			return (SharedBarrier)barriers.get(name);
		}
	}
	public SharedMutex getMutex(String name) throws Exception {
		synchronized(mutex_LOCK) {
			return (SharedMutex)mutexes.get(name);
		}
	}
	
	public SharedVariable newVariable(String name, int type, boolean dirty) throws Exception {
		synchronized(variables_LOCK) {
			SharedVariable v = getVariable(name);
			if (v == null) {
				//new variable
				int vid = getNextId();
				
				v = new SharedVariable(this,name,vid,type,null);
				
				variables.put(new Integer(vid),v);
				variables.put(name,v);
				variable_names.add(name);
			}
			return v;
		}
	}
	public SharedHashMap newHashMap(String name) throws Exception {
		synchronized(maps_LOCK) {
			SharedHashMap h = getHashMap(name);
			if (h == null) {
				//new hashmap
				int hid = getNextId();
				
				h = new SharedHashMap(this,name,hid);
				
				HashMap underlying = new HashMap();
				
				maps.put(new Integer(hid),underlying);
				maps.put(name,underlying);
				
				maps_shared.put(new Integer(hid),h);
				maps_shared.put(name,h);
			}
			return h;
		}
	}

	public SharedVariable getVariable(String name) throws Exception {
		synchronized(variables_LOCK) {
			return (SharedVariable)variables.get(name);
		}
	}
	public SharedHashMap getHashMap(String name) throws Exception {
		synchronized(maps_LOCK) {
			return (SharedHashMap)maps_shared.get(name);
		}
	}
	
	public void signalSemaphore(int id, int n) throws Exception {
		UnqueuedSemaphore sem;
		synchronized(semaphores_LOCK) {
			sem = (UnqueuedSemaphore)semaphore_values.get(new Integer(id));
		}
		sem.doSignal(n);
	}
	public void waitSemaphore(int id, int n) throws Exception {
		UnqueuedSemaphore sem;
		synchronized(semaphores_LOCK) {
			sem = (UnqueuedSemaphore)semaphore_values.get(new Integer(id));
		}
		sem.doWait(n);
	}

	private HashMap getMap(int id) {
		Integer Id = new Integer(id);
		HashMap map = (HashMap)maps.get(Id);
//		if (map == null) {
//			map = new HashMap();
//			maps.put(Id,map);
//		}
		return map;
	}
	public void hashmapSet(int id, String key, Object value) throws Exception {
		synchronized(maps_LOCK) {
			HashMap map = getMap(id);
			map.put(key,value);
		}
	}
	public String[] hashmapGetKeys(int id) throws Exception {
		synchronized(maps_LOCK) {
			HashMap map = getMap(id);
			ArrayList list = new ArrayList(map.keySet());
			String[] keys = new String[list.size()];
			for (int i = 0; i < keys.length; i++) {
				keys[i] = (String)list.get(i);
			}
			return keys;
		}
	}
	public Object hashmapGet(int id, String key) throws Exception {
		synchronized(maps_LOCK) {
			HashMap map = getMap(id);
			return map.get(key);
		}
	}
	public void hashmapClear(int id) throws Exception {
		synchronized(maps_LOCK) {
			HashMap map = getMap(id);
			map.clear();
		}
	}

	public boolean isVariableDirty(int id) throws Exception {
		return false;
	}
	public void storeVariable(int id, int type, Object newval) throws Exception {
		synchronized(variables_LOCK) {
			variable_values.put(new Integer(id),newval);
		}
	}
	public Object fetchVariable(int id, int type) throws Exception {
		synchronized(variables_LOCK) {
			return variable_values.get(new Integer(id));
		}
	}
	
	
	public void doRunnerStackDump() throws Exception {
		synchronized(runners_LOCK) {
			for (int i = 0; i < activeRunners.size(); i++) {
				Runner runner = (Runner)activeRunners.get(i);

				StringBuffer sb = new StringBuffer("MiniEngine - Runner Stack Dump:\n");
				String tmp = runner.getStack();
				sb.append(tmp);

				print(sb.toString());
			}
		}
	}
	public void doRunnerStackTrace(int msg_id) throws Exception {
		synchronized(runners_LOCK) {
			for (int i = 0; i < activeRunners.size(); i++) {
				Runner runner = (Runner)activeRunners.get(i);
				Message m = runner.getStackContents();
				m.setType(msg_id);
				trace(m);
			}
		}
	}

	class RunnerThreadGroup extends ThreadGroup {
		public RunnerThreadGroup() {
			super("Runner Thread Group");	
		}	
		public void uncaughtException(Thread t, Throwable e) {
			//not sure what to do here  
			e.printStackTrace();
		}
	}
	
	class TraceThread extends Thread {
		public TraceThread() {
			setDaemon(true);
		}
		public void run() {
			try {
				while (true) {
					Message m = (Message)trace_buffer.next();
					tracelistener.trace(m);
					
					synchronized(tracelisteners_LOCK) {
						for (int k = 0; k < tracelisteners.size(); k++) {
							((TraceListener)tracelisteners.get(k)).trace(m);
						}
					}
				}
			} catch (IOException e) {
				e.printStackTrace();
			}
		}
	}
	class PrintThread extends Thread {
		public PrintThread() {
			setDaemon(true);
		}
		public void run() {
			try {
				while (true) {
					String s = (String)print_buffer.next();
					tracelistener.print(s);
					
					synchronized(tracelisteners_LOCK) {
						for (int k = 0; k < tracelisteners.size(); k++) {
							((TraceListener)tracelisteners.get(k)).print(s);
						}
					}
				}
			} catch (IOException e) {
				e.printStackTrace();
			}
		}
	}
	class DebugThread extends Thread {
		public DebugThread() {
			setDaemon(true);
		}
		public void run() {
			try {
				while (true) {
					String s = (String)debug_buffer.next();
					tracelistener.debug(s);

					synchronized(tracelisteners_LOCK) {
						for (int k = 0; k < tracelisteners.size(); k++) {
							((TraceListener)tracelisteners.get(k)).debug(s);
						}
					}
				}
			} catch (IOException e) {
				e.printStackTrace();
			}
		}
	}
	
	//
	// testing
	//
	public static void main(String[] args) {
		try {
			byte[] prog = StreamUtils.readAll(new FileInputStream(args[0]));
			Program program = new Program(prog,new byte[0][0], new JARDependency[0]);
			
			DaemonInterface d = EngineFactory.connectToMiniEngineDaemon();
			ControllerInterface c = d.newEngine("MiniEngine "+System.currentTimeMillis(),new PrintTraceListener(),new JARDependency[0],null);
			
			c.setProgram(program);
			Message m = c.launchRunner(1,"activity1",0);
			
			System.out.println(m);
			
			for (int i = 0; i < 6; i++) {
				Thread.sleep(10000);
				System.out.println(i);
			}
			System.exit(0);
		} catch (Exception x) {
			x.printStackTrace();
		}
	}
	
	static class PrintTraceListener implements TraceListener {
		public void trace(Message m) {
			System.out.println(m);
		}
		public void print(String s) {
			System.out.println(s);
		}
		public void debug(String s) {
			System.out.println(s);
		}
	}

}