/*******************************************************************************
 * Copyright (c) 2006-2007 IONA Technologies.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IONA Technologies - initial API and implementation
 *******************************************************************************/
package org.eclipse.stp.ui.xef.editor;

import java.io.ByteArrayInputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.WeakHashMap;

import javax.xml.namespace.QName;


import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.action.IMenuListener;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.action.Separator;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.window.Window;
import org.eclipse.stp.ui.xef.XefPlugin;
import org.eclipse.stp.ui.xef.help.XefHelpView;
import org.eclipse.stp.ui.xef.schema.SchemaElement;
import org.eclipse.stp.xef.ISchemaProvider;
import org.eclipse.stp.xef.SchemaProviderFilterWrapper;
import org.eclipse.stp.xef.ValidationProblem;
import org.eclipse.stp.xef.XMLInstanceElement;
import org.eclipse.stp.xef.XMLModelFactory;
import org.eclipse.stp.xef.XMLSnippet;
import org.eclipse.stp.xef.XMLUtil;
import org.eclipse.stp.xef.XefConstants;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.swt.widgets.Tree;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IViewPart;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.actions.ActionFactory;
import org.eclipse.ui.forms.DetailsPart;
import org.eclipse.ui.forms.IDetailsPage;
import org.eclipse.ui.forms.IDetailsPageProvider;
import org.eclipse.ui.forms.IManagedForm;
import org.eclipse.ui.forms.MasterDetailsBlock;
import org.eclipse.ui.forms.SectionPart;
import org.eclipse.ui.forms.widgets.FormToolkit;
import org.eclipse.ui.forms.widgets.ScrolledForm;
import org.eclipse.ui.forms.widgets.Section;
import org.jdom.Attribute;
import org.jdom.Comment;
import org.jdom.Document;
import org.jdom.Element;
import org.jdom.Namespace;
import org.jdom.input.SAXBuilder;
import org.jdom.output.Format;
import org.jdom.output.XMLOutputter;

public class XefEditMasterDetailsBlock extends MasterDetailsBlock implements IDetailsPageProvider {
    private static final String INITIAL_DESCRIPTION = "";
    private static final String RIGHT_CLICK_DESCRIPTION_TEXT = INITIAL_DESCRIPTION 
        + "Right-Click element to specify additional information";
    private static final String RIGHT_CLICK_DESCRIPTION = "[" + RIGHT_CLICK_DESCRIPTION_TEXT + "]";
    private static final String VERIFIED_LABEL = "verified";
    private static final String CUSTOMIZED_LABEL = "customized";
    private static final String UNVERIFIABLE = "cannot be verified";
    private static SAXBuilder builder = new SAXBuilder();
    static boolean interactive = true; // for testing       
    
    // These fields are package protected to be able to reach them from testing code
    MenuManager menuManager;
    Button addButton;
    AddButtonSelectionListener addButtonListener;
    Button deleteButton;
    
    // Static popup menu actions
    final ChangePrefixAction changePrefixAction;
    final DeleteElementAction deleteAction;
    Action helpAction;

    Element baseElement;
    List<XMLInstanceElement> policies = new LinkedList<XMLInstanceElement>();
    static WeakHashMap<XMLInstanceElement, String> policyLabels = new WeakHashMap<XMLInstanceElement, String>();
    TreeViewer viewer;  
    boolean asyncUpdate = true; // can be set to false for testing purposes

    private final XefEditPage editPage;
    private String original;    
    private Section section;
    private boolean dirty;

    public XefEditMasterDetailsBlock(XefEditPage page) {
        editPage = page;
        changePrefixAction = new ChangePrefixAction(this);
        deleteAction = new DeleteElementAction(this, editPage);        
    }

    @Override
    protected void createMasterPart(final IManagedForm managedForm, final Composite parent) {
        FormToolkit toolkit = managedForm.getToolkit();
        section = toolkit.createSection(parent, Section.DESCRIPTION | Section.TITLE_BAR);
        section.setText("Applied Policies");
        section.setDescription(INITIAL_DESCRIPTION);
        section.marginWidth = 10;
        section.marginHeight = 5;
        
        Composite client = toolkit.createComposite(section, SWT.WRAP);
        GridLayout layout = new GridLayout();
        layout.numColumns = 2;
        layout.marginWidth = 2;
        layout.marginHeight = 2;
        client.setLayout(layout);
        Tree t = toolkit.createTree(client, SWT.SINGLE);        
        GridData gd = new GridData(GridData.FILL_BOTH);
        gd.heightHint = 20;
        gd.widthHint = 100;
        gd.verticalSpan = 2;
        t.setLayoutData(gd);
        toolkit.paintBordersFor(client);
        addButton = toolkit.createButton(client, "Add Policy...", SWT.PUSH);
        gd = new GridData(GridData.VERTICAL_ALIGN_BEGINNING | GridData.HORIZONTAL_ALIGN_FILL);
        addButton.setLayoutData(gd);
        
        deleteButton = toolkit.createButton(client, "Delete", SWT.PUSH);
        gd = new GridData(GridData.VERTICAL_ALIGN_BEGINNING | GridData.HORIZONTAL_ALIGN_FILL);
        deleteButton.setLayoutData(gd);
        
        section.setClient(client);
        final SectionPart spart = new SectionPart(section);
        managedForm.addPart(spart);
        viewer = new TreeViewer(t);
        
        viewer.addSelectionChangedListener(new ISelectionChangedListener() {
            public void selectionChanged(SelectionChangedEvent event) {
		if (!validateCurrentDetailsPage()) {
                    IDetailsPage curPage = getDetailsPart().getCurrentPage();
                    if (curPage instanceof XefDetailsPage) {
                        viewer.setSelection(new StructuredSelection(((XefDetailsPage) curPage).element));
				return;
                    }
		}
		
                managedForm.fireSelectionChanged(spart, event.getSelection());
                ISelection sel = event.getSelection();
                if (sel instanceof IStructuredSelection) {
                    IStructuredSelection ssel = (IStructuredSelection) sel;
                    Object el = ssel.getFirstElement();
                    if (el instanceof XMLInstanceElement) {
                        XMLInstanceElement pi = (XMLInstanceElement) el;
                        updateDescription(pi);
                        setHelpText(pi);                        
                        
                        return;
                    }
                }
            }            
        });

        addButtonListener = new AddButtonSelectionListener(parent);
        addButton.addSelectionListener(addButtonListener);

        deleteButton.addSelectionListener(new SelectionAdapter() {
            @Override
            public void widgetSelected(SelectionEvent e) {
                IStructuredSelection ssel = (IStructuredSelection) viewer.getSelection();
                if (ssel.size() != 1) {
                    MessageDialog dlg = new MessageDialog(parent.getShell(), "Delete Policy", 
                            parent.getShell().getDisplay().getSystemImage(SWT.ICON_INFORMATION), 
                            "Please select the policy to remove first.", MessageDialog.ERROR, new String [] {"OK"}, 0);
                    dlg.open();
                } else {
                    deleteAction.run();
                }
            }            
        });
        
        viewer.setContentProvider(new MasterContentProvider());
        viewer.setLabelProvider(new MasterLabelProvider());
        viewer.setInput(policies);
        
        menuManager = initContextMenu(viewer);
        Menu menu = menuManager.createContextMenu(viewer.getControl());
        viewer.getControl().setMenu(menu);                
    }
    
	private MenuManager initContextMenu(final TreeViewer treeViewer) {
        MenuManager mgr = new MenuManager();
        mgr.setRemoveAllWhenShown(true);
        mgr.addMenuListener(new IMenuListener() {
            public void menuAboutToShow(IMenuManager mgr) {
                fillContextMenu(treeViewer, mgr);
            }
        });
        return mgr;
    }

    void fillContextMenu(TreeViewer treeViewer, IMenuManager mgr) {
        if (mgr == null) {
            return;            
        }
        
        if (!(viewer.getSelection() instanceof IStructuredSelection)) {
            return;
        }
        Object el = ((IStructuredSelection) viewer.getSelection()).getFirstElement();
        
        if (el instanceof XMLInstanceElement) {
            XMLInstanceElement pi = (XMLInstanceElement) el;
            for (final SchemaElement se : XefEditMasterDetailsBlock.getAllowedChildren(pi)) {
                mgr.add(new AddElementAction("Add " + se.getDisplayName(true), pi, se, treeViewer, this));
            }
            
            mgr.add(new Separator());
            mgr.add(deleteAction);
            mgr.add(new Separator());
            mgr.add(changePrefixAction);
            mgr.add(new Separator());
            mgr.add(helpAction);
        } else if (el instanceof SnippetRepresentingXMLInstanceElements) {
            mgr.add(deleteAction);            
        }
    }
    
    protected void updateDescription(XMLInstanceElement pi) {
        if (getAllowedChildren(pi).size() > 0 ) {
            section.setDescription(RIGHT_CLICK_DESCRIPTION);
            viewer.getTree().setToolTipText(getLabel(pi) + " - " + RIGHT_CLICK_DESCRIPTION_TEXT);
        } else {
            section.setDescription(INITIAL_DESCRIPTION);
            viewer.getTree().setToolTipText(INITIAL_DESCRIPTION);
        }
    }

    public TreeViewer getViewer() {
        return viewer;
    }

    IAction getAction(String actionID) {
        if (actionID.equals(ActionFactory.DELETE.getId())) {
            return deleteAction;
        } else {
            return null;
        }
    }

    @Override
    protected void registerPages(DetailsPart detailsPart) {
        detailsPart.setPageLimit(10);
        detailsPart.setPageProvider(this);
    }

    @Override
    protected void createToolBarActions(IManagedForm managedForm) {
        final ScrolledForm form = managedForm.getForm();
        
        Action haction = new Action("hor", Action.AS_RADIO_BUTTON) {
            @Override
            public void run() {
                sashForm.setOrientation(SWT.HORIZONTAL);
                form.reflow(true);
            }            
        };
        haction.setChecked(true);
        haction.setToolTipText("Switch to horizonal orientation");
        haction.setImageDescriptor(XefPlugin.getDefault().getImageRegistry().getDescriptor(
                XefPlugin.IMG_HORIZONTAL));
        
        Action vaction = new Action("ver", Action.AS_RADIO_BUTTON) {
            @Override
            public void run() {
                sashForm.setOrientation(SWT.VERTICAL);
                form.reflow(true);
            }            
        };
        vaction.setChecked(false);
        vaction.setToolTipText("Switch to vertical orientation");
        vaction.setImageDescriptor(XefPlugin.getDefault().getImageRegistry().getDescriptor(
                XefPlugin.IMG_VERTICAL));
        
        helpAction = new Action("Help", Action.AS_PUSH_BUTTON) {
            @Override
            public void run() {
                try {
                    IWorkbenchPage page = PlatformUI.getWorkbench().getActiveWorkbenchWindow().getActivePage();
                    IViewPart view = page.showView(XefConstants.POLICY_HELP_VIEW);
                    if (view instanceof XefHelpView) {
                        XMLInstanceElement pi = getSelectedPolicyInstanceElement();
                        String docText = pi.getTemplate().getDocumentation();
                        if (docText != null) {
                            ((XefHelpView) view).setHTMLText("<h3>" + getLabel(pi) + "</h3>", docText);
                        }
                    }
                } catch (PartInitException e) {
                    e.printStackTrace();
                }
            }            
        };
        helpAction.setToolTipText("Show Policy Help");
        helpAction.setImageDescriptor(XefPlugin.getDefault().getImageRegistry().getDescriptor(
                XefPlugin.IMG_HELP));
        
        form.getToolBarManager().add(haction);
        form.getToolBarManager().add(vaction);
        form.getToolBarManager().add(helpAction);
    }

    final class AddButtonSelectionListener extends SelectionAdapter {
        SchemaSelectionDialog currentDialog;
        final Composite parent;
        final XMLProviderEditorInput input;

        private AddButtonSelectionListener(Composite p) {
            parent = p;
            
            IEditorInput i = editPage.getEditorInput();
            if (i instanceof XMLProviderEditorInput) {
                input = (XMLProviderEditorInput) i;
            } else {
                input = null;
            }
        }

        
        public void widgetSelected(SelectionEvent e) {
            SchemaProviderFilterWrapper wrapper = new SchemaProviderFilterWrapper(input.getSchemaProvider(), policies);
			widgetSelected(new SchemaSelectionDialog(parent.getShell(), wrapper, wrapper.getShadowProvider()), e);
        }
        
        // Separate method to allow for other dialog implementations for testing purposes.
        void widgetSelected(SchemaSelectionDialog dlg, SelectionEvent e) {
            currentDialog = dlg;
            dlg.setBlockOnOpen(interactive);
            
            if (dlg == null) {
                MessageDialog md = new MessageDialog(parent.getShell(), "Add Policy", 
                        parent.getShell().getDisplay().getSystemImage(SWT.ICON_INFORMATION), 
                        "No policies can be added.", MessageDialog.ERROR, new String [] {"OK"}, 0);
                md.open();                
                return;
            }
            
            if (dlg.open() == Window.OK) {
                XMLInstanceElement [] instances = null;
                Object dialogResult = dlg.getFirstResult();
                if (dialogResult instanceof SchemaElement) { 
                    instances = new XMLInstanceElement [] { handleAddPolicy((SchemaElement) dialogResult) };
                } else if (dialogResult instanceof XMLSnippet) {
                    instances = handleAddSnippet((XMLSnippet) dialogResult);
                }
                viewer.refresh();
                if (instances != null && instances.length > 0) {
                    XMLInstanceElement instance = instances[instances.length - 1];
                    if (dialogResult instanceof XMLSnippet) {  
                        // TODO need to select the wrapper that represents the group, if we're dealing with a snippet
                        viewer.setSelection(new StructuredSelection(
                            instance.getData(SnippetRepresentingXMLInstanceElements.class)), true);
                    } else {
                        viewer.setSelection(new StructuredSelection(instance), true);
                    }
                    policiesChanged(instances);
                }
                
                setDirty(true);
            }
        }

        private XMLInstanceElement handleAddPolicy(SchemaElement policy) {
            XMLInstanceElement pi;
            pi = new XMLInstanceElement(policy);
            pi.addMinimalSubElements();
            policies.add(pi);
            return pi;
        }

        private XMLInstanceElement[] handleAddSnippet(XMLSnippet snippet) {
            try {
                String ns = "http://www.example.com/2007/03/pe";
                List<XMLInstanceElement> elements = new LinkedList<XMLInstanceElement>();
                ISchemaProvider schemaProvider = getSchemaProvider();
                String snippetContent = snippet.getSnippet();
                // wrap in a temp container, because that's what XMLModelFactory wants.
                snippetContent = "<pe:pe xmlns:pe='" + ns + "'>" + snippetContent + "</pe:pe>";
                XMLModelFactory.createFromXML(snippetContent, 
                        "{" + ns + "}pe", elements, true, schemaProvider);
                for (XMLInstanceElement element : elements) {
                    element.setBasedOnSnippet(snippet.getName());
                    policies.add(element);
                }
                return elements.toArray(new XMLInstanceElement[elements.size()]);
            } catch (Exception ex) {
                IStatus status = new Status(Status.ERROR, XefPlugin.ID, 0, "Problem applying selection", ex);
                MessageDialog.openError(parent.getShell(), "There was a problem applying the selection", status.toString()); 
                XefPlugin.getDefault().getLog().log(status);
            }
            return null;
        }
    }

    public Object getPageKey(Object object) {
        return object;
    }

    public IDetailsPage getPage(Object key) {
        if (key instanceof XMLInstanceElement) {
            return new XefDetailsPage((XefEditor) editPage.getEditor(), (XMLInstanceElement) key);
        } else if (key instanceof SnippetRepresentingXMLInstanceElements) {
            return new SnippetDetailsPage();
        } else {
            return null;
        }
    }
    
    @SuppressWarnings("unchecked")
    // Synchronized, because it uses the builder
    public synchronized String getXML() {
        Document document;
        try {
            document = builder.build(new ByteArrayInputStream(original.getBytes()));
        } catch (Exception e) {
            document = new Document();
            document.setRootElement(new Element(baseElement.getName(), baseElement.getNamespace()));
        }
        
        Element root = document.getRootElement();
        Element newBase;
        if (root.getName().equals(baseElement.getName()) && root.getNamespace().equals(baseElement.getNamespace())) {
            newBase = root;
        } else {
            newBase = root.getChild(baseElement.getName(), Namespace.getNamespace(baseElement.getNamespaceURI()));
        }
        
        cloneJDOMAttributes(newBase, baseElement);

        for (XMLInstanceElement policy : policies) {            
            newBase.addContent(policy.getJDOMElement());
        }
        
        return new XMLOutputter(Format.getPrettyFormat()).outputString(document);
    }

    @SuppressWarnings("unchecked")
    private static void cloneJDOMAttributes(Element newBase, Element oldBase) {
        newBase.removeContent();
        List<Attribute> attributes = new ArrayList<Attribute>(newBase.getAttributes());
        for (Attribute attr : attributes) {
            newBase.removeAttribute(attr);
        }
        
        for (Attribute attr : (List<Attribute>) oldBase.getAttributes()) {
            newBase.setAttribute(attr);
        }
    }
    
    // Used for testing
    public DetailsPart getDetailsPart() {
        return detailsPart;
    }

    void setDirty(boolean value) {
        dirty = value;
        editPage.getManagedForm().dirtyStateChanged();

        IDetailsPage details = detailsPart.getCurrentPage();
        if (details instanceof IXefDetailsPage) {
            ((IXefDetailsPage) details).setDirty(value);
        }
    }
    
    boolean isDirty() {        
        return dirty;
    }        
        
    boolean validate() {
	if (!validateCurrentDetailsPage()) {
		return false;
	}
	
        for (XMLInstanceElement policy : policies) {
            ValidationProblem vp = policy.validate();
            if (vp != null) {
                XMLInstanceElement inst = vp.getXMLInstanceElement();
                getViewer().reveal(inst);
                getViewer().setSelection(new StructuredSelection(inst));
                
                if (interactive) {
                    new MessageDialog(getViewer().getControl().getShell(), 
                            "Value required", null,
                            vp.getMessage(),  
                            MessageDialog.INFORMATION, new String[] {"OK"}, 0).open();
                }
                
                IDetailsPage details = detailsPart.getCurrentPage();
                if (details instanceof XefDetailsPage) {
                    ((XefDetailsPage) details).setFocus(vp.getProblemObject());
                }
                return false;
            }
        }
        return true;
    }
    
    private boolean validateCurrentDetailsPage() {
        IDetailsPage curPage = getDetailsPart().getCurrentPage();
        if (curPage instanceof XefDetailsPage) {
            return ((XefDetailsPage) curPage).validate();
        }
        // There is no details page, so there is nothing to complain about
		return true;
	}

    XMLInstanceElement getSelectedPolicyInstanceElement() {
        ISelection sel = viewer.getSelection();
        if (!(sel instanceof IStructuredSelection)) {
            return null;
        }
        
        IStructuredSelection ssel = (IStructuredSelection) sel;
        Object el = ssel.getFirstElement();
        if (!(el instanceof XMLInstanceElement)) {
            return null;
        }        
        return (XMLInstanceElement) el;        
    }
    
    static List<SchemaElement> getAllowedChildren(XMLInstanceElement pi) {
        List<SchemaElement> children = new LinkedList<SchemaElement>();
        for (XMLInstanceElement i : pi.getChildren()) {
            children.add(i.getTemplate());
        }
        
        return pi.getTemplate().getNestedElements(children);        
    }

    static String getLabel(Object element) {
        if (element instanceof XMLInstanceElement) {
            XMLInstanceElement pi = (XMLInstanceElement) element;
            synchronized (policyLabels) {
                String label = policyLabels.get(element);
                if (label == null) {
                    SchemaElement se = pi.getTemplate();
                    if (se != null) {
                        label = se.getDisplayName();
                    }
			if (label == null) {
                        label = element.toString();
			}
                    policyLabels.put(pi, label);
                }
                return label;
            }
        } else {
            return element.toString();
        }
    }

    void initializePolicies(String originalXML, Element root, String rootElementQName, List<XMLInstanceElement> newPolicies) {
        QName qn = QName.valueOf(rootElementQName);

        original = originalXML; 
        String prefix = XMLUtil.inferNamespacePrefix(qn.getNamespaceURI());
        baseElement = root == null ? new Element(qn.getLocalPart(), prefix, qn.getNamespaceURI()) : root;
        policies = newPolicies == null ? new LinkedList<XMLInstanceElement>() : newPolicies;
        
        getViewer().setInput(policies);
        policiesChanged(newPolicies.toArray(new XMLInstanceElement[newPolicies.size()]));
    }

    private XMLInstanceElement findApplicableDocumentationElement(XMLInstanceElement pi) {
        if (pi.getTemplate().getDocumentation() == null) {
            if (pi.getParent() != null) {
                return findApplicableDocumentationElement(pi.getParent());
            }
        }
        return pi;
    }
    
    protected String constructPath(XMLInstanceElement pi) {
        String path = "";
        
        if (pi.getParent() != null) {
            path += constructPath(pi.getParent()) + "/";
        }
        path += pi.toString();
        return path;
    }        
    
    void setHelpText(final XMLInstanceElement pi) {                
        Runnable setHelp = new Runnable() {
            public void run() {
                XMLInstanceElement docEl = findApplicableDocumentationElement(pi);
                String path = constructPath(docEl);
                if (path.equals(docEl.toString())) {
                    path = "";
                } else {
                    path = "(" + path + ")"; 
                }
                
                IWorkbenchPage page = editPage.getSite().getWorkbenchWindow().getActivePage();
                IViewPart view = page.findView(XefConstants.POLICY_HELP_VIEW);
                if (view instanceof XefHelpView) {
                    String docText = docEl.getTemplate().getDocumentation();
                    if (docText == null) {
                        docText = "";
                    }
                    ((XefHelpView) view).setHTMLText(
                            "<font size=\"+1\"><b>" + getLabel(docEl) + "</b></font> <font size=\"-1\">" 
                            + path + "</font><p/>", docText);
                }
            }            
        };
        
        if (asyncUpdate) {
            editPage.getSite().getShell().getDisplay().asyncExec(setHelp);
        } else {
            setHelp.run();
        }
    }

    private ISchemaProvider getSchemaProvider() {
        return ((XMLProviderEditorInput)editPage.getEditorInput()).getSchemaProvider();
    }
    
    private volatile boolean decorationJobScheduled = false;
    @SuppressWarnings("unchecked")
    void policiesChanged(final XMLInstanceElement ... updatedPolicies) {        
        if (decorationJobScheduled) {
            return;
        }
        
        if (updatedPolicies.length == 0) {
            return;
        }
        
        Job j = new Job("Inspecting XML snippet bases") {                                
            @Override
            protected IStatus run(IProgressMonitor monitor) {
                try {
                    decorateSnippets((TreeViewer) viewer, updatedPolicies);
                    return Status.OK_STATUS;
                } finally {
                    decorationJobScheduled = false;
                }
            }
        };
        j.setPriority(Job.DECORATE);
        j.schedule();
        decorationJobScheduled = true;
    }
    
    @SuppressWarnings("unchecked")
    void decorateSnippets(final TreeViewer viewer, XMLInstanceElement ... updatedPolicies) {            
        Map<String, List<XMLInstanceElement>> snippets = new HashMap<String, List<XMLInstanceElement>>(); 
        for (XMLInstanceElement policy : updatedPolicies) {
            String basedOn = policy.getBasedOnSnippet();
            if (basedOn != null) {
                List<XMLInstanceElement> l = snippets.get(basedOn);
                if (l == null) {
                    l = new LinkedList<XMLInstanceElement>();
                    snippets.put(basedOn, l);
                }
                l.add(policy);
            }
        }
        
        XMLOutputter outputter = new XMLOutputter(Format.getCompactFormat());
        for (final Map.Entry<String, List<XMLInstanceElement>> entry : snippets.entrySet()) {
            // parse the snippet & lift out the root elements
            // then for every element, reformat it and check for equality
            String snippet = "<policies>" + getSchemaProvider().getSnippet(entry.getKey()) + "</policies>";
            Document snippetDoc;
            // Synchronized as we use the builder object
            synchronized (this) {
                try {
                    snippetDoc = builder.build(new ByteArrayInputStream(snippet.getBytes()));
                } catch (Exception e) {
                    snippetDoc = null;
                }
            }
            if (snippetDoc != null) {
                for (XMLInstanceElement pi : entry.getValue()) {
                    StringBuilder sb = new StringBuilder(pi.toString());
                    sb.append(" (");
                    
                    Element root = snippetDoc.getRootElement();
                    Element el = pi.getJDOMElement();
                    if (elementsEqual(outputter, el, root.getChild(el.getName(), el.getNamespace()))) {
                        sb.append(VERIFIED_LABEL);
                    } else {
                        sb.append(CUSTOMIZED_LABEL);
                    }
                    sb.append(')');
                    policyLabels.put(pi, sb.toString());                
                }
            } else {
                String decoration = " (" + UNVERIFIABLE + ")";
                for (XMLInstanceElement pi : entry.getValue()) {
                    policyLabels.put(pi, pi.toString() + decoration);
                }
            }
            
            
            editPage.getSite().getShell().getDisplay().syncExec(new Runnable() {
                public void run() {
                    viewer.refresh(true);
                }
            });
        }
    }

    private boolean elementsEqual(XMLOutputter outputter, Element el1, Element el2) {
        if (el1 == null && el2 == null) {
            return true;
        } else if (el1 == null || el2 == null) {
            return false;
        }
        
        removeAllComments(el1);
        removeAllComments(el2);
        
        String s1 = outputter.outputString(el1);
        String s2 = outputter.outputString(el2);
        return s1.equals(s2);
    }

    @SuppressWarnings("unchecked")
    private void removeAllComments(Element el) {
        // Create a copy to avoid ConcurrentModificationException
        for (Object obj : new ArrayList(el.getContent())) {
            if (obj instanceof Element) {
                removeAllComments((Element) obj);
            } else if (obj instanceof Comment) {
                el.removeContent((Comment) obj);
            }
        }
    }    
}
