/**********************************************************************
 * Copyright (c) 2005 Scapa Technologies Limited and others
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * Scapa Technologies Limited - Initial API and implementation
 **********************************************************************/

package org.eclipse.stp.b2j.core.jengine.internal.compiler.xpath;

import java.lang.reflect.Field;
import java.util.ArrayList;

/**
 * 
 * @author amiguel
 *
 * Represents a single part of an XPATH expression
 */
public class XPathNode {

	public static final int TYPE_ROOTS_CONTAINER = -1;
	public static final int TYPE_NODE_REFERENCE = -2;
	public static final int TYPE_ROOT_EXPRESSION = -3;
	
	public static final int TYPE_PREDICATE_START = 0;
	public static final int TYPE_PREDICATE = 1;
	public static final int TYPE_NUMBER = 2;
	public static final int TYPE_LITERAL = 3;
	public static final int TYPE_ADDITIVE = 4;
	public static final int TYPE_MULTIPLICATIVE = 5;
	public static final int TYPE_UNARY = 6;
	public static final int TYPE_RELATIONAL = 7;
	public static final int TYPE_EQUALITY = 8;
	public static final int TYPE_ANDEXPR = 9;
	public static final int TYPE_OREXPR = 10;
	public static final int TYPE_BOOLEAN = 11;
	public static final int TYPE_VARIABLEREF = 12;
	public static final int TYPE_FUNCTION_CALL = 13;
	public static final int TYPE_NAME_TEST = 14;
	public static final int TYPE_AXIS_SPECIFIER = 15;
	public static final int TYPE_NODE_TYPE_TEST = 16;

	
	XPathNode ref;
	
	XPathNode parent;
	int type;
	int nodeLeafCount = 0;
	ArrayList children = new ArrayList();
	ArrayList args = new ArrayList();
	
	String hashcode = "";
	
	public XPathNode(int type) {
		this.type = type;
	}

	public XPathNode getParent() {
		return parent;
	}
	
	public String getArg(int index) {
		return (String)args.get(index);
	}
	
	public int getArgCount() {
		return args.size();
	}
	
	public void addArg(Object arg) {
		args.add(String.valueOf(arg));
	}
	
	public void setArg(int index, String replacement) {
		args.set(index,replacement);
	}

	public void removeChild(XPathNode orig) {
		children.remove(orig);
	}
	
	public void replaceChild(XPathNode orig, XPathNode replacement) {
		int n = children.indexOf(orig);
		children.set(n,replacement);
	}
	
	public void insertChild(XPathNode node) {
		children.add(0,node);
		node.parent = this;
	}

	public void addChild(XPathNode node) {
		children.add(node);
		node.parent = this;
	}
	
	public int getChildCount() {
		return children.size();
	}
	
	public int getType() {
		return type;
	}
	
	public XPathNode getChild(int index) {
		return (XPathNode)children.get(index);
	}
	
	public int getNodeLeafCount() {
		return nodeLeafCount;
	}
	
	public String getHashCode() {
		return hashcode;
	}
	
	public void recalculate() {
		StringBuffer hashcode = new StringBuffer();
		hashcode.append(":[TYP/"+type+"]:");
		
		for (int i = 0; i < args.size(); i++) {
			hashcode.append(":[ARG/"+args.get(i)+"]:");
		}
		
		nodeLeafCount = args.size();
		
		for (int i = 0; i < children.size(); i++) {
			XPathNode child = (XPathNode)children.get(i);
			child.recalculate();
			hashcode.append(child.getHashCode());
			
			nodeLeafCount += (1 + child.getNodeLeafCount());
		}
		
		this.hashcode = String.valueOf(999999-nodeLeafCount)+hashcode.toString();
	}

	public static String typeToString(int typ) {
		try {
			Field[] fields = XPathNode.class.getDeclaredFields();
			for (int i = 0; i < fields.length; i++) {
				if (fields[i].getName().startsWith("TYPE_")) {
					if (fields[i].getInt(null) == typ) {
						return fields[i].getName().substring(5);
					}
				}
			}
		} catch (IllegalAccessException e) {
		}
		return "unknown type";
	}
	
	public String toString() {
		return toString(0);
	}
	
	public String toString(int depth) {
		StringBuffer sb = new StringBuffer();
		for (int i = 0; i < depth; i++) {
			sb.append("  ");
		}
		sb.append(typeToString(type));
		sb.append(" ");
		for (int i = 0; i < args.size(); i++) {
			sb.append(args.get(i));
			sb.append(",");
		}
		sb.setLength(sb.length()-1);
		
		for (int i = 0; i < children.size(); i++) {
			sb.append("\n");
			sb.append(((XPathNode)children.get(i)).toString(depth+1));
		}
		
		if (type == TYPE_NODE_REFERENCE) {
			sb.append("\n");
			sb.append(ref.toString(depth+1));
		}
		
		return sb.toString();
	}
	
	public boolean equals(XPathNode node) {
		if (type != node.type) return false;
		if (args.size() != node.args.size()) return false;
		for (int i = 0; i < args.size(); i++) {
			String mine = getArg(i);
			String theirs = node.getArg(i);
			if (!mine.equals(theirs)) return false;
		}
		if (children.size() != node.children.size()) return false;
		for (int i = 0; i < children.size(); i++) {
			XPathNode mine = getChild(i);
			XPathNode theirs = node.getChild(i);
			if (!mine.equals(theirs)) return false;
		}
		return true;
	}
}