/*******************************************************************************
 * Copyright (c) 2006-2007 IONA Technologies.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IONA Technologies - initial API and implementation
 *******************************************************************************/
package org.eclipse.stp.ui.xef.editor;

import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;


import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.jface.viewers.DoubleClickEvent;
import org.eclipse.jface.viewers.IColorProvider;
import org.eclipse.jface.viewers.IDoubleClickListener;
import org.eclipse.jface.viewers.ILabelProvider;
import org.eclipse.jface.viewers.ILabelProviderListener;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.stp.ui.xef.XefPlugin;
import org.eclipse.stp.ui.xef.schema.SchemaElement;
import org.eclipse.stp.ui.xef.schema.SchemaRegistry;
import org.eclipse.stp.xef.ISchemaProvider;
import org.eclipse.stp.xef.IShadowProvider;
import org.eclipse.stp.xef.XMLSnippet;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.ToolBar;
import org.eclipse.swt.widgets.ToolItem;
import org.eclipse.ui.dialogs.ElementTreeSelectionDialog;
import org.eclipse.ui.dialogs.ISelectionStatusValidator;

public class SchemaSelectionDialog extends ElementTreeSelectionDialog {    
    public static final String OTHER_CATEGORY_NAME = "Uncategorized";
    public static final String SNIPPET_CATEGORY_NAME = "Composite Policies";
	
    Combo selectedProduct;
    Map<String, SelectionCategory> categories = new TreeMap<String, SelectionCategory>(new Comparator<String>() {
        // puts "Other" and "Predefined" at the end of the (otherwise sorted) list
        public int compare(String o1, String o2) {
            if (OTHER_CATEGORY_NAME.equals(o1)) {
                if (SNIPPET_CATEGORY_NAME.equals(o2)) {
                    return 1;
                } else {
                    return OTHER_CATEGORY_NAME.equals(o2) ? 0 : 1;
                }
            } else if (SNIPPET_CATEGORY_NAME.equals(o1)) {
                return SNIPPET_CATEGORY_NAME.equals(o2) ? 0 : 1;
            } else if (OTHER_CATEGORY_NAME.equals(o2)) {
                return SNIPPET_CATEGORY_NAME.equals(o1) ? 1 : -1; 
            }
            return o1.compareTo(o2);
        }        
    });
    ISchemaProvider schemaProvider;
    IShadowProvider [] shadowProviders;
    private boolean initialized = false;
    private boolean snippetsInitialized = false;
    
    public SchemaSelectionDialog(Shell parent, ISchemaProvider sp, IShadowProvider ... shadowed) {
        this(parent, sp, new ContentProviderDelegate(), shadowed);
    }
    
    private SchemaSelectionDialog(Shell parent, ISchemaProvider sp, ContentProviderDelegate cp, IShadowProvider ... shadowed) {
        super(parent, getLabelProvider(), cp);
        cp.setDelegate(getContentProvider());
        setTitle("Add Policy");

        schemaProvider = sp;
        shadowProviders = shadowed;
        setInput(categories);
        
        setMessage("Select a Policy");
        setAllowMultiple(false);
        setDoubleClickSelects(false);
        setEmptyListMessage("No policy found that satisfies the criteria");
        setStatusLineAboveButtons(true);
        
        setValidator(new ISelectionStatusValidator() {
            public IStatus validate(Object[] selection) {
                if (selection.length == 1) {
                    if (selection[0] instanceof SchemaElement) {
                        String doc = ((SchemaElement) selection[0]).getDocShort();
                        if (doc == null) {
                            doc = "";
                        }

                        return new Status(IStatus.INFO, XefPlugin.getDefault().getBundle().getSymbolicName(),
                            0, doc, null);
                    } else if (selection[0] instanceof XMLSnippet) {
                        return Status.OK_STATUS;
                    }
                }
                return new Status(IStatus.ERROR, XefPlugin.getDefault().getBundle().getSymbolicName(),
                    1, "Please select a policy", null);
            }            
        });
    }        

    @Override
    protected Point getInitialSize() {
        Point result = super.getInitialSize();
        result.x = (int) (result.x * 1.7);
        return result;
    }

    @Override
    protected Label createMessageArea(Composite composite) {
        Composite titleComposite = new Composite(composite, SWT.NONE);
        GridLayout layout = new GridLayout();
        layout.numColumns = 2;
        layout.marginBottom = 0;
        layout.marginTop = 0;
        layout.marginLeft = 0;
        layout.marginRight = 0;
        layout.marginHeight = 0;
        layout.marginWidth =0;
        titleComposite.setLayout(layout);
        GridData cgd = new GridData();
        cgd.grabExcessHorizontalSpace = true;
        cgd.horizontalAlignment = SWT.FILL;
        titleComposite.setLayoutData(cgd);
        
        Label message = super.createMessageArea(titleComposite);
        GridData gd = new GridData();
        gd.grabExcessHorizontalSpace = true;
        gd.horizontalAlignment = SWT.BEGINNING;
        message.setLayoutData(gd);
        
        Composite toolbarComposite = new Composite(titleComposite, SWT.NONE);
        GridLayout toolbarGrid = new GridLayout();
        toolbarGrid.numColumns = 1;
        toolbarComposite.setLayout(toolbarGrid);
        gd = new GridData();
        gd.grabExcessHorizontalSpace = false;        
        gd.horizontalAlignment = SWT.END;
        toolbarComposite.setLayoutData(gd);
        
        ToolBar tb = new ToolBar(toolbarComposite, SWT.FLAT);
        tb.setBackground(toolbarComposite.getBackground());
        tb.setForeground(toolbarComposite.getForeground());
        
        ToolItem ti = new ToolItem(tb, SWT.PUSH);
        ti.setToolTipText("Refresh");
        ti.setImage(XefPlugin.getDefault().getImageRegistry().get(XefPlugin.IMG_REFRESH));
        ti.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				refreshContents();
			}		
        });
        
        return message;
    }
    
    

    @Override
    protected TreeViewer createTreeViewer(Composite parent) {
        TreeViewer tv = super.createTreeViewer(parent);
        tv.addDoubleClickListener(new IDoubleClickListener() {
            public void doubleClick(DoubleClickEvent event) {
                treeDoubleClicked(event);
            }            
        });
        return tv;
    }
    
    void treeDoubleClicked(DoubleClickEvent event) {
        ISelection sel = event.getSelection();
        if (sel instanceof IStructuredSelection) {
            Object item = ((IStructuredSelection) sel).getFirstElement();
            if (getOkButton().isEnabled() && !(item instanceof SelectionCategory)) {
                okPressed();
            }
        }
    }

    @Override
    public int open() {
        fillContents();        
        return super.open();
    }

    private void fillContents() {
        if (initialized) {
            return;
        }
        initialized = true;

        // Uncategorized policy templates appear under this category.
        // The category is only added to the view *if* there are policies
        // that are uncategorized, i.e. there is never an *empty* category
        // in the view.
        SelectionCategory otherCategory = null;
        
        SchemaRegistry sr = XefPlugin.getDefault().getSchemaRegistry();
        for (String ns : schemaProvider.listSchemaNamespaces(null)) {            
            List<SchemaElement> elements = null;
            try {
                if (ns == null) {                    
                    continue;
                } else {
                    elements = sr.resolveSchemaFromXML(
                        schemaProvider.getSchema(ns), true, schemaProvider);                
                }
            } catch (Exception e) {
                e.printStackTrace();
            } 

            if (elements == null) {
                continue;
            }
            for (SchemaElement se : elements) { 
                String categoryPath = se.getCategory();
                SelectionCategory dialogCategory = null;
                if (categoryPath != null && categoryPath.length() > 0) {
                    dialogCategory = categories.get(categoryPath);
                    if (dialogCategory == null) {
                        dialogCategory = new SelectionCategory(categoryPath);
                        categories.put(categoryPath, dialogCategory);
                    }
                } else {
                    if (otherCategory == null) {
                        otherCategory = new SelectionCategory(OTHER_CATEGORY_NAME);
                        categories.put(OTHER_CATEGORY_NAME, otherCategory);
                    }
                    dialogCategory = otherCategory;
                }
                dialogCategory.addEntry(se);
            }
        }
        
        for (IShadowProvider shp : shadowProviders) {
		for (String category : shp.getShadowCategories()) {				
			SelectionCategory cat = categories.get(category);
			if (SNIPPET_CATEGORY_NAME.equals(cat)) {
				// handle snippets later, when they are triggered
				continue;
			}

			if (cat == null) {
				cat = new SelectionCategory(category);
				categories.put(category, cat);
			}
			
			for (Object shadowed : shp.getShadowed(category)) {
				if (shadowed instanceof SchemaElement) {
					cat.addEntry(new ShadowEntry(((SchemaElement) shadowed).getDisplayName() + " (" + shp.getReason(shadowed) + ")", false));
				} else if (shadowed instanceof XMLSnippet) {
					cat.addEntry(new ShadowEntry(((XMLSnippet) shadowed).toString(), true));
				}
			}
		}
        }
        
        SelectionCategory snippetCategory = new SelectionCategory(SNIPPET_CATEGORY_NAME);
        categories.put(SNIPPET_CATEGORY_NAME, snippetCategory);
        snippetsInitialized = false;
        // We initialize the snippets lazily as it can be expensive
        // The SchemaProviderFilterWrapper.listSnippets() is a relatively
        // expensive operation.    
        
    }
    
    private void refreshContents() {
	initialized = false;
        snippetsInitialized = false;
        XefPlugin.getDefault().getSchemaRegistry().clear();
	schemaProvider.refresh();
	fillContents();
	setInput(categories);
	getTreeViewer().setInput(categories);
    }


    public void setSchemaProvider(ISchemaProvider sp) {
        schemaProvider = sp;
        initialized = false;
    }
    
    public void setShadowProviders(IShadowProvider ... s) {
	shadowProviders = s;
	initialized = false;
    }

    // Dragged into the package for test access.
    @Override
    protected TreeViewer getTreeViewer() {
        return super.getTreeViewer();
    }

    private static ILabelProvider getLabelProvider() {
        class CatalogLabelProvider implements ILabelProvider, IColorProvider {
            public Image getImage(Object element) {
                if (element instanceof SelectionCategory) {
                    return XefPlugin.getDefault().getImageRegistry().get(XefPlugin.IMG_GROUP);
                } else if (element instanceof XMLSnippet) {
                    return XefPlugin.getDefault().getImageRegistry().get(XefPlugin.IMG_SNIPPET);
                } else if (element instanceof ShadowEntry) {			
                    return ((ShadowEntry) element).isSnippet() 
			?  XefPlugin.getDefault().getImageRegistry().get(XefPlugin.IMG_SNIPPET_GREY)
			: XefPlugin.getDefault().getImageRegistry().get(XefPlugin.IMG_INSTANCE_GREY);                    
                } else {
                    return XefPlugin.getDefault().getImageRegistry().get(XefPlugin.IMG_INSTANCE);                    
                }
            }

            public String getText(Object element) {
			return element.toString();
            }

            public boolean isLabelProperty(Object element, String property) {
                return false;
            }

            public void addListener(ILabelProviderListener listener) {}
            public void dispose() {}
            public void removeListener(ILabelProviderListener listener) {}

			public Color getBackground(Object element) {
				return null;
			}

			public Color getForeground(Object element) {
				if (element instanceof ShadowEntry) {
					return Display.getDefault().getSystemColor(SWT.COLOR_GRAY);
				} else {
					return null;
				}
			}            
        };
        
        return new CatalogLabelProvider();
    }

    private ITreeContentProvider getContentProvider() {
        return new ITreeContentProvider() {

            public Object[] getChildren(Object parentElement) {
                if (parentElement instanceof SelectionCategory) {			
                    SelectionCategory sc = (SelectionCategory) parentElement;
                    
                    // We initialize the snippets lazily as it can be expensive
                    // The SchemaProviderFilterWrapper.listSnippets() is a relatively
                    // expensive operation.
                    if (SNIPPET_CATEGORY_NAME.equals(sc.getName())) {
			if (!snippetsInitialized) {
                            for (String n : schemaProvider.listSnippets(null)) {
				XMLSnippet se = new XMLSnippet(n, schemaProvider.getSnippet(n));
				sc.addEntry(se);
                            }
                            
                            for (IShadowProvider shp : shadowProviders) {
				Object[] shadowed = shp.getShadowed(SNIPPET_CATEGORY_NAME);
				if (shadowed != null) {
					for (Object o : shadowed) {
					if (o != null) {						
						sc.addEntry(new ShadowEntry(
								o.toString() + " (" + shp.getReason(o) + ")", true));
					}						
					}
				}
                            }
                            snippetsInitialized = true;
			}
                    }
                    
					return sc.getEntries().toArray();
                } else {
                    return new Object [] {};
                }
            }

            public Object getParent(Object element) {
                return null;
            }

            public boolean hasChildren(Object element) {
                return element instanceof SelectionCategory;
            }

            public Object[] getElements(Object inputElement) {
                if (inputElement instanceof Map) {
                    return ((Map) inputElement).values().toArray();
                } else {
                    return new Object []{};
                }
            }

            public void dispose() {}
            public void inputChanged(Viewer viewer, Object oldInput, Object newInput) {}            
        };
    }
    
    private static class ContentProviderDelegate implements ITreeContentProvider {
	private ITreeContentProvider delegate;
	
	public void setDelegate(ITreeContentProvider d) {
		delegate = d;
	}

		public void dispose() {
			delegate.dispose();
		}

		public Object[] getChildren(Object parentElement) {
			return delegate.getChildren(parentElement);
		}

		public Object[] getElements(Object inputElement) {
			return delegate.getElements(inputElement);
		}

		public Object getParent(Object element) {
			return delegate.getParent(element);
		}

		public boolean hasChildren(Object element) {
			return delegate.hasChildren(element);
		}

		public void inputChanged(Viewer viewer, Object oldInput, Object newInput) {
			delegate.inputChanged(viewer, oldInput, newInput);
		}
    }
}
