/*******************************************************************************
* Copyright (c) 2006 IONA Technologies PLC
* All rights reserved. This program and the accompanying materials
* are made available under the terms of the Eclipse Public License v1.0
* which accompanies this distribution, and is available at
* http://www.eclipse.org/legal/epl-v10.html
* 
* Contributors:
*     IONA Technologies PLC - initial API and implementation
*******************************************************************************/
package org.eclipse.stp.sc.jaxws.builders;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IResourceStatus;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.jdt.core.ICompilationUnit;
import org.eclipse.jdt.core.IType;
import org.eclipse.stp.common.logging.LoggingProxy;
import org.eclipse.stp.sc.common.builders.ScJavaBuilder;
import org.eclipse.stp.sc.common.internal.model.RuntimeCore;
import org.eclipse.stp.sc.common.utils.JDTUtils;
import org.eclipse.stp.sc.common.utils.ResourceUtils;
import org.eclipse.stp.sc.common.workspace.WorkspaceManager;
import org.eclipse.stp.sc.jaxws.runtimeprovider.IJavaToWsdlGenerator;
import org.eclipse.stp.sc.jaxws.runtimeprovider.RuntimeProviderManager;
import org.eclipse.stp.sc.jaxws.workspace.JaxWsWorkspaceManager;

public class ScJavaToWsdlBuilder extends ScJavaBuilder {

    public static final String BUILDER_ID = "org.eclipse.stp.sc.jaxws.builders.javatowsdlbuilder";
    
    private static final LoggingProxy LOG = LoggingProxy.getlogger(ScJavaToWsdlBuilder.class);
    
    public ScJavaToWsdlBuilder() {
        super();
        LOG.debug("JavaToWSDL builder has been created");
    }
    
    /**
     * call the javatowsdl to build one JWS style java source file
     * @param javaFile, the java file with JWS annotation
     */
    protected void buildOneJavaFile(IFile javaFile) throws CoreException {
        try {
            if (javaFile.getFileExtension() == null
                || !javaFile.getFileExtension().equals("java")) {
                return;
            }
            generate(javaFile.getFullPath(), getProject(), null);
        } catch (Exception e) {
            LOG.debug("generation failure", e);

            Status status = new Status(IStatus.ERROR,
                                       ResourcesPlugin.PI_RESOURCES,
                                       IResourceStatus.BUILD_FAILED,
                                       e.getMessage(),
                                       e);
            throw new CoreException(status);
        }
    }
    
    private void generate(IPath path, IProject project, Object data) throws CoreException {

    	boolean isAutoBuild = WorkspaceManager.getWorkspaceAutoBuild();
		try {
			if (isAutoBuild) {
		    	// disable auto-build first before call generator
				LOG.debug("disable workspace autobuild before generating code");
				WorkspaceManager.setWorkspaceAutoBuild(false);
			}
			String runtimeType = RuntimeCore.getRuntimeType(project);
			IJavaToWsdlGenerator generator = RuntimeProviderManager.getInstance()
					.getJavaToWsdlGenerator(runtimeType);
			if (data != null) {
				generator.setInitializationData(null, null, data);
			}
			generator.run(path, project);
			tagGeneratedWsdl(path, project);
		} finally {
			// enable auto-build again after call generator
			if (isAutoBuild) {
				LOG.debug("re-enable workspace autobuild after generating code");
				WorkspaceManager.setWorkspaceAutoBuild(true);
			}
		}
	}

    private void tagGeneratedWsdl(IPath path, IProject project) {
    	try {
    		if (path == null || project == null) {
    			return;
    		} 
    		
    		IFile javaFile = ResourcesPlugin.getWorkspace().getRoot().getFile(path);
    		if (javaFile == null || !javaFile.exists()) {
    			return;
    		}
    		String javaName = ResourceUtils.getFileNameWithoutExt(path);
    		String wsdlName = javaName + ".wsdl";
    		IContainer wsdlLocation = JaxWsWorkspaceManager.getWSDLGenFolder(path, project);
			ICompilationUnit unit = JDTUtils.getJavaUnitFromFile(javaFile);
			if (unit == null || wsdlLocation == null) {
				return;
			}
			IType type = unit.findPrimaryType();
			if (type != null) {

				IFile wsdlFile = (IFile) wsdlLocation.findMember(wsdlName);
				if (wsdlFile != null && wsdlFile.exists()) {
					//set the class or interface property to the generated wsdl file
					boolean isInterface = type.isInterface();
					boolean isPojo = type.isClass();
					String wsClass = type.getFullyQualifiedName();
					if (isInterface) {
						LOG.debug("add interface property to wsdl:" + wsClass);
						wsdlFile.setPersistentProperty(
								JaxWsWorkspaceManager.INTERFACE_PROPERTY,
								wsClass);
					} else if (isPojo) {
						LOG.debug("add class property to wsdl:" + wsClass);
						wsdlFile.setPersistentProperty(
								JaxWsWorkspaceManager.CLASS_PROPERTY, wsClass);
					}
				}

			}
		} catch (Exception e) {
			LOG.error(e);
		}
    }
    
	@Override
	protected boolean checkAnnotation(IFile javaFile)  {
		try {
			if (javaFile.getPersistentProperty(
					JaxWsWorkspaceManager.WSDL_PROPERTY) != null) {
				//this file is generated from wsdl. pass by
				return false;
			}

			// we only build wsdl from java interface
			ICompilationUnit unit = JDTUtils.getJavaUnitFromFile(javaFile);
			if (unit == null) {
				return false;
			}
			IType type = unit.findPrimaryType();
			if (type != null) {
				return JavaDocumentUtils.hasWebServiceAnnotation(javaFile);
			}
			return false;
		} catch (CoreException e) {
			LOG.error(e);
			return false;
		}
	}
	
	/**
     * clean the generated wsdl. This will delete /project/wsdl dir and all contains under that dir
     * @param monitor, the progress monitor
     * @throws CoreException
     */
    protected void clean(IProgressMonitor monitor) throws CoreException {
		LOG.debug("clean for the project:" + getProject().getName());
		IContainer wsdlFolder = JaxWsWorkspaceManager.getWSDLFolder(getProject());
		if(wsdlFolder instanceof IFolder){
			wsdlFolder.delete(true, monitor);
		}
		else if(wsdlFolder instanceof IProject){
			IResource[] resources = ((IProject)wsdlFolder).members();
			for(int i = 0; i < resources.length; i++){
				if("wsdl".equals(resources[i].getFileExtension())){
					resources[i].delete(true, monitor);
				}
			}
		}
		getProject().refreshLocal(IProject.DEPTH_INFINITE, null);
	}

	protected void removeResourceFile(IResource res) throws CoreException {
		if (!(res instanceof IFile)) {
			return;
		}

		IFile file = (IFile) res;

		if (!file.getFileExtension().equals("java")) {
			return;
		}

		if (checkAnnotation(file)) {
			LOG.debug("WebService java file has been removed:"
					+ file.getFullPath());

			String clsName = JDTUtils.getJavaClassNameFromFile(file);

			if (clsName.lastIndexOf(".") > 0) {
				clsName = clsName.substring(clsName.lastIndexOf("."));
			}

			String wsdlName = clsName + ".wsdl";
			LOG.debug("  need to remove:" + wsdlName);

			IContainer folder = JaxWsWorkspaceManager.getWSDLFolder(getProject());
			IResource wsdlResource = folder.findMember(wsdlName);

			if (wsdlResource.exists()) {
				LOG.debug("delete wsdl resource:" + wsdlResource.getFullPath());
				wsdlResource.delete(IResource.FORCE, null);
			}
		}
	}
    
    
}
