/*******************************************************************************
 * Copyright (c) 2003-2006 Sybase, Inc.
 * 
 * All rights reserved. This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License v1.0 which
 * accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Sybase, Inc. - initial API and implementation
 ******************************************************************************/
package org.eclipse.stp.soas.internal.deploy.ui.dialogs;

import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.dialogs.IMessageProvider;
import org.eclipse.stp.soas.internal.deploy.util.DeployUtilPlugin;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Shell;


/**
 * Base dialog implementation.  This object should be used for displaying
 * IDialogPage's.  This object manages apply, ok, and cancel button selections
 * as well as managing text displayed in the title area (title bar, message and
 * error text).  This frees the implementer from having to implement control
 * logic for these common items.  (This is intented to provide developers with
 * a programming paradigm similar to that used by the Eclipse wizard framework.)
 * 
 * Usage:
 * <pre>
 * 		IDialogPage page = new MyDialogPage();
 * 		Dialog dialog = new Dialog(shell,page);
 * 		dialog.open();
 * </pre>
 * 
 * @author rcernich
 */
public class Dialog
	extends org.eclipse.jface.dialogs.TitleAreaDialog
	implements IDialogPageContainer {

	public static final int APPLY_ID = 100;
	public static final String APPLY_LABEL = DeployUtilPlugin.getDefault().getResourceString("Dialog.LABEL.apply"); //$NON-NLS-1$

	private IDialogPage mPage;
	private String mPageMessage;
	private String mPageDescription;
	private int mPageMessageType = IMessageProvider.NONE;

	private Button mApply;
	
	private boolean mWasModified = false;

	/**
	 * Constructor for BPDialog.
	 * @param parentShell
	 */
	public Dialog(Shell parent,IDialogPage page) {
		super(parent.getShell());
		mPage = page;
		mPage.setContainer(this);
	}
	
	public Dialog(Shell parent,IDialogPage page, int style) {
		this(parent.getShell(), page);
		setShellStyle(getShellStyle() | style);
	}	

	public boolean wasModified() {
		return mWasModified;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.stp.soas.internal.deploy.ui.dialogs.IDialogPageContainer#closeCancel()
	 */
	public void closeCancel() {
		cancelPressed();
	}

	/* (non-Javadoc)
	 * @see org.eclipse.stp.soas.internal.deploy.ui.dialogs.IDialogPageContainer#closeOK()
	 */
	public void closeOK() {
		okPressed();
	}

	protected boolean performOk() {
		boolean success = mPage.performOk();
		if (success) {
			mWasModified = true;
		}
		return success;
	}

	protected boolean performCancel() {
		return mPage.performCancel();
	}

	protected void performApply() {
		if (mPage.performApply()) {
			mWasModified = true;
		}
	}

	protected void buttonPressed(int buttonId) {
		if (APPLY_ID == buttonId) {
			applyPressed();
		}
		else {
			super.buttonPressed(buttonId);
		}
	}

	protected void okPressed() {
		if (performOk()) {
			super.okPressed();
		}
	}

	protected void cancelPressed() {
		if (performCancel()) {
			super.cancelPressed();
		}
	}

	protected void applyPressed() {
		performApply();
	}

	protected void createButtonsForButtonBar(Composite parent) {
		// create OK and Cancel buttons by default
		super.createButtonsForButtonBar(parent);
		if (mPage.needsApplyButton()) {
			mApply = createButton(parent, APPLY_ID, APPLY_LABEL, false);
		}
	}

	public void updateButtons() {
		boolean pageValid = mPage.isValid();

		getButton(IDialogConstants.OK_ID).setEnabled(pageValid);
		getButton(IDialogConstants.CANCEL_ID).setEnabled(true);
		if (mApply != null) {
			mApply.setEnabled(pageValid && mPage.isModified());
		}
	}

	public void updateMessage() {
		if (getShell() == null)
			return;

		mPageMessage = mPage.getMessage();
		if (mPageMessage != null && mPage instanceof IMessageProvider)
			mPageMessageType = ((IMessageProvider) mPage).getMessageType();
		else
			mPageMessageType = IMessageProvider.NONE;

		if (mPageMessage == null)
			setMessage(mPageDescription);
		else
			setMessage(mPageMessage, mPageMessageType);
		setErrorMessage(mPage.getErrorMessage());
	}

	public void updateTitleBar() {
		if (getShell() == null)
			return;

		String s = mPage.getTitle();
		if (s == null)
			s = ""; //$NON-NLS-1$
		setTitle(s);

		setTitleImage(mPage.getImage());
		updateDescriptionMessage();
		updateMessage();
	}

	private void updateDescriptionMessage() {
		mPageDescription = mPage.getDescription();
		if (mPageMessage == null)
			setMessage(mPage.getDescription());
	}

	private void update() {
		// Update the window title
		updateWindowTitle();

		// Update the title bar
		updateTitleBar();

		// Update the buttons
		updateButtons();
	}

	private void updateWindowTitle() {
		if (getShell() == null)
			// Not created yet
			return;

		String title = mPage.getTitle();
		if (title == null)
			title = ""; //$NON-NLS-1$

		getShell().setText(title);
	}

	/**
	 * @see org.eclipse.jface.window.Window#createContents(Composite)
	 */
	protected Control createContents(Composite parent) {
		Control contents = super.createContents(parent);

		mPage.createControl((Composite) getDialogArea());

		update();

		return contents;
	}

	/**
	 * @see org.eclipse.jface.window.Window#close()
	 */
	public boolean close()
	{
		mPage.dispose();

		// Prevent memory leaks.
		mPage.setContainer(null);
		return super.close();
	}

}
