/*******************************************************************************
 * Copyright (c) 2006-2007 IONA Technologies.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IONA Technologies - initial API and implementation
 *******************************************************************************/
package org.eclipse.stp.ui.xef.schema;

import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import javax.xml.XMLConstants;

import org.apache.xerces.dom.DocumentImpl;
import org.apache.xerces.xs.XSAnnotation;
import org.apache.xerces.xs.XSAttributeUse;
import org.apache.xerces.xs.XSComplexTypeDefinition;
import org.apache.xerces.xs.XSConstants;
import org.apache.xerces.xs.XSElementDeclaration;
import org.apache.xerces.xs.XSModelGroup;
import org.apache.xerces.xs.XSObject;
import org.apache.xerces.xs.XSObjectList;
import org.apache.xerces.xs.XSParticle;
import org.apache.xerces.xs.XSTerm;
import org.apache.xerces.xs.XSTypeDefinition;
import org.apache.xerces.xs.XSWildcard;
import org.eclipse.stp.xef.XMLUtil;
import org.eclipse.stp.xef.XefConstants;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;

public class SchemaElement extends AbstractAnnotatedElement implements AnnotatedElement, Comparable {
    final XSElementDeclaration ctx;
    final XSParticle particle;
    final XSModelGroup group;
    private AnyElement anySequenceElement;
    SchemaElement(XSElementDeclaration obj, XSParticle part, XSModelGroup grp) {
        ctx = obj;
        particle = part;
        group = grp;
        
        handleAnnotations(ctx.getAnnotation());
    }
    
    @Override
    public String getDisplayName() {
        return getDisplayName(false);
    }

    public String getDisplayName(boolean isMenu) {
        String dn = super.getDisplayName();

        if (dn == null) {
    		if (!isMenu && ctx.getScope() == 1) {
    			dn = this.toRawString();
    		} else {
                dn = this.getName();
    		}
        }
        return dn;
    }
    
    public String getName() {
        return ctx.getName();
    }   
    
    public String getShortNameSpace() {
        return XMLUtil.inferNamespacePrefix(ctx.getNamespace());
    }
    
    public String getNameSpace() {
        return ctx.getNamespace();
    }

    /** Qualifiers are identifications associated with schema elements to divide these elements
     * into groups. Certain groups only allow a single element from within that group to be present.
     * Multiple qualifiers can be attached to a single element.
     * @return A map with all the qualifiers as keys. The values are booleans that mark whether a certain
     * qualifier allows multiple elements or not.
     */
    public Map<String, Boolean> getQualifiers() {
        return qualifiers;
    }
    
    /** Returns the list of lists of requirements. Every element on the outer list is in itself a list.
     * The outer lists should be interpreted as 'or' statement. The inner lists are 'and' statements. So
     * The following pseudo literal means either soap+http or corba is required for this element to be 
     * applicable:
     *  ((soap, http), (corba))
     * @return The requirement preconditions for this element.
     */
    public List<List<String>> getRequires() {
        return requires;
    }
    
    public List<SchemaAttribute> getAttributes() {
        List<SchemaAttribute> l = new LinkedList<SchemaAttribute>();
        
        XSTypeDefinition td = ctx.getTypeDefinition();
        if (td instanceof XSComplexTypeDefinition) {
            XSComplexTypeDefinition ctd = (XSComplexTypeDefinition) td;
            XSObjectList attrs = ctd.getAttributeUses();
            for (int i = 0; i < attrs.getLength(); i++) {
                XSObject obj = attrs.item(i);
                if (obj instanceof XSAttributeUse) {
                    XSAttributeUse attr = (XSAttributeUse) obj;
                    SchemaAttribute sa = new SchemaAttribute(attr.getAttrDeclaration());
                    sa.setRequired(attr.getRequired());
                    if (attr.getConstraintType() == XSConstants.VC_DEFAULT) {
                        sa.setDefault(attr.getConstraintValue());
                    }
                    if (attr.getConstraintType() == XSConstants.VC_FIXED) {
                        sa.setFixed(attr.getConstraintValue());
                    }
                    
                    handleAttributeAnnotations(attr, sa);
                    l.add(sa);
                }
            }
        }
        return l;
    }

    private void handleAttributeAnnotations(XSAttributeUse attr, SchemaAttribute sa) {
        XSAnnotation annotation = attr.getAttrDeclaration().getAnnotation();
        if (annotation != null) {
            Document domDoc = new DocumentImpl();
            annotation.writeAnnotation(domDoc, XSAnnotation.W3C_DOM_DOCUMENT); // parse the annotation
            NodeList docNodes = domDoc.getElementsByTagNameNS(XMLConstants.W3C_XML_SCHEMA_NS_URI, "documentation");
            sa.setDocumentation(getI18nTextValue(docNodes));            
            
            sa.setCategory(getTextValue(domDoc.getElementsByTagNameNS(XefConstants.XEF_NS, "category")));
            sa.setDisplayName(getTextValue(domDoc.getElementsByTagNameNS(XefConstants.XEF_NS, "displayName")));
            sa.setDocShort(getI18nTextValue(domDoc.getElementsByTagNameNS(XefConstants.XEF_NS, "docShort")));
            sa.setExample(getTextValue(domDoc.getElementsByTagNameNS(XefConstants.XEF_NS, "example")));
            sa.setFilterId(getTextValue(domDoc.getElementsByTagNameNS(XefConstants.XEF_NS, "filter")));
            sa.setFieldEditorId(getTextValue(domDoc.getElementsByTagNameNS(XefConstants.XEF_NS, "fieldEditor")));
            sa.setPattern(getTextValue(domDoc.getElementsByTagNameNS(XefConstants.XEF_NS, "pattern")));            
            sa.setUnits(getTextValue(domDoc.getElementsByTagNameNS(XefConstants.XEF_NS, "units")));
            String widgetType = getTextValue(domDoc.getElementsByTagNameNS(XefConstants.XEF_GUI_NS, "widget"));
            if (widgetType != null) {
                try {                
                    sa.setWidget(WidgetType.valueOf(widgetType.toUpperCase()));
                } catch (Throwable th) {
                    th.printStackTrace();
                }
            }
            NodeList contextElems = domDoc.getElementsByTagNameNS(XefConstants.XEF_GUI_NS, "context");
            if (contextElems.getLength() > 0 && contextElems.item(0) instanceof Element) {
                String contextType = getTextValue(((Element)contextElems.item(0)).getElementsByTagNameNS(XefConstants.XEF_GUI_NS, "values"));
                if (contextType != null) {
                    sa.setContextValueID(contextType);
                }
                String contextXPath = getTextValue(((Element)contextElems.item(0)).getElementsByTagNameNS(XefConstants.XEF_GUI_NS, "dependency"));
                if (contextXPath != null) {
                    sa.setContextXPath(contextXPath);
                }
            }
        }
    }
    
           
    public SchemaAttribute getAttribute(String name) {
        for (SchemaAttribute attr : getAttributes()) {
            if (attr.getName().equals(name)) {
                return attr;
            }
        }
        return null;
    }
    
    public boolean hasText() {
        XSTypeDefinition type = ctx.getTypeDefinition();
        if (type instanceof XSComplexTypeDefinition) {
            return ((XSComplexTypeDefinition) type).getContentType() == XSComplexTypeDefinition.CONTENTTYPE_MIXED;
        }
        return false;
    }
        
    public SchemaElement getNestedElement(String name) {
        for (SchemaElement nested : getNestedElements(new LinkedList<SchemaElement>())) {
            if (nested.getName().equals(name)) {
                return nested;
            }
        }
        
        return null;
    }

    public List<SchemaElement> getNestedElements(List<SchemaElement> present) {
        XSTypeDefinition td = ctx.getTypeDefinition();
        if (td instanceof XSComplexTypeDefinition) {
            XSComplexTypeDefinition ctd = (XSComplexTypeDefinition) td;
            XSParticle part = ctd.getParticle();
            if (part != null) {
                XSTerm groupTerm = part.getTerm();
                if (groupTerm instanceof XSModelGroup) {
                    return getNestedElements((XSModelGroup) groupTerm, present);
                }
            }
        }
        return new ArrayList<SchemaElement>(0);
    }
    
    public List<SchemaElement> getMinimalSisterElementsList(List<SchemaElement> present) {
        List<SchemaElement> ret = new LinkedList<SchemaElement>();
        if (group != null) {
            switch (group.getCompositor()) {
            case XSModelGroup.COMPOSITOR_ALL:
            case XSModelGroup.COMPOSITOR_SEQUENCE:
                List<SchemaElement> sisters = getElements(group);
                for (SchemaElement sister : sisters) {
                    if (!sister.equals(this)) {
                        if (sister.particle != null) {
                            int pCount = sister.particle.getMinOccurs();
                            for (SchemaElement p : present) {
                                if (p.equals(sister)) {
                                    pCount--;
                                }
                            }
                            for (; pCount > 0; pCount--) {
                                ret.add(sister);
                            }
                        }
                    }
                }
            }
        }
        return ret;
    }
    
    public List<SchemaElement> getMinimalSubElementList() {
        XSTypeDefinition td = ctx.getTypeDefinition();
        if (td instanceof XSComplexTypeDefinition) {
            XSComplexTypeDefinition ctd = (XSComplexTypeDefinition) td;
            XSParticle particle = ctd.getParticle();
            if (particle != null) {
                XSTerm groupTerm = particle.getTerm();           
                if (groupTerm instanceof XSModelGroup) {
                    XSModelGroup group = (XSModelGroup) groupTerm;
                    switch (group.getCompositor()) {
                    case XSModelGroup.COMPOSITOR_ALL:
                    case XSModelGroup.COMPOSITOR_SEQUENCE:
                        List<SchemaElement> availableElements = getElements(group);
                        List<SchemaElement> minimalElements = new LinkedList<SchemaElement>();
                        
                        for (SchemaElement available : availableElements) {
                            if (available.particle != null) {
                                for (int i = 0; i < available.particle.getMinOccurs(); i++) {
                                    minimalElements.add(available);
                                }
                            }
                        }
                        return minimalElements;
                    default:
                        return new ArrayList<SchemaElement>(0);
                    }                
                }
            }
        }        
        return new ArrayList<SchemaElement>(0);
    }

    private List<SchemaElement> getNestedElements(XSModelGroup group, List<SchemaElement> present) {
        List<SchemaElement> ret = new ArrayList<SchemaElement>();
        
        List<SchemaElement> availableElements = getElements(group);
        List<XSModelGroup> subGroups = getSubGroups(group);
        
        switch (group.getCompositor()) {
        case XSModelGroup.COMPOSITOR_CHOICE:
            ret.addAll(getChoiceElements(availableElements, subGroups, present));
            break;
        case XSModelGroup.COMPOSITOR_SEQUENCE:
            ret.addAll(getSequenceElements(availableElements, present));
            break;
        case XSModelGroup.COMPOSITOR_ALL:
            ret.addAll(availableElements);
            break;
        default:
            break;
        }
        
        for (XSModelGroup subGroup : subGroups) {
            ret.addAll(getNestedElements(subGroup, present));
        }
        
        return ret;
    }
    
    private List<SchemaElement> getChoiceElements(List<SchemaElement> availableElements,
            List<XSModelGroup> subGroups, List<SchemaElement> presentElements) {
        
        for (XSModelGroup subGroup : subGroups) {
            List<SchemaElement> subElems = getAllElements(subGroup);
            for (SchemaElement present : presentElements) {
                if (subElems.contains(present)) {
                    subGroups.clear();
                    subGroups.add(subGroup);
                    return new ArrayList<SchemaElement>();
                }
            }
        }
        
        for (SchemaElement present : presentElements) {
            if (availableElements.contains(present)) {
                List<SchemaElement> ret = new ArrayList<SchemaElement>();
                if (present.particle != null) {
                    if (present.particle.getMaxOccursUnbounded()
                            || getElementCount(presentElements, present) < present.particle.getMaxOccurs()) {
                        ret.add(present);
                    }
                }
                subGroups.clear();
                return ret;
            }
        }

        return availableElements;
    }

    private List<SchemaElement> getSequenceElements(List<SchemaElement> availableElements, 
            List<SchemaElement> presentElements) {
        List<SchemaElement> l = new ArrayList<SchemaElement>(availableElements.size());
        
        for (SchemaElement available : availableElements) {
            if (available.particle != null) {
                if (available.particle.getMaxOccursUnbounded()
                        || getElementCount(presentElements, available) < available.particle.getMaxOccurs()) {
                    l.add(available);
                }
            }
        }
        return l;
    }
    
    private int getElementCount(List<SchemaElement> list, SchemaElement element) {
        int count = 0;
        
        for (SchemaElement el : list) {
            if (el.equals(element)) {
                count++;
            }
        }
        return count;
    }

    private List<SchemaElement> getElements(XSModelGroup group) {
        List<SchemaElement> l = new LinkedList<SchemaElement>();

        XSObjectList list = group.getParticles();
        for (int i = 0; i < list.getLength(); i++) {
            XSObject item = list.item(i);
            if (item instanceof XSParticle) {
                XSParticle part = (XSParticle) item;
                XSTerm term = part.getTerm();
                if (term instanceof XSElementDeclaration) {
                    l.add(new SchemaElement((XSElementDeclaration) term, part, group));
                }
            }
        }
        return l;
    }

    private List<SchemaElement> getAllElements(XSModelGroup group) {
        List<SchemaElement> l = new LinkedList<SchemaElement>();

        XSObjectList list = group.getParticles();
        for (int i = 0; i < list.getLength(); i++) {
            XSObject item = list.item(i);
            if (item instanceof XSParticle) {
                XSParticle part = (XSParticle) item;
                XSTerm term = part.getTerm();
                if (term instanceof XSElementDeclaration) {
                    l.add(new SchemaElement((XSElementDeclaration) term, part, group));
                } else if (term instanceof XSModelGroup) {
                    l.addAll(getAllElements((XSModelGroup)term));
                }

            }
        }
        return l;
    }

    private List<XSModelGroup> getSubGroups(XSModelGroup group) {
        List<XSModelGroup> l = new LinkedList<XSModelGroup>();

        XSObjectList list = group.getParticles();
        for (int i = 0; i < list.getLength(); i++) {
            XSObject item = list.item(i);
            if (item instanceof XSParticle) {
                XSParticle part = (XSParticle) item;
                XSTerm term = part.getTerm();
                if (term instanceof XSModelGroup) {
                    l.add((XSModelGroup)term);
                }
            }
        }
        return l;
    }
    
    public boolean isSequenceOfAny() {
        if (anySequenceElement != null) {
            return !anySequenceElement.equals(AnyElement.NO_ANY_ELEMENT);
        }
        
        XSTypeDefinition type = ctx.getTypeDefinition();
        if (type instanceof XSComplexTypeDefinition &&
                !XMLConstants.W3C_XML_SCHEMA_NS_URI.equals(type.getNamespace())) {
            // the embedded piece must not have the http://www.w3.org/2001/XMLSchema namespace
            // because that means that all the content is defaulted, thats what the previous
            // namespace check is for.
            // So a sequence of any is only really considered for elements defined as follows:
            // <xs:element name="xxx">
            //   <xs:complexType>
            //      <xs:sequence>
            //         <xs:any/>
            //      </xs:sequence>
            //   </xs:complexType>
            // </xs:element>
            // 
            // The following element is not considered to be a sequence of any, although
            // the Xerces XML Schema parser treats it as such
            // <xs:element name="yyy"/>
            
        	XSParticle part = ((XSComplexTypeDefinition) type).getParticle();
        	if (part != null) {
        		XSTerm term = part.getTerm();
        		if (term instanceof XSModelGroup) {
        			XSObjectList subPartList = ((XSModelGroup) term).getParticles();
        			if (subPartList.getLength() == 1) {
        				XSObject subPart = subPartList.item(0);
        				if (subPart instanceof XSParticle) {
        					XSTerm subTerm = ((XSParticle) subPart).getTerm();
        					if (subTerm instanceof XSWildcard) {
        					    XSWildcard wildCard = (XSWildcard) subTerm; 
        						if (wildCard.getConstraintType() == XSWildcard.NSCONSTRAINT_ANY) {
        						    anySequenceElement = new AnyElement(wildCard);
        						    return true;
        						}
        					}
        				}
        			}
        		}
        	}
        }
    	
        anySequenceElement = AnyElement.NO_ANY_ELEMENT;
        return false;
    }
    
    public AnyElement getAnySequenceElement() {
        if (isSequenceOfAny()) {
            return anySequenceElement;
        } else {
            return null;
        }
    }

    @Override
    public boolean equals(Object obj) {
        if (obj == this) {
            return true;
        }

        if (!(obj instanceof SchemaElement)) {
            return false;
        }
        
        SchemaElement se = (SchemaElement) obj;
        return ctx.equals(se.ctx);
    }

    @Override
    public int hashCode() {
        return ctx.hashCode();
    }

    @Override
    public String toString() {
	return getDisplayName();
    }

    private String toRawString() {
        String shortNS = getShortNameSpace();
        return shortNS == null ? getName() : shortNS + ":" + getName();
    }

    public int compareTo(Object o) {
        if (o instanceof SchemaElement) {
            return getName().compareTo(((SchemaElement) o).getName());
        }
        return 0;
    }
}
