/*******************************************************************************
 * Copyright (c) 2006-2007 IONA Technologies.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IONA Technologies - initial API and implementation
 *******************************************************************************/
package org.eclipse.stp.xef;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.net.URI;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.xpath.XPath;
import javax.xml.xpath.XPathConstants;
import javax.xml.xpath.XPathFactory;

import org.xml.sax.SAXException;

import org.w3c.dom.DOMConfiguration;
import org.w3c.dom.Document;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.w3c.dom.bootstrap.DOMImplementationRegistry;
import org.w3c.dom.ls.DOMImplementationLS;
import org.w3c.dom.ls.LSInput;
import org.w3c.dom.ls.LSOutput;
import org.w3c.dom.ls.LSParser;
import org.w3c.dom.ls.LSSerializer;

import org.eclipse.stp.xef.util.InputStreamHelper;
import org.jdom.Attribute;
import org.jdom.Element;
import org.jdom.input.SAXBuilder;
import org.jdom.output.Format;
import org.jdom.output.XMLOutputter;

public class XMLUtil {
    // DOM static instances
    private static final DocumentBuilderFactory DOCUMENT_BUILDER_FACTORY = DocumentBuilderFactory.newInstance();
    private static final DOMImplementationLS DOM_LS;
    private static final XPathFactory XPATH_FACTORY = XPathFactory.newInstance();
    private static final LSSerializer LS_SERIALIZER;
    private static LSParser LS_PARSER;
    
    // JDOM static instances
    private static final SAXBuilder SAX_BUILDER = new SAXBuilder();
    private static final XMLOutputter XML_OUTPUTTER = new XMLOutputter(Format.getPrettyFormat());
    
    static {
        System.setProperty(DOMImplementationRegistry.PROPERTY, "org.apache.xerces.dom.DOMXSImplementationSourceImpl");
        try {
            DOMImplementationRegistry registry = DOMImplementationRegistry.newInstance();
            DOM_LS = (DOMImplementationLS) registry.getDOMImplementation("LS");
            LS_PARSER = DOM_LS.createLSParser(DOMImplementationLS.MODE_SYNCHRONOUS, null);

            LS_SERIALIZER = DOM_LS.createLSSerializer();
            DOMConfiguration config = LS_SERIALIZER.getDomConfig();
            config.setParameter("format-pretty-print", true);            
        } catch (Exception e) {
            e.printStackTrace();
            throw new RuntimeException(e);
        }
    }
    
    public static String getXMLSnippet(File sourceXML, String xpath) throws IOException {
        // We try to guess the parent XPath
        String parentXPath = getParentXPath(xpath);
        return getXMLSnippet(sourceXML, xpath, parentXPath);
    }

    private static String getParentXPath(String xpath) {
        String parentXPath = null;
        int idx = xpath.lastIndexOf('/');
        if (idx >= 0) {
            parentXPath = xpath.substring(0, idx);
        }
        return parentXPath;
    }    
    
    public synchronized static String getXMLSnippet(File sourceXML, String xpath, String parentXPath) throws IOException {
        try {
            final Document doc = LS_PARSER.parseURI(sourceXML.toURI().toString());
            DocumentNamespaceContext dnc = new DocumentNamespaceContext(doc);
            XPath xp = XPATH_FACTORY.newXPath();
            xp.setNamespaceContext(dnc);
            NodeList nodes = (NodeList) xp.evaluate(xpath, doc, XPathConstants.NODESET);
            Node parent = getParent(nodes);
            
            if (parent == null) {
                XPath xp2 = XPATH_FACTORY.newXPath();
                xp2.setNamespaceContext(dnc);
                parent  = (Node) xp2.evaluate(parentXPath, doc, XPathConstants.NODE);

                if (parent == null) {
                    return null;
                }
            }
                        
            DocumentBuilder builder = DOCUMENT_BUILDER_FACTORY.newDocumentBuilder();
            Document newDoc = builder.newDocument();
            Node newParent = newDoc.adoptNode(parent.cloneNode(false));
            newDoc.appendChild(newParent);
            
            for (int i=0; i < nodes.getLength(); i++) {
                Node node = newDoc.adoptNode(nodes.item(i));
                newParent.appendChild(node);
            }
            
            String result = LS_SERIALIZER.writeToString(newDoc);
            
            // davidb: by default the serializer creates XML documents that are UTF-16
            // and strangely enough the XML parser barfs on that later on with a 'no content
            // allowed in prolog'. Funny enough its happy with UTF-8. For the moment I'm 
            // just stripping of the prolog to avoid the problem.
            // REVISIT!
            result = result.replaceAll("<\\?(.*?)\\?>", "");
            
			return result;
        } catch (Exception e) {
            IOException ioe = new IOException(e.getMessage());
            ioe.initCause(e);
            throw ioe;
        }
    }

    private static Node getParent(NodeList nodes) {
        for (int i=0; i < nodes.getLength(); i++) {
            Node cur = nodes.item(i);
            Node parent = cur.getParentNode();
            if (parent != null) {
                return parent;
            }
        }
        return null;
    }
    
    @SuppressWarnings("serial")
    public static void putXMLSnippet(String snippetXML, File targetXML, String xpath) throws IOException {
        OutputStream os = null;
        try {
            LSInput input = DOM_LS.createLSInput();
            input.setStringData(snippetXML);
            Document snippetDoc = LS_PARSER.parse(input);
            Document targetDoc = LS_PARSER.parseURI(targetXML.toURI().toString());
            XPath xp = XPATH_FACTORY.newXPath();
            DocumentNamespaceContext dnc = new DocumentNamespaceContext(targetDoc);
            xp.setNamespaceContext(dnc);
            NodeList nodes = (NodeList) xp.evaluate(xpath, targetDoc, XPathConstants.NODESET);
            
            if (nodes.getLength() == 0) {
                String parentXPath = getParentXPath(xpath);
                if (parentXPath == null) {
                    return;
                }
                XPath xp2 = XPATH_FACTORY.newXPath();
                xp2.setNamespaceContext(dnc);
                Node parent = (Node) xp2.evaluate(parentXPath, targetDoc, XPathConstants.NODE);
                NodeList newChildren = snippetDoc.getDocumentElement().getChildNodes();
                for (int i=0; i < newChildren.getLength(); i++) {
                    Node node = targetDoc.adoptNode(newChildren.item(i).cloneNode(true));
                    parent.appendChild(node);                                        
                }
            } else {
                NodeList newChildren = snippetDoc.getDocumentElement().getChildNodes();
                Node firstEl = nodes.item(0);
                Node parent = firstEl.getParentNode();
                if (parent == null) {
                    return;
                }
                for (int i=0; i < newChildren.getLength(); i++) {
                    Node node = targetDoc.adoptNode(newChildren.item(i).cloneNode(true));
                    parent.insertBefore(node, firstEl);                    
                }
                for (int i=0; i < nodes.getLength(); i++) {
                    parent.removeChild(nodes.item(i));
                }
            }
                        
            // write to file targetXML;
            LSOutput output = DOM_LS.createLSOutput();
            os = new FileOutputStream(targetXML);
            output.setByteStream(os);
            LS_SERIALIZER.write(targetDoc, output);
        } catch (Exception e) {
            IOException ioe = new IOException(e.getMessage());
            ioe.initCause(e);
            throw ioe;
        } finally {
            InputStreamHelper.close(os);
        }
    }
    
    /** Normalizes and reformats a string containing XML. This can be useful when 
     * comparing two pieces of XML, as the output will be identical if the content 
     * is the same. <p/>
     * 
     * Note: comments are ignored in the comparization.
     * @param xml The XML to normalize.
     * @return The normalized XML.
     * @throws Exception If anything goes wrong, generally if the string passed in is not XML. 
     */
    public static synchronized String normalizeXML(String xml) throws Exception {
        String s2 = stripComment(xml);
        String s3 = stripProlog(s2);

        return XML_OUTPUTTER.outputString(SAX_BUILDER.build(new ByteArrayInputStream(s3.getBytes())));
    }
    
    private static String stripComment(String s) { 
        return s.replaceAll("<!--(.*?)-->", "");
    }
    
    public static String stripProlog(String s) {
        return s.replaceAll("<\\?(.*?)\\?>", "");
    }
    
    /**
     * Canonicalizes the JDom element passed in. This process is carried out in accordance with 
     * the canonicalization specification {@link http://www.w3.org/TR/xml-c14n}.
     * @param el The element to be canonicalized.
     * @return The resulting XML as a string.
     * @throws IOException
     * @throws SAXException
     */
    public static String canonicalize(Element el) throws IOException, SAXException {
        Element cel = (Element) el.clone();
        canonicalizeElement(cel);
        
        String formatted = new XMLOutputter(Format.getPrettyFormat()).outputString(cel);
        
        String f2 = formatted.replaceAll("<\\s*(\\S+)\\s*(\\s[^<>]*?)?\\s*/>", "<$1$2></$1>");
        
        return f2;
    }    

    @SuppressWarnings("unchecked")
    private static void canonicalizeElement(Element el) {
        // Sort the attributes
        // The tree map will do the sorting for us.
        Map<String, Attribute> attrs = new TreeMap<String, Attribute>();
        
        for (Attribute a : new ArrayList<Attribute>(el.getAttributes())) {
            attrs.put(a.getName(), a);
            el.removeAttribute(a);                
        }
        
        for (String attrName : attrs.keySet()) {
            el.setAttribute(attrs.get(attrName));
        }

        for (Element e : (List<Element>) el.getChildren()) {
            canonicalizeElement(e);
        }
    }

    @SuppressWarnings("unchecked")
    /** Merges selections of modified XML back to the original XML
     */ 
    public static String mergeXMLBack(String originalXML, String selection, String newXML) throws Exception {
        org.jdom.Document orgdoc = SAX_BUILDER.build(new ByteArrayInputStream(normalizeXML(originalXML).getBytes()));
        org.jdom.Document seldoc = SAX_BUILDER.build(new ByteArrayInputStream(normalizeXML(selection).getBytes()));
        org.jdom.Document newdoc = SAX_BUILDER.build(new ByteArrayInputStream(newXML.getBytes()));
    
        Element orgRoot = orgdoc.getRootElement();
        List<Element> elements = (List<Element>) orgRoot.getChildren();
        
        Element orgRootCopy = (Element) orgRoot.clone();
        List<Element> originalElements = (List<Element>) orgRootCopy.getChildren();

        List<String> elementXMLs = new ArrayList<String>(elements.size());
        for (Element el : elements) {
            elementXMLs.add(canonicalize(el));
        }
        
        for (Element el : (List<Element>) seldoc.getRootElement().getChildren()) {
            int idx = elementXMLs.indexOf(canonicalize(el));
            if (idx != -1) {
                elements.remove(idx);
                elementXMLs.remove(idx);
            }
        }
        
        List<Element> newElements = (List<Element>) newdoc.getRootElement().cloneContent();
        for (Element el : newElements) {
            int insertIdx = findInsertLocation(originalElements, elements, el, newElements);
            if (insertIdx == -1) {
                orgRoot.addContent(el);
            } else {
                orgRoot.addContent(insertIdx, el);
            }
        }
        return new XMLOutputter(Format.getPrettyFormat()).outputString(orgdoc);        
    }

    private static int findInsertLocation(List<Element> originalElements, List<Element> currentElements, Element element, List<Element> newElements) {
        int orgElIdx = -1;
        for (int i=0; i < originalElements.size(); i++) {
            Element el = originalElements.get(i);
            if (elementNamesEqual(element, el)) {
                orgElIdx = i;
                break;
            }
        }
        
        if (orgElIdx == -1) {
            Element nextEl = findNextNewElement(element, newElements);
            if (nextEl != null) {
                return findInsertBeforeLocation(originalElements, currentElements, nextEl, newElements);
            } else {
                return -1;
            }
        }
        
        int orgNextElIdx = -1;
        for (int i=orgElIdx+1; i < originalElements.size(); i++) {
            Element el = originalElements.get(i);
            if (!elementNamesEqual(element, el)) {
                orgNextElIdx = i;
                break;
            }
        }
        
        if (orgNextElIdx == -1) {
            return -1;
        }
        
        // We found the insert location in the original document: just before orgNextElIdx, now map that to the new contents
        for (int i=orgNextElIdx; i < originalElements.size(); i++) {
            Element orgNextEl = originalElements.get(i);
            for (int j=0; j < currentElements.size(); j++) {
                Element el = currentElements.get(j);
                if (elementNamesEqual(orgNextEl, el)) {
                    // found
                    return el.getParentElement().indexOf(el); // find the exact location in the parent element to insert
                }
            }
        }
        return -1;
    }

    private static Element findNextNewElement(Element element, List<Element> newElements) {
        int sz = newElements.size();
        for (int i=0; i < sz; i++) {
            if (newElements.get(i).equals(element)) {
                while(i < sz && newElements.get(i).equals(element)) {
                    i++;
                }
                if (i < sz) {
                    return newElements.get(i);
                } else {
                    return null;
                }
            }
        }
        return null;
    }

    private static int findInsertBeforeLocation(List<Element> originalElements, List<Element> currentElements, Element element, List<Element> newElements) {
        for (Element el : (List<Element>) originalElements) {
            if (elementNamesEqual(element, el)) {
                Element parent = el.getParentElement();
                if (parent != null) {
                    // The parent could be null if we just added it to the list
                    return parent.indexOf(el); // found it!
                }       
            }
        }
        
        Element nextEl = findNextNewElement(element, newElements);
        if (nextEl != null) {
            return findInsertBeforeLocation(originalElements, currentElements, nextEl, newElements);
        } else {
            return -1;
        }
    }
    
    private static boolean elementNamesEqual(Element el1, Element el2) {
        return el2.getName().equals(el1.getName()) && 
                el2.getNamespace().equals(el1.getNamespace());
    }        
    
    /** Utility to dream up a short namespace prefix for a given namespace URI 
     * @param uri The namespace URI
     * @return A prefix, or null if no prefix can be inferred
     */
    public static String inferNamespacePrefix(String uri) {
        if (uri == null) {
            return null;
        }

        try {
            URI u = new URI(uri);
            String host = u.getHost();
            if (!host.endsWith(".org")) {
                int idx = host.indexOf('.');
                int idx2 = host.indexOf('.', idx + 1);
                return host.substring(idx + 1, idx2);
            } else {
                int idx = uri.lastIndexOf('/');
                return uri.substring(idx + 1);
            }
        } catch (Throwable th) {
        }
        return null;
    }    
}
