/*******************************************************************************
 * Copyright (c) 2006-2007 IONA Technologies.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IONA Technologies - initial API and implementation
 *******************************************************************************/
package org.eclipse.stp.xef;

import java.io.IOException;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.NullProgressMonitor;

/**
 * This IXMLProvider provides a really easy way to create a XEF editor for a piece of XML, possibly part
 * of a larger XML document. The XPathXMLProvider provides the XML that is selected through an XPath expression. <p/>
 * 
 * Once the XML is modified by the editor, the {@link #setXML(String)} call will only replace the piece selected by
 * the the original XPath in the original document. <p/>
 *    
 * This way, the XPathXMLProvider can be used to create editors for specific sections of XML, such as a WSDL extension 
 * by simply providing a schema and an XPath to select the relevant portions. <p/>
 * 
 * Example:
 * <pre>
 * IXMLProvider xpathXMLProvider = 
 *     new XPathXMLProvider(IFile, "//*[local-name()='logging' and namespace-uri()='http://www.example.com/xsd/2006/02/logging']")
 * XMLProviderEditorInput editorInput = new XMLProviderEditorInput("My Editor Title", xpathXMLProvider, null, null);
 * IDE.openEditor(getPage(), editorInput, XefConstants.XEF_EDITOR);
 * </pre>
 */
public class XPathXMLProvider implements IXMLProvider {
    private final IFile file;
    private final String xpath;
    private final String rootQName;

    public XPathXMLProvider(IFile f, String xp, String root) {
        file = f;
        xpath = xp;
        rootQName = root;
    }        

    public String getRootElementQName() {
        return rootQName;
    }

    public String getXML() {
        try {
            return XMLUtil.getXMLSnippet(file.getLocation().toFile(), xpath, null);
        } catch (IOException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
            return null;
        }
    }    
    
    public void setXML(String newXML) {
        try {
            XMLUtil.putXMLSnippet(newXML, file.getLocation().toFile(), xpath);
            file.refreshLocal(IResource.DEPTH_ZERO, new NullProgressMonitor());
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

}
