/*******************************************************************************
* Copyright (c) 2006 IONA Technologies PLC
* All rights reserved. This program and the accompanying materials
* are made available under the terms of the Eclipse Public License v1.0
* which accompanies this distribution, and is available at
* http://www.eclipse.org/legal/epl-v10.html
* 
* Contributors:
*     IONA Technologies PLC - initial API and implementation
*******************************************************************************/
package org.eclipse.stp.sc.jaxwsri.generators;

import java.util.ArrayList;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IPath;
import org.eclipse.jdt.core.ICompilationUnit;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.IType;
import org.eclipse.stp.common.logging.LoggingProxy;
import org.eclipse.stp.sc.common.utils.JDTUtils;
import org.eclipse.stp.sc.common.utils.LaunchUtil;
import org.eclipse.stp.sc.jaxws.preferences.PreferencesAccessor;
import org.eclipse.stp.sc.jaxws.preferences.SCPreferenceConstants;
import org.eclipse.stp.sc.jaxws.runtimeprovider.IJavaToWsdlGenerator;
import org.eclipse.stp.sc.jaxws.workspace.JaxWsWorkspaceManager;
import org.eclipse.stp.sc.jaxwsri.properties.SCJAXWSRIPropertyPage;

public class JAXWSRIJavaToWsdlGenerator implements IJavaToWsdlGenerator
{
    
    private static final LoggingProxy LOG = LoggingProxy.getlogger(JAXWSRIJavaToWsdlGenerator.class);
    
    public void setInitializationData(IConfigurationElement config,
        String propertyName, Object data) throws CoreException {
    }

    public void run(IPath path, IProject project) throws CoreException {

        try {
            IFile javaFile = ResourcesPlugin.getWorkspace().getRoot().getFile(path);
            ICompilationUnit cu = JDTUtils.getJavaUnitFromFile(javaFile);
            IType type = cu.findPrimaryType();
            if(type.isInterface()){
            	//JAXWS-RI can only generate wsdl from a SEI implementation
            	LOG.info("can not generate WSDL from interface " + type.getElementName());
            	return;
            }

            LOG.info("generate WSDL for " + path.toString());

        	IContainer wsdlLocation = JaxWsWorkspaceManager.getWSDLGenFolder(path, project);
            
            LOG.info("WSDL location is: " + wsdlLocation);
            
            IJavaProject javaProject = JDTUtils.findJavaProject(project.getName());

            ArrayList<String> args = new ArrayList<String>();
            
            //resource dir
            args.add(ToolConstants.JAVA2WSDL_PARA_RESOURCE_DIR);
            args.add(wsdlLocation.getLocation().toOSString());
            
            //source dir
            String srcDir = JaxWsWorkspaceManager.getSrcFolder(project).getLocation().toOSString();
            args.add(ToolConstants.JAVA2WSDL_PARA_SOURCE_DIR);
            args.add(srcDir);
            
            //classpath 
            String classPath = JDTUtils.getProjectClasspathAsString(project);
            LOG.debug("classpaths passed to j2w:" + classPath);
            args.add(ToolConstants.JAVA2WSDL_PARA_CLASS_PATH);
            args.add(classPath);

            //soap1.2 flag
            String soap12 = project.getPersistentProperty(
            		SCJAXWSRIPropertyPage.PROPERTY_KEY_SOAP_OPTION);
            LOG.debug("soap12 from project: " + soap12);
            boolean isSOAP12 = false;
            if (soap12 == null) {
            	if (PreferencesAccessor.getSoap12Option()) {
            		soap12 = "anyValue";
            		LOG.debug("soap12 is needed.");
            		isSOAP12 = true;
            	}
            } else if (soap12.equals(SCPreferenceConstants.KEY_SOAP12)) {
                LOG.debug("soap12 is needed by project");
                isSOAP12 = true;
            }
            if(isSOAP12){
            	args.add(ToolConstants.JAVA2WSDL_PARA_EXTENSION);
            	args.add(ToolConstants.JAVA2WSDL_PARA_PARA_SOAP12);
            }else{
            	args.add(ToolConstants.JAVA2WSDL_PARA_PARA_SOAP11);
            }

            //target class
            String clsName = JDTUtils.getJavaClassNameFromFile(javaFile);
            LOG.debug("target cls name:" + clsName);
            if (clsName == null) {
                throw new Exception("failed to get java class name");
            }
            args.add(clsName);
            
            LaunchUtil.launchJavaProgram(javaProject, ToolConstants.JAVA2WSDL_GENERATOR_CLASS, null, args.toArray(new String[0]), null);
            LOG.debug("merge the geneared wsdl and schema file");
            
            project.refreshLocal(IProject.DEPTH_INFINITE, null);
        } catch (Throwable e) {
            LOG.error("JAX-WS RI Java To Wsdl Generation failure", e);
        }
    }
}
