/*******************************************************************************
 * Copyright (c) 2004-2006 Sybase, Inc.
 * 
 * All rights reserved. This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License v1.0 which
 * accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Sybase, Inc. - initial API and implementation
 ******************************************************************************/
package org.eclipse.stp.soas.internal.deploy.ui;

import java.io.IOException;
import java.net.URL;
import java.text.MessageFormat;
import java.util.Locale;
import java.util.MissingResourceException;
import java.util.ResourceBundle;

import org.eclipse.core.runtime.FileLocator;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Status;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.swt.graphics.Image;
import org.eclipse.ui.plugin.AbstractUIPlugin;

/**
 * This class serves as a common abstract class adding logging
 * and resource management functionality.
 * 
 * @author shongxum
 */

public abstract class AbstractUIPluginBase extends AbstractUIPlugin 
{
    public static final int INTERNAL_ERROR = 10001;

    public static final String RESOURCES_FILE = "resources"; //$NON-NLS-1$

    private ResourceBundle resourceBundle = null;

    public AbstractUIPluginBase() {
        super();
    }
    
    /**
     * In the body, add something like: return
     * ResourceBundle.getBundle("....resources")
     * 
     * @return ResourceBundle
     * @throws MissingResourceException
     */
    public ResourceBundle loadResourceBundle() throws MissingResourceException {
    	String resources = getClass().getPackage().getName() + '.'
                + RESOURCES_FILE;
        return ResourceBundle.getBundle(resources, Locale.getDefault(), getClass().getClassLoader());
    }

    private ResourceBundle getResourceBundle() throws MissingResourceException {
        if (resourceBundle == null) resourceBundle = loadResourceBundle();
        return resourceBundle;
    }

    private void setResourceBundle(ResourceBundle resourceBundle) {
        this.resourceBundle = resourceBundle;
    }

	// Get the image
	//
	public ImageDescriptor getImageDescriptor(String relativePath) {
		String iconPath = "icons/"; //$NON-NLS-1$

		try {
			//ToolboxesPlugin plugin = ToolboxesPlugin.getDefault();
			return ImageDescriptor.createFromURL(getBundle().getEntry(
					iconPath + relativePath));
		} catch (RuntimeException e) {
			return ImageDescriptor.getMissingImageDescriptor();
		}
	}

	public Image getImage(String relativePath) {
		return getImageDescriptor(relativePath).createImage();
	}

    /**
     * Retrieves given resource from the bundle.
     * 
     * @param key
     *            Resource key.
     * @return String
     */
    public String getResourceString(String key) {
        try {
            ResourceBundle resBundle = getResourceBundle();
            if (resBundle == null) {
                resBundle = loadResourceBundle();
                setResourceBundle(resBundle);
            }

            String bundleString = resBundle.getString(key);
            return bundleString;
        } catch (MissingResourceException e) {
            return key;
        }
    }

    /**
     * Retrieves given resource from the bundle with arguments.
     * 
     * @param key
     *            Resource key.
     * @param arguments
     *            Arguments for the resource.
     * @return String
     */
    public String getResourceString(String key, Object[] arguments) {
        MessageFormat f = new MessageFormat(getResourceString(key));

        for (int i = 0; i < arguments.length; i++) {
            if (arguments[i] == null) {
                arguments[i] = "null"; //$NON-NLS-1$
            } else if (!(arguments[i] instanceof String)
                    && !(arguments[i] instanceof Number)
                    && !(arguments[i] instanceof java.util.Date)) {
                arguments[i] = arguments[i].toString();
            }
        }

        return f.format(arguments);
    }

    /**
     * Logs runtime status.
     * 
     * @param status
     *            Runtime status.
     */
    public void log(IStatus status) {
        getLog().log(status);
    }

    /**
     * Logs error message.
     * 
     * @param message
     *            Error message.
     */
    public void log(String message) {
        log(createErrorStatus(message));
    }

    /**
     * Logs and exception.
     * 
     * @param e
     *            Exception.
     */
    public void log(Throwable e) {
        log(createErrorStatus(e));
    }

    /**
     * Returns the installed plugin's location
     * 
     * @return IPath return the absolute location of the plugin.
     */
    public IPath getInstalledLocation() throws IOException {
        IPath installedDir = null;

        URL installedURL;
        URL resolvedURL;

        installedURL = getBundle().getEntry("/"); //$NON-NLS-1$
        resolvedURL = FileLocator.resolve(installedURL);
        installedDir = (new Path(resolvedURL.getFile()))
                .removeTrailingSeparator();

        return installedDir;
    }
    
    public IStatus createErrorStatus(String message) {
    	return new Status(IStatus.ERROR, getBundle().getSymbolicName(),
                INTERNAL_ERROR, message, null);
    }
    
    public IStatus createErrorStatus(Throwable e) {
    	return new Status(IStatus.ERROR, getBundle().getSymbolicName(),
                INTERNAL_ERROR, getResourceString("plugin.internal_error"), e); //$NON-NLS-1$
    }
}