/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.stp.core.saf.handler;

import org.eclipse.core.resources.IContainer;
import org.eclipse.stp.core.saf.IUIContext;
import org.eclipse.stp.core.saf.exception.InvalidOperationException;
import org.eclipse.stp.core.saf.exception.SAFException;
import org.eclipse.stp.core.sca.EntryPoint;
import org.eclipse.stp.core.sca.SCAObject;
import org.eclipse.wst.common.frameworks.datamodel.IDataModelOperation;

/**
 * This interface must be implemented by all entry point handlers. Most handlers
 * should extend the abstract class
 * {@link org.eclipse.stp.core.saf.handler.AbstractEntryPointHandler AbstractEntryPointHandler}
 * which provides certain convenient methods.
 */
public interface IEntryPointHandler extends IHandler {

	// ======================
	// Create entry point bindings
	// ======================
	
	/**
	 * Queries if an entry point of the kind supported by this handler can be
	 * created for the specified model object. If this method returns
	 * <code>true</code>, the handler's
	 * {@link #createEntryPointFor(Object, IContainer, IUIContext)} method may
	 * be called to create an entry point for the component or external service.
	 * <p>
	 * This method should be quick to execute, because it may be called to populate a
	 * context menu or a user dialog.
	 * 
	 * @param service
	 *            an object representing the service to be exposed
	 * @return <code>true</code> if the handler can create an
	 *         <code>EntryPoint</code> object from the specified model object;
	 *         <code>false</code> otherwise
	 */
	public boolean canCreateEntryPointFor(Object service);
	
	/**
	 * Creates an entry point with a binding of the kind supported by this
	 * handler for the specified service. All interfaces of the target are
	 * exposed. To expose specific interfaces, call
	 * {@link #createEntryPointBindingFor(EntryPoint, IUIContext)} instead.
	 * Optionally, this method may create other objects required to support the
	 * new entry point (such as a component). This method is not expected to
	 * associate a resource with the returned {@link EntryPoint} or supporting
	 * object.
	 * 
	 * @param service
	 *            an object representing the service being exposed
	 * @param container
	 *            the container that holds the module, subsystem, composite, etc.
	 *            that will use the entry point object returned by this method.
	 *            The handler could use its location as a base to place any
	 *            binding-related files that need to be generated.
	 * @param callback
	 *            callback object that the handler can use to obtain additional
	 *            information needed to create the entry point. 
	 *            Callers may pass in a null value.
	 * @return an array of {@link SCAObject} containing the new entry point and
	 *         any other SCA objects generated by the handler to
	 *         support the new entry point
	 * @exception SAFException
	 *                An unexpected exception occurred.
	 * @exception InvalidOperationException
	 *                An entry point cannot be created from the given service
	 *                object.
	 * @exception InterruptedException
	 *                The caller did not provide a callback object needed by the
	 *                handler, or the callback request was canceled or aborted.
	 */
	public SCAObject[] createEntryPointFor(Object service,
			IContainer container, IUIContext callback)
			throws SAFException, InterruptedException, InvalidOperationException;

	/**
	 * Queries if the entry point handler provides support for creating an entry
	 * point binding using the method
	 * {@link #createEntryPointBindingFor(EntryPoint, IUIContext) createEntryPointBindingFor()}.
	 * 
	 * @return <code>true</code> if support is provided for creating a
	 *         binding; <code>false</code> otherwise
	 */
	public boolean canCreateEntryPointBinding();
	
	/**
	 * Queries if the entry point handler provides support for creating more
	 * than one binding instance for an entry point.
	 * 
	 * @return <code>true</code> if support is provided for creating more than
	 *         one binding of the supported kind; <code>false</code> otherwise
	 */
	public boolean canCreateManyEntryPointBindings();

	/**
	 * Adds a binding of the kind supported by this handler to the specified
	 * {@link EntryPoint EntryPoint} object, based on the specified target and
	 * interfaces in the entry point definition. The handler is expected to
	 * update the entry point definition, if necessary. If the entry point has an
	 * existing binding, the handler can add to or replace the binding. The
	 * handler can use the callback to query for more information from the user.
	 * If the binding requires associated files, this method should generate
	 * those files. In this case, this method should be aware that the files it
	 * generates may replace some already being managed in a team environment.
	 * The {@link EntryPoint EntryPoint} object should have a resource
	 * associated with it. The handler can use this location to generate
	 * relative file references if needed or use the location for storing
	 * binding-related files that need to be generated. The resource can be
	 * retrieved by casting the {@link EntryPoint EntryPoint} object to an
	 * {@link org.eclipse.emf.ecore.EObject EObject} and calling
	 * {@link org.eclipse.emf.ecore.EObject#eResource() EObject.eResource()}.
	 * 
	 * @param theEntryPoint
	 *            an existing entry point definition, which may contain
	 *            interfaces and a binding
	 * @param callback
	 *            callback object that the handler can use to obtain additional
	 *            information needed to create the binding. Callers
	 *            may pass in a null value.
	 * @return an IDataModelOperation instance that, once executed by the
	 *         caller, creates the entry point binding
	 * @exception SAFException
	 *                An unexpected exception occurred.
	 * @exception InvalidOperationException
	 *                A binding cannot be created for the given entry point.
	 * @exception InterruptedException
	 *                The caller did not provide a callback object needed by the
	 *                handler, or the callback request was canceled or aborted.
	 */
	public IDataModelOperation createEntryPointBindingFor(
			EntryPoint theEntryPoint, IUIContext callback)
			throws SAFException, InterruptedException, InvalidOperationException;

	/**
	 * Queries if an entry point of the kind supported by this handler
	 * supports interfaces of the kind specified by the given interface handler
	 * in its interface set.
	 * 
	 * @param interfaceHandler
	 *            the interface handler that will be used to manipulate the
	 *            interface. The implementation of this method can obtain the
	 *            interface kind by invoking {@link IInterfaceHandler#getKind()}
	 *            when performing the evaluation.
	 * @return <code>true</code> if the entry point supports interfaces of the
	 *         specified interface kind; <code>false</code> otherwise
	 * @throws SAFException
	 *             An unexpected exception occurred in the handler.
	 */
	public boolean isInterfaceKindSupported(IInterfaceHandler interfaceHandler)
			throws SAFException;
	
	/**
	 * Removes a given binding kind from the entry point. Generally, existing
	 * interfaces will be preserved. No clean-up is done on existing
	 * binding-related files. If the entry point does not contain the given
	 * binding type, this method performs no work.
	 * 
	 * @param theEntryPoint
	 *            the entry point being modified
	 * @return an {@link IDataModelOperation} instance that, once executed by
	 *         the caller, resets the entry point kind
	 * @exception SAFException
	 *                No handler for the specified kind is found, or the
	 *                handler throws an exception.
	 * @exception InvalidOperationException
	 *                The entry point kind cannot be removed.
	 */
	public IDataModelOperation removeKind(EntryPoint theEntryPoint)
			throws SAFException, InvalidOperationException;

}
