/*******************************************************************************
 * Copyright (c) 2003 IBM Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.stp.core.introspection;

import java.util.List;

import org.eclipse.core.runtime.IStatus;
import org.eclipse.emf.common.notify.Adapter;
import org.eclipse.stp.core.sca.Implementation;

/**
 * 
 * Clients should provide an implementation which updates the individual
 * features on a ComponentType when {@link #introspect(int, List)} is called.
 * 
 * <p>
 * Implementations are encouraged to only reflect the specific field requested,
 * and delay the introspection of the entire type, if possible.
 * </p>
 * 
 * <p>
 * The {@link #init(Implementation)} method will be called before the
 * introspector is used and after the introspector has been added to the
 * {@link org.eclipse.stp.core.sca.ComponentType}. Implementations are required
 * to save the implementation supplied when the {@link #init(Implementation)}
 * method is called.
 * </p>
 * 
 * <p>
 * Implementations may subclass
 * {@link org.eclipse.stp.core.introspection.AbstractComponentTypeIntrospector}
 * for convenience.
 * </p>
 * 
 * <p>
 * Clients may implement this interface.
 * </p>
 * 
 * @see org.eclipse.stp.core.introspection.AbstractComponentTypeIntrospector
 * 
 * @since 1.0
 * 
 */
public interface IComponentTypeIntrospector extends Adapter {

   /**
    * The ADAPTER_TYPE is used for searching for IComponentTypeIntrospectors
    * attached to the eAdapter() list of an EObject.
    * 
    * Clients are not expected to need this access this adapter directly.
    */
   String ADAPTER_TYPE = IComponentTypeIntrospector.class.getName();

   /**
    * init() will be called exactly once with the implementation model from a
    * Component. If no component context is present, then an
    * UnknownImplementation will be passed with its URI intialized to that of
    * the IFile that triggered it.
    * 
    * This method is required to handle two cases. The first case is when
    * theImplementation is loaded through a resource directly, and there is no
    * Component present. In this case, the type of theImplementation will be an
    * UnknownImplementation with a String based path to the file being loaded.
    * In the second case, a substitution group specific to the introspector
    * (like "implementation.java") will be passed in. Then the introspector can
    * use the attributes and elements of the implementation element to determine
    * how the ComponentType should be resolved.
    * 
    * The ComponentTypeIntrospector should save the Implementation for later
    * invocations on introspectXXX() methods.
    * 
    * @param theImplementation
    *           An implementation URI that uses a scheme or extension which this
    *           IComponentTypeIntrospector is interested in.
    */
   void init(Implementation theImplementation);

   /**
    * This method will only be called after init() and after the introspector
    * has been added to the ComponentType as an adapter, so clients will also
    * have access to a the ComponentType model object via
    * {@link AbstractComponentTypeIntrospector#getComponentType()} if clients
    * choose to subclass that the {@link AbstractComponentTypeIntrospector}
    * type.
    * 
    * @return An OK Status if the component type is valid, or an error status
    *         otherwise. The status will automatically be added to the model
    *         object if it is !isOK()
    */
   IStatus hasValidComponentType();

   /**
    * The introspector should add or remove values as necessary to synchronize
    * the current list available on the ComponentType with the underlying
    * implementation model.
    * 
    * <p>
    * The integer value will indicate which field requires introspection. The
    * values are available as:
    * </p>
    * 
    * <ul>
    * <li>{@link org.eclipse.stp.core.sca.SCAPackage#COMPONENT_TYPE__SERVICES}:
    * Indicates a list of {@link org.eclipse.stp.core.sca.Service}s.</li>
    * <li>{@link org.eclipse.stp.core.sca.SCAPackage#COMPONENT_TYPE__REFERENCES}:
    * Indicates a list of {@link org.eclipse.stp.core.sca.Reference}s.</li>
    * <li>{@link org.eclipse.stp.core.sca.SCAPackage#COMPONENT_TYPE__PROPERTIES}:
    * Indicates a list of {@link org.eclipse.stp.core.sca.Property}s.</li>
    * </ul>
    * 
    * <p>
    * The current state of the list will be passed by reference. Clients may add
    * and remove to this list and return it (without need for creating a new
    * list). Clients should use the value of the theFieldType parameter to
    * determine what type of values exist in the list. The possible types are
    * described below.
    * </p>
    * 
    * 
    * 
    * @param theCurrentValues
    *           The current List of values known in the canonical model. The
    *           list will be empty the first time this method is called.
    * @param theFieldType
    *           The field that should be introspected. The types of fields are
    *           available as {@link org.eclipse.stp.core.sca.SCAPackage#SERVICE},
    *           {@link org.eclipse.stp.core.sca.SCAPackage#COMPONENT_TYPE__REFERENCES},
    *           or
    *           {@link org.eclipse.stp.core.sca.SCAPackage#COMPONENT_TYPE__PROPERTIES}.
    * 
    * @see org.eclipse.stp.core.sca.Service
    * @see org.eclipse.stp.core.sca.Reference
    * @see org.eclipse.stp.core.sca.Property
    * 
    * @return A status to indicate whether the introspection was successful.
    * 
    */
   IStatus introspect(int theFieldType, List theCurrentValues);

   /**
    * Respond to a change in the list of values for a given field type. The
    * types of fields are available as
    * {@link org.eclipse.stp.core.sca.SCAPackage#SERVICE},
    * {@link org.eclipse.stp.core.sca.SCAPackage#COMPONENT_TYPE__REFERENCES},
    * or {@link org.eclipse.stp.core.sca.SCAPackage#COMPONENT_TYPE__PROPERTIES}.
    * 
    * 
    * @param theFieldType
    *           One of {@link org.eclipse.stp.core.sca.SCAPackage#SERVICE},
    *           {@link org.eclipse.stp.core.sca.SCAPackage#COMPONENT_TYPE__REFERENCES},
    *           or
    *           {@link org.eclipse.stp.core.sca.SCAPackage#COMPONENT_TYPE__PROPERTIES}.
    *           Indicates the type of changed object ({@link org.eclipse.stp.core.sca.SCAPackage#SERVICE}
    *           for {@link org.eclipse.stp.core.sca.Service},
    *           {@link org.eclipse.stp.core.sca.SCAPackage#COMPONENT_TYPE__REFERENCES}
    *           for {@link org.eclipse.stp.core.sca.Reference}, and
    *           {@link org.eclipse.stp.core.sca.SCAPackage#COMPONENT_TYPE__PROPERTIES}
    *           for {@link org.eclipse.stp.core.sca.Property}).
    * @param theChangeType
    *           One of {@link org.eclipse.emf.common.notify.Notification#ADD} or
    *           {@link org.eclipse.emf.common.notify.Notification#REMOVE}.
    * @param theChangedObject
    *           The object that was added or removed. Check the value of
    *           <code>theFieldType</code> to determine the type.
    * @return A status to indicate whether the update to the implementation
    *         model succeeded.
    */
   IStatus onModelChange(int theFieldType, int theChangeType,
         Object theChangedObject);

   /**
    * Persist the changes made to the underlying model through this adaptation.
    * 
    * @return A status to indicate whether the save succeeded or failed.
    */
   IStatus save();

   /**
    * Release any resources that were required to introspect the implementation
    * model.
    * 
    * @return A status to indicate whether the save succeeded or failed.
    */
   IStatus dispose();

   /**
    * @return The Implementation for the underlying ComponentType.
    */
   Object resolveImplementation();

}
