/**********************************************************************
 * Copyright (c) 2005 Scapa Technologies Limited and others
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * Scapa Technologies Limited - Initial API and implementation
 **********************************************************************/

package org.eclipse.stp.b2j.core.jengine.internal.compiler.wsdl;

import java.io.ByteArrayInputStream;
import java.util.ArrayList;
import java.util.HashMap;

import org.eclipse.stp.b2j.core.jengine.internal.compiler.TranslatorLog;
import org.eclipse.stp.b2j.core.jengine.internal.compiler.Util;
import org.eclipse.stp.b2j.core.jengine.internal.compiler.wsdlmap.WSDLMap;
import org.eclipse.stp.b2j.core.jengine.internal.compiler.xmlns.NamespaceException;
import org.eclipse.stp.b2j.core.jengine.internal.compiler.xmlns.NamespaceTranslator;
import org.eclipse.stp.b2j.core.jengine.internal.compiler.xsdmap.XSDMap;
import org.eclipse.stp.b2j.core.misc.internal.XMLUtil;
import org.eclipse.stp.b2j.core.publicapi.extension.wsdlbinding.WSDLBindingTranslator;
import org.eclipse.stp.b2j.core.publicapi.extension.wsdlbinding.WSDLBindingTranslatorException;
import org.eclipse.stp.b2j.core.publicapi.extension.wsdlbinding.WSDLTypeTranslator;
import org.eclipse.stp.b2j.core.publicapi.extension.wsdlbinding.XSDTypeTranslator;
import org.eclipse.stp.b2j.core.publicapi.extension.wsdlbinding.XSDTypeTranslatorException;
import org.eclipse.stp.b2j.core.publicapi.jcompiler.CompilerUnit;
import org.eclipse.stp.b2j.core.xml.internal.w3c.Document;
import org.eclipse.stp.b2j.core.xml.internal.w3c.DocumentBuilder;
import org.eclipse.stp.b2j.core.xml.internal.w3c.Element;

/**
 * 
 * @author amiguel
 *
 * Translates WSDL files into any necessary classes which become part of
 * an engine Program (e.g. Messages)
 */
public class WSDLTranslator {

//MessageCodec codec = new MessageCodec();

XSDTypeTranslator[] codecs;

XSDMap xsdmap;
WSDLMap wsdlmap;

NamespaceTranslator nt;

TranslatorLog log;

String LOG_SOURCE = "WSDLTranslator";

Util compiler_util;

class WSDLState implements CompilerUnit {
	int SCOPE = 1;
	StringBuffer decl = new StringBuffer();
	StringBuffer prog = new StringBuffer();

	String pkg_name;
	String class_name;
	String base_class;
	String src_file;
	
	boolean isMessage = false;
	boolean iface = false;

	ArrayList ifaces = new ArrayList();
	ArrayList imports = new ArrayList();

	ArrayList field_types = new ArrayList();
	ArrayList field_names = new ArrayList();
	ArrayList field_elements = new ArrayList();

	public String getSourceFile() {
		return src_file==null?"(source file unknown)":src_file;
	}
	
/*	public WSDLState(String src_file, String pkg_name, String class_name) {
		this.pkg_name = pkg_name;
		this.class_name = class_name;
		this.src_file = src_file;
	}*/

	public WSDLState(String src_file, String pkg_name, String class_name, String extend) {
		this.pkg_name = pkg_name;
		this.class_name = class_name;
		this.base_class = extend;
		this.src_file = src_file;
	}

	public void setMessage(boolean b) {
		isMessage = b;
	}
	
	public void setInterface(boolean b) {
		iface = b;	
	}

	private void addInterface(String iface) {
		ifaces.add(iface);
	}

	private void addInterfaces(String[] ifaces) {
		for (int i = 0; i < ifaces.length; i++) {
			addInterface(ifaces[i]);
		}
	}

	private void addImport(String iport) {
		imports.add(iport);
	}
	
	private void addImports(String[] ifaces) {
		for (int i = 0; i < ifaces.length; i++) {
			addImport(ifaces[i]);
		}
	}

	private void addField(String type, String name, String elemname) {
		field_types.add(type);
		field_names.add(name);
		field_elements.add(elemname);
	}
	
	public String getQualifiedName() {
		return pkg_name+"."+class_name;	
	}

	public String getPackageName() {
		return pkg_name;	
	}

	public String getClassName() {
		return NamespaceTranslator.getJavaClassName(class_name);
	}

	private void appendLine(String s) {
		for (int i = 0; i < SCOPE; i++) {
			prog.append("  ");	
		}
		prog.append(s).append("\n");
	}
	
	private void incrementScope() {
		SCOPE++;
	}
	private void decrementScope() {
		if (SCOPE >= 1) SCOPE--;	
	}

	public void declToString(StringBuffer buf) {

		//append my declarations
		buf.append(decl.toString());
	}
	
	public void progToString(StringBuffer buf) {

		String clazz = "class";
		if (iface) clazz = "interface";


		buf.append("//@@START_STRIP\n");
		//append my logic
		//for whatever reasons the source file has newlines so this breaks stuff.
//		buf.append("//GENERATED FROM SOURCE WSDL FILE: "+src_file.trim());
		buf.append("package "+pkg_name+";\n");
		buf.append("import "+Util.jengine_package+".utils.*;\n");
		buf.append("import "+Util.jengine_package+".core.*;\n");
		buf.append("import "+Util.jengine_package+".message.Message;\n");
		
		for (int i = 0; i < imports.size(); i++) {
			buf.append("import "+imports.get(i)+";\n");
		}
		
		if (base_class == null) {
			buf.append("public "+clazz+" "+NamespaceTranslator.getJavaClassName(class_name)+"  \n");
		} else {
			buf.append("public "+clazz+" "+NamespaceTranslator.getJavaClassName(class_name)+" extends "+base_class+" \n");
		}
		if (ifaces.size() > 0) {
			buf.append(" implements\n");	
		}
		for (int i = 0; i < ifaces.size(); i++) {
			buf.append("  "+ifaces.get(i));	
			if (i < ifaces.size()-1) {
				buf.append(",");	
			}
			buf.append("\n");
		}
		buf.append("{\n");	
		buf.append("//@@END_STRIP\n");
		
		buf.append("\n");
		buf.append("  //WSDL messages have no tag name\n");
		buf.append("  public String "+NamespaceTranslator.getElementTagName()+" = null;\n");
		buf.append("\n");

		buf.append(prog.toString());

		buf.append("\n");
		buf.append("public static String getStackTrace(Throwable t) {\n");
		buf.append("    try {\n");
		buf.append("  	  java.io.ByteArrayOutputStream bout = new java.io.ByteArrayOutputStream();\n");
		buf.append("	  t.printStackTrace(new java.io.PrintStream(bout));\n");
		buf.append("	  return new String(bout.toByteArray());\n");
		buf.append("    } catch (Exception e) { return t.toString(); }\n");
		buf.append("}\n");
		buf.append("\n");
		
		/*

		buf.append("\n");
		buf.append("  public String toString() {\n");
		buf.append("    try {\n");
		buf.append("      return String.valueOf(toEngineMessage(this));\n");
		buf.append("    } catch (Exception e) {\n");
		buf.append("      return e.toString();\n");
		buf.append("    }\n");
		buf.append("  }\n");

		buf.append("  public void appendXpathStringValue(StringBuffer sb) {\n");
		buf.append("    try {\n");
		for (int i = 0; i < field_names.size(); i++) {
			String fname = (String)field_names.get(i);
			String ftype = (String)field_types.get(i);
			buf.append("    for (int i = 0; i < "+fname+".length; i++) {\n");
			buf.append("      if ("+fname+"[i] != null) "+fname+"[i].appendXpathStringValue(sb);\n");
			buf.append("    }\n");
		}
		buf.append("    } catch (Exception e) {\n");
		buf.append("      sb.append(e.toString());\n");
		buf.append("    }\n");
		buf.append("  }\n");
		buf.append("  public String toXpathStringValue() {\n");
		buf.append("    try {\n");
		buf.append("      StringBuffer sb = new StringBuffer();\n");
		buf.append("      appendXpathStringValue(sb);\n");
		buf.append("      return sb.toString();\n");
		buf.append("    } catch (Exception e) {\n");
		buf.append("      return e.toString();\n");
		buf.append("    }\n");
		buf.append("  }\n");
*/		
		if (isMessage) xpathToString(buf);
		
		buf.append("//@@START_STRIP\n");
		buf.append("}\n");
		buf.append("//@@END_STRIP\n\n");

	}

	public void xpathToString(StringBuffer buf) {
		buf.append("public XNodeSet getFieldSet() {\n");
		buf.append("  XNodeSet fields = new XNodeSet("+field_names.size()+");\n");
		for (int i = 0; i < field_names.size(); i++) {
			String fieldname = String.valueOf(field_names.get(i));
			String elemname = String.valueOf(field_elements.get(i));
////			String realname = NamespaceTranslator.getRealName(fieldname);
//			String realname = elemname;
			
			buf.append("  if ("+fieldname+" != null)\n");
			buf.append("    for (int i = 0; i < "+fieldname+".length; i++)\n");
//			buf.append("      fields.add(new XNode(\""+realname+"\","+fieldname+"[i],"+fieldname+",i,null));\n");
//TODO use the element name when filtering xpath expressions? or the real name?
			buf.append("      fields.add(new XNode(\""+elemname+"\","+fieldname+"[i],"+fieldname+",i,null));\n");
		}
		buf.append("  return fields;\n");
		buf.append("}\n");
		buf.append("public XNodeSet getFieldSet(XNode parent) {\n");
		buf.append("  XNodeSet fields = new XNodeSet("+field_names.size()+");\n");
		for (int i = 0; i < field_names.size(); i++) {
			String fieldname = String.valueOf(field_names.get(i));
			String elemname = String.valueOf(field_elements.get(i));
////			String realname = NamespaceTranslator.getRealName(fieldname);
//			String realname = elemname;
			
			buf.append("  if ("+fieldname+" != null)\n");
			buf.append("    for (int i = 0; i < "+fieldname+".length; i++)\n");
//			buf.append("      fields.add(new XNode(\""+realname+"\","+fieldname+"[i],"+fieldname+",i,parent));\n");
//			TODO use the element name when filtering xpath expressions? or the real name?
			buf.append("      fields.add(new XNode(\""+elemname+"\","+fieldname+"[i],"+fieldname+",i,parent));\n");
		}
		buf.append("  return fields;\n");
		buf.append("}\n");
	}


	public String toString() {
		StringBuffer buf = new StringBuffer();
		
		buf.append("//WSDL CLASS\n");
		declToString(buf);
		progToString(buf);
		
		return buf.toString();	
	}
	
//	public String toCompilerString() {
//		StringBuffer buf = new StringBuffer();
//		
//		buf.append("//WSDL CLASS\n");
//		buf.append(decl);
//		buf.append(prog);
//		
//		return buf.toString();	
//	}
	
}

	public WSDLTranslator(Util compiler_util, TranslatorLog log, XSDMap xsdmap, WSDLMap wsdlmap) {
		this.xsdmap = xsdmap;
		this.wsdlmap = wsdlmap;
		this.log = log;
		this.compiler_util = compiler_util;
		nt = compiler_util.createNamespaceTranslator();
	}
	
	public void translateMessage(ArrayList states, Element elem) throws WSDLTranslatorException, NamespaceException {		
		
//		String bname = "WSDLMessage_"+elem.getAttribute("name");
		String bname = elem.getAttribute("name");

/*		if (bname.endsWith("AllObservationsResponse")) {
			System.out.println(this.tmp);
		}
System.out.println("translate Message "+bname);*/
		
		String pkg = nt.getPackage(bname,true);
		String name = NamespaceTranslator.getName(bname);

//		WSDLState state = new WSDLState(pkg,name,"java.lang.Exception");
		WSDLState state = new WSDLState(currentWsdlFile,pkg,name,"java.lang.Object");
		state.addInterface(Util.jengine_package+".utils.XPathAccessible");
		state.setMessage(true);
		
		ArrayList parts = Util.getAllElements(elem,"part");
		String[] part_types = new String[parts.size()];
		String[] part_names = new String[parts.size()];
		String[] part_elements = new String[parts.size()];

		for (int i = 0; i < parts.size(); i++) {
			Element part = (Element)parts.get(i);
			String pname = part.getAttribute("name");
			String pnamealias = pname;
			String ptype = part.getAttribute("type");
			if (ptype.length() == 0) {
				//we can treat type and element as the same thing
				ptype = part.getAttribute("element");	
				pnamealias = NamespaceTranslator.getName(part.getAttribute("element"));
			}
			
			String qtype = nt.qualify(ptype,false);
			
			if (xsdmap.getType(qtype) == null) {
				throw new WSDLTranslatorException(compiler_util,part,"Type "+ptype+" not found for WSDL Message part "+pname);
			}
			
			part_types[i] = qtype;
			part_names[i] = pname;
			part_elements[i] = pnamealias;
			
			state.addField(qtype,NamespaceTranslator.getElement(pname),pnamealias);
			
//			state.appendLine("public "+qtype+"[] "+pname+" = new "+qtype+"[] {new "+qtype+"()};");	
//			state.appendLine("public "+qtype+"[] "+pname+" = new "+qtype+"[1];");
			//this is undefined and should therefore be an array of length 0 rather than 
			//an array of length 1 with a null element which means its there but xsi:nil=true	
			state.appendLine("public "+qtype+"[] "+NamespaceTranslator.getElement(pname)+" = new "+qtype+"[0];");	
		}
		
		for (int i = 0; i < codecs.length; i++) {
			if (codecs[i] instanceof WSDLTypeTranslator) {
				try {
					codecs[i].translateComplex(nt,name,new String[0],new String[0],part_types,part_names,part_elements,elem);
			
					state.appendLine(codecs[i].getMethods());
					state.addInterfaces(codecs[i].getImplementations());
					state.addImports(codecs[i].getImports());
				} catch (XSDTypeTranslatorException e) {
					e.printStackTrace();
				}
			}
		}
		
		System.out.println("Added state "+pkg+"."+name);
		states.add(state);
	}

	public void translatePortType(ArrayList states, Element elem) throws NamespaceException {
//		String name = "WSDLPortType_"+elem.getAttribute("name");
		
		String bname = elem.getAttribute("name");

		String pkg = nt.getPackage(bname,true);
		String name = NamespaceTranslator.getName(bname);

		WSDLState state = new WSDLState(currentWsdlFile,pkg,name,null);
		state.setInterface(true);

		ArrayList ops = Util.getAllElements(elem,"operation");
		for (int i = 0; i < ops.size(); i++) {
			Element op = (Element)ops.get(i);

			translateOperation(state,op);
			
		}

		log.logInfo(LOG_SOURCE,"Didn't add port interface "+bname+" to the compilation list because it doesn't have a purpose");
//		states.add(state);
	}
	
	public void translateOperation(WSDLState state, Element elem) throws NamespaceException {
		String opname = elem.getAttribute("name");

		Element input = Util.getFirstElement(elem,"input");
		Element output = Util.getFirstElement(elem,"output");
		ArrayList faults = Util.getAllElements(elem,"fault");		
		
		String inputname = "";
		if (input != null) {
			inputname = input.getAttribute("message");
			inputname = nt.qualify(inputname,false);
		}
		
		String outputname = "void";
		if (output != null) {
			outputname = output.getAttribute("message");
			outputname = nt.qualify(outputname,false);	
		}
		
		if (faults.size() > 0) {

			state.appendLine("public "+outputname+" "+opname+"("+inputname+" input) throws ");
			state.incrementScope();
			for (int i = 0; i < faults.size(); i++) {
				Element fault = (Element)faults.get(i);
				String faultname = fault.getAttribute("message");
				faultname = nt.qualify(faultname,false); 
				state.appendLine(faultname);
			}
			state.appendLine(";");
			state.decrementScope();

		} else {

			state.appendLine("public "+outputname+" "+opname+"("+inputname+" input);");
			
		}
	}

	public void translateDefinitions(ArrayList states, Element definitions) throws WSDLTranslatorException, NamespaceException {

		nt.addNamespaces(definitions);

		ArrayList elems = Util.getAllElements(definitions);
		for (int i = 0; i < elems.size(); i++) {
			Element def = (Element)elems.get(i);
			String name = NamespaceTranslator.getName(def);
			
			if (name.equals("message")) {
				translateMessage(states,def);
			} else if (name.equals("portType")) {
				translatePortType(states,def);
			} else if (name.equals("partnerLinkType")) {
				//ignored
			} else if (name.equals("import")) {
				//ignored
			} else if (name.equals("types")) {
				//ignored
			} else if (name.equals("binding")) {
				//ignored
			} else if (name.equals("service")) {
				//ignored
			} else {
				log.logWarning(LOG_SOURCE,"Unknown definition child \""+name+"\"");	
			}	
		}

		nt.removeNamespaces(definitions);
	}
	
	String currentWsdlFile;

	public WSDLState[] getJava(String[] wsdl, WSDLBindingTranslator[] bindings, XSDTypeTranslator[] codecs, String[] wsdl_files) throws Exception {

		this.codecs = codecs;
		
		ArrayList states = new ArrayList();

		for (int w = 0; w < wsdl.length; w++) {
			DocumentBuilder builder = XMLUtil.getDocumentBuilder();
			String sourceFile = "(Unknown Source File)";
			if (w < wsdl_files.length) {
				sourceFile = wsdl_files[w];
			}
			Document doc = builder.parse(new ByteArrayInputStream(wsdl[w].getBytes()),sourceFile);
			
			Element definitions = doc.getDocumentElement();
			if (!NamespaceTranslator.getName(definitions).equals("definitions")) {
				throw new WSDLBindingTranslatorException("Expected top level <definitions> element but found "+definitions.getTagName());
			}
			
			for (int k = 0; k < codecs.length; k++) {
				try {
					codecs[k].readWSDL(definitions);
				} catch (Exception e) {
					throw new Exception("Binding Type Translator "+codecs[k].getClass().getName()+" failed to read WSDL file "+wsdl_files[w],e);
				}
			}

			for (int k = 0; k < bindings.length; k++) {
				try {
					bindings[k].readWSDL(definitions);
				} catch (Exception e) {
					throw new Exception("Binding "+bindings[k].getClass().getName()+" failed to read WSDL file "+wsdl_files[w],e);
				}
			}
			
//			codec.readWSDL(definitions);
/*			
			for (int i = 0; i < bindings.length; i++) {
				//have the bindings read the WSDL binding stuff before we do anything with it
				bindings[i].readWSDL(definitions);
				
				XSDCodecTranslator[] codecs = bindings[i].getCodecs();
				for (int k = 0; k < bindings.length; k++) {
					codecs[k].readWSDL(definitions);
				}
			}
*/	
			if (w < wsdl_files.length) {
				currentWsdlFile = wsdl_files[w];//+"\n\n"+wsdl[w];
			}
			translateDefinitions(states,definitions);
		}
		
		HashMap warn = new HashMap();
		HashMap test = new HashMap();
		for (int i = 0; i < states.size(); i++) {
			WSDLState state = (WSDLState)states.get(i);
			WSDLState testy = (WSDLState)test.get(state.getPackageName()+"."+state.getClassName());
			WSDLState warny = (WSDLState)warn.get(state.getClassName());
			if (testy != null) {
				//found an exact match, check if they are the same file
				if (!testy.getSourceFile().equals(state.getSourceFile())) {
					//different files - redefinition!
					String src1 = state.toString();
					String src2 = testy.toString();
					
					src1 = src1.substring(src1.indexOf("package"));
					src2 = src2.substring(src2.indexOf("package"));
					
					if (src1.equals(src2)) {
						//redefinition, but SAME definition
						log.logWarning(LOG_SOURCE, state.getPackageName()+"."+state.getClassName()+" defined twice in file "+testy.getSourceFile()+" and "+state.getSourceFile()+", but making this a WARNING since definitions are exactly the same");
					} else {
						//two different definitions
	
						//XXX note that at the moment these will always differ because of the NSPACE statically incremented
						//variable in XMLCodec
//						System.out.println("----SOURCE1:\n"+src1);
//						System.out.println("----SOURCE2:\n"+src2);
						throw new Exception(state.getPackageName()+"."+state.getClassName()+" defined twice in file "+testy.getSourceFile()+" and "+state.getSourceFile());
					}
				} else if (testy.src_file == null) {
					//same file, but both unknown files
					log.logWarning(LOG_SOURCE, state.getPackageName()+"."+state.getClassName()+" defined twice in unknown files");
				}
				if (!warny.getSourceFile().equals(state.getSourceFile())) {
					//different files - redefinition!
					log.logWarning(LOG_SOURCE, state.getClassName()+" defined twice (but under different namespaces) in file "+warny.getSourceFile()+" and "+state.getSourceFile());
				} else if (warny.src_file == null) {
					//same file, but both unknown files
					log.logWarning(LOG_SOURCE, state.getClassName()+" defined twice (but under different namespaces) in unknown files");
				}
				states.remove(i--);
			} else {
				test.put(state.getPackageName()+"."+state.getClassName(),state);
				warn.put(state.getClassName(), state);
			}
		}

		WSDLState[] ret = new WSDLState[states.size()];
		for (int i = 0; i < ret.length; i++) {
			ret[i] = (WSDLState)states.get(i);	
		}
		return ret;
	}
	
	public CompilerUnit[] getDefinitionSources(String[] wsdl, WSDLBindingTranslator[] bindings, XSDTypeTranslator[] codecs, String[] wsdl_files) throws Exception {

		WSDLState[] java = getJava(wsdl,bindings, codecs,wsdl_files);
		
		return java;
	}

}