/*******************************************************************************
* Copyright (c) 2006 IONA Technologies PLC
* All rights reserved. This program and the accompanying materials
* are made available under the terms of the Eclipse Public License v1.0
* which accompanies this distribution, and is available at
* http://www.eclipse.org/legal/epl-v10.html
* 
* Contributors:
*     IONA Technologies PLC - initial API and implementation
*******************************************************************************/
package org.eclipse.stp.sc.sca.java.wizards;

import java.io.FileOutputStream;
import java.lang.reflect.InvocationTargetException;
import java.util.jar.Attributes;
import java.util.jar.Manifest;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExecutableExtension;
import org.eclipse.core.runtime.IExtension;
import org.eclipse.core.runtime.IExtensionRegistry;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.SubProgressMonitor;
import org.eclipse.jdt.core.ICompilationUnit;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.wizard.IWizardPage;
import org.eclipse.jface.wizard.Wizard;
import org.eclipse.stp.common.logging.LoggingProxy;
import org.eclipse.stp.sc.common.utils.CXFUtil;
import org.eclipse.stp.sc.sca.java.ScaJavaResources;
import org.eclipse.stp.sc.sca.java.extensions.IJavaLaunchConfigGenerator;
import org.eclipse.stp.sc.sca.java.utils.JavaClassGenerator;
import org.eclipse.stp.sc.sca.java.workspace.ScaWorkspaceManager;
import org.eclipse.stp.sc.sca.scdl.ScdlBuilder;
import org.eclipse.ui.INewWizard;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.dialogs.WizardNewProjectCreationPage;
import org.eclipse.ui.wizards.newresource.BasicNewProjectResourceWizard;



public class ScaJavaProjectWizard extends Wizard implements INewWizard, IExecutableExtension {
    // page identifiers used by this wizard
    public static final String GENERAL_PAGE = "generalPage";
    public static final String COMP_PAGE = "compPage";
    public static final String RMI_PAGE = "rmiPage";
    public static final String WS_PAGE = "wsPage";
    public static final String RUNTIME_PAGE = "runtimePage";
    
    private static final LoggingProxy LOG = LoggingProxy.getlogger(ScaJavaProjectWizard.class);
    
    protected IConfigurationElement fConfigElement;
    WizardNewProjectCreationPage genPage;
    ScaComponentWizardPage compPage;
    ScaRMIBindingWizardPage rmiPage;
    ScaWSBindingWizardPage wsPage;
    RuntimeSelectionPage runtimePage;
    
    IProject project;
    
    String componentName = null;
    
    public ScaJavaProjectWizard() {
        super();

        String title = ScaJavaResources.getString("wizard.project.title");
        setWindowTitle(title);
    }
    
    /*
     * Stores the configuration element for the wizard.  The config element will be used
     * in <code>performFinish</code> to set the result perspective.
     */
    public void setInitializationData(IConfigurationElement cfig, String propertyName, Object data) {
        fConfigElement = cfig;
    }

    public void addPages() {
        genPage = new WizardNewProjectCreationPage(GENERAL_PAGE);
        genPage.setTitle(ScaJavaResources.getString("wizard.project.generalpage.title"));
        genPage.setDescription(ScaJavaResources.getString("wizard.project.generalpage.desc"));

        ImageDescriptor banner = ScaJavaResources.getImageDescriptor("wizban/sca_wiz.gif");
        genPage.setImageDescriptor(banner);
        addPage(genPage);
        
        runtimePage = new RuntimeSelectionPage(RUNTIME_PAGE, "", banner);
        addPage(runtimePage);
        
        compPage = new ScaComponentWizardPage(COMP_PAGE, this);
        addPage(compPage);
        
        rmiPage = new ScaRMIBindingWizardPage(RMI_PAGE, this);
        addPage(rmiPage);
        
        wsPage = new ScaWSBindingWizardPage(WS_PAGE, this);
        addPage(wsPage);
    }

    @Override
    public synchronized boolean performFinish() {
        try {
            project = createProject(null);
            createJavaClass();
            createSCDL();
            BasicNewProjectResourceWizard.updatePerspective(fConfigElement);
            project.refreshLocal(IProject.DEPTH_INFINITE, null);
            if (componentName != null) {
                createLaunchConfig(componentName, getMainClassFullName());
            }
        } catch (Exception e) {
        	LOG.error("error during create project", e);
            return false;
        }

        return true;
    }
    
    private void createManifest(String mainClsFullName) {
    	try {
			Manifest mfst = new Manifest();
			Attributes attrs = mfst.getMainAttributes();
			attrs.put(Attributes.Name.MANIFEST_VERSION, "1.0");
			attrs.put(Attributes.Name.MAIN_CLASS, mainClsFullName);
			IFile file = ScaWorkspaceManager.getManiInfFile(project);
			FileOutputStream fos = new FileOutputStream(file.getRawLocation()
					.toOSString());
			mfst.write(fos);
			fos.close();
		} catch (Exception e) {
    		LOG.debug("error try to create manifest file.", e);
    	}
    	
    	
    }
    
    private void createJavaClass() {
    	String implClsName = compPage.getImplName();
    	String interfaceName = compPage.getInterfaceName(); 
    	String packageName = compPage.getPackage();
    	
    	if ( implClsName ==null
    			|| implClsName.equals("")) {
    	    return;			
    	}
    	
    	if (packageName == null) {
    		packageName = "";
    	}
    	
    	
    	JavaClassGenerator generator = new JavaClassGenerator();
    	ICompilationUnit infUnit = null;
    	if (interfaceName != null && !interfaceName.equals("")) {
    	    LOG.debug("create interface:" + packageName + "." + interfaceName);
    	    infUnit = generator.createJavaClass(packageName, 
        			interfaceName, true, null, project);
    	    if (infUnit != null) {
    	    	generator.addInterfaceDefaultAnnoation(infUnit);
    	    }
    	    generator.createScaServerMain(packageName, interfaceName, project);
    	    createManifest(getMainClassFullName());
    	}
    	LOG.debug("create impl class:" + packageName + "." + implClsName);
    	ICompilationUnit clsUnit = generator.createJavaClass(packageName, 
    			implClsName, false, interfaceName, project);
    	if (clsUnit != null && infUnit !=null) {
	    	generator.addClsDefaultAnnoation(clsUnit, infUnit.findPrimaryType().getFullyQualifiedName());
	    }
    }
    
    private void createSCDL() {
    	componentName = compPage.getComponentName();
    	if (componentName == null || componentName.equals("")) {
    		return;
    	}
    	ScdlBuilder builder = new ScdlBuilder();
    	IFolder scdlFolder = ScaWorkspaceManager.getScdlFileFolder(project);
    	builder.setFilePath(scdlFolder.getRawLocation().toOSString());
		
		builder.setCompositeName(project.getName() + "Composite");
		
		builder.setCompName(compPage.getComponentName());
		builder.setCompLan(ScdlBuilder.Language.JAVA);
		builder.setCompClass(compPage.getImplFullName());
		
        builder.setServiceName(compPage.getInterfaceName());
        
        String nameSpace = CXFUtil.getNamespace(compPage.getPackage());
        if(nameSpace.endsWith("/")){
        	nameSpace = nameSpace.substring(0, nameSpace.length() - 1);
        }
        builder.setWSDLNameSpace(nameSpace);
        
        
        String bindingType = compPage.getBindingType();
        if (!bindingType.equals("none")) {
        	if (bindingType.equals(ScaComponentWizardPage.BINDING_RMI)) {
        		builder.setInterfaceType(ScdlBuilder.InfType.JAVA);
        		builder.setBindingType(ScdlBuilder.BindingType.RMI);
                builder.setBindingAttrs(rmiPage.getBindingAttrs());
                builder.setInterfaceName(compPage.getInterfaceFullName());
        	} else if (bindingType.equals(ScaComponentWizardPage.BINDING_WS)) {
        		builder.setInterfaceType(ScdlBuilder.InfType.WSDL);
        		builder.setUrl(wsPage.composite.txtURL.getText());
        		builder.setBindingType(ScdlBuilder.BindingType.WS);
                builder.setBindingAttrs(wsPage.getBindingAttrs());
                builder.setInterfaceName(compPage.getServiceName() + "PortType");
        	} 
        }
        
        builder.createScdlFile();
        	
    }

    public void init(IWorkbench workbench, IStructuredSelection selection) {
        // TODO Auto-generated method stub
    }
    
    

    @Override
	public boolean canFinish() {
    	IWizardPage page = this.getContainer().getCurrentPage();
    	if (page != null) {
    	    if (page instanceof WizardNewProjectCreationPage) {
    	    	return page.canFlipToNextPage();
    	    }
    	}
    	return super.canFinish();
	}

	/**
     * actually create the project
     * @param newMonitor
     * @throws InvocationTargetException
     */
    private IProject createProject(IProgressMonitor newMonitor) throws InvocationTargetException {

        IProgressMonitor checkedMonitor = (newMonitor != null) ? newMonitor
                                                               : new NullProgressMonitor();
        checkedMonitor.beginTask("", 100);
        
        IProgressMonitor createProjectMonitor = new SubProgressMonitor(checkedMonitor, 20);
        String runtimeType = runtimePage.getRuntimeType();
	    String runtimeID = runtimePage.getRuntimeID();
        return ScaWorkspaceManager.createProject(createProjectMonitor,
                                       genPage.getProjectName(),
                                       genPage.getLocationPath(),
                                       runtimeType,
                                       runtimeID);
                                       

    }
    
    
    private void createLaunchConfig(String componentName, String mainClassName) {
    	try {
    	    IJavaLaunchConfigGenerator generator = getJavaLaunchConfigGenerator();
    	    generator.createScaComponentLaunchConfig(project, componentName, mainClassName);
    	} catch (Exception e) {
    		e.printStackTrace();
    	}
    }
    
    private IJavaLaunchConfigGenerator getJavaLaunchConfigGenerator(){
    	IExtensionRegistry reg = Platform.getExtensionRegistry();
        IExtension[] exts = reg.getExtensionPoint(
       		 IJavaLaunchConfigGenerator.EXT_POINT_SC_RUNTIME_PROVIDER).getExtensions();
        if (exts == null || exts.length == 0) {
       	    return null;
        }
        
        IConfigurationElement[] extElts = exts[0].getConfigurationElements();
        
        for (int extIndex = 0; extIndex < extElts.length; extIndex++) {
            IConfigurationElement cfgElt = extElts[extIndex];
            if (cfgElt.getName().equals("javaLaunchConfigGenerator")) {
                try {
                    Object generatorObj = cfgElt.createExecutableExtension("class");
                    if (generatorObj instanceof IJavaLaunchConfigGenerator) {
                        return (IJavaLaunchConfigGenerator)generatorObj;
                    }
                } catch (CoreException e) {
                    LOG.error("can't create class", e);
                }
            }
        }
        return null;
    }
    
    private String getMainClassFullName(){
    	String rtnValue = null;
    	String interfaceName = compPage.getInterfaceName(); 
    	String packageName = compPage.getPackage();
    	if(interfaceName != null && interfaceName.trim().length() > 0){
    		if(packageName != null && packageName.trim().length() > 0){
    			rtnValue = packageName + "." + interfaceName + "Server";
    		}else{
    			rtnValue = interfaceName + "Server";
    		}
    		
    	}
    	return rtnValue;
    }
}
