package org.eclipse.stp.soas.internal.deploy.core;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.stp.soas.deploy.core.DeployCorePlugin;
import org.eclipse.stp.soas.deploy.core.DeploymentExtensionManager;
import org.eclipse.stp.soas.deploy.core.IDeployTarget;
import org.eclipse.stp.soas.deploy.core.ILogicalPackage;
import org.eclipse.stp.soas.deploy.core.IPackageConfiguration;
import org.eclipse.stp.soas.deploy.core.IPackageConstructor;
import org.eclipse.stp.soas.deploy.core.IPackageConstructorExtension;
import org.eclipse.stp.soas.deploy.core.IPackageCreationContext;
import org.eclipse.stp.soas.deploy.core.IPackageOutputDescriptor;
import org.eclipse.stp.soas.deploy.core.IPackageValidator;
import org.eclipse.stp.soas.deploy.core.IRuntimeDefinition;
import org.eclipse.stp.soas.deploy.core.IRuntimeType;
import org.eclipse.stp.soas.deploy.core.IServerType;
import org.eclipse.stp.soas.deploy.core.ISupportValidator;
import org.eclipse.stp.soas.deploy.core.ISupportedRuntimeType;
import org.eclipse.stp.soas.deploy.core.ISupportedServerType;
import org.eclipse.stp.soas.deploy.core.ISupportedTechnologyType;
import org.eclipse.stp.soas.deploy.core.ITechnologyType;
import org.eclipse.stp.soas.deploy.core.IVersion;
import org.eclipse.stp.soas.deploy.core.MatchRule;
import org.eclipse.stp.soas.deploy.core.ServerType;
import org.eclipse.stp.soas.deploy.core.Version;
import org.eclipse.stp.soas.deploy.core.utils.DeploymentUtil;
import org.eclipse.wst.server.core.IServer;
import org.eclipse.wst.server.core.ServerCore;

public class PackageConstructorExtension implements
		IPackageConstructorExtension {

	public static final String ATTR_ID = "id"; //$NON-NLS-1$
	public static final String ATTR_PROFILE = "profile"; //$NON-NLS-1$
	public static final String ATTR_CLASS = "class"; //$NON-NLS-1$
	public static final String ATTR_NAME = "label"; //$NON-NLS-1$	
	public static final String ELEM_SERVER = "supportedServer"; //$NON-NLS-1$
	public static final String ELEM_PACKAGE_CONSTRUCTOR = "packageConstructor"; //$NON-NLS-1$
	public static final String ELEM_SUPPORTED_TECHNOLOGY = "supportedTechnology"; //$NON-NLS-1$
	public static final String ELEM_SUPPORTED_SERVER = "supportedServer"; //$NON-NLS-1$
	public static final String ELEM_SUPPORTED_RUNTIME = "supportedRuntime"; //$NON-NLS-1$
	public static final String ATTR_SUPPORT_VALIDATOR = "supportValidator"; //$NON-NLS-1$
	public static final String ATTR_PACKAGE_VALIDATOR = "packageValidator"; //$NON-NLS-1$
	
	public static final String ATTR_TYPE = "type"; //$NON-NLS-1$
	public static final String ATTR_MIN_VERSION = "minVersion"; //$NON-NLS-1$
	public static final String ATTR_MAX_VERSION = "maxVersion"; //$NON-NLS-1$
	public static final String ATTR_MATCH_RULE = "matchRule"; //$NON-NLS-1$
	

	private IConfigurationElement mElement;
	private String mID;
	private String mName;
	private IPackageConstructor mConstructor;
	private ISupportValidator mSupportValidator;
	private IPackageValidator mPackageValidator;
	private ISupportedTechnologyType mSupportedTechnology;
	private List<ISupportedServerType> mSupportedServers;
	private List<ISupportedRuntimeType> mSupportedRuntime;

	public PackageConstructorExtension(IConfigurationElement element)
			throws PackageExtensionException {
		super();
		init(element);
	}

	public IConfigurationElement getConfigurationElement() {
		return mElement;
	}

	public String getID() {
		return mID;
	}

	public String getName() {
		return mName;
	}

	public IPackageOutputDescriptor createPackage(ILogicalPackage pkg,
			IPackageCreationContext context, IPackageConfiguration configuration)
			throws CoreException {
		IPackageOutputDescriptor retVal;
		initConstructor();
		if (mConstructor == null) {
			IStatus status = new Status(
					IStatus.ERROR,
					DeployCorePlugin.getDefault().getBundle().getSymbolicName(),
					-1,
					DeployCorePlugin
							.getDefault()
							.getResourceString(
									"DeployDriverExtension.error.couldNotLoadPackageConstructor"), null); //$NON-NLS-1$
			throw new CoreException(status);
		} else {
			retVal = mConstructor.createPackage(pkg, context, configuration);
		}

		return retVal;
	}

	public boolean supportsPackage(ILogicalPackage pkg) {
		boolean retVal;
		retVal = getSupportedTechnologyType().supportsTechnology(
				pkg.getTechnologyType());

		if (pkg.getRuntimeType() != null
				&& !this.supportsRuntimeType(pkg.getRuntimeType())) {
			retVal = false;
		}

		if (retVal) {
			initSupportValidator();
			if (mSupportValidator != null) {
				retVal = retVal
						&& mSupportValidator.supportsFile(pkg.getFile());
			}
		}
		return retVal;
	}

	public boolean supportsServer(IServerType serverType) {
		List serverTypes = getSupportedServerType();
		for (Iterator<ISupportedServerType> it = serverTypes.iterator(); it
				.hasNext();) {
			ISupportedServerType isst = it.next();
			if (isst.supportsServer(serverType)) {
				return true;
			}
		}
		return false;
	}


	public boolean supportsRuntimeType(IRuntimeType type) {
		 
		for(Iterator<ISupportedRuntimeType> it = getSupportedRuntimeType().iterator(); it.hasNext();){
			ISupportedRuntimeType isrt = it.next();
			if(isrt.supportsRuntime(type)){
				return true;
			}
		}
		return false;
	}

	public boolean supportsTechnology(ITechnologyType type) {
		// TODO Auto-generated method stub
		return false;
	}
	
	
	public IStatus[] validatePackage(ILogicalPackage pkg,
			IPackageConfiguration configuration, IDeployTarget target) {
		IStatus[] retVal;
		initPackageValidator();
		if (mPackageValidator == null) {
			retVal = new IStatus[0];
		} else {
			retVal = mPackageValidator.validate(pkg, configuration, target);
		}
		return retVal;
	}

	public ISupportedTechnologyType getSupportedTechnologyType() {
		return mSupportedTechnology;
	}

	public List<ISupportedRuntimeType> getSupportedRuntimeType() {
		return mSupportedRuntime;
	}

	public List<ISupportedServerType> getSupportedServerType() {
		return mSupportedServers;
	}

	private void init(IConfigurationElement element)
			throws PackageExtensionException {
		mElement = element;

		mID = element.getAttribute(ATTR_ID);
		mName = element.getAttribute(ATTR_NAME);

		if (mSupportedServers == null) {
			mSupportedServers = new ArrayList<ISupportedServerType>();
		}
		
		if (mSupportedRuntime == null) {
			mSupportedRuntime = new ArrayList<ISupportedRuntimeType>();
		}

		processTechnology(element);
		processServers(element);
		processRuntime(element);

	}

	private void processTechnology(IConfigurationElement element)
			throws PackageExtensionException {
		IConfigurationElement[] supportedTechnologies = element
				.getChildren(ELEM_SUPPORTED_TECHNOLOGY);
		if (supportedTechnologies.length < 1) {
			if (DeploymentExtensionManager.DEBUG_DEPLOYMENT_EXTENSION) {
				System.err
						.println(DeployCorePlugin
								.getDefault()
								.getResourceString(
										"DeployDriverExtension.trace.error.supportedTechnologyNotSpecified", //$NON-NLS-1$
										new Object[] { mID }));
				System.err.flush();
			}
			throw new PackageExtensionException(
					DeployCorePlugin
							.getDefault()
							.getResourceString(
									"DeployDriverExtension.exception.supportedTechnologyNotSpecified")); //$NON-NLS-1$
		} else if (supportedTechnologies.length > 1) {
			if (DeploymentExtensionManager.DEBUG_DEPLOYMENT_EXTENSION) {
				System.err
						.println(DeployCorePlugin
								.getDefault()
								.getResourceString(
										"DeployDriverExtension.trace.error.multipleSupportedTechnologyElements", //$NON-NLS-1$
										new Object[] { mID }));
				System.err.flush();
			}
			throw new PackageExtensionException(
					DeployCorePlugin
							.getDefault()
							.getResourceString(
									"DeployDriverExtension.exception.multipleSupportedTechnologyElements")); //$NON-NLS-1$
		}
		mSupportedTechnology = new SupportedTechnologyType(
				supportedTechnologies[0]);
	}

	private void processServers(IConfigurationElement element)
			throws PackageExtensionException {
		IConfigurationElement[] supportedServers = element
				.getChildren(ELEM_SUPPORTED_SERVER);
		if (supportedServers.length < 1) {
			//Todo:Johnson comment out the follow code.
			//during the soas refactor, we still want to support old deploy dirver/package constructor extensions.
			//the follow code will be added back once we stop to support old deploy driver
			/*
			if (DeploymentExtensionManager.DEBUG_DEPLOYMENT_EXTENSION) {
				System.err
						.println(DeployCorePlugin
								.getDefault()
								.getResourceString(
										"DeployDriverExtension.trace.error.supportedServerNotSpecified", //$NON-NLS-1$
										new Object[] { mID }));
				System.err.flush();
			}
			throw new PackageExtensionException(
					DeployCorePlugin
							.getDefault()
							.getResourceString(
									"DeployDriverExtension.exception.supportedServerNotSpecified")); //$NON-NLS-1$
			 */
		}
		for (int i = 0; i < supportedServers.length; i++) {
			try {
			    mSupportedServers.add(new SupportedServerType(supportedServers[i]));
			} catch (PackageExtensionException e) {
				//we need to make sure have at least one valid server
				if (mSupportedServers.size() == 0) {
					throw e;
				}
			}
		}

	}

	private void processRuntime(IConfigurationElement element)
			throws PackageExtensionException {
		IConfigurationElement[] supportedRuntimes = element
				.getChildren(ELEM_SUPPORTED_RUNTIME);
		if (supportedRuntimes.length < 1) {
			return;
			//Todo:Johnson comment out the follow code.
			//during the soas refactor, we still want to support old deploy dirver/package constructor extensions.
			//the follow code will be added back once we stop to support old deploy driver

			/*
			if (DeploymentExtensionManager.DEBUG_DEPLOYMENT_EXTENSION) {
				System.err
						.println(DeployCorePlugin
								.getDefault()
								.getResourceString(
										"DeployDriverExtension.trace.error.supportedRuntimeNotSpecified", //$NON-NLS-1$
										new Object[] { mID }));
				System.err.flush();
			}
			throw new PackageExtensionException(
					DeployCorePlugin
							.getDefault()
							.getResourceString(
									"DeployDriverExtension.exception.supportedRuntimeNotSpecified")); //$NON-NLS-1$
			 */
		} else if (supportedRuntimes.length > 1) {
			if (DeploymentExtensionManager.DEBUG_DEPLOYMENT_EXTENSION) {
				System.err
						.println(DeployCorePlugin
								.getDefault()
								.getResourceString(
										"DeployDriverExtension.trace.error.multipleSupportedRuntimeElements", //$NON-NLS-1$
										new Object[] { mID }));
				System.err.flush();
			}
			throw new PackageExtensionException(
					DeployCorePlugin
							.getDefault()
							.getResourceString(
									"DeployDriverExtension.exception.multipleSupportedRuntimeElements")); //$NON-NLS-1$
		}
		
		
		
		initSupportedRuntime(supportedRuntimes[0]);
	}

	private void initSupportedRuntime(IConfigurationElement rtElement) {
		MatchRule mr;
		String mRuntimeDef = rtElement.getAttribute(ATTR_TYPE);
		Version mMinVersion = Version.valueOf(rtElement
				.getAttribute(ATTR_MIN_VERSION));
		Version mMaxVersion = Version.valueOf(rtElement
				.getAttribute(ATTR_MAX_VERSION));
		String matchRule = rtElement.getAttribute(ATTR_MATCH_RULE);
		if(matchRule != null && !"".equals(matchRule)){
			mr = MatchRule.valueOf(rtElement.getAttribute(ATTR_MATCH_RULE));
		}
		else{
			mr = null;
		}
		List<IRuntimeDefinition> irds = DeploymentExtensionManager
				.getInstance().getRuntimeDefinitionByType(mRuntimeDef);
		for(Iterator<IRuntimeDefinition> it = irds.iterator(); it.hasNext();){
			IRuntimeDefinition ird = it.next(); 
			String runId = ird.getID();
			
			IVersion runVersion = DeploymentUtil.getSTPVersionByWTPRuntimeType(ServerCore.findRuntimeType(runId));
			if(runVersion.compareTo(mMinVersion) >= 0 && runVersion.compareTo(mMaxVersion) <= 0){
				mSupportedRuntime.add(new SupportedRuntimeType(runId, mMinVersion, mMaxVersion, mr));
			}
		}
		

	}
	private void initConstructor() {
		if (mConstructor == null) {
			try {
				mConstructor = (IPackageConstructor) mElement
						.createExecutableExtension(ATTR_CLASS);
			} catch (CoreException e) {
				if (DeploymentExtensionManager.DEBUG_DEPLOYMENT_EXTENSION) {
					System.err
							.println(DeployCorePlugin
									.getDefault()
									.getResourceString(
											"DeployDriverExtension.trace.error.couldNotLoadPackageConstructor", //$NON-NLS-1$
											new Object[] {
													getID(),
													mElement
															.getAttribute(ATTR_CLASS) }));
					System.err.flush();
				}
			}
		}
	}

	private void initSupportValidator() {
		if (mSupportValidator == null) {
			try {
				mSupportValidator = (ISupportValidator) mElement
						.createExecutableExtension(ATTR_SUPPORT_VALIDATOR);
			} catch (CoreException e) {
				if (DeploymentExtensionManager.DEBUG_DEPLOYMENT_EXTENSION) {
					System.err
							.println(DeployCorePlugin
									.getDefault()
									.getResourceString(
											"DeployDriverExtension.trace.error.couldNotLoadSupportValidator", //$NON-NLS-1$
											new Object[] {
													getID(),
													mElement
															.getAttribute(ATTR_SUPPORT_VALIDATOR) }));
					System.err.flush();
				}
			}
		}
	}

	private void initPackageValidator() {
		if (mPackageValidator == null) {
			try {
				mPackageValidator = (IPackageValidator) mElement
						.createExecutableExtension(ATTR_PACKAGE_VALIDATOR);
			} catch (CoreException e) {
				if (DeploymentExtensionManager.DEBUG_DEPLOYMENT_EXTENSION) {
					System.err
							.println(DeployCorePlugin
									.getDefault()
									.getResourceString(
											"DeployDriverExtension.trace.error.couldNotLoadPackageValidator", //$NON-NLS-1$
											new Object[] {
													getID(),
													mElement
															.getAttribute(ATTR_SUPPORT_VALIDATOR) }));
					System.err.flush();
				}
			}
		}
	}

	public IDeployTarget adaptServer(IServer server) {
		IDeployTarget idt;
		String serverTypeId = server.getServerType().getId();
		IServerType serverType = new ServerType(serverTypeId, DeploymentUtil
				.getSTPVersionFromWTPServer(server));

		if (this.supportsServer(serverType)) {
			idt = new DeployTarget(this, server);
		} else {
			idt = null;
		}
		return idt;
	}


}