/**********************************************************************
 * Copyright (c) 2005 Scapa Technologies Limited and others
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * Scapa Technologies Limited - Initial API and implementation
 **********************************************************************/

package org.eclipse.stp.b2j.core.jengine.internal.core.api;

import org.eclipse.stp.b2j.core.jengine.internal.api.Program;
import org.eclipse.stp.b2j.core.jengine.internal.core.datapool.SharedHashMap;
import org.eclipse.stp.b2j.core.jengine.internal.core.datapool.SharedVariable;
import org.eclipse.stp.b2j.core.jengine.internal.core.sync.SharedBarrier;
import org.eclipse.stp.b2j.core.jengine.internal.core.sync.SharedMutex;
import org.eclipse.stp.b2j.core.jengine.internal.core.sync.SharedSemaphore;
import org.eclipse.stp.b2j.core.jengine.internal.message.Message;

/**
 * 
 * @author amiguel
 * 
 * Part of the generic engine client API, used to represent an engine Controller 
 * (kind of like a client) which can run an engine Program.
 */
public interface ControllerInterface {

	/**
	 * Set the general logging level in the engine
	 * @param error turn on error logging
	 * @param warning turn on warning logging
	 * @param info turn on info logging
	 * @throws Exception if an error occurs while making this transaction on the engine instance
	 */
	public void setLogLevel(boolean error, boolean warning, boolean info) throws Exception;

	/**
	 * Set whether the engine should keep running when the client terminates the connection
	 * @param headless whether the engine should keep running when the client terminates the connection
	 * @throws Exception if an error occurs while making this transaction on the engine instance
	 */
	public void setHeadless(boolean headless) throws Exception;
	
	/**
	 * Add a trace listener to listen for printing and tracing events from this engine instance
	 * @param listener the TraceListener to receive events from the engine
	 */
	public void addTraceListener(TraceListener listener);
	
	/**
	 * Remove a trace listener from this engine instance
	 * @param listener the TraceListener to remove from the engine
	 */
	public void removeTraceListener(TraceListener listener);
	
	/**
	 * Set the Program this engine should run (NOTE an engine instance can run only ONE program)
	 * @param p the Program this engine will run
	 * @return the clock offset in milliseconds for any relative timestamps returned by the engine
	 * @throws Exception if an error occurs while making this transaction on the engine instance
	 */
	public long setProgram(Program p) throws Exception;
	
	/**
	 * Launch a Runner on the engine to run part of the engine Program
	 * @param count the number of runners to launch
	 * @param method the method in the Program to run
	 * @param host the host to launch the Runner(s) on
	 * @return a Message containing a number of Long objects representing the IDs of the launched runners
	 * @throws Exception if an error occurs while making this transaction on the engine instance
	 */
	public Message launchRunner(int count, String method, int host_index) throws Exception;
	
	/**
	 * Launch a Runner on the engine to run part of the engine Program
	 * @param count the number of runners to launch
	 * @param method the method in the Program to run
	 * @param host the host to launch the Runner(s) on
	 * @param args an array of String arguments to pass to this runner
	 * @return a Message containing a number of Long objects representing the IDs of the launched runners
	 * @throws Exception if an error occurs while making this transaction on the engine instance
	 */
	public Message launchRunner(int count, String method, int host_index, String[] args) throws Exception;

	/**
	 * Have the Runners trace any stack information they have built up
	 * @throws Exception if an error occurs while making this transaction on the engine instance
	 */
	public void doRunnerStackDump() throws Exception;
	
	/**
	 * Have the Runners trace any stack information they have built up
	 * @param stack_message_id any traced stack messages will be given a message type of stack_message_id
	 * @throws Exception if an error occurs while making this transaction on the engine instance
	 */
	public void doRunnerStackTrace(int stack_message_id) throws Exception;
	
	/**
	 * Wait for the Runner with the specified ID to complete
	 * @param remote_id the ID of the Runner to join with
	 * @throws Exception if an error occurs while making this transaction on the engine instance
	 */
	public void joinRunner(Long remote_id) throws Exception;
	
	/**
	 * Forcibly close this connecion to the engine instance
	 */
	public void closeConnection();

	/**
	 * Terminate this engine instance
	 * @throws Exception if an error occurs while making this transaction on the engine instance
	 */
	public void terminate() throws Exception;
	
	public String[] getHosts() throws Exception;
	
	/**
	 * Get a list of all the variables containde within the engine
	 * @return an array containing the names of all variables within the engine
	 * @throws Exception if an error occurs while making this transaction on the engine instance
	 */
	public String[] getVariableNames() throws Exception;

	public SharedVariable getVariable(String name) throws Exception;
	public SharedSemaphore getSemaphore(String name) throws Exception;
	public SharedHashMap getHashMap(String name) throws Exception;
	public SharedBarrier getBarrier(String name) throws Exception;
	public SharedMutex getMutex(String name) throws Exception;

	public SharedVariable newVariable(String name, int type, boolean dirty) throws Exception;
	public SharedSemaphore newSemaphore(String name, int initial) throws Exception;
	public SharedHashMap newHashMap(String name) throws Exception;
	public SharedBarrier newBarrier(String name, int size) throws Exception;
	public SharedMutex newMutex(String name) throws Exception;
}