/**********************************************************************
 * Copyright (c) 2005 Scapa Technologies Limited and others
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * Scapa Technologies Limited - Initial API and implementation
 **********************************************************************/

package org.eclipse.stp.b2j.core.jengine.internal.extensions.wsdlbinding.wsif;

import java.util.ArrayList;
import java.util.HashMap;

import org.eclipse.stp.b2j.core.jengine.internal.compiler.TranslatorLog;
import org.eclipse.stp.b2j.core.jengine.internal.compiler.Util;
import org.eclipse.stp.b2j.core.jengine.internal.compiler.wsdlmap.WSDLMap;
import org.eclipse.stp.b2j.core.jengine.internal.compiler.wsdlmap.WSDLMessage;
import org.eclipse.stp.b2j.core.jengine.internal.compiler.wsdlmap.WSDLOperation;
import org.eclipse.stp.b2j.core.jengine.internal.compiler.wsdlmap.WSDLPortType;
import org.eclipse.stp.b2j.core.jengine.internal.compiler.xmlns.NamespaceException;
import org.eclipse.stp.b2j.core.jengine.internal.compiler.xmlns.NamespaceTranslator;
import org.eclipse.stp.b2j.core.jengine.internal.compiler.xsdmap.XSDMap;
import org.eclipse.stp.b2j.core.publicapi.extension.wsdlbinding.WSDLBindingTranslator;
import org.eclipse.stp.b2j.core.publicapi.extension.wsdlbinding.WSDLBindingTranslatorException;
import org.eclipse.stp.b2j.core.publicapi.extension.wsdlbinding.XSDTypeTranslator;
import org.eclipse.stp.b2j.core.xml.internal.w3c.Element;

/**
 * 
 * @author amiguel
 *
 * A translator which translates WSIF Java binding specific information in WSDL
 * files into parts of an engine Program
 */
public class WSIFJavaBindingTranslator implements WSDLBindingTranslator {

public static final String NAMESPACE_JAVABINDING = "http://schemas.xmlsoap.org/wsdl/java/";
	
public static int METHOD_TYPE_INSTANCE = 0;
public static int METHOD_TYPE_STATIC = 1;
public static int METHOD_TYPE_CONSTRUCTOR = 2;

private static final String LOG_SOURCE = "Java Binding: ";

private static int XSDID = 0;

private static String getNextXsdId() {
	return "wsifjava_xsdtype_"+XSDID++;
}

private static String getNextPortId() {
	return "wsifjava_port_"+XSDID++;
}

class Binding {
	String qname;
	String qportType;
	HashMap operations = new HashMap();
	
	String bindingName;
	
	String portName;
	String className;
	String classPath;
	String classLoader;
}
class Operation {
	String operationName;
	String methodName;
	String[] parameterOrder;
	String returnPart;
	int methodType;
}

String portHolderType = WSIFJavaBinding.class.getName();
String portHolder = "org_eclipse_stp_b2j_core_jengine_internal_extensions_wsif_JavaBinding";
	
WSIFFormatTypeCodec codec_formatType;
XSDTypeTranslator[] codecs;

HashMap bindings = new HashMap();
XSDMap xsdmap;
WSDLMap wsdlmap;

Util compiler_util;
TranslatorLog log;

ClassLoader dependancy_loader;

	public String getID() {
		return getClass().getName();
	}
	
	public String getHumanReadableName() {
		return "Default Java Binding";
//		return B2jPlugin.getString("BINDING_TYPE_WSIF");
	}


//	public String getProviderPluginName() {
//		return B2jPlugin.getDefault().getBundle().getSymbolicName();
//	}
	
	public void init(Util compiler_util, TranslatorLog log, XSDMap xsdmap, WSDLMap wsdlmap, ClassLoader dependancy_loader) {
		this.xsdmap = xsdmap;
		this.wsdlmap = wsdlmap;
		this.dependancy_loader = dependancy_loader;
		this.compiler_util = compiler_util;
		this.log = log;
		
		codec_formatType = new WSIFFormatTypeCodec();
	
		codecs = new XSDTypeTranslator[] {
			codec_formatType,	
		};
	}
	
	private boolean isBindingOurs(NamespaceTranslator nt, Element elem) throws NamespaceException {
		ArrayList elems = Util.getAllElements(elem);
		for (int i = 0; i < elems.size(); i++) {
			Element jbinding = (Element)elems.get(i);
			nt.addNamespaces(jbinding);
			if (nt.checkQName(jbinding,NAMESPACE_JAVABINDING,"binding")) {
				return true;
			}
			nt.removeNamespaces(jbinding);
		}
		return false;
	}
	
	private void readOperations(NamespaceTranslator nt, Binding binding, Element elem) throws NamespaceException, WSDLBindingTranslatorException {
		ArrayList wsops = Util.getAllElements(elem);
		for (int i = 0; i < wsops.size(); i++) {
			Element wsop = (Element)wsops.get(i);
			nt.addNamespaces(wsop);
			
			if (NamespaceTranslator.getName(wsop).equals("operation")) {
			
				ArrayList jops = Util.getAllElements(wsop);
				for (int k = 0; k < jops.size(); k++) {
					Element jop = (Element)jops.get(k);
					nt.addNamespaces(jop);
					if (nt.checkQName(jop,NAMESPACE_JAVABINDING,"operation")) {
						
						Operation op = new Operation();
						op.operationName = wsop.getAttribute("name");
						op.methodName = jop.getAttribute("methodName");

						WSDLPortType wsdlpt = wsdlmap.getPortType(binding.qportType);
						WSDLOperation wsdlop = wsdlpt.getOperation(op.operationName);
						if (wsdlop == null) {
							throw new WSDLBindingTranslatorException("WSIF Java Binding specified operation "+op.operationName+" on portType "+compiler_util.getQNameFromQName(binding.qportType)+" not found");
						}
						
						if (op.methodName == null) {
							throw new WSDLBindingTranslatorException("methodName attribute is required for WSIF Java binding operation");
						}
						if (op.methodName.length() == 0) {
							throw new WSDLBindingTranslatorException("methodName attribute is required for WSIF Java binding operation");
						}
						
						op.parameterOrder = Util.split(jop.getAttribute("parameterOrder"));
						op.returnPart = jop.getAttribute("returnPart");
						
						WSDLMessage inputmsg = wsdlmap.getMessage(wsdlop.getMessageType("input"));
						for (int z = 0; z < op.parameterOrder.length; z++) {
							if (inputmsg.getPartType(op.parameterOrder[z]) == null) {
								throw new WSDLBindingTranslatorException("WSIF Java Binding found no part \""+op.parameterOrder[z]+"\" as specified in parameter list of portType "+compiler_util.getQNameFromQName(binding.qportType)+", operation "+op.operationName);
							}
						}

						boolean returnRequired = wsdlop.getType() == WSDLOperation.TYPE_REQUEST_RESPONSE;
						
						if (op.returnPart == null && returnRequired) {
							throw new WSDLBindingTranslatorException("WSIF Java Binding specified no returnPart attribute but WSDL operation type is request-response");
						}
						if (op.returnPart.length()==0 && returnRequired) {
							log.logWarning(LOG_SOURCE,"WSIF Java Binding return type required but no return type specified");
							returnRequired = false;
//							throw new BindingTranslatorException("WSIF Java Binding specified no returnPart attribute but WSDL operation type is request-response");
						}
						
						if (returnRequired) {
							WSDLMessage outputmsg = wsdlmap.getMessage(wsdlop.getMessageType("output"));
							if (outputmsg.getPartType(op.returnPart) == null) {
								throw new WSDLBindingTranslatorException("WSIF Java Binding found no part \""+op.returnPart+"\" in message "+compiler_util.getQNameFromQName(wsdlop.getMessageType("output"))+" as specified in return list of portType "+compiler_util.getQNameFromQName(binding.qportType)+", operation "+op.operationName);
							}
						}
						
						String mtype = jop.getAttribute("methodType");
						if (mtype == null) mtype = "";
						if (mtype.length() > 0) {
							if (mtype.equals("instance")) {
								op.methodType = METHOD_TYPE_INSTANCE;
							} else if (mtype.equals("static")) {
								op.methodType = METHOD_TYPE_STATIC;
							} else if (mtype.equals("constructor")) {
								op.methodType = METHOD_TYPE_CONSTRUCTOR;
							} else {
								throw new WSDLBindingTranslatorException("methodType attribute must be one of 'instance', 'static' or 'constructor'");
							}
						} else {
							op.methodType = METHOD_TYPE_INSTANCE;
						}
						
						//add this operation mapping to the binding
						binding.operations.put(op.operationName,op);
					}
					nt.removeNamespaces(jop);
				}

			}
			
			nt.removeNamespaces(wsop);
		}
	}
	
	private void readEPR(NamespaceTranslator nt, Element port, Binding binding) throws NamespaceException, WSDLBindingTranslatorException {
		String portName = port.getAttribute("name");
		ArrayList jports = Util.getAllElements(port);
		for (int i = 0; i < jports.size(); i++) {
			Element jport = (Element)jports.get(i);
			nt.addNamespaces(jport);
			if (nt.checkQName(jport,NAMESPACE_JAVABINDING,"address")) {
				binding.portName = portName;
				binding.className = jport.getAttribute("className");
				binding.classPath = jport.getAttribute("classPath");
				binding.classLoader = jport.getAttribute("classLoader");
				
				if (binding.className == null) {
					throw new WSDLBindingTranslatorException("WSIF Java Binding service port binding specified but no className attribute - className attribute is required");
				}
				if (binding.className.length() == 0) {
					throw new WSDLBindingTranslatorException("WSIF Java Binding service port binding specified but no className attribute - className attribute is required");
				}
			}
			nt.removeNamespaces(jport);
		}
	}
	
	public void readWSDL(Element wsdl_definitions) throws NamespaceException, WSDLBindingTranslatorException {
		NamespaceTranslator nt = compiler_util.createNamespaceTranslator();
		nt.addNamespaces(wsdl_definitions);
		
		ArrayList elems = Util.getAllElements(wsdl_definitions);
		for (int i = 0; i < elems.size(); i++) {
			Element elem = (Element)elems.get(i);
			nt.addNamespaces(elem);

			if (NamespaceTranslator.getName(elem).equals("binding")) {
				//we've found a binding
				//get the port type
				String name = elem.getAttribute("name");
				String ptype = elem.getAttribute("type");
				if (isBindingOurs(nt,elem)) {
					Binding binding = new Binding();
					binding.qname = nt.qualify(name,true);
					binding.qportType = nt.qualify(ptype,false);
					binding.bindingName = nt.qualify(name,true);
					
					WSDLPortType wsdlpt = wsdlmap.getPortType(binding.qportType);
					if (wsdlpt == null) throw new WSDLBindingTranslatorException("WSIF Java Binding specified portType "+compiler_util.getQNameFromQName(binding.qportType)+" not found");
					
					readOperations(nt,binding,elem);
					
					//add two entries, one referencing it via the port type, one via the binding type
					bindings.put(binding.qportType,binding);
					bindings.put(binding.qname,binding);
					bindings.put(binding.bindingName,binding);
				}
			}
			
			nt.removeNamespaces(elem);
		}
		for (int i = 0; i < elems.size(); i++) {
			Element elem = (Element)elems.get(i);
			nt.addNamespaces(elem);

			if (NamespaceTranslator.getName(elem).equals("service")) {
				ArrayList ports = Util.getAllElements(elem);
				for (int k = 0; k < ports.size(); k++) {
					Element port = (Element)ports.get(k);
					nt.addNamespaces(port);
					
					if (NamespaceTranslator.getName(port).equals("port")) {

						String portName = port.getAttribute("name");
						String qbindingType = nt.qualify(port.getAttribute("binding"),false);
						
						Binding binding = (Binding)bindings.get(qbindingType);
						
						if (binding != null) {
							//its one of ours
							readEPR(nt,port,binding);
						}
					}
					
					nt.removeNamespaces(port);
				}
			}
			
			nt.removeNamespaces(elem);
		}
	}

	public XSDTypeTranslator[] getCodecs() {
		return codecs;
	}
	
	private Binding checkForBindingNamed(String name) throws WSDLBindingTranslatorException {
		if (name == null) return null;
		Binding binding = (Binding)bindings.get(name);
		if (binding == null) {
			throw new WSDLBindingTranslatorException("no WSIF Java binding found named '"+compiler_util.getQNameFromQName(name)+"'");
		}
		return binding;
	}

	private Binding checkForBinding(String qportType) throws WSDLBindingTranslatorException {
		Binding binding = (Binding)bindings.get(qportType);
		if (binding == null) {
			throw new WSDLBindingTranslatorException("no WSIF Java binding found for port type "+compiler_util.getQNameFromQName(qportType));
		}
		return binding;
	}
	
	public String translateInvocation(NamespaceTranslator nt, String qportType, String bindingName, String operation, String eprFieldName, String correlationSetFieldNameIn, String correlationSetFieldNameOut, String inputFieldName, String outputFieldName) throws NamespaceException, WSDLBindingTranslatorException {
		StringBuffer sb = new StringBuffer();

		checkForBindingNamed(bindingName);
		Binding binding = checkForBinding(qportType);
		
		//get the mapped java operation
		Operation jop = (Operation)binding.operations.get(operation);
		if (jop == null) throw new WSDLBindingTranslatorException("java operation mapping not found for portType "+compiler_util.getQNameFromQName(qportType)+", operation "+operation+"");
		
		//get references to the WSDL messages and operations we're using
		WSDLOperation wsop = wsdlmap.getPortType(qportType).getOperation(operation);
		String qinputType = wsop.getMessageType("input");
		String qoutputType = wsop.getMessageType("output");
		
		WSDLMessage inputMessage = wsdlmap.getMessage(qinputType);
		WSDLMessage outputMessage = wsdlmap.getMessage(qoutputType);
		
		boolean returnRequired = false;
		
		if (wsop.getType() == WSDLOperation.TYPE_REQUEST_RESPONSE) {
			returnRequired = true;
		} else if (wsop.getType() == WSDLOperation.TYPE_ONEWAY) {
			returnRequired = false;
		} else {
			throw new WSDLBindingTranslatorException("WSIF Java Binding Translator - only WSDL operations of type One-way or Request-response are supported");
		}

		if (inputMessage == null) throw new WSDLBindingTranslatorException("WSIF Java Binding Translator - input message required for operation "+operation+" in port type "+compiler_util.getQNameFromQName(qportType));
		if (outputMessage == null && returnRequired) throw new WSDLBindingTranslatorException("WSIF Java Binding Translator - output message required for operation "+operation+" in port type "+compiler_util.getQNameFromQName(qportType));
		
		//get the XSD types of the parameter parts and the return part
		String[] paramTypes = new String[jop.parameterOrder.length];
		for (int i = 0; i < paramTypes.length; i++) {
			paramTypes[i] = inputMessage.getPartType(jop.parameterOrder[i]);
		}
		String returnType = null;
		String returnName = null;
		if (returnRequired) {
			returnType = outputMessage.getPartType(jop.returnPart);
			returnName = getNextXsdId();
		}

		if (inputFieldName == null) {
			throw new WSDLBindingTranslatorException("WSIF Java Binding Translator - input variable required for operation "+operation+" in port type "+compiler_util.getQNameFromQName(qportType));
		}
		if (inputFieldName.length() == 0) {
			throw new WSDLBindingTranslatorException("WSIF Java Binding Translator - input variable required for operation "+operation+" in port type "+compiler_util.getQNameFromQName(qportType));
		}
		
		if (returnRequired) {
			if (outputFieldName == null) {
				throw new WSDLBindingTranslatorException("WSIF Java Binding Translator - output variable required for operation "+operation+" in port type "+compiler_util.getQNameFromQName(qportType));
			}
			if (outputFieldName.length() == 0) {
				throw new WSDLBindingTranslatorException("WSIF Java Binding Translator - output variable required for operation "+operation+" in port type "+compiler_util.getQNameFromQName(qportType));
			}
		}
		
		String paramsName = getNextXsdId();
		
		sb.append("Object[] ");
		sb.append(paramsName);
		sb.append(" = ");
		sb.append("new Object[] {\n");
		for (int i = 0; i < paramTypes.length; i++) {
			String toJavaFunction = codec_formatType.getToJavaFunction(paramTypes[i],qportType);
			sb.append(paramTypes[i]);
			sb.append(".");
			sb.append(toJavaFunction);
			sb.append("(");
			sb.append(inputFieldName);
			sb.append(".");
			sb.append(NamespaceTranslator.getElement(jop.parameterOrder[i]));
			sb.append("[0]");
			sb.append(")");
			sb.append(",\n");
		}
		sb.append("};\n");

		String portName = getNextPortId();
		
		sb.append(WSIFJavaPort.class.getName()+" ");
		sb.append(portName);
		sb.append(" = ("+WSIFJavaPort.class.getName()+")");
		sb.append(portHolder);
		sb.append(".getPortGroupForEndpointReference("+eprFieldName+","+correlationSetFieldNameOut+")");
		sb.append(".getPort();\n");
		
		if (returnRequired && returnType == null) {
			log.logWarning(LOG_SOURCE,"WSIF Java Binding return type required but no return type specified");
			returnRequired = false;
		}
		
		if (returnRequired) {
			//invoke
			sb.append("Object "+returnName+" = "+portName+".invoke(\""+jop.methodName+"\","+paramsName+");\n");

			//copy the output into the variable field
			String fromJavaFunction = codec_formatType.getFromJavaFunction(returnType,qportType);
			sb.append(outputFieldName+"."+NamespaceTranslator.getElement(jop.returnPart)+" = new "+returnType+"[1];\n");
			sb.append(outputFieldName+"."+NamespaceTranslator.getElement(jop.returnPart)+"[0] = ("+returnType+") "+returnType+"."+fromJavaFunction+"(("+codec_formatType.getJavaType(returnType,qportType)+")"+returnName+");\n");
			
		} else {
			//invoke
			sb.append(portName+".invoke(\""+jop.methodName+"\","+paramsName+");\n");
			
			//no output
		}

		return sb.toString();
	}
	public String translateReceive(NamespaceTranslator nt, String qportType, String bindingName, String operation, String messageExchange, String eprFieldName, String correlationSetFieldName, String inputFieldName) throws NamespaceException, WSDLBindingTranslatorException {
		throw new WSDLBindingTranslatorException("WSIF Java Binding: cannot translate <receive> activity for a Java port");
	}
	public String translateReply(NamespaceTranslator nt, String qportType, String bindingName, String operation, String messageExchange, String eprFieldName, String correlationSetFieldName, String outputFieldName) throws NamespaceException, WSDLBindingTranslatorException {
		throw new WSDLBindingTranslatorException("WSIF Java Binding: cannot translate <reply> activity for a Java port");
	}
	public String translateDefaultEPR(NamespaceTranslator nt, String qportType, String bindingName) throws NamespaceException, WSDLBindingTranslatorException {
		StringBuffer sb = new StringBuffer();

		checkForBindingNamed(bindingName);
		Binding binding = checkForBinding(qportType);
		
		if (bindingName != null) {
			//use a specific binding (and therefore use a specific binding's EPR)
			binding = (Binding)bindings.get(bindingName);
		}
		
		if (binding.className == null) {
			throw new WSDLBindingTranslatorException("no default classname found for WSIF Java Binding of portType "+compiler_util.getQNameFromQName(qportType));
		}
		
		sb.append("new WSEndpointReference(\""+binding.className+"\")");

		return sb.toString();
	}
	
	public void translatePickStart(int onMessageCount, int onAlarmCount, String timeoutFieldLongMs) throws NamespaceException, WSDLBindingTranslatorException {
		throw new WSDLBindingTranslatorException("WSIF Java Binding: cannot translate <pick> activity for a Java port");
	}
	public void translatePickOnMessage(NamespaceTranslator nt, String qportType, String bindingName, String operation, String messageExchange, String eprFieldName, String correlationSetFieldName, String inputFieldName) throws NamespaceException, WSDLBindingTranslatorException {
		throw new WSDLBindingTranslatorException("WSIF Java Binding: cannot translate <pick> activity for a Java port");
	}
//	public void translatePickOnAlarm(String timeoutFieldLongMs) throws NamespaceException, BindingTranslatorException {
//		throw new BindingTranslatorException("WSIF Java Binding: cannot translate <pick> activity for a Java port");
//	}
	public String translatePickFinish(String indexFieldName)  throws NamespaceException, WSDLBindingTranslatorException {
		throw new WSDLBindingTranslatorException("WSIF Java Binding: cannot translate <pick> activity for a Java port");
	}
	
	
	public String getDeclarations() {
		//create a static copy of the WSIF Java binding
		StringBuffer decl = new StringBuffer();
		decl.append("static ");
		decl.append(portHolderType);
		decl.append(" ");
		decl.append(portHolder);
		decl.append(" = new ");
		decl.append(portHolderType);
		decl.append("();\n");
		return decl.toString();
	}
	public String[] getImports() {
		return new String[] {
				portHolderType,
		};
	}
}