/**********************************************************************
 * Copyright (c) 2005 Scapa Technologies Limited and others
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * Scapa Technologies Limited - Initial API and implementation
 **********************************************************************/

package org.eclipse.stp.b2j.core.jengine.internal.mainengine.api;

import java.io.IOException;
import java.util.ArrayList;

import org.eclipse.stp.b2j.core.jengine.internal.api.Program;
import org.eclipse.stp.b2j.core.jengine.internal.core.PrintHandler;
import org.eclipse.stp.b2j.core.jengine.internal.core.Runner;
import org.eclipse.stp.b2j.core.jengine.internal.core.api.ControllerInterface;
import org.eclipse.stp.b2j.core.jengine.internal.core.api.TraceListener;
import org.eclipse.stp.b2j.core.jengine.internal.mainengine.Controller;
import org.eclipse.stp.b2j.core.jengine.internal.mainengine.ControllerConnection;
import org.eclipse.stp.b2j.core.jengine.internal.mainengine.NonCriticalThreadGroup;
import org.eclipse.stp.b2j.core.jengine.internal.mainengine.TransactionFactory;
import org.eclipse.stp.b2j.core.jengine.internal.message.Message;
import org.eclipse.stp.b2j.core.jengine.internal.message.MessageReader;
import org.eclipse.stp.b2j.core.jengine.internal.message.MessageWriter;
import org.eclipse.stp.b2j.core.jengine.internal.multiplex.MultiplexerInputStream;
import org.eclipse.stp.b2j.core.jengine.internal.multiplex.MultiplexerOutputStream;
import org.eclipse.stp.b2j.core.jengine.internal.transport.session.Session;
import org.eclipse.stp.b2j.core.jengine.internal.transport.session.SessionFactory;
import org.eclipse.stp.b2j.core.jengine.internal.utils.Logger;
import org.eclipse.stp.b2j.core.publicapi.JARDependency;
import org.eclipse.stp.b2j.core.publicapi.transport.session.SessionAddress;

/**
 * 
 * @author amiguel
 *
 * The distributed engine implementation of a Controller Connection (engine client)
 */
public class ControllerConnector extends ControllerConnection implements ControllerInterface {

long clock = 0;

//Socket[] sockets;
Session session;

//String host;
//int port;

MessageReader notify_in;
MessageWriter notify_out;

TraceListener tlistener;

Object tracelisteners_LOCK = new Object();
ArrayList tracelisteners = new ArrayList();

	public void addTraceListener(TraceListener listener) {
		synchronized(tracelisteners_LOCK) {
			tracelisteners.add(listener);
		}
	}
	public void removeTraceListener(TraceListener listener) {
		synchronized(tracelisteners_LOCK) {
			tracelisteners.remove(listener);
		}
	}

	public ControllerConnector(SessionAddress session_address, TraceListener nlisten) throws Exception {
		super(0);
		this.tlistener = nlisten;

		session = SessionFactory.newSession(session_address,true);
		session.begin();

		MultiplexerInputStream mxin = new MultiplexerInputStream(session.getInputStream(Session.DEFAULT_STREAM_INDEX));
		MultiplexerOutputStream mxout = new MultiplexerOutputStream(session.getOutputStream(Session.DEFAULT_STREAM_INDEX));
		
		short server = 0;
		short client = 1;
		short notify = 2;

		ts = TransactionFactory.getTransactionServer(mxin.getInputStream(server),mxout.getOutputStream(server),this,new NonCriticalThreadGroup(this),"JEngine API [Controller Server Reader] Thread ("+session_address+")");
		tc = TransactionFactory.getTransactionClient(mxin.getInputStream(client),mxout.getOutputStream(client),new NonCriticalThreadGroup(this),"JEngine API [Controller Client Reader] Thread ("+session_address+")");

		notify_in = new MessageReader(mxin.getInputStream(notify),new NonCriticalThreadGroup(),"JEngine API [Controller Notify Reader] Thread ("+session_address+")");
		notify_out = new MessageWriter(mxout.getOutputStream(notify));
		
		Message m = notify_in.read();
		id = m.getType();
		
		Reader reader = new Reader(notify_in,new NonCriticalThreadGroup());
		//reader.setDaemon(true);
		reader.start();
		
	}

	public void closeConnection() {

		try {
			terminate();
//			Message m = new Message(Controller.CONTROLLER_ABORT);
//			m = tc.doTransaction(m);
		} catch (Exception e) {
			Logger.warning("unable to do controller abort transaction",e);
		}
//		for (int i = 0; i < sockets.length; i++) {
			try {
				session.end();
//				sockets[i].close();
			} catch (Exception e) {
			}
//		}	
	}

	public long setProgram(Program p) throws Exception {
		Message m;
		JARDependency[] deps = p.getDependencies();
		
		//
		// Set program
		//		
		m = new Message(Controller.CONTROLLER_SET_PROGRAM);
		m.append(p.writeProgram());
		
		Message cacheCheck = new Message(Controller.CONTROLLER_CHECK_DEPS_CACHE);
		for (int i = 0; i < deps.length; i++) {
			cacheCheck.append(deps[i].getCacheKey());
		}
		cacheCheck = tc.doTransaction(cacheCheck);
		if (cacheCheck.getType() != Controller.CONTROLLER_CHECK_DEPS_CACHE_OK) {
			throw new Exception("Controller check dependencies cache failed "+cacheCheck);			
		}  
		
		//
		// Send dependencies
		//
		Message dependencies = new Message();
		for (int i = 0; i < deps.length; i++) {
			String key = (String)cacheCheck.get(i);
			if (key == null) {
				System.out.println("Client Dependency "+i+" of "+deps.length+" - actual JAR");
				//append the JAR
				dependencies.append(JARDependency.toMessage(deps[i]));
			} else {
				System.out.println("Client Dependency "+i+" of "+deps.length+" - cache key "+key+" ("+deps[i].size()+")");
				//append the Cache Key
				dependencies.append(key);
			}
		}
		m.append(dependencies);

		System.out.println("\nSetting Engine Program ("+((byte[])m.get(0)).length/1024+" k)");
		long pt = System.currentTimeMillis();
		
		m = tc.doTransaction(m);

		pt = System.currentTimeMillis()-pt;
		System.out.println("\nSet Engine Program ("+(pt/1000)+"s)");
		
		if (m.getType() != Controller.CONTROLLER_SET_PROGRAM_OK) {
			throw new Exception("Controller set program failed "+m);			
		}  

		//
		// Sync clock
		//

		//find out how long a transaction takes
		long t = System.currentTimeMillis();
			m = new Message(Controller.CONTROLLER_EMPTY);
			m = tc.doTransaction(m);
		t = System.currentTimeMillis()-t;
		t = t/2;	//divide it by two (expect same delay up as down)

		//set up the message with the network offset
		m = new Message(Controller.CONTROLLER_SYNC_CLOCK);
		m.append(t);

		//reset the local clock and do the transaction
		clock = System.currentTimeMillis();
		m = tc.doTransaction(m);

		if (m.getType() != Controller.CONTROLLER_SYNC_CLOCK_OK) {
			throw new Exception("Controller sync clock failed "+m);			
		}  
//Logger.info("CLIENT CLOCK = "+clock);		

		return clock;  
	}

	public void setHeadless(boolean headless) throws Exception {
		Message m = new Message(Controller.CONTROLLER_SET_HEADLESS);
		
		if (headless) {
			m.append(1);
		} else {
			m.append(0);
		}
		
		m = tc.doTransaction(m);
		
		if (m.getType() != Controller.CONTROLLER_SET_HEADLESS_OK) {
			throw new Exception("Controller set headless failed "+m);			
		}  
	}
	
	public void setLogLevel(boolean error, boolean warning, boolean info) throws Exception {
		Logger.PRINT_ERROR = error;
		Logger.PRINT_WARNING = warning;
		Logger.PRINT_INFO = info;
		
		Message m = new Message(Controller.CONTROLLER_SET_LOG_LEVEL);
		if (error) {
			m.append(1);
		} else {
			m.append(0);
		}
		if (warning) {
			m.append(1);
		} else {
			m.append(0);
		}
		if (info) {
			m.append(1);
		} else {
			m.append(0);
		}
		
		m = tc.doTransaction(m);

		if (m.getType() != Controller.CONTROLLER_SET_LOG_LEVEL_OK) {
			throw new Exception("Controller set log level failed "+m);			
		}  
		
	}
	
/*
	public long runProgram(Program p) throws Exception {
		
		Message m;
		
		//
		// Launch program
		//
		m = new Message(Controller.CONTROLLER_LAUNCH_PROGRAM);

		m = tc.doTransaction(m);

		if (m.getType() != Controller.CONTROLLER_LAUNCH_PROGRAM_OK) {
			throw new Exception("Controller set program failed "+m);			
		}
		
	}
*/
	//
	// Messages coming from Daemon or notifies coming from subcontroller
	//
	class Reader extends Thread {

		MessageReader min;
		ThreadGroup tg;
		
		Reader(MessageReader min) {
			this.min = min;
		}

		Reader(MessageReader min, ThreadGroup tg) {
			super(tg,"JEngine Client [Client Notify Reader] Thread (PORT: unknown)");
			this.min = min;
			this.tg = tg;
		}
	
		public void run() {
			Thread th = Thread.currentThread();
			th.setName("JEngine Client [Client Notify Reader] Thread");
						
			try {
				while (true) {
					
					Message m = min.read();

					int type = m.getType();

					if (type == Controller.CONTROLLER_TRACE) {
//						Debugger.info("CONTROLLER: Trace");
						try {
							Message val = (Message)m.get(0);
							
							tlistener.trace(val);						
							
							synchronized(tracelisteners_LOCK) {
								for (int k = 0; k < tracelisteners.size(); k++) {
									((TraceListener)tracelisteners.get(k)).trace(val);
								}
							}
						} catch (Throwable t) {
							Logger.direct("Error in trace listener - "+t);
						}

					} else if (type == Controller.CONTROLLER_PRINT) {
//						Debugger.info("CONTROLLER: Print");
						
						String val = (String)m.get(0);
						
						tlistener.print(val);						

						synchronized(tracelisteners_LOCK) {
							for (int k = 0; k < tracelisteners.size(); k++) {
								((TraceListener)tracelisteners.get(k)).print(val);
							}
						}
						
					} else if (type == Controller.CONTROLLER_DEBUG) {
//						Debugger.info("CONTROLLER: Debug Print");
						String val = (String)m.get(0);

						tlistener.debug(val);						

						synchronized(tracelisteners_LOCK) {
							for (int k = 0; k < tracelisteners.size(); k++) {
								((TraceListener)tracelisteners.get(k)).debug(val);
							}
						}
						
					} else {
						
						throw new IOException("unrecognised message "+m);
						
					}
					
				}
			} catch (Throwable t) {
				Logger.direct("Notification (Trace+Print) Reader Thread Exited");
				
				ThreadGroup tg = getThreadGroup();
				tg.uncaughtException(this,t);
//				t.printStackTrace();	
			}
		}
	}

	public void notifyRunnerDeath(long id, Runner runner) throws Exception {
		throw new Exception("function not available");
	}

	protected PrintHandler ph;
	
	public void setPrintHandler(PrintHandler ph) {
		this.ph = ph;
	}
	
	public void trace(Message m) throws Exception {
		throw new Exception("function not available");
	}
	public void print(String s) throws Exception {
		throw new Exception("function not available");
	}
	public void debug(String msg) throws Exception {
		throw new Exception("function not available");
	}
	public ArrayList launchRunnerLocal(int count, String method, ArrayList args) throws Exception {
		throw new Exception("function not available");
	}
	public long getClock() {
		return 0;
	}
	public String getClientHost() {
		return "localhost";
	}
	
}