/**********************************************************************
 * Copyright (c) 2005 Scapa Technologies Limited and others
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * Scapa Technologies Limited - Initial API and implementation
 **********************************************************************/

package org.eclipse.stp.b2j.core.xpath.internal;
import java.util.ArrayList;

////////////////////////////////////////////////////////
//  Default Implementation of the XPathVisitor
//
//  The visitor assumes that every result generated
//  is pushed back onto the stack, as a single element,
//  in order for it to be accessible to higher expressions
//  in the grammar. The final 'push' statements of each 
//	method indicate this and should be completed once
//  the required processing is performed.
//
//	The expression that each method processes is 
//  presented as the first comment in the method.
//
//	Scapa Technologies
//	17/03/2004
////////////////////////////////////////////////////////

public class XPathVisitorAdapter extends XPathVisitor
{
	public XPathVisitorAdapter()
	{
		super();
	}

	public XPathVisitorAdapter(boolean debug)
	{
		super(debug);
	}
	
	public void processNumber()
	{
		// Process the top element of the stack as a number
		String num = (String) pop();
		
		if (debug == true)
			System.out.println("PROCESS NUMBER "+num);

		push(new Double(num));
	}
	
	public void processLiteral()
	{
		// Process the top element of the stack as a number
		String lit = (String) pop();
		lit = lit.substring(1,lit.length()-1);
		
		if (debug == true)
			System.out.println("PROCESS LITERAL "+lit);

		push(lit);
	}

	public void processSlash()
	{
		String op = (String) pop();
		//ArrayList list = new ArrayList();
		//push relative reference

		if (debug == true)
			System.out.println("PROCESS SLASH "+op);		
		
		push(op);
	}

	public void processAbsoluteSlash()
	{
		// Set the document root element
		String slash = (String) pop();
		// push absolute reference
		if (debug == true)
			System.out.println("PROCESS ABSOLUTE SLASH "+slash);		
		
		push(slash);		
	}

	public void setContextNode()
	{
		// Push the context node onto the stack
	}

	public void prePredicateEval()
	{
		// Token Builder mimic setup stuff in here
	}

	public void postPredicateEval()
	{
		// Token Builder processing in here
	}

	public void evaluateNodeTypeTest()
	{
		// NodeType '(' ')'
		Object closeb, openb, nodetype;
		
		closeb = pop();
		openb = pop();
		nodetype = pop();
		
		if (debug == true)
			System.out.println("NODE TYPE TEST "+nodetype+"()");
		
		push(""+nodetype+openb+closeb);
	}
	
	public void evaluateAdditiveExpr()
	{
		// MultiplicativeExpr (+ | -) Multiplicative
		Object expression1, expression2, op;
		
		expression2 = pop();
		op = pop();
		expression1 = pop();

		if (debug==true) 
			System.out.println("ADDITIVE EXPR "+expression1 + op + expression2);

		push(""+expression1 + op + expression2);
	}

	public void evaluateMultiplicativeExpr()
	{
		// UnaryExpr (* | DIV | MOD) UnaryExpr

		Object expression1, expression2, op;
		
		expression2 = pop();
		op = pop();
		expression1 = pop();
		
		if (debug==true) 
			System.out.println("MULTIPLICATIVE EXP | "+expression1 + op + expression2);

		push("" + expression1 + op + expression2);
	}

	public void evaluateUnaryExpr()
	{
		// <MINUS> UnaryExpr

		Object expression, minus;
		
		expression = pop();
		minus = pop();

		if (debug==true) 
			System.out.println("UNARY EXPR | "+minus+expression);

		push(""+minus+expression);	
	}

	public void evaluateRelationalExpr()
	{
		// AdditiveExp (LT | GT | LTE | GTE) AdditiveExp

		Object expression1, expression2, op;
		
		expression2 = pop();
		op = pop();
		expression1 = pop();

		if (debug==true) 
			System.out.println("RELATIONAL EXPR | "+expression1 + op + expression2); 

		push(""+expression1+op+expression2);
	}

	public void evaluateEqualityExpr()
	{
		// RelationalExpr (<EQ> | <NEQ>) RelationalExpr

		Object expression1, expression2, op;
		
		expression2 = pop();
		op = pop();
		expression1 = pop();

		if (debug==true) 
			System.out.println("EQUALITY EXPR | "+expression1 + op + expression2);

		push(""+expression1 + op + expression2);
	}

	public void evaluateAndExpr()
	{
		// AndExpr <AND> AndExpr

		Object expression1, expression2, op;
		
		expression2 = pop();
		op = pop();
		expression1 = pop();

		if (debug==true) 
			System.out.println("AND EXPR | "+expression1+op+expression2);

		push(""+expression1+op+expression2);
	}

	public void evaluateOrExpr()
	{
		// EqualityExpr <OR> EqualityExpr

		Object expression1, expression2, op;
		
		expression2 = pop();
		op = pop();
		expression1 = pop();
		
		if (debug==true) 
			System.out.println("OR EXPR | "+expression1+op+expression2);
			
		push(""+expression1+op+expression2);		
	}

	public void evaluateRelativePathPattern()
	{
		// Step Pattern <SLASH|SLASHSLASH> Step Pattern

		Object pattern1, pattern2, op;

		pattern2 = pop();
		op = pop();
		pattern1 = pop();

		if (debug==true) 
			System.out.println("REL. PATH. PAT | " + pattern1 + op + pattern2);

		push(""+pattern1+op+pattern2);		
	}

	public void evaluateRelativeLocationPath()
	{
		// Step (<SLASH> | <SLASHSLASH>) RelativeLocationPath

		Object path1, op, path2;

		path2 = pop();
		op = pop();
		path1 = pop();

		if (debug==true) 
			System.out.println("REL. LOC. PATH | "+path1+op+path2);

		push(""+path1+op+path2);
	}

	public void evaluateStepPredicate()
	{
		// AxisSpecifier NodeTest (Predicate)*
		
		ArrayList predicates = new ArrayList();
		Object closedbracket, openbracket, predicate, node;
		String axis;
		
		if (peek() instanceof String)
		{
			String temp = (String)peek();
	
			while (temp != null && temp instanceof String && temp.equals("]"))
			{
				closedbracket = pop();
				predicate = pop();
				predicates.add(0,predicate);
				openbracket = pop();
				if (peek() instanceof String)
					temp = (String)peek();
				else
					break;
			}			
		}

		node = pop();
		axis = "";
		
		Object temp = peek();
		
		if (temp != null && temp instanceof String)
		{
			axis = (String)temp;
			if (axis.equals("@") || axis.endsWith("::"))
				axis = (String) pop();
			else
				axis = "";
		}

		if (debug == true)
		{
			String s = axis+node;
			for (int i = 0 ; i < predicates.size();i++)
				s = s + "["+predicates.get(i)+"]";
			System.out.println("STEP PRED | "+s);
		}
		
		String s = axis+node;
		for (int i = 0 ; i < predicates.size();i++)
			s = s + "["+predicates.get(i)+"]";
		System.out.println("STEP PRED | "+s);

		push(s);
	}

	public void evaluateQName()
	{
		// NCName ":" NCName
		// The ":" token is NOT added to the stack, as it is assumed implicitly

		Object name2, name1;

		name2 = pop();
		name1 = pop();
		
		if (debug==true) 
			System.out.println("QNAME | "+name1+":"+name2);

		push(""+name1+":"+name2);
	}

	public void evaluateQNameWONodeType()
	{
		// QNameWONodeType ":" QNameWONodeType
		// The ":" token is NOT added to the stack

		Object name1, name2;

		name2 = (String)pop();
		name1 = (String)pop();
		
		if (debug==true) 
			System.out.println("QNAME WO NODE | "+name1+":"+name2);

		// push(QName Without NodeType);
		push(""+name1+":"+name2);		
	}

	public void evaluatePattern()
	{
		// LocationPathPattern <UNION> LocationPathPattern

		Object path1, path2, op;
		
		path2 = pop();
		op = pop();
		path1 = pop();
		
		if (debug==true) 
			System.out.println("PATTERN | "+path1+op+path2);

		// push(Union Pattern);
		push(""+path1+op+path2);		
	}

	public void evaluateLocationPathPattern()
	{
		// <SLASH> LocationPath || <SLASH>

		Object slash, path, temp;
		String tempString;

		temp = peek();
		path = null;
		slash = null;

		if (temp instanceof String)
		{
			tempString = (String) temp;

			if (tempString.equals("/"))
			{
				path = null;
				slash = pop();
			}
			else
			{
				path = pop();
				slash = pop();
			}
		}
		
		if (debug==true) 
		{
			if (path == null)
				System.out.println("LOC. PATH. PATT | "+slash);	
			else
				System.out.println("LOC. PATH. PATT | "+slash+path);
		}
		
		if (path != null)
			push(""+slash+path);
		else
			push(""+slash);
	}

	public void evaluateLocationIDPathPattern()
	{
		// IdKeyPattern ( (<SLASH>|<SLASHSLASH>) RelativPathPattern)?
		
		Object path, op, temp, id;
		String tempString;
		
		path = null;
		op = null;
		temp = peek();

		if (temp instanceof String)
		{
			tempString = (String)temp;
			if (!tempString.equals("//") || !tempString.equals("/"))	// This means that we are reading the end of an ID, and hence no relative path here
			{
				path = pop();
				op = pop();
			}
		}

		// Now lets deal with the ID
		id = pop();

		if (debug==true) 
			System.out.println("LOCATIONIDPATHPATTERN | "+id+" "+op+" "+path);

		push(id+" "+op+" "+path);
	}

	public void evaluateLocationRelPathPattern()
	{
		// "//"? RelativePathPattern

		Object path, op;
		
		path = pop();
		op = pop();
		
		if (debug==true) 
			System.out.println("LOC. REL. PATH PATTERN | "+op+path);

		push(""+op+path);
	}

	public void evaluateSingleIDKeyPattern()
	{
		// ID "(" LITERAL ")"
		Object closebracket, openbracket, literal, id;

		closebracket = pop();
		literal = pop();
		openbracket = pop();
		id = pop();

		if (debug==true) 
			System.out.println("EVALUATESINGLEIDKEYPATTERN | " + id+openbracket+literal+closebracket);

		// push(Single ID Key Pattern);
		push(""+id+openbracket+literal+closebracket);
	}

	public void evaluateDoubleIDKeyPattern()
	{
		// ID "(" LITERAL , LITERAL ")"
		Object closeb, openb, literal1, literal2, id, comma;

		closeb = pop();
		literal2 = pop();
		comma = pop();
		literal1 = pop();
		openb = pop();
		id = pop();
		
		if (debug==true) 
			System.out.println("DOUBLEIDKEYPATTERN | "+id+openb+literal1+comma+literal2+closeb);

		// push(Double ID Key Pattern);
		push(""+id+openb+literal1+comma+literal2+closeb);
	}

	public void evaluateVariableReference()
	{
		// <VARIABLE> QName

		Object qname, var;
		
		qname = pop();
		var = pop();
		
		if (debug==true) 
			System.out.println("VARIABLE REF | "+qname+" "+var);
			
		// push(Variable Reference);
		push(qname+" "+var);
	}

	public void evaluateWildCardQName()
	{
		// QName()
	}

	public void evaluateWildCardStar()
	{
		if (debug==true) 
			System.out.println("*");
			
		push("*");
	}

	public void evaluateWildCardNCName()
	{
		// NCName() : *

		Object name;
		
		name = pop();
		
		if (debug == true)
			System.out.print("WILDCARD NC NAME | "+name+":"+"*");
			
		push(""+name+":"+"*");
	}

	public void evaluatePrimaryFilterExpr()
	{
		// Primary Expr -- a convenience method that does not require implementation for correct parsing

		Object primaryExpression;
		
		primaryExpression = pop();
		
		if (debug==true) 
			System.out.println("PRIMARY EXPR | "+primaryExpression);

		push(primaryExpression);
	}

/*	
	public void evaluatePredicateFilterExpr()
	{
		// Predicate  -- convenience method
		
		Object predicate;

		predicate = pop();
		
		if (debug==true) 
			System.out.println("PREDICATE EXPR | "+predicate);

		// push(Predicate Filter Expression);	DEFAULT : push(predicate);
	}
*/

	public void evaluatePathExpr()
	{
		// Filter Expr (<SLASH> | <SLASHSLASH>) Relative Location Path

		Object relativeLocPath, op, filterExp;
		
		relativeLocPath = pop();
		op = pop();
		filterExp = pop();

		if (debug==true) 
			System.out.println("PATH EXPR | "+filterExp+op+relativeLocPath);
			
		// push(Path Expression);
		push(""+filterExp+op+relativeLocPath);
	}	
	
	public void evaluateUnionExpr()
	{
		// Path Expr <UNION> Path Expr
		
		Object path1, path2, op;

		path2 = pop();
		op = pop();
		path1 = pop();
		
		if (debug==true) 
			System.out.println("UNION EXPR | "+path1 + op + path2);

		push(""+path1+op+path2);
	}

	public void evaluateFunctionCall()
	{
		// FunctionName ( Arg1, arg2, arg3 ... )

		Object closebracket, arg, functionName;
		
		ArrayList arglist = new ArrayList();
		
		String argString;
		
		pop();
		
		closebracket = pop();
		
		arg = pop();
		
		if (arg instanceof String)
			argString = (String) arg;
		else
			argString = null;

		while (argString == null || !argString.equals("("))
		{
			arglist.add(arg);	

			arg = pop();

			if (arg instanceof String)
				argString = (String) arg;
			else
				argString = null;
		}			
		
		functionName = pop();

		if (debug == true)
		{
			String s = functionName+"(";

			for (int i = 0 ; i < arglist.size(); i++)
			{
				if (i == 0)
					s = s + arglist.get(i);
				else
					s = s + ","+arglist.get(i);
			}
			s = s + ")";

			System.out.println("FUNCTION CALL | "+s);
		}
		
		String s = functionName+"(";

		for (int i = 0 ; i < arglist.size(); i++)
		{
			if (i == 0)
				s = s + arglist.get(i);
			else
				s = s + ","+arglist.get(i);
		}
		s = s + ")";

		System.out.println("FUNCTION CALL | "+s);		
		push(s);
	}

	public void evaluateAbsoluteLocationPath()
	{
		// <SLASH> (RelativeLocationPath)?	

		Object path, slash, temp;
		String tempString = null;
		
		temp = peek();
		slash = null;
		path = null;
		
		if (temp instanceof String)
			tempString = (String) temp;
			
		if (tempString != null && tempString.equals("/"))
			slash = pop();
		else
		{
			path = pop();
			slash = pop();
		}
		
		if (debug==true) 
			System.out.println("ABS. LOC. PATH | "+slash+path);

		push(""+slash+path);
	}	

	public void evaluateProcessingInstruction()
	{
		// <PI> "(" <Literal> ")"
		
		Object closedbracket, openbracket, pi, literal;

		closedbracket = pop();
		literal = pop();
		openbracket = pop();
		pi = pop();
		
		if (debug==true) 
			System.out.println("PROCESSING INFO | "+pi+"("+literal+")");
		
		push(""+pi+"("+literal+")");
	}

	public void evaluateAbbreviatedAbsoluteLocationPath()
	{
		// <SLASHSLASH> RelativeLocationPath
		
		Object path;
		Object slashslash;
		
		path = pop();
		slashslash = pop();
		
		if (debug==true) 
			System.out.println("ABBREV. ABS. LOC. PATH | "+"//"+path);

		// push( Absolute Location Path);
		push("//"+path);
	}

	public void evaluateAbbreviatedRelativeLocationPath()
	{
		// RelativeLocationPath <SLASHSLASH> Step

		Object step;
		Object relativeLocPath;
		Object slashslash;
		
		step = pop();
		slashslash = pop();
		relativeLocPath = pop();
		
		if (debug==true) 
			System.out.println("ABBREV. REL. LOC. PATH | "+relativeLocPath+"//"+step);

		// push(Relative Location);
		push(""+relativeLocPath+"//"+step);
	}
	
	public void processAbbreviatedAbsoluteSlash()
	{
		if (debug == true)
			System.out.println("PROCESSING ABBREVIATEDABSOLUTESLASH");
	}
	
	public void processAbbreviatedRelativeSlash()
	{
		if (debug == true)
			System.out.println("PROCESSING ABBREVIATEDRELATIVESLASH");		
	}
	
	public void evaluateNameTest()
	{
		if (debug == true)
			System.out.println("PROCESSING NAMETEST");				
	}
	
	public void evaluateAxisSpecifier()
	{
		if (debug == true)
			System.out.println("PROCESSING AXISSPECIFIER");		
	}
	
}

