/*******************************************************************************
* Copyright (c) 2006 IONA Technologies PLC
* All rights reserved. This program and the accompanying materials
* are made available under the terms of the Eclipse Public License v1.0
* which accompanies this distribution, and is available at
* http://www.eclipse.org/legal/epl-v10.html
* 
* Contributors:
*     IONA Technologies PLC - initial API and implementation
*******************************************************************************/
package org.eclipse.stp.sc.cxf.generators;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.PrintWriter;
import java.net.MalformedURLException;
import java.util.ArrayList;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.Map;

import javax.wsdl.Definition;
import javax.wsdl.Port;
import javax.wsdl.PortType;
import javax.wsdl.Service;
import javax.xml.namespace.QName;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IPath;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.stp.common.logging.LoggingProxy;
import org.eclipse.stp.sc.common.utils.CXFUtil;
import org.eclipse.stp.sc.common.utils.JDTUtils;
import org.eclipse.stp.sc.common.utils.LaunchUtil;
import org.eclipse.stp.sc.cxf.launchers.JavaLaunchConfig;
import org.eclipse.stp.sc.jaxws.runtimeprovider.IWsdlToJavaGenerator;
import org.eclipse.stp.sc.jaxws.workspace.JaxWsWorkspaceManager;
import org.eclipse.stp.sc.jaxws.wsdl.WsdlUtils;

public class CeltixWsdlToJavaGenerator implements IWsdlToJavaGenerator {
    
	private static final String INTERFACE_SUFFIX = "Port";
	
    private static final LoggingProxy LOG = LoggingProxy.getlogger(CeltixWsdlToJavaGenerator.class);
    
    private Hashtable toolParams = new Hashtable();
    
    private String pojoClass = null;
    private String seiClass = null;
    private String portTypeName  = null;
    
    private boolean genServer = false;
    private boolean genClient = false;

    public void setInitializationData(IConfigurationElement config,
                                      String propertyName,
                                      Object data) throws CoreException {
        if (data instanceof Hashtable) {
        	toolParams = (Hashtable)data;
        }
    }
    
    public void run(IPath path, IProject project) throws CoreException {
        String url;
        try {
            url = path.toFile().toURL().toExternalForm();
            run(url, project);
        } catch (MalformedURLException e) {
        	LOG.error(e);
        }
        
    }

    
      public void run(String url, IProject project) throws CoreException {
        try {
        	
        	IJavaProject javaProject = JDTUtils.findJavaProject(project.getName());

        	//program arguments
        	ArrayList<String> args = new ArrayList<String>();

        	//fe
        	args.add(ToolConstants.WSDL2JAVA_PARA_FRONT_END);
            args.add("jaxws");
            
            //data binding
            args.add(ToolConstants.WSDL2JAVA_PARA_DATA_BINDING);
            args.add("jaxb");
            
            //output dir
            String outDir = (String)toolParams.get(IWsdlToJavaGenerator.GEN_OUTPUTDIR);
            LOG.debug("Wsdl to Java output dir: " + outDir);
            File file = new File(outDir);
            if (!file.exists()) {
            	file.mkdirs();
            }
            args.add(ToolConstants.WSDL2JAVA_PARA_OUTPUT_DIR);
            args.add(outDir);
            
            Map specificParams = (Map)toolParams.get(IWsdlToJavaGenerator.TOOL_SPECIFIC_OPS);
            
            String genOps = (String)specificParams.get(ToolConstants.WSDL2JAVA_PARA_GEN_IMPL);
            //if impl
            if (genOps != null) {
                LOG.debug("The impl option has been set");
                args.add(ToolConstants.WSDL2JAVA_PARA_GEN_IMPL);
            }
            genOps = (String)specificParams.get(ToolConstants.WSDL2JAVA_PARA_GEN_SERVER);
            //if server
            if (genOps != null) {
            	genServer = true;
                LOG.debug("The server option has been set");
                if(!args.contains(ToolConstants.WSDL2JAVA_PARA_GEN_IMPL)){
                	args.add(ToolConstants.WSDL2JAVA_PARA_GEN_IMPL);
                }
                args.add(ToolConstants.WSDL2JAVA_PARA_GEN_SERVER);
            }
            genOps = (String)specificParams.get(ToolConstants.WSDL2JAVA_PARA_GEN_CLIENT);
            //if client
            if (genOps != null) {
            	genClient = true;
                LOG.debug("The client option has been set");
                args.add(ToolConstants.WSDL2JAVA_PARA_GEN_CLIENT);
            }

            //binding files
            ArrayList bindingParams = (ArrayList)specificParams.get(CeltixWsdlToJavaParameterPage.KEY_BINDING_FILE_LIST);
            if (bindingParams != null) {
                for (Object bf : bindingParams) {
                    LOG.debug("BINDing files: " + bf);
                }
            }
            
            //exsh
            String soapOps = (String)specificParams.get(ToolConstants.WSDL2JAVA_PARA_EXTEND_SOAP_HEADER);
            if (soapOps != null) {
                LOG.debug("Extra soap header option selected");
                args.add(ToolConstants.WSDL2JAVA_PARA_EXTEND_SOAP_HEADER);
                args.add("true");
            }else{
                args.add(ToolConstants.WSDL2JAVA_PARA_EXTEND_SOAP_HEADER);
                args.add("false");
            }
            //dns
            soapOps = (String)specificParams.get(ToolConstants.WSDL2JAVA_PARA_DEFAULT_NAMESPACE);
            if (soapOps != null) {
                LOG.debug("default namespace option selected");
                args.add(ToolConstants.WSDL2JAVA_PARA_DEFAULT_NAMESPACE);
                args.add("true");                
            }else{
            	//argsMap.put(ToolConstants.PARA_DEFAULT_NAMESPACE, "false");
                args.add(ToolConstants.WSDL2JAVA_PARA_DEFAULT_NAMESPACE);
                args.add("false");  
            }
            
            //dex
            soapOps = (String)specificParams.get(ToolConstants.WSDL2JAVA_PARA_DEFAULT_EXCLUDE_NAMESPACE);
            if (soapOps != null) {
                LOG.debug("default  default excludes option selected");
                args.add(ToolConstants.WSDL2JAVA_PARA_DEFAULT_EXCLUDE_NAMESPACE);
                args.add("true");  
            }else{
                args.add(ToolConstants.WSDL2JAVA_PARA_DEFAULT_EXCLUDE_NAMESPACE);
                args.add("false");  
            }
            
            //ant and verbose
            String miscOps = (String)specificParams.get(ToolConstants.WSDL2JAVA_PARA_GEN_ANT);
            if (miscOps != null) {
                LOG.debug("The ant option has been set");
                args.add(ToolConstants.WSDL2JAVA_PARA_GEN_ANT);
            }
            miscOps = (String)specificParams.get(ToolConstants.WSDL2JAVA_PARA_VERBOSE);
            //verbose
            if (miscOps != null) {
                LOG.debug("The verbose option has been set");
                args.add(ToolConstants.WSDL2JAVA_PARA_VERBOSE);
            }
            
            processWsdlProperty(url, project);
            processLaunchConfig(url, project);
            
            if (pojoClass != null) {
                processPojoCustomize(url, pojoClass, args);
            } else if (seiClass != null) {
            	processSEICustomize(url, seiClass, args);
            }
            
            args.add(url);
            
            LaunchUtil.launchJavaProgram(javaProject, ToolConstants.WSDL2JAVA_GENERATOR_CLASS, null, args.toArray(new String[0]), null);
            project.refreshLocal(IProject.DEPTH_INFINITE, null);
            
        } catch (Exception e) {
            LOG.error("wsdl to java generation error", e);
        }
    }
      
    private void processLaunchConfig(String wsdlName, IProject project) throws Exception {
        Definition wsdlDef = WsdlUtils.readWSDL(wsdlName);
        
        Iterator sItor = wsdlDef.getServices().values().iterator();
        while (sItor.hasNext()) {
        	Service s = (Service)sItor.next();
        	Iterator pItor = s.getPorts().values().iterator();
        	while (pItor.hasNext()) {
        		Port port = (Port)pItor.next();
        		processOnePort(port, project, wsdlName);
        	}
        	
        }
        
    }
        
     
    private void processOnePort(Port port, IProject project, String wsdlName) {
		
    	PortType portType = port.getBinding().getPortType();
		portTypeName = portType.getQName().getLocalPart();
		String packageName = CXFUtil.getPackageName(portType.getQName()
				.getNamespaceURI());
		String seiClsName = packageName + "."
				+ portType.getQName().getLocalPart();

		if (pojoClass != null) {
			packageName = getPackageNameFromCls(pojoClass);
			seiClsName = getSEINameFromPojo(pojoClass);
		} else if (seiClass != null) {
			packageName = getPackageNameFromCls(seiClass);
			seiClsName = seiClass;
		}

		createLaunchConfig(genServer, genClient, getSimpleName(seiClsName), packageName,
				wsdlName, port.getName(), project);
	}
    
      /**
		 * try to read the persistent property from the wsdl file
		 * 
		 * @param wsdlUrl
		 * @param project
		 */
    private void processWsdlProperty(String wsdlUrl, IProject project) throws Exception {
    	pojoClass = null;
    	seiClass = null;
    	String prjLoc = project.getLocation().toOSString();
		if (wsdlUrl.indexOf(prjLoc) < 0) {
			//if here, the wsdl maybe a link to external file
			return;
		}
		String relativePath = wsdlUrl.substring(project.getLocation().toOSString().length(),
				wsdlUrl.length());
		IFile wsdlFile = project.getFile(relativePath);
		
		if (!wsdlFile.exists()) {
			return;
		}

		pojoClass = wsdlFile
				.getPersistentProperty(JaxWsWorkspaceManager.CLASS_PROPERTY);
		seiClass = wsdlFile.getPersistentProperty(JaxWsWorkspaceManager.INTERFACE_PROPERTY);
    }
      
    /**
     * create server and client launch configuration
     * @param env
     * @param portType
     * @param packageName
     * @param wsdlName
     */
    private void createLaunchConfig(boolean genServer, boolean genClient, String seiClassName, String packageName, String wsdlName,
    		String portName, IProject project) {
        if(genServer){    
            // generate a client launcher for each defined PortType
            String args = "\"" + wsdlName + "\"";
            createClientLaunchConfiguration(seiClassName + "_" + portName+ "_Client",
                    packageName, args, project);
        }

        if(genClient){     
            // if generating a complete server stub, make sure implementation
            // class is generated too.
            createServerLaunchConfiguration(seiClassName+ "_" + portName+ "_Server",
                    packageName, null, project);
        }
    }
    
    /**
     * generate customized binding if this wsdl is created from sei
     * in this case, we only need to setup package name mapping if required
     * @param wsdlUrl
     * @param env
     * @param sei
     * @throws Exception
     */
    private void processSEICustomize(String wsdlUrl, String sei, ArrayList<String> argsMap) throws Exception {
    	String pkgMapping = getCustomizePackage(sei, wsdlUrl);
    	if (pkgMapping != null) {
    		argsMap.add(ToolConstants.WSDL2JAVA_PARA_PACKAGE);
    		argsMap.add(pkgMapping);
    	}
        
        //check if sei class is different from porttype
        String seiName = getSimpleName(sei);
        if (!seiName.equals(portTypeName)) {
            String bindingFile = createBindingFile(wsdlUrl, 
				seiName, null);
    		// need to setup customized binding file
			argsMap.add(ToolConstants.WSDL2JAVA_PARA_BINDING_NAME);
			argsMap.add(bindingFile);
        }
    }
    
    private String getSimpleName(String clsName) {
    	if (clsName.indexOf(".") >= 0) {
    		return clsName.substring(clsName.lastIndexOf(".") + 1);
    	}
    	return clsName;
    }
    
    private String getCustomizePackage(String cls, String wsdlUrl) throws Exception {
    	String pkgName = getPackageNameFromCls(cls);
    	String namespace  = getNamespaceFromWsdl(wsdlUrl);
    	String wsdlPkgName = null;
    	if (namespace != null) {
    	    wsdlPkgName = CXFUtil.getPackageName(namespace);
    	}
    	
    	if (pkgName != null && wsdlPkgName != null) {
    		if (!pkgName.equals(wsdlPkgName)) {
    			LOG.debug("sei package is:" + pkgName);
    			LOG.debug("wsdl namespace-> package is:" + wsdlPkgName);
    			String mapping = namespace + "=" + pkgName;
    			LOG.debug("need to set ns->package mapping:" + mapping);
    			return mapping;
    		}
    	}
    	return null;
    }
    
    private String getPackageNameFromCls(String cls) {
    	if (cls.indexOf(".") >= 0) {
			return cls.substring(0, cls.lastIndexOf("."));
		}
    	return null;
    }
   
    
    private String getNamespaceFromWsdl(String wsdlUrl) throws Exception {
    	Definition wsdlDef = WsdlUtils.readWSDL(wsdlUrl);
        Iterator iPortTypes = wsdlDef.getPortTypes().keySet().iterator();

        while (iPortTypes.hasNext()) {
            QName portType = (QName)iPortTypes.next();
            return portType.getNamespaceURI();
        }
    	return null;
    }
    
    
    /**
     * generate customized binding for POJO
     * @param env
     */
    private void processPojoCustomize(String wsdlUrl, String pojo, ArrayList<String> argsMap) {
		try {
			
			String pkgMapping = getCustomizePackage(pojo, wsdlUrl);
	    	if (pkgMapping != null) {
	    		argsMap.add(ToolConstants.WSDL2JAVA_PARA_PACKAGE);
	    		argsMap.add(pkgMapping);
	    	}
			
			String bindingFile = createBindingFile(wsdlUrl, 
					getSEINameFromPojo(pojoClass), null);
			
			// since it is from pojo, we don't need to generte impl
			argsMap.remove(ToolConstants.WSDL2JAVA_PARA_GEN_IMPL);
			
			// need to setup customized binding file
			argsMap.add(ToolConstants.WSDL2JAVA_PARA_BINDING_NAME);
			argsMap.add(bindingFile);
		} catch (Exception e) {
			LOG.error("error during generate customized binding file", e);
		}
	}
      
    public String getSEINameFromPojo(String pojo) {
    	String implName = "impl";
    	if (pojo.endsWith(implName)) {
    		return pojo.substring(0, pojo.length() - implName.length());
    	}
    	return pojo +  INTERFACE_SUFFIX;
    }
    /**
	 * if the wsdl is generated from POJO, we need to create customized binding
	 * to specify the porttype class. Otherwise, the default result will
	 * overwrite POJO class with SEI
	 * also need to specifiy the package name sometime
	 * 
	 * @return String, the bind file path. if return null, no building file has
	 *         been created
	 * @throws CoreException
	 */  
    private String createBindingFile(String wsdlUrl, String portTypeClass, String packageName) {

		try {
			if (portTypeClass != null && portTypeClass.indexOf(".") >= 0) {
			    portTypeClass = getSimpleName(portTypeClass);
			    
			}
			
			String wsdlUri = new File(wsdlUrl).toURI().toString();
			File bindingFile = File.createTempFile("binding", "xml");
			PrintWriter pw = new PrintWriter(bindingFile);

			pw.println("<bindings");
			pw.println("    xmlns:xsd=\"http://www.w3.org/2001/XMLSchema\"");
			pw.println("    xmlns:wsdl=\"http://schemas.xmlsoap.org/wsdl/\"");
			pw.println("    xmlns=\"http://java.sun.com/xml/ns/jaxws\"");
			pw.println("    wsdlLocation=\""
					+ wsdlUri + "\">");
			if (packageName != null) {
				//specify the package name for wsdl2java mapping
		        pw.println("     <package name=\"" + packageName 
		    		+ "\"/>");
			}
			if (portTypeClass != null) {
                //this wsdl is created from pojo class
				//we need create customized binding for port type. 
				//thus the generated sei won't overwrite the pojo class
			    pw.println("     <bindings node=\"wsdl:definitions/wsdl:portType\">");
			    pw.println("         <class name=\"" + portTypeClass+ "\"/>");
			    pw.println("     </bindings>");
			}
			pw.println("</bindings>");
			pw.close();
			bindingFile.deleteOnExit();
			// print the binding file for debug
			BufferedReader br = new BufferedReader(new FileReader(bindingFile));
			String line;
			LOG.debug("customized binding file************");
			while ((line = br.readLine()) != null) {
				LOG.debug(line);
			}
			br.close();
			LOG.debug("customized binding file end*******");
            return bindingFile.toURL().toExternalForm();
		} catch (Exception e) {
			LOG.error("error during create customized binding file", e);
		}

		return null;
	}
      
    /**
	 * This method creates a client launch configuration to launch the produced
	 * client code, if any, from within the eclipse environment
	 */
    private void createClientLaunchConfiguration(String clientName, 
        String packageName, String progArgs, IProject project) {
    	String projectName = project.getName();
        //create the java launch configuration
        JavaLaunchConfig launchConfig = new JavaLaunchConfig();
        launchConfig.setProgramArgs(progArgs);
        launchConfig.setConfigBaseName(clientName);
        launchConfig.setProjectName(projectName);
        launchConfig.setClientMainClassName(packageName + "." + clientName);
        launchConfig.createClientLaunchConfig();
        
    }

    /**
     * This method creates a server launch configuration to launch the produced client
     * code, if any, from within the eclipse environment  
     */
    private void createServerLaunchConfiguration(String serverName, 
        String packageName, String progArgs, IProject project) {
    	String projectName = project.getName();

        //create the java launch configuration
        JavaLaunchConfig launchConfig = new JavaLaunchConfig();
        launchConfig.setProgramArgs(progArgs);
        launchConfig.setConfigBaseName(serverName);
        launchConfig.setProjectName(projectName);
        launchConfig.setServerMainClassName(packageName + "." + serverName);
        launchConfig.createServerLaunchConfig();
    }     
}
