/**********************************************************************
 * Copyright (c) 2005 Scapa Technologies Limited and others
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * Scapa Technologies Limited - Initial API and implementation
 **********************************************************************/

package org.eclipse.stp.b2j.core.publicapi;

import java.io.BufferedInputStream;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;

import org.eclipse.stp.b2j.core.jengine.internal.message.Message;
import org.eclipse.stp.b2j.core.jengine.internal.message.MessageReader;
import org.eclipse.stp.b2j.core.jengine.internal.message.MessageWriter;
import org.eclipse.stp.b2j.core.jengine.internal.utils.FileMD5;
import org.eclipse.stp.b2j.core.misc.internal.HexData;

/**
 * 
 * @author amiguel
 *
 * Represents a single JAR dependency which the engine must include in the program
 * or in remote engines' classloaders
 */
public class JARDependency {
	private byte[] jar_dat;
	private String path;
	private String name;
	private boolean enabled = true;
	
	//transient - does not need to be saved or loaded
	private String[] namespaces = null;
	private String plugin_name = null;
	
	public long size() {
		return jar_dat.length;
	}
	
	private JARDependency() {
	}
	/**
	 * Create a new JAR dependency
	 * @param dat the JAR data as a byte array
	 * @param path the java File path to the source JAR file
	 */
	public JARDependency(byte[] dat, String path) {
		this.jar_dat = dat;
		this.path = path;
		this.name = path.substring( Math.max(path.lastIndexOf('/'),path.lastIndexOf('\\')) + 1 );
	}
	
	/**
	 * Set the file path of this JAR dependency
	 * @param path the java File path to the source JAR file
	 */
	public void setFilePath(String path) {
		this.path = path;
		this.name = path.substring( Math.max(path.lastIndexOf('/'),path.lastIndexOf('\\')) + 1 );
	}
	
	/**
	 * Get the source java File path for this JAR dependency
	 * @return the java File path to the source JAR file
	 */
	public String getFilePath() {
		return path;
	}
	
	/**
	 * Get the name part of the source java File path for this JAR dependency
	 * @return the name of the source JAR file
	 */
	public String getFileName() {
		return name;
	}
	
	/**
	 * Get the JAR data 
	 * @return the source JAR file as a byte array
	 */
	public byte[] getJarData() {
		return jar_dat;
	}
	
	/**
	 * Set the plugin providing this JAR dependency
	 * @param name the name of the plugin providing this JAR dependency
	 */
	public void setPluginName(String name) {
		plugin_name = name;
	}
	
	/**
	 * Get the name of the plugin providing this JAR dependency
	 * @return the name of the plugin providing this JAR dependency
	 */
	public String getPluginName() {
		return plugin_name;
	}
	
	/**
	 * Set any namespaces associated with this JAR dependency
	 * @param namespaces any namespaces for bindings etc which require this JAR
	 */
	public void setAssociatedNamespaces(String[] namespaces) {
		this.namespaces = namespaces;
	}
	
	/**
	 * Get any namespaces associated with this JAR dependency (used to filter out
	 * unused JAR dependencies based on the used namespaces in the BPEL)
	 * @return any associated namespaces which require this JAR
	 */
	public String[] getAssociatedNamespaces() {
		return namespaces;
	}
	
	/**
	 * Set whether this JAR dependency is enabled (not ignored)
	 * @param enabled whether this JAR dependency is enabled (should be included in the dependencies)
	 */
	public void setEnabled(boolean enabled) {
		this.enabled = enabled;
	}
	
	/**
	 * Get whether this JAR dependency is enabled (not ignored)
	 * @return whether this JAR dependency is enabled (should be included in the dependencies)
	 */
	public boolean getEnabled() {
		return enabled;
	}
	
	public String getCacheKey() throws IOException {
		return getJarDataCacheKey(jar_dat);
	}

	//
	// Static utility methods
	//
	
	public static String getJarDataCacheKey(byte[] jar_dat) throws IOException {
		try {
			MessageDigest md = MessageDigest.getInstance("MD5");

			int N = 0;
			
			ZipInputStream zin = new ZipInputStream(new ByteArrayInputStream(jar_dat));
			
			ZipEntry zent = zin.getNextEntry();
			while (zent != null) {
				N++;
				
				//update the md5sum with the name of the entry
				md.update(zent.getName().getBytes("UTF8"));
				
				byte[] buf = new byte[10000];
				int n = 0;
				while (n != -1) {
					n = zin.read(buf,0,10000);
					if (n > 0) {
						md.update(buf,0,n);
					}
				}
				
/*
				long t = zent.getCrc();
				
 				if (zent.getName().indexOf("OrchestrationData") != -1) 
					System.out.println("CRC for "+zent.getName()+" is "+t);
				
				md.update((byte)t);
				md.update((byte)(t>>8));
				md.update((byte)(t>>16));
				md.update((byte)(t>>24));
				md.update((byte)(t>>32));
				md.update((byte)(t>>40));
				md.update((byte)(t>>48));
				md.update((byte)(t>>54));*/
				
//				md.update((byte)N);
//				md.update((byte)(N>>8));
//				md.update((byte)(N>>16));
//				md.update((byte)(N>>24));
				
				zent = zin.getNextEntry();
			}
			
			zin.close();
			
			return jar_dat.length+"_"+FileMD5.md5ToString(md.digest());
		} catch (IOException e) {
			throw e;
		} catch (NoSuchAlgorithmException e) {
			IOException ee = new IOException("Failed to load MD5 algorithm");
			ee.initCause(e);
			throw ee;
		}

//			return jar_dat.length+"_"+FileMD5.md5ToString(FileMD5.getMD5Sum(jar_dat));
	}
	
	/**
	 * Serialise this JAR dependency as a hexadecimal string
	 * @param j the JARDependency to serialise as a string
	 * @return a String hexadecimal representation of this JAR dependency
	 * @throws IOException if an error occurs during the serialisation
	 */
	public static String toHex(JARDependency j) throws IOException {
		ByteArrayOutputStream bout = new ByteArrayOutputStream();
		MessageWriter mwrite = new MessageWriter(bout);
		mwrite.write(toMessage(j));
		return HexData.byteArrayToHexString(bout.toByteArray());
	}
	
	/**
	 * Deserialise this JAR dependency from a hexadecimal string
	 * @param s the hexadecimal string to deserialise into a JAR dependency
	 * @return the JARDependency represented by this hexadecimal string
	 * @throws IOException if an error occurs during the deserialisation
	 */
	public static JARDependency fromHex(String s) throws IOException {
		ByteArrayInputStream bin = new ByteArrayInputStream(HexData.hexStringToByteArray(s));
		MessageReader mread = new MessageReader(bin);
		Message m = mread.read();
		return fromMessage(m);
	}
	
	public static Message toMessage(JARDependency j) {
		Message m = new Message();
		m.append(j.jar_dat);
		m.append(j.path);
		m.append(j.name);
		if (j.enabled) {
			m.append(1);
		} else {
			m.append(0);
		}
		return m;
	}
	
	public static JARDependency fromMessage(Message m) {
		JARDependency j = new JARDependency();
		j.jar_dat = (byte[])m.get(0);
		j.path = (String)m.get(1);
		j.name = (String)m.get(2);
		if (m.length() > 3) {
			j.enabled = ((Integer)m.get(3)).intValue() == 1;
		}
		return j;
	}
	
	public static JARDependency writeTemporaryDependency(JARDependency dep) throws IOException {
		File tmpjar = File.createTempFile("B2jEngineDependency_",".jar");
		FileOutputStream fout = new FileOutputStream(tmpjar);
		
		fout.write(dep.jar_dat);
		fout.flush();
		fout.close();
		
		tmpjar.deleteOnExit();
		
		dep.path = tmpjar.getCanonicalPath();
		
		return dep;
	}
	
}