/**********************************************************************
 * Copyright (c) 2005 Scapa Technologies Limited and others
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * Scapa Technologies Limited - Initial API and implementation
 **********************************************************************/

package org.eclipse.stp.b2j.ui.internal.launchconfig;

import java.util.ArrayList;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.debug.core.ILaunchConfigurationWorkingCopy;
import org.eclipse.debug.ui.AbstractLaunchConfigurationTab;
import org.eclipse.debug.ui.ILaunchConfigurationTab;
import org.eclipse.stp.b2j.ui.B2jImageManager;
import org.eclipse.stp.b2j.ui.UiPlugin;
import org.eclipse.stp.b2j.ui.internal.extensionpoints.BPELProviderLoader;
import org.eclipse.stp.b2j.ui.publicapi.extensionpoints.bpelprovider.BPELProviderCategory;
import org.eclipse.stp.b2j.ui.publicapi.extensionpoints.bpelprovider.BPELProviderNode;
import org.eclipse.stp.b2j.ui.publicapi.extensionpoints.bpelprovider.PropertySource;
import org.eclipse.stp.b2j.ui.publicapi.extensionpoints.bpelprovider.PropertySourceUI;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.CTabFolder;
import org.eclipse.swt.custom.CTabItem;
import org.eclipse.swt.events.KeyEvent;
import org.eclipse.swt.events.KeyListener;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Tree;
import org.eclipse.swt.widgets.TreeItem;
import org.w3c.dom.Document;

public class LaunchConfigProviderTab extends AbstractLaunchConfigurationTab implements SelectionListener, ILaunchConfigurationTab, KeyListener, PropertySource {

	private static final String SELECTED_PROVIDER_ID = "selected_provider_id";

	private static final int TABCOUNT = 1;
	
CTabFolder tabfolder;

CTabItem provider_tab;

	Tree provider_tree;
	
	String message;
	String error_message;

ArrayList allproviders = new ArrayList();
	
BPELProviderNode current = null;
PropertySourceUI current_ui = null;
CTabItem[] current_tabs = new CTabItem[0];

LaunchConfigPropertySource propsource = new LaunchConfigPropertySource();

	private void selectProvider(String id) {
		for (int i = 0; i < allproviders.size(); i++) {
			TreeItem item = (TreeItem)allproviders.get(i);
			BPELProviderNode provnode = (BPELProviderNode)item.getData();
			
			if (provnode.getId().equals(id)) {
				provider_tree.setSelection(new TreeItem[]{item});
	
				setCurrentProvider(provnode,false,true);
			}
		}
	}
	
	public void checkForErrors() {
		error_message = null;
	}

	public boolean isValid(ILaunchConfiguration launchConfig) {
		return error_message == null;
	}
	
	public String getErrorMessage() {
		return error_message;
	}
	
	public String getMessage() {
		return message;
	}
	
	public void setDirty(boolean b) {
		super.setDirty(b);
		checkForErrors();
		updateLaunchConfigurationDialog();
	}
	
	private void buildTree(BPELProviderCategory cat, TreeItem parent) {
		TreeItem item;
		if (parent == null) {
			item = new TreeItem(provider_tree,SWT.NONE|SWT.SINGLE);
		} else {
			item = new TreeItem(parent,SWT.NONE);
		}
		
		item.setText(cat.getName());
		if (cat.getImage() != null) {
			item.setImage(cat.getImage());
		} else {
//TODO default image?
//			item.setImage(B2jPlugin.imageManager.getImage(B2jImageManager.IMG_SERVICES));
		}
		item.setData(cat);
		
		for (int i = 0; i < cat.getProviderCount(); i++) {
			BPELProviderNode prov = cat.getProvider(i);
			
			TreeItem provitem = new TreeItem(item,SWT.NONE);
			provitem.setText(prov.getName());
			if (prov.getImage() != null) {
				provitem.setImage(prov.getImage());
			} else {
//				TODO default image?
//				provitem.setImage(B2jPlugin.imageManager.getImage(B2jImageManager.IMG_SERVICEFILE));
			}
			provitem.setData(prov);
			
			allproviders.add(provitem);
		}
		
		for (int i = 0; i < cat.getCategoryCount(); i++) {
			buildTree(cat.getCategory(i),item);
		}
	}
	
	public void createControl(Composite parent) {
		
		tabfolder = new CTabFolder(parent,SWT.BOTTOM);
		
		provider_tree = new Tree(tabfolder,SWT.BORDER);

		provider_tab = new CTabItem(tabfolder,SWT.NONE);
		provider_tab.setControl(provider_tree);
		provider_tab.setText(UiPlugin.getString("PROVIDER_SUBTAB_NAME"));
		
		try {
			BPELProviderCategory cat = BPELProviderLoader.getProviders();
			System.out.println(cat.getTreeString());
			
			for (int i = 0; i < cat.getCategoryCount(); i++) {
				buildTree(cat.getCategory(i),null);
			}
			
		} catch (Exception e) {
			UiPlugin.DBG.logVisibleError(e,"Failed to load BPEL provider",false);
			e.printStackTrace();
		}
		
		tabfolder.setSelection(provider_tab);
		
//		selectProvider(B2jPlugin.EXTENSION_BPELPROVIDER_WORKSPACEFILE);
		
		//
		// event handling
		//
		
		provider_tree.addSelectionListener(this);
	}
	
	public Control getControl() {
		return tabfolder;
	}

	public void setDefaults(ILaunchConfigurationWorkingCopy configuration) {
		configuration.setAttribute(SELECTED_PROVIDER_ID,UiPlugin.EXTENSION_BPELPROVIDER_WORKSPACEFILE);
	}

	public void initializeFrom(ILaunchConfiguration configuration) {
		try {
			propsource.initializeFrom(configuration);

			String id = configuration.getAttribute(SELECTED_PROVIDER_ID,UiPlugin.EXTENSION_BPELPROVIDER_WORKSPACEFILE);
			
//System.out.println("INITIALIZING - SELECT PROVIER "+id+" ("+current_ui+")");			
			selectProvider(id);
			
		} catch (CoreException e) {
			e.printStackTrace();
		}
		checkForErrors();
	}

	public void performApply(ILaunchConfigurationWorkingCopy configuration) {
		if (current != null) {
			if (current_ui != null) {
				current_ui.saveToPropertySource();
			}
			
			configuration.setAttribute(SELECTED_PROVIDER_ID,current.getId());
			
			propsource.performApply(configuration);
		}
	}

	public void activated(ILaunchConfigurationWorkingCopy workingCopy) {
		initializeFrom(workingCopy);
	}
	
	public void deactivated(ILaunchConfigurationWorkingCopy workingCopy) {
		performApply(workingCopy);
	}
	
	public String getName() {
		return UiPlugin.getString("PROVIDER_TAB_NAME");
	}

	public Image getImage() {
		return UiPlugin.imageManager.getImage(B2jImageManager.IMG_RUN);
	}
	
	public void dispose() {
		try {
			tabfolder.dispose();
		} catch (Exception e) {
		}
	}
	
	public void setCurrentProvider(BPELProviderNode bpnode, boolean save, boolean force) {
		System.out.println("SET CURRENT PROVIDER: "+bpnode.getId());
		
		if (force || bpnode != current) {
			
			if (save && current_ui != null) {
				try {
					current_ui.saveToPropertySource();
				} catch (Exception e) {
				}
			}
			
			current = bpnode;

			int prev_sel_index = tabfolder.getSelectionIndex();
			
			//remove current provider tabs
			for (int i = 0; i < current_tabs.length; i++) {
				current_tabs[i].getControl().dispose();
				current_tabs[i].dispose();
			}
			
			//set up new provider tabs
			current_ui = bpnode.getProvider().getPropertySourceUI();
			current_ui.setPropertySource(this);
			
			current_tabs = new CTabItem[current_ui.getTabCount()];
			
			for (int i = 0; i < current_tabs.length; i++) {
				Composite panel = new Composite(tabfolder,SWT.NONE);
				current_ui.createTab(i,panel);
				
				current_tabs[i] = new CTabItem(tabfolder,SWT.NONE);
				current_tabs[i].setText(current_ui.getTabName(i));
				current_tabs[i].setControl(panel);
			}

			if (prev_sel_index != -1 && prev_sel_index < current_tabs.length+TABCOUNT) {
				tabfolder.setSelection(prev_sel_index);
			} else {
				tabfolder.setSelection(provider_tab);
			}
			current_ui.loadFromPropertySource();
			
			tabfolder.update();
			tabfolder.redraw();
			
			setDirty(true);
		}
	}

	//
	// Event handlers
	//
	
	public void keyPressed(KeyEvent e) {
		setDirty(true);
	}

	public void keyReleased(KeyEvent e) {
		setDirty(true);
	}

	public void widgetSelected(SelectionEvent e) {
		Object o = e.getSource();
		
		if (o == provider_tree) {
			TreeItem[] items = provider_tree.getSelection();
			if (items.length == 1) {
				TreeItem item = items[0]; 
				
				Object dat = item.getData();
				if (dat instanceof BPELProviderNode) {
					setCurrentProvider((BPELProviderNode)item.getData(),true,false);
				}
			}
		}
		
		checkForErrors();
	}

	public void widgetDefaultSelected(SelectionEvent e) {
		widgetSelected(e);
	}

	public Document createProperty(String doctype) {
		return propsource.createProperty(doctype);
	}

	public Document getProperty(String name) {
		return propsource.getProperty(name);
	}

	public void setProperty(String name, Document doc) {
		propsource.setProperty(name,doc);
	}
}