/**********************************************************************
 * Copyright (c) 2005 Scapa Technologies Limited and others
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * Scapa Technologies Limited - Initial API and implementation
 **********************************************************************/

package org.eclipse.stp.b2j.ui.internal.launchconfig;

import java.util.ArrayList;
import java.util.HashMap;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.debug.core.ILaunchConfigurationWorkingCopy;
import org.eclipse.debug.ui.AbstractLaunchConfigurationTab;
import org.eclipse.debug.ui.ILaunchConfigurationTab;
import org.eclipse.jface.dialogs.IInputValidator;
import org.eclipse.jface.viewers.CellEditor;
import org.eclipse.jface.viewers.CheckboxCellEditor;
import org.eclipse.jface.viewers.DialogCellEditor;
import org.eclipse.jface.viewers.ICellModifier;
import org.eclipse.jface.viewers.ILabelProviderListener;
import org.eclipse.jface.viewers.IStructuredContentProvider;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.ITableLabelProvider;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.TextCellEditor;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.stp.b2j.core.publicapi.B2jPlatform;
import org.eclipse.stp.b2j.core.publicapi.JARDependency;
import org.eclipse.stp.b2j.core.publicapi.extension.sessiontransport.SessionTransportProvider;
import org.eclipse.stp.b2j.ui.B2jImageManager;
import org.eclipse.stp.b2j.ui.UiPlugin;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.KeyEvent;
import org.eclipse.swt.events.KeyListener;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.FileDialog;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.swt.widgets.TableItem;

public class LaunchConfigDependenciesTab extends AbstractLaunchConfigurationTab implements ILaunchConfigurationTab, SelectionListener, KeyListener, IStructuredContentProvider, ITableLabelProvider, ICellModifier  {

	//
	// Should have only session level options.   The transport is just something the user selects
	//
	
	public static final int COL_JAR = 0;
	public static final int COL_ENABLED = 1;
	public static final int COL_PATH = 2;

	static Object[][] columns = new Object[][] {
		new Object[] {UiPlugin.getString("COL_JAR"), new Integer(SWT.LEFT), new Integer(200)},
		new Object[] {UiPlugin.getString("COL_ENABLED"), new Integer(SWT.CENTER), new Integer(90)},
		new Object[] {UiPlugin.getString("COL_PATH"), new Integer(SWT.CENTER), new Integer(250)},
	};
	
	static String[] properties = new String[] {
			(String)columns[COL_JAR][0],
			(String)columns[COL_ENABLED][0],
			(String)columns[COL_PATH][0],
	};
		
	Composite main_panel;
	
	Label hosts_label;
	//List hosts_list;
	TableViewer hosts_viewer;
	
	
	Button hosts_add;
	Button hosts_remove;
	
	String message;
	String error_message;
	
	JARDependency[] deps = new JARDependency[0];
	
	SessionTransportProvider[] providers;

	public void checkForErrors() {
		error_message = null;
	}

	public boolean isValid(ILaunchConfiguration launchConfig) {
		return error_message == null;
	}
	
	public String getErrorMessage() {
		return error_message;
	}
	
	public String getMessage() {
		return message;
	}
	
	public void setDirty(boolean b) {
		super.setDirty(b);
		checkForErrors();
		updateLaunchConfigurationDialog();
	}
	
	public void createControl(Composite parent) {
		
		main_panel = new Composite(parent,0);
		main_panel.setLayout(new GridLayout(4,false));
		
		hosts_label = new Label(main_panel,SWT.NONE);
		hosts_label.setText(UiPlugin.getString("DEPENDENCY_LIST"));
		hosts_label.setLayoutData(new GridData(GridData.BEGINNING,GridData.END,false,false,4,1));
		
		hosts_viewer = new TableViewer(new Table(main_panel, SWT.FULL_SELECTION | SWT.SINGLE | SWT.HIDE_SELECTION));
		hosts_viewer.getTable().setLayoutData(new GridData(GridData.FILL,GridData.FILL,true,true,3,4));
		
		hosts_add = new Button(main_panel,SWT.NONE);
		hosts_add.setText(UiPlugin.getString("ADD"));
		hosts_add.setLayoutData(new GridData(GridData.FILL,GridData.FILL,false,false,1,1));
		
		hosts_remove = new Button(main_panel,SWT.NONE);
		hosts_remove.setText(UiPlugin.getString("REMOVE"));
		hosts_remove.setLayoutData(new GridData(GridData.FILL,GridData.FILL,false,false,1,1));
		
		//table stuff
		hosts_viewer.getTable().setHeaderVisible(true); 
		hosts_viewer.getTable().setLinesVisible(true);
		
		hosts_viewer.setContentProvider(this);
		hosts_viewer.setLabelProvider(this);

		for (int i = 0; i < columns.length; i++) {
			String col_name = (String)columns[i][0];
			int col_hints = ((Integer)columns[i][1]).intValue();
			int col_width = ((Integer)columns[i][2]).intValue();
			
			TableColumn tc = new TableColumn(hosts_viewer.getTable(),col_hints);
			tc.setText(col_name);
			tc.setWidth(col_width);
		}
		
		CellEditor[] editors = new CellEditor[] {
				new DialogCellEditor(hosts_viewer.getTable()) {
					public Object openDialogBox(Control cellEditorWindow) {
						FileDialog dialog = new FileDialog(cellEditorWindow.getShell(),SWT.OPEN);
						String path = dialog.open();
						return path;
					}
				}
				,
				new CheckboxCellEditor(hosts_viewer.getTable()),
				new TextCellEditor(hosts_viewer.getTable()),
		};
		
		
		hosts_viewer.setCellEditors(editors);
		hosts_viewer.setCellModifier(this);
		hosts_viewer.setColumnProperties(properties);
		
		//
		// Event handling
		//
		hosts_add.addSelectionListener(this);
		hosts_remove.addSelectionListener(this);
	}
	
	public Control getControl() {
		return main_panel;
	}

	public void setDefaults(ILaunchConfigurationWorkingCopy configuration) {
		configuration.setAttribute("jar_list",new ArrayList());
	}

	public static String getExtensionDependencyKey(JARDependency dep) {
		return dep.getPluginName()+":"+dep.getFilePath();
	}
	
	public void initializeFrom(ILaunchConfiguration configuration) {
		try {
			java.util.List jar_refs = configuration.getAttribute("jar_list",new ArrayList());
			
			HashMap uniquenessMap = new HashMap();
			
			//load all standard dependencies
			ArrayList list = new ArrayList();
			JARDependency[] exts = B2jPlatform.toDependencyJARs(B2jPlatform.getAllDependencyInfo());

			for (int i = 0; i < exts.length; i++) {
				
				if (uniquenessMap.get(getExtensionDependencyKey(exts[i])) == null) {
				
					//load whether each imported jar dependency is enabled or not
					boolean enabled = configuration.getAttribute("jar_enabled_"+getExtensionDependencyKey(exts[i]),true);
					exts[i].setEnabled(enabled);
					
					list.add(exts[i]);
					
					uniquenessMap.put(getExtensionDependencyKey(exts[i]),new Object());
				}
			}
			
			//load all user dependencies
			for (int i = 0; i < jar_refs.size(); i++) {
				JARDependency dep = JARDependency.fromHex((String)jar_refs.get(i));
				if (uniquenessMap.get(getExtensionDependencyKey(dep)) == null) {
					list.add(dep);
					uniquenessMap.put(getExtensionDependencyKey(dep),new Object());
				}
			}
			
			JARDependency[] items = new JARDependency[list.size()];
			list.toArray(items);
			
			deps = items;
			
			hosts_viewer.setInput(deps);
			
		} catch (CoreException e) {
			 e.printStackTrace();
		} catch (Exception e) {
			 e.printStackTrace();
		}
		checkForErrors();
	}

	public void performApply(ILaunchConfigurationWorkingCopy configuration) {

		ArrayList list = new ArrayList();
		
		for (int i = 0; i < deps.length; i++) {
			try {
				if (deps[i].getAssociatedNamespaces() == null) {
					list.add(JARDependency.toHex(deps[i]));
				} else {
					configuration.setAttribute("jar_enabled_"+getExtensionDependencyKey(deps[i]),deps[i].getEnabled());
				}
			} catch (Exception e) {
				e.printStackTrace();
			}

			if (deps[i].getAssociatedNamespaces() == null) {
				configuration.setAttribute("jar_list_enabled_"+deps[i].getFilePath(),list);
			} else {
				configuration.setAttribute("jar_list_enabled_"+deps[i].getPluginName()+":"+deps[i].getFilePath(),list);
			}
		}
		
		configuration.setAttribute("jar_list",list);
	}

	public void activated(ILaunchConfigurationWorkingCopy workingCopy) {
		initializeFrom(workingCopy);
	}
	
	public void deactivated(ILaunchConfigurationWorkingCopy workingCopy) {
		performApply(workingCopy);
	}
	
	public String getName() {
		return UiPlugin.getString("DEPENDENCIES_TAB_NAME");
	}

	public Image getImage() {
		return UiPlugin.imageManager.getImage(B2jImageManager.IMG_JAR);
	}
	
	public void dispose() {
		try {
			main_panel.dispose();
		} catch (Exception e) {
		}
	}

	//
	// Event handlers
	//
	
	private void addJAR(String jarpath) {
		JARDependency[] tmp = new JARDependency[deps.length+1];
		System.arraycopy(deps,0,tmp,0,deps.length);
		tmp[tmp.length-1] = new JARDependency(new byte[0],jarpath);
		deps = tmp;
	}
	
	public void keyPressed(KeyEvent e) {
		setDirty(true);
	}

	public void keyReleased(KeyEvent e) {
		setDirty(true);
	}

	public void widgetSelected(SelectionEvent e) {
		Object o = e.getSource();
		if (o == hosts_add) {
			
			FileDialog dialog = new FileDialog(getShell(),SWT.OPEN);
			String path = dialog.open();
			
			addJAR(path);
			
			hosts_viewer.setInput(deps);
			
			setDirty(true);

		} else if (o == hosts_remove) {
			
			ArrayList list = new ArrayList();
			
			IStructuredSelection selection = (IStructuredSelection)hosts_viewer.getSelection();
			Object[] removals = selection.toArray();
			
			for (int k = 0; k < deps.length; k++) {
				boolean remove = false;
				for (int i = 0; i < removals.length; i++) {
					if (deps[k] == removals[i] && deps[k].getAssociatedNamespaces() == null) {
						//selected for removal
						remove = true;
						break;
					}
				}
				if (!remove) {
					list.add(deps[k]);
				}
			}
			
			JARDependency[] tmp = new JARDependency[list.size()];
			list.toArray(tmp);
			
			if (deps.length != tmp.length) {
				setDirty(true);
			}
			
			deps = tmp;

			hosts_viewer.setInput(deps);

		}
		
		checkForErrors();
	}

	public void widgetDefaultSelected(SelectionEvent e) {
		widgetSelected(e);
	}
	
	class IgnoredInputValidator implements IInputValidator {

		public String isValid(String newText) {
			return null;
		}
	
	}
	
	//
	// Table content provider
	//

	public void inputChanged(Viewer currviewer, Object oldInput, Object newInput) {
	}
	
	
	public Object[] getElements(Object newInput) {
		return (JARDependency[])newInput;
	}
	
	//
	// Label provider
	//
	
	public Image getColumnImage(Object element, int columnIndex) {
		JARDependency address = (JARDependency)element;

		if (columnIndex == COL_JAR) {
			if (address.getAssociatedNamespaces() != null) {
				return UiPlugin.imageManager.getImage(B2jImageManager.IMG_JAR_IMPORT);
			} else {
				return UiPlugin.imageManager.getImage(B2jImageManager.IMG_JAR);
			}
		}
		return null;
	}
	public String getColumnText(Object element, int columnIndex) {
		JARDependency address = (JARDependency)element;
		
		if (columnIndex == COL_JAR) {
			return address.getFileName();
		} else if (columnIndex == COL_ENABLED) {
			if (address.getEnabled()) {
				return UiPlugin.getString("ENABLED");
			} else {
				return UiPlugin.getString("DISABLED");
			}
		} else if (columnIndex == COL_PATH) {
			if (address.getAssociatedNamespaces() == null) {
				return address.getFilePath();
			} else {
				return UiPlugin.getString("PLUGIN_DIR")+address.getFilePath();
			}
		}
		return "unknown";
	}
	
	public void addListener(ILabelProviderListener listener) {
	}
	public boolean isLabelProperty(Object element, String property) {
		return false;
	}
	public void removeListener(ILabelProviderListener listener) {
	}

	//
	// Cell editor
	//
	
	private int getIndexForProperty(String property) {
		for (int i = 0; i < properties.length; i++) {
			if (property.equals(properties[i])) {
				return i;
			}
		}
		return -1;
	}
	
	public boolean canModify(Object element, String property) {
		JARDependency address = (JARDependency)element;
		int index = getIndexForProperty(property);
		
		if (index == COL_ENABLED) {
			return true;
		} else if (index == COL_JAR) {
			if (address.getAssociatedNamespaces() == null) {
				return true;
			}
		}
		
		return false;
	}

	public Object getValue(Object element, String property) {
		JARDependency address = (JARDependency)element;
		int index = getIndexForProperty(property);
		
		if (index == COL_JAR) {
			return address.getFilePath();
		} else if (index == COL_ENABLED) {
			return new Boolean(address.getEnabled());
		}
		return null;
	}

	public void modify(Object element, String property, Object value) {
		if (value == null) return;
		
		TableItem item = (TableItem)element;
		JARDependency address = (JARDependency)item.getData();
		int index = getIndexForProperty(property);
		if (index == COL_JAR) {
			address.setFilePath(""+value);
			setDirty(true);
			hosts_viewer.setInput(deps);
		} else if (index == COL_ENABLED) {
			address.setEnabled(((Boolean)value).booleanValue());
			setDirty(true);
			hosts_viewer.setInput(deps);
		}
	}
}