/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.stp.core.saf;

import org.eclipse.stp.core.saf.handler.IComponentHandler;
import org.eclipse.stp.core.saf.handler.IEntryPointHandler;
import org.eclipse.stp.core.saf.handler.IExternalServiceHandler;
import org.eclipse.stp.core.saf.handler.IInterfaceHandler;
import org.eclipse.stp.core.sca.Component;
import org.eclipse.stp.core.sca.EntryPoint;
import org.eclipse.stp.core.sca.ExternalService;
import org.eclipse.stp.core.sca.Interface;

/**
 * This interface allows interaction with the SOA Assembly Framework. It
 * is implemented by a singleton class that can be accessed by the
 * {@link org.eclipse.stp.core.saf.ISAFManager#INSTANCE INSTANCE}
 * element. This interface is not intended to be implemented by client tools.
 */
public interface ISAFManager {

	/**
	 * Use this interface to access the {@link ISAFManager} singleton
	 * implementation.
	 */
	public ISAFManager INSTANCE = org.eclipse.stp.core.internal.saf.impl.SAFManager
			.getInstance();

	/**
	 * Returns an array of {@link IComponentHandler} for all supported component
	 * kinds.
	 * 
	 * @param context
	 *            the object that can be used to
	 *            filter the resulting set of component handlers 
	 *            to show only the ones that are valid for the given context
	 * @return an array of component handlers
	 */
	public IComponentHandler[] getAllComponentHandlers(Context context);

	/**
	 * Returns an array of {@link IExternalServiceHandler} for all supported
	 * external service kinds.
	 * 
	 * @param context
	 *            the object that can be used to
	 *            filter the resulting set of external service handlers
	 *            to show only the ones that are valid for the given context
	 * @return an array of external service handlers
	 */
	public IExternalServiceHandler[] getAllExternalServiceHandlers(Context context);

	/**
	 * Returns an array of {@link IEntryPointHandler} for all supported entry
	 * point kinds.
	 * 
	 * @param context
	 *            the object that can be used to
	 *            filter the resulting set of entry point handlers to show
	 *            only the ones that are valid for the given context
	 * @return an array of entry point handlers
	 */
	public IEntryPointHandler[] getAllEntryPointHandlers(Context context);

	/**
	 * Returns an array of {@link IInterfaceHandler} for all available interface
	 * kinds.
	 * 
	 * @return an array of interface handlers
	 */
	public IInterfaceHandler[] getAllInterfaceHandlers();

	// =========
	// Component
	// =========

	/**
	 * This method calls the
	 * {@link org.eclipse.stp.core.saf.handler.IComponentHandler#canCreateComponentFor(Object) canCreateComponentFor()}
	 * method of all component handlers, passing the implementation model
	 * object. Each of the handlers that returns <code>true</code> is returned
	 * in the resulting array. Each handler in the array represents
	 * a kind of component that can be created for the specified implementation
	 * model object. An empty array is returned if no matching handler
	 * is found.
	 * 
	 * @param implementationObj
	 *            an object that could provide the implementation for a
	 *            component
	 * @param context
	 *            the object that can be used to
	 *            filter the resulting set of kind component handlers to show
	 *            only the ones that are valid for the given context
	 * @return an array of component handlers
	 */
	public IComponentHandler[] getComponentHandlersFor(Context context, Object implementationObj);

	/**
	 * Returns the QName string for the default component implementation-neutral
	 * type.
	 * 
	 * @return QName string for the appropriate component type
	 */
	public String getDefaultComponentKind();

	/**
	 * Returns the {@link IComponentHandler} for the kind of the specified
	 * component or returns <code>null</code> if the component has no kind. The handler provides
	 * information about the component kind that could be presented to an end
	 * user. If no component handler supports the component kind,
	 * <code>null</code> is returned.
	 * 
	 * @param theComponent
	 *            the component whose handler is returned
	 * @return the handler for the specified component
	 */
	public IComponentHandler getHandlerFor(Component theComponent);

	/**
	 * Returns the {@link IInterfaceHandler} for the kind of the specified
	 * interface or returns <code>null</code> if the interface has no kind. The handler provides
	 * information about the interface kind that could be presented to an end
	 * user. If no interface handler supports the interface kind,
	 * <code>null</code> is returned.
	 * 
	 * @param theInterface
	 *            the interface whose handler is returned
	 * @return the handler for the specified interface
	 */
	public IInterfaceHandler getHandlerFor(Interface theInterface);

	// ========================
	// Component Implementation
	// ========================

	/**
	 * This method calls the
	 * {@link IExternalServiceHandler#canCreateExternalServiceFor(Object)}
	 * method of all external service handlers, passing the service model
	 * object. Each of the handlers that returns <code>true</code> is returned
	 * in the resulting array. Each handler in the array
	 * represents a kind of external service binding that can be created for the
	 * specified service model object. Each handler provides information about
	 * the external service kind that could be presented to an end user. 
	 * An empty array is returned if no matching handler is found.
	 * 
	 * @param context
	 *            the object that can be used to
	 *            filter the resulting set of handlers, depending on which ones
	 *            are valid for the given context
	 * @param service
	 *            the object representing a service to be imported
	 * @return an array of external service handlers
	 */
	public IExternalServiceHandler[] getExternalServiceHandlersFor(Context context, Object service);

	/**
	 * Returns a {@link IExternalServiceHandler} for the external service kind
	 * of the specified external service or returns <code>null</code> if the external service has no
	 * binding. The handler provides information about the binding kind that
	 * could be presented to an end user. If no external service handler
	 * supports the external service kind, <code>null</code> is returned.
	 * 
	 * @param theExternalService
	 *            the external service whose handler is returned
	 * @return an array of handlers for the specified external service
	 */
	public IExternalServiceHandler[] getHandlersFor(ExternalService theExternalService);

	/**
	 * This method calls the
	 * {@link IEntryPointHandler#canCreateEntryPointFor(Object)} method of all
	 * external service handlers, passing the service model object. Each of the
	 * handlers that returns <code>true</code> is returned in the resulting
	 * array. Each handler in the array represents
	 * a kind of entry point binding that can be created for the specified
	 * service model object. Each handler provides information about the entry
	 * point kind that could be presented to an end user. 
	 * An empty array is returned if no matching handler is found.
	 * 
	 * @param context
	 *            the object that can be used to
	 *            filter the resulting set of handlers, depending on which ones
	 *            are valid for the given context
	 * @param service
	 *            the object representing a service to be exposed
	 * @return an array of entry point handlers
	 */
	public IEntryPointHandler[] getEntryPointHandlersFor(Context context, Object service);

	/**
	 * Returns a {@link IEntryPointHandler} for the entry point kind of the
	 * specified entry point or returns <code>null</code> if the entry point has no binding. The
	 * handler provides information about the binding kind that could be
	 * presented to an end user. If no entry point handler supports the entry
	 * point kind, <code>null</code> is returned.
	 * 
	 * @param theEntryPoint
	 *            the entry point whose handler is returned
	 * @return an array of handlers for the specified entry point
	 */
	public IEntryPointHandler[] getHandlersFor(EntryPoint theEntryPoint);

	/**
	 * This method calls the
	 * {@link IInterfaceHandler#canCreateInterfaceFor(Object)} method of all
	 * interface handlers, passing the object. Each of the handlers that returns
	 * <code>true</code> is returned in the resulting array. 
	 * Each handler in the array represents
	 * a kind of interface that can be created for the specified model object.
	 * Each handler provides information about the interface kind that could be
	 * presented to an end user. An empty array is
	 * returned if no matching handler is found.
	 * 
	 * @param interfaceObj
	 *            the object representing the interface
	 * @return an array of interface handlers for the specified object
	 */
	public IInterfaceHandler[] getInterfaceHandlersFor(Object interfaceObj);

	/**
	 * Flag representing an interface set. This value is provided for use with
	 * {@link #isInterfaceKindSupportedForComponent(String,String,boolean)
	 * isInterfaceKindSupportedForComponent()}.
	 */
	public static final boolean SERVICE = true;

	/**
	 * Flag representing a reference set. This value is provided for use with
	 * {@link #isInterfaceKindSupportedForComponent(String,String,boolean)
	 * isInterfaceKindSupportedForComponent()}.
	 */
	public static final boolean REFERENCE = false;

}
