/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.stp.core.saf.handler;

import java.net.URL;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.stp.core.saf.IUIContext;
import org.eclipse.stp.core.saf.exception.InvalidOperationException;
import org.eclipse.stp.core.saf.exception.SAFException;
import org.eclipse.stp.core.sca.Interface;

/**
 * This interface must be implemented by all interface handlers.
 */
public interface IInterfaceHandler extends IHandler {

	/**
	 * The {@link IInterfaceLabelProvider} maps an interface object to an
	 * optional image and optional text string used to display the element in
	 * the viewer's control.
	 */
	public interface IInterfaceLabelProvider {

		/**
		 * Returns the name for the label of the given interface object.
		 * 
		 * @param interfaceObject
		 *            the interface object whose label is required
		 * @return interface name
		 */
		public String getName(Object interfaceObject);

		/**
		 * Returns the name of the qualifier for the label of the given
		 * interface object.
		 * 
		 * @param interfaceObject
		 *            the interface object whose label is required
		 * @return qualifier name
		 */
		public String getQualifier(Object interfaceObject);

		/**
		 * Returns the {@link URL} for the label of the given interface object.
		 * 
		 * @param interfaceObject
		 *            the interface object whose label is required
		 * @return interface image {@link URL}
		 */
		public URL getIconURL(Object interfaceObject);

		/**
		 * Returns the file containing the given interface object.
		 * 
		 * @param interfaceObject
		 *            the interface object whose file is required
		 * @return the new interface
		 */
		public IFile getFile(Object interfaceObject);

	}
	
	/**
	 * Queries if an interface of the kind supported by this handler can be
	 * created for the specified interface object. If this method returns
	 * <code>true</code>, the handler's
	 * {@link #createInterfaceFor(Object) createInterfaceFor()} method may be
	 * called to create an interface for the file.
	 * <p>
	 * This method should be quick to execute as it may be called to populate a
	 * context menu or a user dialog. When interfaceObj is a container, a
	 * quick check should be performed to verify that an SCA interface can be
	 * created from one of the objects contained by interfaceObj. For example,
	 * if interfaceObj is a file, perhaps the file extension can be checked
	 * rather than opening the file to inspect its contents. If this method
	 * returns <code>true</code> for an invalid container, the
	 * {@link #createInterfaceFor(Object, IUIContext) } can throw an
	 * {@link InvalidOperationException} to let the caller know that an
	 * interface could not be created from it.
	 * 
	 * @param interfaceObj
	 *            an object representing the interface
	 * @return <code>true</code> if the handler can create an
	 *         interface from the specified object;
	 *         <code>false</code> otherwise
	 */
	public boolean canCreateInterfaceFor(Object interfaceObj);

	/**
	 * Queries if this handler can create an interface object for the supported
	 * type. If this method returns <code>true</code>, the handler's
	 * {@link #createInterfaceObject(IProject, IUIContext)} method may be called
	 * to create an interface object.
	 * 
	 * @return <code>true</code> if the handler can create an
	 *         interface object; <code>false</code> otherwise
	 */
	public boolean canCreateInterfaceObject();

	/**
	 * Creates an SCA interface from the given interface object.
	 * <p>
	 * The interface object can be an object supported by the
	 * {@link #canCreateInterfaceFor(Object) } or returned by the
	 * {@link #getInterfaceObjects(IResource) } method.
	 * 
	 * @param interfaceObj
	 *            the object representing the interface
	 * @param callback
	 *            callback object that the handler can use to obtain additional
	 *            information needed to create the interface object. For
	 *            example, what object can be used when creating the SCA interface if
	 *            interfaceObj is a container, such as a file.
	 * @return the new interface
	 * @exception SAFException
	 *                An unexpected exception occurred.
	 * @exception InvalidOperationException
	 *                An interface cannot be created from the given interface
	 *                object.
	 */
	public Interface createInterfaceFor(Object interfaceObj, IUIContext callback)
			throws SAFException, InvalidOperationException;
	
	/**
	 * Creates a new interface object to be used by an SCA interface. This
	 * method takes an optional interface name that the handler can choose to
	 * use if appropriate. Also, the IUIContext is optionally used by this
	 * method to display the corresponding wizard used to create the new
	 * interface object.
	 * 
	 * @param defaultLocation
	 *            the default project to use when creating the interface
	 * @param inerfaceName
	 *            an optional name to be used by this handler when creating the
	 *            interface object
	 * @param callback
	 *            callback object that the handler can use to obtain additional
	 *            information needed to create the interface object
	 * @return the new interface object
	 * @exception SAFException
	 *                An unexpected exception occurred.
	 * @exception InvalidOperationException
	 *                The interface object cannot be created.
	 * @exception InterruptedException
	 *                The caller did not provide a callback object needed by the
	 *                handler, or the callback request was canceled or aborted.
	 */
	public Object createInterfaceObject(IProject defaultLocation, String interfaceName,
			IUIContext callback)
			throws SAFException, InvalidOperationException, InterruptedException;

	/**
	 * Returns an array of {@link Object} representing all interfaces found
	 * within the given search scope. The handler determines how it will use the
	 * search scope. Depending on the interface type being sought, the handler
	 * may look at a single project or folder or look in the workspace or in a
	 * project's classpath.
	 * <p>
	 * The objects returned by this method must be supported as the argument to
	 * the {@link #canCreateInterfaceFor(Object) } method.
	 * 
	 * @param searchScope
	 *            the {@link IResource} used to determine the search scope used
	 *            by this handler
	 * @return an array of {@link Object}
	 * @exception SAFException
	 *                The search cannot be completed.
	 */
	public Object[] getInterfaceObjects(IResource searchScope) throws SAFException;

	/**
	 * This method opens a registered editor for the given interface type.
	 * 
	 * @param interfaceObj
	 *            the interface to open
	 * @throws SAFException
	 *                An unexpected exception occurred.
	 * @exception InvalidOperationException
	 *                The interface cannot be opened.
	 */
	public void open(Interface interfaceObj)
			throws SAFException, InvalidOperationException;
	
	/**
	 * This method returns the handler's instance of
	 * {@link IInterfaceLabelProvider}. The provider is reponsible for
	 * generating label information for objects returned by
	 * {@link #getInterfaceObjects(IResource) }.
	 * 
	 * @return the {@link IInterfaceLabelProvider} instance
	 */
	public IInterfaceLabelProvider getInterfaceLabelProvider();

}
